/****************************************************************************
    $Id: vpseq.cpp 501.0 1995/03/07 12:26:28 RON Exp $

    Copyright (c) 1993-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLVPSeq.

    $Log: vpseq.cpp $
    Revision 501.0  1995/03/07 12:26:28  RON
    Updated for TLX 5.01
    Revision 1.8  1995/01/31 16:31:50  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.7  1995/01/06  15:58:53  ron
    Corrected Revision keyword

    Revision 1.6  1994/11/16  15:46:02  ron
    Added module info; rearranged #include directives

    Revision 1.5  1994/10/05  18:46:42  ron
    Formatting changes

    Revision 1.4  1994/09/28  14:47:45  ron
    Removed Macintosh-style #include references

    Revision 1.3  1994/09/27  20:23:30  ron
    Changed path separator from / to \

    Revision 1.2  1994/09/26  15:51:33  ron
    Renamed SetSize() to Resize()

    Revision 1.1  1994/08/16  18:13:23  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <string.h>		// For memcpy(), memmove()
#include <tlx\501\except.h>		// Exception handling
#include <tlx\501\vparrays.h>	// Class declaration

/*-------------------------------------------------------------------------*/
    TLVPSeq::TLVPSeq(
    	size_t 		aSize, 	// Initial capacity
    	size_t 		aDelta	// Initial expansion factor
    )

/*  Constructor, creates a sequence of specific initial capacity and
    expansion factor. Also doubles as default constructor, creating
    and empty and nonexpandable sequence.
---------------------------------------------------------------------------*/
: mVector(aSize), mCount(0), mDelta(aDelta),
  mCompare(TLVPVector::DefaultCompare)
{
    TLX_ASSERT(Size() == aSize);
}

/*-------------------------------------------------------------------------*/
    TLVPSeq::TLVPSeq(void *aPtr)

/*  Constructor, creating a single element sequence that is not expandable.
---------------------------------------------------------------------------*/
: mVector(aPtr), mCount(1), mDelta(0), mCompare(TLVPVector::DefaultCompare)
{
    TLX_ASSERT(Size() == 1);
    TLX_ASSERT(Size() == Count());
    TLX_ASSERT(mVector.PeekAt(0) == aPtr);
    mCount = Size();		// Security for our nondebugging version
}

/*-------------------------------------------------------------------------*/
    TLVPSeq::TLVPSeq(void **aVector, size_t aSize)

/*  Constructor creating a sequence that contains a copy of a C-style vector.
    The sequence is not expandable.
---------------------------------------------------------------------------*/
: mVector(aVector, aSize), mCount(aSize), mDelta(0),
  mCompare(TLVPVector::DefaultCompare)
{
    TLX_ASSERT(Size() == aSize);
    TLX_ASSERT(Size() == Count());
    mCount = Size();		// Security for our nondebugging version
}

/*-------------------------------------------------------------------------*/
    TLVPSeq::TLVPSeq(const TLVPSeq &aSeq)

/*  Copy constructor. Creates a copy of another sequence.
---------------------------------------------------------------------------*/
: mVector(aSeq.mVector), mCount(aSeq.mCount), mDelta(aSeq.mDelta),
  mCompare(aSeq.mCompare)
{
    TLX_ASSERT(Size() == aSeq.Size());
    TLX_ASSERT(Count() == aSeq.Count());
    TLX_ASSERT(Delta() == aSeq.Delta());
}

/*-------------------------------------------------------------------------*/
    TLVPSeq::~TLVPSeq()

/*  Destructor. Not really necessary (the compiler generated version
    would work as well because of the mVector destructor), but useful
    to do a few extra checks and to set breakpoints at.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(Count() <= Size());
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Append(void *aPtr)

/*  Adds a single element at the end of the sequence. If necessary and
    allowed, the sequence is expanded.
---------------------------------------------------------------------------*/
{
    InsertAt(Maxi() + 1, aPtr);
    TLX_ASSERT(PeekLast() == aPtr);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Append(void **aVector, size_t aSize)

/*  Adds a C-style vector at the end of the sequence. If necessary and
    allowed, the sequence is expanded. It is an error to add the current
    sequence to itself.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aVector != mVector.StorageVector());
    InsertAt(Maxi() + 1, aVector, aSize);
    TLX_ASSERT(aSize == 0 || PeekLast() == aVector[aSize - 1]);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Append(const TLVPSeq &aSeq)

/*  Adds another sequence at the end of the sequence. If necessary and
    allowed, the sequence is expanded. It is an error to add the current
    sequence to itself.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(this != &aSeq);
    InsertAt(Maxi() + 1, aSeq);
    TLX_ASSERT(PeekLast() == aSeq.PeekLast());
}

/*-------------------------------------------------------------------------*/
    size_t TLVPSeq::Available() const

/*  Returns the amount of free space in the sequence, i.e. the difference
    between the physical and the logical length.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(Count() <= Size());
    return Size() - Count();
}

/*-------------------------------------------------------------------------*/
    bool TLVPSeq::Contains(const void *aPtr) const

/*  Checks for the presence of an element, returning nonzero if it appears
    in the sequence.
---------------------------------------------------------------------------*/
{
    return IsValidIndex(IndexOf(aPtr));
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Remove(void *aPtr)

/*  Removes the indicated pointer from the sequence, and deletes the
    asociated object if we are the owner.
---------------------------------------------------------------------------*/
{
    void *vptr = Extract(aPtr);
    if (IsOwner()) DeletePtr(vptr);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::RemoveAll()

/*  Removes all elements from the sequence and set the contents of the
    sequence to 0. If we are owner of the pointed to objects, they are
    delete during the operation.
---------------------------------------------------------------------------*/
{
    // Removal comes down to setting the count to 0 and clearing the vector.
    mVector.RemoveAll();
    mCount = 0;
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::RemoveAt(index_t aIndex, size_t aCount)

/*  Removes a number of elements from the vector without returning them.
    It is not an error to remove more elements than there are left in the
    sequence; only the remainder is removed in that case.

    If we are the owner of the pointed to objects, they are deleted.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));
    while (aCount-- > 0 && IsValidIndex(aIndex)) {
	void *vptr = Extract(aIndex);
	if (IsOwner()) DeletePtr(vptr);
    }
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::RemoveFirst(size_t aLength)

/*  Removes the first 'aLength' elements from the vector without returning
    them. It is not an error to remove more elements than there are left in
    the sequence; only the remainder is removed in that case.

    If we are the owner of the pointed to objects, they are deleted.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    RemoveAt(Mini(), aLength);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::RemoveLast(size_t aLength)

/*  Removes the last 'aLength' elements from the vector without returning
    them. It is not an error to remove more elements than there are left in
    the sequence; only the remainder is removed in that case.

    If we are the owner of the pointed to objects, they are deleted.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    if (aLength > Count())
	aLength = Count();
    RemoveAt(Maxi() - aLength + 1, aLength);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::Extract(void *aPtr)

/*  Removes a single value from the sequence and returns that value.
---------------------------------------------------------------------------*/
{
    index_t i = IndexOf(aPtr);
    if (!IsValidIndex(i))
	THROW(TLXNotFound(LOCUS));
    return Extract(i);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::Extract(index_t aIndex)

/*  Removes a single value from the sequence and returns that value.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));
    void *vPtr = PeekAt(aIndex);

    // Close the gap that is left by the element just removed
    if (aIndex < Maxi())
	memmove(&PeekAt(aIndex), &PeekAt(aIndex + 1),
		(Maxi() - aIndex) * sizeof(void *));

    // Zero out the position that has been vacated
    PeekAt(Maxi()) = 0;
    TLX_ASSERT(Count() > 0);
    mCount--;
    return vPtr;
}

/*-------------------------------------------------------------------------*/
    TLVPSeq TLVPSeq::Extract(index_t aIndex, size_t aCount)

/*  Removes a number of pointers from the sequence and returns them
    in another sequence. If the remaining length of the current sequence
    is less than the requested number, only the remaining elements are
    removed. This is not considered an error.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));

    // Adjust number of items to remove, if necessary
    if (aCount > (tSize)(Maxi() - aIndex + 1))
    	aCount = Maxi() - aIndex + 1;

    // Create a copy of the subsequence, then remove the elements
    TLVPSeq vCopy(&PeekAt(aIndex), aCount);
    TLX_ASSERT(aCount > 0);
    TLX_ASSERT(aCount <= Count());

    // Close the gap that is left by the elements to be removed
    if (aIndex < Maxi())
	memmove(&PeekAt(aIndex), &PeekAt(aIndex + aCount),
		(Maxi() - aIndex - aCount + 1) * sizeof(void *));

    // Zero out the positions that have been vacated
    memset(&PeekAt(Maxi() - aCount + 1), 0, aCount * sizeof(void *));
    mCount -= aCount;
    return vCopy;
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::ExtractFirst()

/*  Extracts and returns the first element of the sequence.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    return Extract(Mini());
}

/*-------------------------------------------------------------------------*/
    TLVPSeq TLVPSeq::ExtractFirst(size_t aLength)

/*  Removes a number of pointers from the sequence and return them in
    another sequence. If the remaining length of the current sequence
    is less than the requested number, only the remaining elements are
    removed. This is not considered an error.
---------------------------------------------------------------------------*/
{
    return Extract(Mini(), aLength);
}

/*-------------------------------------------------------------------------*/
    TLVPSeq TLVPSeq::ExtractLast(size_t aLength)

/*  Removes a number of pointers from the sequence and returns them
    in another sequence. If the remaining length of the current sequence
    is less than the requested number, only the remaining elements are
    removed. This is not considered an error.
---------------------------------------------------------------------------*/
{
    if (aLength > Count())
	aLength = Count();
    return Extract(Maxi() - aLength + 1, aLength);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::ExtractLast()

/*  Extracts and returns the last element of the sequence.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    return Extract(Maxi());
}

/*-------------------------------------------------------------------------*/
    index_t TLVPSeq::IndexOf(const void *aPtr) const

/*  Finds the position of a pointer in the sequence. The function performs
    a linear search, and only finds the first occurrence of the pointer
    value (if any). It returns the index of the pointer, or Maxi() + 1
    if the pointer does not appear in the sequence.
---------------------------------------------------------------------------*/
{
    return mVector.IndexOf(aPtr) + 1;
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Insert(void *aPtr)

/*  Inserts a new pointer value into the (sorted) sequence at the correct
    position according to the comparison function.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aPtr);
    index_t aIndex;
    Search(aPtr, aIndex);
    InsertAt(aIndex, aPtr);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::InsertAt(index_t aIndex, void *aPtr)

/*  Inserts a single pointer at the indicated position. The previous
    element at that position and all subsequent ones are moved up
    one position. If necessary, the physical vector is resized.

    Legal indices for the insertion point range from Mini() to Maxi() + 1;
    in the latter case, the new element is added to the end of the sequence.
---------------------------------------------------------------------------*/
{
    InsertAt(aIndex, &aPtr, 1);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::InsertAt(index_t aIndex, void **aVector, size_t aSize)

/*  Inserts a C-style vector at the indicated position. The previous
    element at that position and all subsequent ones are moved up.
    If necessary, the physical vector is resized.

    Legal indices for the insertion point range from Mini() to Maxi() + 1;
    in the latter case, the new elements are added to the end of the
    sequence.
---------------------------------------------------------------------------*/
{
    if (aIndex < Mini() || aIndex > Maxi() + 1)
	THROW(TLXIndex(LOCUS, aIndex));

    // Check if we have enough room; if not, expand
    if (Available() < aSize) {
	if (IsFrozen())
	    THROW(TLXFull(LOCUS));

	// Increment the size in multiples of mDelta
	mVector.ExpandBy(((aSize - Available()) % mDelta + 1) * mDelta);
    }
    TLX_ASSERT(Available() >= aSize);

    // Open up the gap for the new elements
    if (aIndex <= Maxi())
	memmove(&mVector.PeekAt(aIndex + aSize - 1),
		&mVector.PeekAt(aIndex - 1),
	    	(Maxi() - aIndex + 1) * sizeof(void *));

    // Copy the new elements to their positions
    memcpy(&mVector.PeekAt(aIndex - 1), aVector, aSize * sizeof(void *));
    mCount += aSize;
    TLX_ASSERT(Count() <= Size());
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::InsertAt(index_t aIndex, const TLVPSeq &aSeq)

/*  Inserts another sequence at the indicated position. The previous
    element at that position and all subsequent ones are moved up.
    If necessary, the physical vector is resized.

    It is an error to insert the current sequence in itself.

    Legal indices for the insertion point range from Mini() to Maxi() + 1;
    in the latter case, the sequence is added to the end of the current
    sequence.
---------------------------------------------------------------------------*/
{
    if (this != &aSeq)
    	InsertAt(aIndex, aSeq.mVector.StorageVector(), aSeq.Count());
}

/*-------------------------------------------------------------------------*/
    bool TLVPSeq::IsValidIndex(index_t aIndex) const

/*  Tests whether a particular index is valid for the sequence in its
    current configuration, returning nonzero if it is.
---------------------------------------------------------------------------*/
{
    return (aIndex >= Mini() && aIndex <= Maxi());
}

/*-------------------------------------------------------------------------*/
    void *&TLVPSeq::operator [](index_t aIndex)

/*  Returns a reference to the aIndex-th element in the sequence. The
    index is range checked, and an TLXIndex exception is thrown if it is
    out of range.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));
    return PeekAt(aIndex);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::operator [](index_t aIndex) const

/*  Returns the value of the aIndex-th element in the sequence. The
    index is range checked, and an TLXIndex exception is thrown if it is
    out of range.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));
    return PeekAt(aIndex);
}

/*-------------------------------------------------------------------------*/
    TLVPSeq &TLVPSeq::operator =(const TLVPSeq &aSeq)

/*  Overloading of assignment operator for class TLVPSeq. The overloading
    is not really necessary (the compiler generated version would work as
    well), but is useful for extra tests and breakpoints.
---------------------------------------------------------------------------*/
{
    if (this != &aSeq) {
	mVector = aSeq.mVector;
	mCount  = aSeq.mCount;
	mDelta  = aSeq.mDelta;
    }
    return *this;
}

/*-------------------------------------------------------------------------*/
    TLVPSeq &TLVPSeq::operator =(void *aPtr)

/*  Overloading of assignment operator that allows a single pointer to be
    assigned to the sequence. The sequence consists of a single element
    afterwards.

    NOTE: If the current sequence has currently a physical size of zero
    and is not allowed to expand, this function fails.
---------------------------------------------------------------------------*/
{
    RemoveAll();
    Append(aPtr);
    TLX_ASSERT(Size() == 1);
    TLX_ASSERT(PeekAt(1) == aPtr);
    return *this;
}

/*-------------------------------------------------------------------------*/
    void *&TLVPSeq::PeekAt(index_t aIndex)

/*  Returns a reference to the aIndex-th pointer in the sequence. There is
    no range checking.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aIndex >= Mini());
    TLX_ASSERT(aIndex <= Maxi());
    return mVector.PeekAt(aIndex - 1);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::PeekAt(index_t aIndex) const

/*  Returns the aIndex-th pointer value in the sequence. There is no range
    checking.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aIndex >= Mini());
    TLX_ASSERT(aIndex <= Maxi());
    return mVector.PeekAt(aIndex - 1);
}

/*-------------------------------------------------------------------------*/
    void *&TLVPSeq::PeekFirst()

/*  Returns a reference to the first element in the sequence. The sequence
    must be non empty.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    return PeekAt(1);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::PeekFirst() const

/*  Returns the value of the first element in the sequence. The sequence
    must be non empty.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    return PeekAt(1);
}

/*-------------------------------------------------------------------------*/
    void *&TLVPSeq::PeekLast()

/*  Returns a reference to the last element in the sequence. The sequence
    must be non empty.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    return PeekAt(Maxi());
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::PeekLast() const

/*  Returns the value of the last element in the sequence. The sequence
    must be non empty.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));
    return PeekAt(Maxi());
}

/*-------------------------------------------------------------------------*/
    void *&TLVPSeq::PeekReverse(index_t aIndex)

/*  Returns a reference to the pointer at the aIndex-th item from the *end*
    of the sequence.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aIndex >= Mini());
    TLX_ASSERT(aIndex <= Maxi());

    // Convert to normal index value
    aIndex = Maxi() - aIndex + 1;
    return PeekAt(aIndex);
}

/*-------------------------------------------------------------------------*/
    void *TLVPSeq::PeekReverse(index_t aIndex) const

/*  Returns the value of the element at the aIndex-th item from the *end*
    of the sequence.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aIndex >= Mini());
    TLX_ASSERT(aIndex <= Maxi());

    // Convert to normal index value
    aIndex = Maxi() - aIndex + 1;
    return PeekAt(aIndex);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Prepend(void *aPtr)

/*  Adds a single element at the start of the sequence. If necessary and
    allowed, the sequence is expanded.
---------------------------------------------------------------------------*/
{
    InsertAt(Mini(), aPtr);
    TLX_ASSERT(PeekFirst() == aPtr);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Prepend(void **aVector, size_t aSize)

/*  Adds a C-style vector at the start of the sequence. If necessary and
    allowed, the sequence is expanded. It is an error to add the current
    sequence to itself.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aVector != mVector.StorageVector());
    InsertAt(Mini(), aVector, aSize);
    TLX_ASSERT(aSize == 0 || PeekFirst() == aVector[0]);
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Prepend(const TLVPSeq &aSeq)

/*  Adds another sequence at the start of the sequence. If necessary and
    allowed, the sequence is expanded. It is an error to add the current
    sequence to itself.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(this != &aSeq);
    InsertAt(Mini(), aSeq);
    TLX_ASSERT(PeekFirst() == aSeq.PeekFirst());
}

/*-------------------------------------------------------------------------*/
    index_t TLVPSeq::QPart(index_t low, index_t up)

/*  Partitions a subarray of the sequence for quick-sort sorting.
---------------------------------------------------------------------------*/
{
    void *pivot = PeekAt(low);
    index_t i 	= low - 1;		// Rover in lower part
    index_t j 	= up + 1;		// Rover in upper part

    TLX_ASSERT_PTR(mCompare);
    for (;;) {
	// Find subsequence in upper part
	do j--; while ((*mCompare)(PeekAt(j), pivot) > 0);

	// Find subsequence in lower part
	do i++; while ((*mCompare)(PeekAt(i), pivot) < 0);

	// Swap elements, or return partition position
	if (i < j) {
	    void *vp = PeekAt(i);
	    PeekAt(i) = PeekAt(j);
	    PeekAt(j) = vp;
	} else
	    return j;		// Done
    }
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::QSort(index_t low, index_t up)

/*  Performs Quicksort on a partition of the sequence.
---------------------------------------------------------------------------*/
{
    if (low >= up) return;		// Empty or single-item partition
    index_t mid = QPart(low, up);	// Make partitions
    QSort(low, mid);			// Sort lower part
    QSort(mid + 1, up);			// Sort upper part
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Replace(index_t aIndex, void *aPtr)

/*  Replaces a single element by another. The length of the sequence does
    not change.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));
    PeekAt(aIndex) = aPtr;
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Replace(index_t aIndex, void **aVector, size_t aSize)

/*  Replaces a part of the sequence by a C-style vector. The length of the
    sequence does not change. If the C-style vector is longer than the
    remaining positions in the sequence from aIndex onwards, only the
    remainder is replaced.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));
    if (aSize > (tSize)(Maxi() - aIndex))
	aSize = Maxi() - aIndex;
    memmove(&PeekAt(aIndex), aVector, aSize * sizeof(void *));
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Replace(index_t aIndex, const TLVPSeq &aSeq)

/*  Replaces a part or whole of the current sequence by the contents
    of another sequence. The length of the current sequence does not
    change. The length of the other sequence must be less than or equal
    to the (logical) length of the part that is replaced.

    It is an error to replace a sequence by (a part of) itself.
---------------------------------------------------------------------------*/
{
    if (this != &aSeq)
    	Replace(aIndex, aSeq.mVector.StorageVector(), aSeq.Count());
}

/*-------------------------------------------------------------------------*/
    bool TLVPSeq::Search(const void *aPtr, index_t &aIndex) const

/*  Searches the sequence for a given pointer value. The search is
    performed with the aid of the comparison function.

    The function performs a binary search which assumes that the
    sequence is proprly sorted according to the comparison function.

    If the element is not found, the index argument is set to the index
    value where the element should be inserted if the sequence were
    to remain sorted.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mCompare);
    TLX_ASSERT_PTR(aPtr);

    // Initialize invariant:
    //
    //  low <= pos(t) <= up...
    index_t low = 1;
    index_t up  = mCount;
    aIndex     = (low + up) / 2;

    while (low <= up) {
	int result = (*mCompare)(aPtr, PeekAt(aIndex));

	if (result < 0)			// 't' less than _vect[i]
	    up = aIndex - 1;
	else if (result == 0)		// Found a match
	    return true;
	else				// 't' greater than _vect[i]
	    low = aIndex + 1;

	aIndex = (low + up) / 2;	// Adjust position
    }

    aIndex++;		// VPSeq::InsertAt() inserts BEFORE position 'i'
    return false;
}

/*-------------------------------------------------------------------------*/
    tCompareFunc TLVPSeq::SetCompare(tCompareFunc aFunc)

/*  Sets a new comparison function and return a pointer to the previous one.
    If 0 is passed for the comparison function, the default function is
    restored.
---------------------------------------------------------------------------*/
{
    tCompareFunc oldF = mCompare;
    mCompare = aFunc ? aFunc : TLVPVector::DefaultCompare;
    return oldF;
}

/*-------------------------------------------------------------------------*/
    size_t TLVPSeq::SetDelta(size_t aDelta)

/*  Change the expansion factor of the sequence. Returns the original factor.
---------------------------------------------------------------------------*/
{
    size_t oldDelta = mDelta;
    mDelta = aDelta;
    return oldDelta;
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Resize(size_t aSize)

/*  Changes the physical size of the sequence. The new size must be able
    to accomodate all current elements; if necessary, the requested size
    is adjusted.
---------------------------------------------------------------------------*/
{
    mVector.Resize(tlMax(aSize, Count()));
}

/*-------------------------------------------------------------------------*/
    void TLVPSeq::Sort()

/*  Performs Quicksort on the sequence.
---------------------------------------------------------------------------*/
{
    QSort(Mini(), Maxi());
}

