/****************************************************************************
    $Id: vparray.cpp 501.0 1995/03/07 12:26:26 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLVPArray. Class TLVPArray is largely a wrapper
    around its TLVPVector data member 'mVector', who provides most of the
    services. TLVPArray adds an index base.

    $Log: vparray.cpp $
    Revision 501.0  1995/03/07 12:26:26  RON
    Updated for TLX 5.01
    Revision 1.7  1995/01/31 16:30:32  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.6  1995/01/06  15:58:49  ron
    Corrected Revision keyword

    Revision 1.5  1994/11/16  15:45:49  ron
    Added module info; rearranged #include directives

    Revision 1.4  1994/09/28  14:47:26  ron
    Removed Macintosh-style #include references

    Revision 1.3  1994/09/27  20:23:25  ron
    Changed path separator from / to \

    Revision 1.2  1994/09/26  15:50:55  ron
    Renamed SetSize() to Resize()

    Revision 1.1  1994/08/16  18:13:21  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <tlx\501\except.h>		// Exception handling
#include <tlx\501\vparrays.h>	// Class declaration

/*-------------------------------------------------------------------------*/
    TLVPArray::TLVPArray(size_t aSize)

/*  Constructor that creates 0-based array of specified size. Also doubles
    as default constructor, building an empty array.
---------------------------------------------------------------------------*/
: mVector(aSize), mBase(0)
{
    TLX_ASSERT(Size() == aSize);
}

/*-------------------------------------------------------------------------*/
    TLVPArray::TLVPArray(index_t aFrom, index_t aTo)

/*  Constructor creating an array with specified lower and upper bounds.
---------------------------------------------------------------------------*/
: mVector(aTo - aFrom + 1), mBase(aFrom)
{
    // TLVector has been constructed already, but we check a few assumptions
    // here anyway. (We chose not to check them on beforehand so as save on
    // runtime. After all, we can trust the programmer, can't we...?)

    TLX_ASSERT(aFrom <= aTo);
    TLX_ASSERT((uint32)Size() == (uint32)((long)aTo - (long)aFrom + 1L));
}

/*-------------------------------------------------------------------------*/
    TLVPArray::TLVPArray(void *aPtr)

/*  Constructor creating a 0-based, single element array with just one
    pointer in it.
---------------------------------------------------------------------------*/
: mVector(aPtr), mBase(0)
{
    TLX_ASSERT(Size() == 1);
}

/*-------------------------------------------------------------------------*/
    TLVPArray::TLVPArray(void **aVector, size_t aSize)

/*  Constructor creating a 0-based array that is a copy of the C-style
    vector of the given size.
---------------------------------------------------------------------------*/
: mVector(aVector, aSize), mBase(0)
{
    TLX_ASSERT(Size() == aSize);
}

/*-------------------------------------------------------------------------*/
    TLVPArray::TLVPArray(const TLVPArray &aVector)

/*  Copy constructor. Not really necessary (compiler generated version
    would work equally well, since mVector's copy constructor does the
    right thing), but can be used for extra checks and to set breakpoints.
---------------------------------------------------------------------------*/
: mVector(aVector.mVector), mBase(aVector.mBase)
{
    TLX_ASSERT(Size() == aVector.Size());
    TLX_ASSERT(Mini() == aVector.Mini());
}

/*-------------------------------------------------------------------------*/
    TLVPArray::~TLVPArray()

/*  Destructor. Not really necessary (compiler generated version would work
    equally well, since mVector's destructor does the right thing), but can
    be used for extra checks and for breakpoints.
*/
{
}

/*-------------------------------------------------------------------------*/
    void TLVPArray::ExpandBy(size_t aDelta)

/*  Increases the size of the array by a given amount.
---------------------------------------------------------------------------*/
{
    // Make sure that all elements remain accessible. The index type
    // is an integer, so we can use INT_MAX to determine its range.

    TLX_ASSERT(aDelta <= ((uint32)INT_MAX - (uint32)Size()));
    mVector.ExpandBy(aDelta);
}

/*-------------------------------------------------------------------------*/
    index_t TLVPArray::Maxi() const

/*  Returns the maximum valid index for the array. If the array is empty,
    it returns Mini() - 1.
---------------------------------------------------------------------------*/
{
    // If the array is empty and the index base is INT_MIN, we get in trouble
    // because Mini() - 1 will then wrap around to INT_MAX on most
    // platforms.

    TLX_ASSERT(Size() > 0 || Mini() > INT_MIN);
    return Mini() + Size() - 1;
}

/*-------------------------------------------------------------------------*/
    index_t TLVPArray::IndexOf(const void *aPtr) const

/*  Finds the array index of a specified value. The function performs a
    linear search of pointer values. If the pointer does not appear in the
    array, Maxi() + 1 is returned.
---------------------------------------------------------------------------*/
{
    return mVector.IndexOf(aPtr) + mBase;
}

/*-------------------------------------------------------------------------*/
    bool TLVPArray::IsValidIndex(index_t aIndex) const

/*  Tests the validity of a given index.
---------------------------------------------------------------------------*/
{
    return aIndex >= Mini() && aIndex <= Maxi();
}

/*-------------------------------------------------------------------------*/
    TLVPArray &TLVPArray::operator =(const TLVPArray &aVector)

/*  Overloaded assignment operator for class TLVPArray. Not really
    necessary (compiler generated version would work as well, because
    mVector's assignment operator does the right thing), but useful
    for extra checks and breakpoints.
---------------------------------------------------------------------------*/
{
    if (this != &aVector)
    {
	mVector = aVector.mVector;
	mBase = aVector.mBase;

	TLX_ASSERT(Size() == aVector.Size());
    }
    return *this;
}

/*-------------------------------------------------------------------------*/
    TLVPArray &TLVPArray::operator =(void *aPtr)

/*  Overloading of the assignment operator for class TLVPArray that allows
    assignment of a single (void *) to the array. The index base is left
    unchanged.
---------------------------------------------------------------------------*/
{
    mVector = aPtr;

    TLX_ASSERT(Size() == 1);
    TLX_ASSERT(mVector.PeekAt(0) == aPtr);

    return *this;
}

/*-------------------------------------------------------------------------*/
    void *&TLVPArray::operator [](index_t aIndex)

/*  Returns a referenceto the aIndex-th element in the array. If the index
    is invalid, a TLXIndex exception is thrown.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));

    return PeekAt(aIndex);
}

/*-------------------------------------------------------------------------*/
    void *TLVPArray::operator [](index_t aIndex) const

/*  Returns the value of the aIndex-th element in the array. If the index
    is invalid, a TLXIndex exception is thrown.
---------------------------------------------------------------------------*/
{
    if (!IsValidIndex(aIndex))
	THROW(TLXIndex(LOCUS, aIndex));

    return PeekAt(aIndex);
}

/*-------------------------------------------------------------------------*/
    void *&TLVPArray::PeekAt(index_t aIndex)

/*  Returns a reference to the aIndex-th element in the array. There is
    no range checking.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aIndex >= Mini());
    TLX_ASSERT(aIndex <= Maxi());

    return mVector.PeekAt(aIndex - mBase);
}

/*-------------------------------------------------------------------------*/
    void *TLVPArray::PeekAt(index_t aIndex) const

/*  Returns the value of the element at the aIndex-th position. There is no
    range checking.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aIndex >= Mini());
    TLX_ASSERT(aIndex <= Maxi());

    return mVector.PeekAt(aIndex - mBase);
}

/*-------------------------------------------------------------------------*/
    void TLVPArray::SetMini(index_t aBase)

/*  Sets the base index of the array.
---------------------------------------------------------------------------*/
{
    // Make sure that all elements remain accessible. The index type
    // is an integer, so we can use INT_MAX to determine its range.

    TLX_ASSERT((uint32)aBase <= ((uint32)INT_MAX - (uint32)Size()));
    mBase = aBase;
}

/*-------------------------------------------------------------------------*/
    void TLVPArray::Resize(size_t aSize)

/*  Changes the size of the array to a given value.
---------------------------------------------------------------------------*/
{
    // Make sure that all elements remain accessible. The index type
    // is an integer, so we can use INT_MAX to determine its range.

    TLX_ASSERT(aSize <= ((uint32)INT_MAX - (uint32)Mini()));
    mVector.Resize(aSize);
}

/*-------------------------------------------------------------------------*/
    void TLVPArray::ShrinkBy(size_t aDelta)

/*  Reduces the size of the array by a given amount.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(aDelta <= Size());
    mVector.ShrinkBy(aDelta);
}

