/****************************************************************************
    $Id: trace.cpp 501.0 1995/03/07 12:26:24 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of tracing functions.

    $Log: trace.cpp $
    Revision 501.0  1995/03/07 12:26:24  RON
    Updated for TLX 5.01
    Revision 1.11  1995/02/28 15:15:22  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.10  1995/01/18  19:04:13  ron
    Added support for GUI/non-GUI output

    Revision 1.9  1995/01/06  15:58:38  ron
    Corrected Revision keyword

    Revision 1.8  1995/01/05  15:31:04  ron
    Changes in tracing output format

    Revision 1.7  1994/11/16  15:45:06  ron
    Made global tracing functions into static members of TLTracer
    Added module info; rearranged #include directives

    Revision 1.6  1994/10/05  18:45:11  ron
    Renamed TLx...() functions to tl...()

    Revision 1.5  1994/09/28  14:46:55  ron
    Removed Macintosh-style #include references

    Revision 1.4  1994/09/27  20:23:16  ron
    Changed path separator from / to \

    Revision 1.3  1994/09/26  15:49:45  ron
    Small formatting changes

    Revision 1.2  1994/09/12  14:57:34  ron
    Changed wording and layout of diagnostic trace messages slightly

    Revision 1.1  1994/08/16  18:13:19  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <iostream.h>		// For ostream
#include <stdio.h>		// For vsprintf()
#include <stdlib.h>		// For getenv()
#include <string.h>		// For strlen()/

#if OS_WINXXX && defined(_GUI)
  #define STRICT
  #include <windows.h>
#endif

#ifndef TLX_MAX_MSGLEN
// TLX_MAX_MSGLEN is defined in <tlx\tracelib.h> for Windows
#define TLX_MAX_MSGLEN	512
#endif

/*---------------------------------------------------------------------------
    Global variables
---------------------------------------------------------------------------*/

// Try TLTRACE environment variable for trace level

static char *	_gEnv 	      = getenv("TLTRACE");
int		_gTraceLevel  = _gEnv ? atoi(_gEnv) : INT_MAX;
ostream *	_gTraceStream = &cerr;

bool		TLTracer::sEnabled = true;
TLTracer	gTraceGrpTLX("TLX");

/*-------------------------------------------------------------------------*/
    TLTracer::TLTracer(const char *aName)

/*  Constructor. Creates a tracing group with the given name.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aName);

    mGroupName = aName;
    mMaxLevel  = INT_MAX;
    mEnabled   = true;
    mOStream   = &cerr;
    mFileName  = "";
    mLineNo    = 0;
}

/*-------------------------------------------------------------------------*/
    TLTracer::~TLTracer()

/*  Destructor. Does nothing currently.
---------------------------------------------------------------------------*/
{
}

/*-------------------------------------------------------------------------*/
    void TLTracer::Disable()

/*  Disables tracing output for the current group.
---------------------------------------------------------------------------*/
{
    mEnabled = false;
}

/*-------------------------------------------------------------------------*/
    void TLTracer::DisableAll()

/*  Disables tracing output for all tracing groups at once.
---------------------------------------------------------------------------*/
{
    sEnabled = false;
}

/*-------------------------------------------------------------------------*/
    void TLTracer::Enable()

/*  Enables tracing output for the current group.
---------------------------------------------------------------------------*/
{
    mEnabled = true;
}

/*-------------------------------------------------------------------------*/
    void TLTracer::EnableAll()

/*  Restores the individual tracing on/off settings for all groups.
---------------------------------------------------------------------------*/
{
    sEnabled = true;
}

/*-------------------------------------------------------------------------*/
    void __cdecl TLTracer::ErrorFmt(const char *aFmt, ...)

/*  Prints a formatted error message, provided the current maximum tracing
    level >= 1 and tracing is enabled both globally and locally.
---------------------------------------------------------------------------*/
{
    if (!CanTrace(1))
	return;

    va_list args;
    va_start(args, aFmt);
    VPrintf("Err", aFmt, args);
    va_end(args);
}

/*-------------------------------------------------------------------------*/
    void __cdecl TLTracer::FatalFmt(const char *aFmt, ...)

/*  Prints a formatted fatal message, provided the current maximum tracing
    level >= 0 and tracing is enabled both globally and locally.
---------------------------------------------------------------------------*/
{
    if (!CanTrace(0))
	return;

    va_list args;
    va_start(args, aFmt);
    VPrintf("Fatal", aFmt, args);
    va_end(args);
}

/*-------------------------------------------------------------------------*/
    void __cdecl TLTracer::InfoFmt(const char *aFmt, ...)

/*  Prints a formatted info message, provided the current maximum tracing
    is >= 3 level and tracing is enabled both globally and locally.
---------------------------------------------------------------------------*/
{
    if (!CanTrace(3))
	return;

    va_list args;
    va_start(args, aFmt);
    VPrintf("Info", aFmt, args);
    va_end(args);
}

/*-------------------------------------------------------------------------*/
    TLTracer &TLTracer::SetLocus(const char *aFile, int aLine)

/*  Helper function to store locus information prior to the output of a
    trace message.
---------------------------------------------------------------------------*/
{
    mFileName = aFile;
    mLineNo   = aLine;
    return *this;
}

/*-------------------------------------------------------------------------*/
    void TLTracer::SetMaxLevel(int aLevel)

/*  Sets the maximum level of trace output for the current group.
---------------------------------------------------------------------------*/
{
    mMaxLevel = aLevel;
}

/*-------------------------------------------------------------------------*/
    void TLTracer::SetStream(ostream &os)

/*  Installs the given stream as the output stream for the tracing group.
---------------------------------------------------------------------------*/
{
    mOStream = &os;
}

/*-------------------------------------------------------------------------*/
    void __cdecl TLTracer::TraceFmt(int aLevel, const char *aFmt, ...)

/*  Prints a formatted trace message, provided its level is <= the current
    maximum tracing level and tracing is enabled both globally and locally.
---------------------------------------------------------------------------*/
{
    if (!CanTrace(aLevel))
	return;

    char buf[20];
    itoa(aLevel, buf, 10);

    va_list args;
    va_start(args, aFmt);
    VPrintf(buf, aFmt, args);
    va_end(args);
}

/*-------------------------------------------------------------------------*/
    void TLTracer::VPrintf(const char *aPre, const char *aFmt, va_list args)

/*  Prints a tracing message consisting of a the group's name, a prefix,
    the actual vprintf-style formatted message, and the locus of the
    message originator.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aPre);
    TLX_ASSERT_PTR(mOStream);
    TLX_ASSERT_PTR(mGroupName);
    TLX_ASSERT_PTR(mFileName);

    // Build the trace message in a local buffer

    char buffer[TLX_MAX_MSGLEN];

    sprintf(buffer, "%s %s %04d: [%s] ", aPre, mFileName, mLineNo, mGroupName);
    vsprintf(&buffer[strlen(buffer)], aFmt, args);

    size_t msglen = strlen(buffer);
    TLX_ASSERT(msglen < TLX_MAX_MSGLEN - 1);

    if (buffer[msglen - 1] != '\n')
	strcat(buffer, "\n");

  #ifndef _GUI
    *mOStream << buffer;
  #elif OS_WINXXX
    OutputDebugString(buffer);          // Windows debug output
  #else
    #error Unsupported platform
  #endif
}

/*-------------------------------------------------------------------------*/
    void __cdecl TLTracer::WarningFmt(const char *aFmt, ...)

/*  Prints a formatted warning message, provided the current maximum tracing
    level >= 2 and tracing is enabled both globally and locally.
---------------------------------------------------------------------------*/
{
    if (!CanTrace(2))
	return;

    va_list args;
    va_start(args, aFmt);
    VPrintf("Warn", aFmt, args);
    va_end(args);
}

