/****************************************************************************
    $Id: strings.cpp 501.0 1995/03/07 12:26:24 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLString. This class represents a character
    string, i.e. a sequence of characters of known length. The string
    resizes itself each time an expansion is necessary. (Contractions are
    only performed on request.)

    The string is not a 0-terminated C-style string, although conversions
    from and to these strings are implemented. Character indexing starts at
    0, just as with C-style strings.

    TODO:

    - check whether change from 1-based to 0-based indexing is correct
    - make sure that C-style functions result in a C-style string
    - check logic for invisible extra character

    $Log: strings.cpp $
    Revision 501.0  1995/03/07 12:26:24  RON
    Updated for TLX 5.01
    Revision 1.8  1995/01/31 16:30:28  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.7  1995/01/06  15:58:31  ron
    Corrected Revision keyword

    Revision 1.6  1994/11/16  15:43:57  ron
    Added module info; rearranged #include directives

    Revision 1.5  1994/10/05  18:43:42  ron
    Renamed TLx...() functions to tl...()

    Revision 1.4  1994/09/28  14:22:48  ron
    Removed Macintosh-style #include references

    Revision 1.3  1994/09/27  20:23:08  ron
    Changed path separator from / to \

    Revision 1.2  1994/09/26  15:48:16  ron
    Renamed SetSize() to Resize()

    Revision 1.1  1994/08/16  18:13:16  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

//-----	System headers

#include <ctype.h>		// For tolower(), toupper()
#include <iostream.h>
#include <stdlib.h>		// For malloc() etc.
#include <string.h>		// C string-handling functions

//----- Library headers

#include <tlx\501\except.h>
#include <tlx\501\strings.h>        // class TLString

/*---------------------------------------------------------------------------
    Implementation notes:
    ---------------------

1.  We use the malloc() family of memory allocations rather than operators
    new and delete, because the frequent reallocations might benefit from
    the availability of realloc().

2.  We silently allocate 1 extra character at the end of each string buffer.
    This extra character ensures that we will always be able to append a
    '\0' terminator to each string. The extra character is not included in
    any count, not even in 'mSize'.
---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
    TLString::TLString(size_t aSize)

/*  Constructor, initializing the string as a character buffer of the
    specified length. The buffer's contents are set to 0.
---------------------------------------------------------------------------*/
{
    Create(0, 0, aSize);
    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Size() == aSize);
}

/*-------------------------------------------------------------------------*/
    TLString::TLString(char c, size_t extra)

/*  Constructor initializing the string as a character buffer with a length
    of 1 + 'extra'. The first character of the buffer is set to 'c'. The
    length of the string is set to 1.
---------------------------------------------------------------------------*/
{
    Create(&c, 1, extra);
    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Size() == extra + 1);
    TLX_ASSERT(Length() == 1);
}

/*-------------------------------------------------------------------------*/
    TLString::TLString(const char *s, size_t extra)

/*  Constructor initializing the string as a character buffer with a length
    of n = strlen(s) + 'extra'. The first n characters are copied from 's'.
    The length of the string is set to n.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(s);

    Create(s, strlen(s), extra);
    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Size() == strlen(s) + extra);
    TLX_ASSERT(Length() == strlen(s));
}

/*-------------------------------------------------------------------------*/
    TLString::TLString(const TLString &s)

/*  Copy constructor. Initializes the string as a character buffer with a
    length of n = s.mLen.
---------------------------------------------------------------------------*/
{
    Create(s.mString, s.mLen, s.mSize - s.mLen);
    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Size() == s.Size());
}

/*-------------------------------------------------------------------------*/
    TLString::~TLString()

/*  Destructor. Frees the character buffer.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);
    free(mString);
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::Assign(const char *s, size_t n)

/*  Assigns the character string to the string as its contents. If the
    character buffer size is less than required, an attempt is made to
    expand the buffer. In any case, only as many characters as will fit
    in the buffer are copied.

    This member function does not determine the string length itself, but
    relies on the value provided in the 'n' parameter. This allows us to
    capture the code that is common to several assignment operations.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(s);
    TLX_ASSERT_PTR(mString);

    if (mSize < n) Resize(n);          // Try to resize
    if (mSize == 0) return *this;
    if (n > mSize) n = mSize;           // Truncate if necessary

    memcpy(mString, s, n);
    mLen = n;
    Terminate();
    return *this;
}

/*-------------------------------------------------------------------------*/
    void TLString::Clear()

/*  Clears the contents of the string by setting its length to 0 and NUL-
    terminating the character string.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    mLen = 0;
    Terminate();
}

/*-------------------------------------------------------------------------*/
    void TLString::CompactStorage()

/*  Reduces the size of the character buffer to the size required by the
    string stored in it.
---------------------------------------------------------------------------*/
{
    Resize(mLen);
}

/*-------------------------------------------------------------------------*/
    TLString TLString::Copy(size_t pos, size_t n)

/*  Creates a new string that contains a substring from the current string.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (pos >= mLen) return TLString(0u);
    if (pos + n >= mLen) n = mLen - pos;

    TLString result(n);
    if (result.mString) {
    	memcpy(result.mString, mString + pos, n);
	result.mLen = n;
	result.Terminate();
    }
    return result;
}

/*-------------------------------------------------------------------------*/
    void TLString::Create(const char *s, size_t n, size_t extra)

/*  Creates a character buffer of the specified size, and initializes the
    other data members accordingly. The first 'n' characters of the buffer
    are copied from 's'.

    NOTE: this function can only be used safely by constructors.
---------------------------------------------------------------------------*/
{
    mString = 0;
    mLen    =
    mSize   = 0;

    Resize(n + extra);
    TLX_ASSERT_PTR(mString);

    if (s) memcpy(mString, s, n);
    mLen = n;
    Terminate();
}

/*-------------------------------------------------------------------------*/
    void TLString::Remove(size_t pos, size_t n)

/*  Removes 'n' characters from the string, starting at position 'pos'.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (pos >= mLen) return;
    if (pos + n >= mLen) n = mLen - pos;

    memmove(mString + pos, mString + pos + n, mLen - pos - n);
    mLen -= n;
}

/*-------------------------------------------------------------------------*/
    TLString TLString::Extract(size_t pos, size_t n)

/*  Removes and returns a substring from the current string.
---------------------------------------------------------------------------*/
{
    TLString result(Copy(pos, n));
    Remove(pos, n);
    return result;
}

/*-------------------------------------------------------------------------*/
    void TLString::Fill(char c, size_t pos, size_t len)

/*  Fills the character buffer with the specified character. If pos >= mSize,
    nothing happens; if len > remainder, only the remainder is filled.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (pos >= mSize) return;
    if (len > mSize - pos) len = mSize - pos;
    memset(mString + pos, c, len);
}

/*-------------------------------------------------------------------------*/
    size_t TLString::IndexOf(char c, size_t pos)

/*  Returns the index of the first occurrence of 'c', starting at position
    'pos'. It returns 0 if not found.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (pos >= mLen) return 0;
    char *cptr = (char *) memchr(mString + pos, c, mLen - pos);
    ptrdiff_t i = cptr ? cptr - mString + 1 : 0;

    TLX_ASSERT(i >= 0);
    TLX_TODO(Change to 0-based indexing; use index_t for return type);

    return (size_t) i;
}

/*-------------------------------------------------------------------------*/
    size_t TLString::IndexOf(const char *s, size_t pos)

/*  Returns the index of the first occurrence of 's', starting at position
    'pos'. It returns 0 if not found.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (pos >= mLen) return 0;
    char *sptr = strstr(mString + pos, s);
    ptrdiff_t i = sptr ? sptr - mString + 1 : 0;

    TLX_ASSERT(i >= 0);
    TLX_TODO(Change to 0-based indexing; use index_t for return type);

    return (size_t) i;
}

/*-------------------------------------------------------------------------*/
    size_t TLString::IndexOf(const TLString &s, size_t pos)

/*  Returns the index of the first occurrence of 's', starting at position
    'pos'. It returns 0 if not found.
---------------------------------------------------------------------------*/
{
    return IndexOf(s.mString, pos);
}

/*-------------------------------------------------------------------------*/
    void TLString::Insert(char c, size_t pos)

/*  Inserts the character at the indicated position. The previous character
    at that position and all remaining characters are moved up. If necessary
    and allowed, the character buffer is expanded.
---------------------------------------------------------------------------*/
{
    Insert(&c, 1, pos);
}

/*-------------------------------------------------------------------------*/
    void TLString::Insert(const char *s, size_t pos)

/*  Inserts the character string at the indicated position.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(s);
    Insert(s, strlen(s), pos);
}

/*-------------------------------------------------------------------------*/
    void TLString::Insert(const TLString &s, size_t pos)

/*  Inserts the other string at the indicated position.
---------------------------------------------------------------------------*/
{
    Insert(s.mString, s.mLen, pos);
}

/*-------------------------------------------------------------------------*/
    void TLString::Insert(const char *s, size_t n, size_t pos)

/*  Inserts 'n' characters from 's' into the string. If necessary, the
    string is first expanded to make room for the new characters.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(s);
    TLX_ASSERT_PTR(mString);

    // First try to expand the string to the required size
    if (mSize < mLen + n) Resize(mLen + n);

    // Truncate the copy operation if necessary
    if (mLen >= mSize) return;
    if (n > mSize - mLen) n = mSize - mLen;

    // Make sure the insert position is reasonable
    if (pos > mLen) pos = mLen;

    // Create a hole, then copy the characters into it
    memmove(mString + pos + n, mString + pos, mSize - pos - n);
    memcpy(mString + pos, s, n);
    mLen += n;
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::operator =(char c)

/*  Assigns the character to the string as its single character.
---------------------------------------------------------------------------*/
{
    return Assign(&c, 1);
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::operator =(const char *s)

/*  Assigns the character string to the string as its contents. If the
    character buffer size is less than required, the buffer is resized.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(s);
    TLX_ASSERT_PTR(mString);

    if (s != mString)
	Assign(s, strlen(s));

    return *this;
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::operator =(const TLString &s)

/*  Assigns the contents of the other string to the current one. The _flex
    parameter is NOT copied. If the character buffer size is less than
    required, the buffer is expanded.

    NOTE: s.mLen is used as the length of the string to be copied.
---------------------------------------------------------------------------*/
{
    if (&s != this)
	Assign(s.mString, s.mLen);

    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Length() == strlen(mString));

    return *this;
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::operator +=(char c)

/*  Appends the character at the end of the character buffer. The buffer
    is supposed to end at position 'mLen', i.e. 'c' is inserted at that
    position.
---------------------------------------------------------------------------*/
{
    Insert(&c, 1, mLen);

    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Length() == strlen(mString));

    return *this;
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::operator +=(const char *s)

/*  Appends the character string at the end of the character buffer. The
    buffer is supposed to end at position 'mLen', i.e. the contents of 's'
    are inserted started at that position.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(s);

    Insert(s, strlen(s), mLen);

    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Length() == strlen(mString));

    return *this;
}

/*-------------------------------------------------------------------------*/
    TLString &TLString::operator +=(const TLString &s)

/*  Appends the other string at the end of the character buffer. The buffer
    is supposed to end at position 'mLen', i.e. the contents of 's' are
    inserted started at that position.

    NOTE: s.mLen is used as the length of the string to be copied.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    Insert(s.mString, s.mLen, mLen);

    TLX_ASSERT(Length() == strlen(mString));

    return *this;
}

#if 0
TLString::operator char *()
/*
    Turns the string into a C-style 0-terminated string, and returns a pointer
    to it. A '\0' is inserted at position mString[mLen], which is guaranteed to
    exist (we have a hidden extra character at the end of each buffer).
*/
{
    if (mString) mString[mLen] = '\0';
    return mString;
}
#endif

/*-------------------------------------------------------------------------*/
    char &TLString::operator [](size_t i)

/*  Returns a reference to the 'i'th character, or throws a TLXIndex
    exception if the index is invalid.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (!IsValidIndex(i))
	THROW(TLXIndex(LOCUS, i));

    return mString[i];
}

/*-------------------------------------------------------------------------*/
    char TLString::operator [](size_t i) const

/*  Returns the value of the 'i'th character, or throws a TLXIndex
    exception if the index is invalid.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    if (!IsValidIndex(i))
	THROW(TLXIndex(LOCUS, i));

    return mString[i];
}

/*-------------------------------------------------------------------------*/
    void TLString::Reserve(size_t sz)

/*  Expands the string buffer to hold at least 'sz' characters. Space for
    the terminating 0 is always added silently.
---------------------------------------------------------------------------*/
{
    if (Size() < sz)
	Resize(sz);
}

/*-------------------------------------------------------------------------*/
    void TLString::Resize(size_t size)

/*  Resizes an existing string to the requested size. If the operation
    succeeds, the mSize data member is adjusted. If not, a TLXAlloc
    exception is thrown. The contents that the reallocated character buffer
    has in common with the old buffer remain unchanged; the new positions
    (if any) are uninitialized.

    NOTE: we silently allocate an extra character at the end of each buffer.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(size <= MaxSize());

    // Perform reallocation, but keep mString intact. Allocated size is
    // always > 0.
    char *tmp = (char *) realloc(mString, size + 1);
    if (!tmp)
	THROW(TLXAlloc(LOCUS, size + 1));

    mString = tmp;
    mSize   = size;
}

/*-------------------------------------------------------------------------*/
    void TLString::Reverse()

/*  Reverses the contents of the buffer.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    size_t n    = mLen/2;
    char *head = mString;
    char *tail = mString + mLen - 1;

    while (n--) {
        char c  = *head;
        *head++ = *tail;
        *tail-- = c;
    }
    TLX_ASSERT(Length() == strlen(mString));
}

/*-------------------------------------------------------------------------*/
    void TLString::Sync()

/*  Brings the mLen data member up to date with the position of the first
    '\0' character in the buffer. If there is no such character, mLen
    becomes equal to mSize.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    char *nul = (char *) memchr(mString, 0, mSize);
    ptrdiff_t len = nul ? nul - mString : mSize;

    TLX_ASSERT(len >= 0);
    mLen = (size_t) len;
}

/*-------------------------------------------------------------------------*/
    void TLString::Terminate()

/*  Appends a '\0' character to terminate the string according to C usage.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);
    TLX_ASSERT(Size() >= mLen);

    mString[mLen] = '\0';	// Always at least 1 position allocated
}

/*-------------------------------------------------------------------------*/
    void TLString::ToLower()

/*  Converts all characters up to mLen to lowercase.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    size_t n    = mLen;
    char *cptr = mString;

    while (n--) {
        *cptr = (char)tolower(*cptr);
        cptr++;
    }
}

/*-------------------------------------------------------------------------*/
    void TLString::ToUpper()

/*  Converts all characters up to mLen to uppercase.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mString);

    size_t n    = mLen;
    char *cptr = mString;

    while (n--) {
	*cptr = (char)toupper(*cptr);
	cptr++;
    }
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator ==(const TLString &aStr1, const TLString &aStr2)

/*  Compares two strings for equality, returning nonzero if the are equal.
    The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1.mString, aStr2.mString) == 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator ==(const TLString &aStr1, const char *aStr2)

/*  Compares two strings for equality, returning nonzero if the are equal.
    The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2);
    return strcmp(aStr1.mString, aStr2) == 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator ==(const char *aStr1, const TLString &aStr2)

/*  Compares two strings for equality, returning nonzero if the are equal.
    The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1, aStr2.mString) == 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator !=(const TLString &aStr1, const TLString &aStr2)

/*  Compares two strings for inequality, returning zero if the are equal.
    The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1.mString, aStr2.mString) != 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator !=(const TLString &aStr1, const char *aStr2)

/*  Compares two strings for inequality, returning zero if the are equal.
    The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2);
    return strcmp(aStr1.mString, aStr2) == 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator !=(const char *aStr1, const TLString &aStr2)

/*  Compares two strings for inequality, returning zero if the are equal.
    The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1, aStr2.mString) == 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator <=(const TLString &aStr1, const TLString &aStr2)

/*  Compares two strings for less than or equal, returning nonzero if the
    first string is less than or equal to the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1.mString, aStr2.mString) <= 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator <=(const TLString &aStr1, const char *aStr2)

/*  Compares two strings for less than or equal, returning nonzero if the
    first string is less than or equal to the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2);
    return strcmp(aStr1.mString, aStr2) <= 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator <=(const char *aStr1, const TLString &aStr2)

/*  Compares two strings for less than or equal, returning nonzero if the
    first string is less than or equal to the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1, aStr2.mString) <= 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator < (const TLString &aStr1, const TLString &aStr2)

/*  Compares two strings for less than, returning nonzero if the first
    string is less than the second one. The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1.mString, aStr2.mString) < 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator < (const TLString &aStr1, const char *aStr2)

/*  Compares two strings for less than, returning nonzero if the first
    string is less than the second one. The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2);
    return strcmp(aStr1.mString, aStr2) < 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator < (const char *aStr1, const TLString &aStr2)

/*  Compares two strings for less than, returning nonzero if the first
    string is less than the second one. The comparison is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1, aStr2.mString) == 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator >=(const TLString &aStr1, const TLString &aStr2)

/*  Compares two strings for greater than or equal, returning nonzero if the
    first string is greater than or equal to the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1.mString, aStr2.mString) >= 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator >=(const TLString &aStr1, const char *aStr2)

/*  Compares two strings for greater than or equal, returning nonzero if the
    first string is greater than or equal to the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2);
    return strcmp(aStr1.mString, aStr2) >= 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator >=(const char *aStr1, const TLString &aStr2)

/*  Compares two strings for greater than or equal, returning nonzero if the
    first string is greater than or equal to the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1, aStr2.mString) >= 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator > (const TLString &aStr1, const TLString &aStr2)

/*  Compares two strings for greater than, returning nonzero if the
    first string is greater than the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1.mString, aStr2.mString) > 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator > (const TLString &aStr1, const char *aStr2)

/*  Compares two strings for greater than, returning nonzero if the
    first string is greater than the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1.mString);
    TLX_ASSERT_PTR(aStr2);
    return strcmp(aStr1.mString, aStr2) > 0;
}

/*-------------------------------------------------------------------------*/
    int _TLXFUNC operator > (const char *aStr1, const TLString &aStr2)

/*  Compares two strings for greater than, returning nonzero if the
    first string is greater than the second one. The comparison
    is case sensitive.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aStr1);
    TLX_ASSERT_PTR(aStr2.mString);
    return strcmp(aStr1, aStr2.mString) > 0;
}

/*-------------------------------------------------------------------------*/
    ostream & _TLXFUNC operator <<(ostream &os, const TLString &s)

/*  Output operator for strings.
---------------------------------------------------------------------------*/
{
    // The call to os.write() causes problems when linking with BC++ 4.0
    // DLL RTL. It appears that the os.write() is not properly exported
    // under these circumstances, and the linker therefore fails to find it.
    // Until we have sorted this out, we remove the call.
    //
    // The BC++ 4.0 compiler has __BORLANDC__ version # 0x452

#if defined(OS_WINXXX) && defined(__BORLANDC__) && (__BORLANDC__ == 0x452)
    // Warn our users
    TLX_NOTIMPLEMENTED(operator <<(ostream &os, const TLString &s));
    return os;
#else
    // Normal behavior
    return os.write(s.mString, s.mLen);
#endif
}

/*-------------------------------------------------------------------------*/
    void _TLXFUNC tlSwap(TLString &s1, TLString &s2)

/*  Swaps the contents of two strings efficiently.
---------------------------------------------------------------------------*/
{
    if (&s1 == &s2) return;

    char *tmps = s1.mString;
    s1.mString = s2.mString;
    s2.mString = tmps;

    size_t tmpsz = s1.mSize;
    s1.mSize = s2.mSize;
    s2.mSize = tmpsz;

    size_t tmpl = s1.mLen;
    s1.mLen = s2.mLen;
    s2.mLen = tmpl;
}
