/****************************************************************************
    $Id: slcbase.cpp 501.0 1995/03/07 12:26:22 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLSLCBase.

    $Log: slcbase.cpp $
    Revision 501.0  1995/03/07 12:26:22  RON
    Updated for TLX 5.01
    Revision 1.7  1995/01/31 16:30:24  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.6  1995/01/06  15:58:19  ron
    Corrected Revision keyword

    Revision 1.5  1994/11/16  15:42:59  ron
    Added module info; rearranged #include directives

    Revision 1.4  1994/09/28  14:21:39  ron
    Removed Macintosh-style #include references

    Revision 1.3  1994/09/27  20:22:59  ron
    Changed path separator from / to \

    Revision 1.2  1994/09/26  15:47:23  ron
    Changed include file references

    Revision 1.1  1994/08/16  18:13:13  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <tlx\501\except.h>		// Exception handling
#include <tlx\501\slists.h>		// Class declaration

/*-------------------------------------------------------------------------*/
    TLSLCBase::TLSLCBase(bool aOwner)

/*  Constructor, also doubles as default constructor. Creates an empty list.
---------------------------------------------------------------------------*/
: mTail(0), mCount(0), mOwner(aOwner)
{
}

/*-------------------------------------------------------------------------*/
    TLSLCBase::TLSLCBase(const TLSLCBase &)

/*  Copy constructor. Does not copy the list.
---------------------------------------------------------------------------*/
: mTail(0), mCount(0), mOwner(false)
{
}

/*-------------------------------------------------------------------------*/
    TLSLCBase::~TLSLCBase()

/*  Destructor. If the list is owner of its elements, they are deleted.
---------------------------------------------------------------------------*/
{
    if (mOwner) CleanUp();
}

/*-------------------------------------------------------------------------*/
    void TLSLCBase::Append(TLSLink *aLink)

/*  Adds an element at the end of the list. The element pointer must be
    nonzero, and the element must not currently be part of a list.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_NULL(aLink->mNext);

    if (mTail)
    {
	aLink->Append(mTail->Next());
	mTail->Append(aLink);
    }
    else			// Create a single element list
    {
	TLX_ASSERT(mCount == 0);
	aLink->Append(aLink);
    }
    mTail = aLink;

    mCount++;
}

/*-------------------------------------------------------------------------*/
    void TLSLCBase::Append(TLSLink *aLink, TLSLink *aPos)

/*  Inserts a link after another link in the list. The link to insert must
    not be 0 and must not currently be part of a list.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_PTR(aPos);
    TLX_ASSERT_NULL(aLink->mNext);
    TLX_ASSERT(Contains(aPos));

    aPos->Append(aLink);
    aLink->Append(aPos->Next());

    if (aPos == mTail)
	mTail = aLink;

    mCount++;
}

/*-------------------------------------------------------------------------*/
    bool TLSLCBase::BecomeOwner(bool aOwner)

/*  Changes the owner status, returning the previous status.
---------------------------------------------------------------------------*/
{
    bool oldstat = mOwner;
    mOwner = aOwner;
    return oldstat;
}

/*-------------------------------------------------------------------------*/
    void TLSLCBase::CleanUp()

/*  Deletes all elements in the list. The list should be owner for
    this function to be called.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(mOwner);
    if (!mTail) return;

    TLSLink *rover = mTail;
    do
    {
	TLX_ASSERT_PTR(rover);

	TLSLink *tmp = rover;
	rover = rover->Next();
	tmp->Unlink();
	delete tmp;
    }
    while (rover != mTail);
}

/*-------------------------------------------------------------------------*/
    bool TLSLCBase::Contains(TLSLink *aLink) const

/*  Tests whether a given element appears in the list, returning nonzero
    if it does.
---------------------------------------------------------------------------*/
{
    if (!aLink) return false;

    for (TLSLink *rover = mTail->Next(); rover; rover = rover->Next())
    {
	if (rover == aLink)
	    return true;
	if (rover == mTail)
	    break;
    }
    return false;
}

/*-------------------------------------------------------------------------*/
    void TLSLCBase::RemoveAll()

/*  Removes all elements from the list. If the list owns its items, they
    are deleted; else they are simply forgotten.
---------------------------------------------------------------------------*/
{
    if (mOwner) CleanUp();
    mTail  = 0;
    mCount = 0;
}

/*-------------------------------------------------------------------------*/
    TLSLink *TLSLCBase::Extract(TLSLink *aLink)

/*  Removes and returns a specific link from the list.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mTail);
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT(Contains(aLink));

    // Find the element preceding aLink

    for (TLSLink *rover = mTail->Next(); rover; rover = rover->Next())
    {
	if (rover->Next() == aLink)
	{
	    // rover points to the element preceding aLink, which could be
	    // aLink itself. In the latter case, the list will become empty.

	    if (rover == aLink)
	    {
		// List becomes empty
		mTail = 0;
	    }
	    else
	    {
		rover->Append(aLink->Next());
		if (aLink == mTail)
		    mTail = rover;
	    }
	    break;
	}
	else if (rover == mTail)
	    THROW(TLXNotFound(LOCUS));
    }

    aLink->Unlink();
    TLX_ASSERT(mCount > 0);
    mCount--;

    return aLink;
}

/*-------------------------------------------------------------------------*/
    TLSLink *TLSLCBase::ExtractHead()

/*  Removes and returns the first element of the list.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));

    return Extract(mTail->Next());
}

/*-------------------------------------------------------------------------*/
    TLSLink *TLSLCBase::ExtractTail()

/*  Removes and returns the last element of the list.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));

    return Extract(mTail);
}

/*-------------------------------------------------------------------------*/
    bool TLSLCBase::IsEmpty() const

/*  Checks if the list is currently empty, returning nonzero if it is.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(mCount > 0 || mTail == 0);
    return mCount == 0;
}

/*-------------------------------------------------------------------------*/
    TLSLCBase &TLSLCBase::operator =(const TLSLCBase &)

/*  Assignment operator, does not copy anything.
---------------------------------------------------------------------------*/
{
    return *this;
}

/*-------------------------------------------------------------------------*/
    void TLSLCBase::Prepend(TLSLink *aLink)

/*  Adds a new link at the start of the current list. The new link
    must not be 0 and must not be part of a list yet.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_NULL(aLink->mNext);

    if (mTail)
    {
	aLink->Append(mTail->Next());
	mTail->Append(aLink);
    }
    else			// Make a single element list
    {
	aLink->Append(aLink);
	mTail = aLink;
    }

    mCount++;
}

/*-------------------------------------------------------------------------*/
    void TLSLCBase::Prepend(TLSLink *aLink, TLSLink *aPos)

/*  Inserts a link before another one. The link to insert must not be 0
    and must not be part of a list yet.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_PTR(aPos);
    TLX_ASSERT_NULL(aLink->mNext);
    TLX_ASSERT(Contains(aPos));

    // Find the element preceding aPos

    for (TLSLink *rover = mTail->Next(); rover; rover = rover->Next())
    {
	if (rover->Next() == aPos)
	{
	    // rover points to the element preceding aPos, which could be
	    // aPos itself.

	    aLink->Append(aPos);
	    rover->Append(aLink);
	    break;
	}
	else if (rover == mTail)
	    THROW(TLXNotFound(LOCUS));
    }

    mCount++;
}

