/****************************************************************************
    $Id: refptr.cpp 501.0 1995/03/07 12:26:20 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLRefPtr.

    $Log: refptr.cpp $
    Revision 501.0  1995/03/07 12:26:20  RON
    Updated for TLX 5.01
    Revision 1.8  1995/01/31 16:30:22  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.7  1995/01/06  15:58:16  ron
    Corrected Revision keyword

    Revision 1.6  1994/11/16  15:42:32  ron
    Added module info; rearranged #include directives

    Revision 1.5  1994/09/28  14:20:55  ron
    Removed Macintosh-style #include references

    Revision 1.4  1994/09/27  20:22:53  ron
    Changed path separator from / to \

    Revision 1.3  1994/09/26  15:46:22  ron
    Changed include file references

    Revision 1.2  1994/09/06  14:11:35  ron
    Adapted to change from TLRef to TLRefCount

    Revision 1.1  1994/08/16  18:13:11  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <tlx\501\except.h>		// Exception handling
#include <tlx\501\refcnt.h>		// Class declaration

/*-------------------------------------------------------------------------*/
    TLRefPtr::TLRefPtr()

/*  Default constructor. Initializes smart pointer to 0.
---------------------------------------------------------------------------*/
: mPtr(0)
{
}

/*-------------------------------------------------------------------------*/
    TLRefPtr::TLRefPtr(TLRefCount *aPtr)

/*  Constructor that provides for conversion from ordinary TLRefCount pointers.
    It attaches the smart pointer to the pointed to object.
---------------------------------------------------------------------------*/
: mPtr(0)
{
    Attach(aPtr);
    TLX_ASSERT(aPtr == mPtr);
}

/*-------------------------------------------------------------------------*/
    TLRefPtr::TLRefPtr(const TLRefPtr &aPtr)

/*  Copy constructor. Attaches the smart pointer to the same object as the
    other smart pointer points to.
---------------------------------------------------------------------------*/
: mPtr(0)
{
    Attach(aPtr.mPtr);
    TLX_ASSERT(aPtr.mPtr == mPtr);
}

/*-------------------------------------------------------------------------*/
    TLRefPtr::~TLRefPtr()

/*  Destructor. Calls Delete(), which has the effect of detaching the
    smart pointer from the pointed to object, and deleting the pointed
    to object if that was the last reference to it.
---------------------------------------------------------------------------*/
{
    Delete();
}

/*-------------------------------------------------------------------------*/
    TLRefCount *TLRefPtr::Attach(TLRefCount *aPtr)

/*  Attaches the smart pointer to an object to be managed. If the object
    exists (i.e. if the pointer is non-0), its reference count is
    incremented.

    This function should only be called if the smart pointer is not
    currently attached to an object.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_NULL(mPtr);

    if ((mPtr = aPtr) != 0)
	mPtr->Protect();

    return mPtr;
}

/*-------------------------------------------------------------------------*/
    TLRefCount *TLRefPtr::Detach()

/*  Detaches the smart pointer from an object it is associated with. The
    pointed to object is not destroyed, not even if we just removed the
    last reference.
---------------------------------------------------------------------------*/
{
    if (mPtr)
	mPtr->Unprotect();

    TLRefCount *tmp = mPtr;
    mPtr = 0;
    return tmp;
}

/*-------------------------------------------------------------------------*/
    void TLRefPtr::Delete()

/*  Detaches the smart pointer from an object it is associated with, then
    delete the object previously pointed to if we were the last pointer to
    it.
---------------------------------------------------------------------------*/
{
    if (mPtr)
    {
	mPtr->Unprotect();
	if (!mPtr->IsProtected())
	    delete mPtr;
    	mPtr = 0;
    }
}

/*-------------------------------------------------------------------------*/
    TLRefPtr &TLRefPtr::operator =(TLRefCount *aPtr)

/*  Overloading of assignment operator to allow assignment of ordinary
    TLRefCount pointers to the smart pointer class. If the smart pointer was
    pointing to another object before the assignment, that attachment
    is broken (possibly deleting the old object in the process).
---------------------------------------------------------------------------*/
{
    if (aPtr != mPtr)
    {
	Delete();
	Attach(aPtr);
    }
    return *this;
}

/*-------------------------------------------------------------------------*/
    TLRefPtr &TLRefPtr::operator =(const TLRefPtr &aPtr)

/*  Overloading of assignment operator to handle assignment of smart
    pointers to each other. If the current smart pointer was pointing to
    another object before the assignment, that attachment is broken
    (possibly deleting the old object in the process).
---------------------------------------------------------------------------*/
{
    if (mPtr != aPtr.mPtr)
    {
	Delete();
	Attach(aPtr.mPtr);
    }
    return *this;
}

/*-------------------------------------------------------------------------*/
    TLRefCount &TLRefPtr::operator *() const

/*  Overloading of pointer dereference operator, in order to make it
    possible to use TLRefPtr instances as if they were ordinary pointers.
---------------------------------------------------------------------------*/
{
    if (!mPtr)
	THROW(TLXNullPtr(LOCUS));

    return *mPtr;
}

