/****************************************************************************
    $Id: propac3.cpp 501.0 1995/03/07 12:26:18 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:    Tarma Library for C++ V5.0
    Author:     Ron van der Wal

    Implementation of the TLPropagatorAC3 class. This class implements
    the AC-3 arc consistency algorithm.

    $Log: propac3.cpp $
    Revision 501.0  1995/03/07 12:26:18  RON
    Updated for TLX 5.01
    Revision 1.15  1995/02/28 15:12:40  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.14  1995/01/06  15:58:07  ron
    Corrected Revision keyword

    Revision 1.13  1995/01/05  15:26:34  ron
    Added standard Propagate() functions

    Revision 1.12  1994/11/16  15:41:45  ron
    Added module info; rearranged #include directives

    Revision 1.11  1994/10/10  16:53:04  ron
    Changed to <tlx\solve\csp.h>

    Revision 1.10  1994/10/07  17:02:51  ron
    Changed Unregister...() to Deregister...()

    Revision 1.9  1994/10/05  18:41:31  ron
    Changed from collections to iterators

    Revision 1.8  1994/09/28  14:20:06  ron
    Removed Macintosh-style #include references

    Revision 1.7  1994/09/27  20:22:41  ron
    Changed path separator from / to \

    Revision 1.6  1994/09/26  15:44:22  ron
    Adapted to changes in constraint check counting
    Implemented support to remember first failed constraint

    Revision 1.5  1994/09/13  10:13:09  ron
    Added call to reset the failed constraint pointer prior to
    the propagation process.

    Revision 1.4  1994/09/07  15:43:21  ron
    No changes

    Revision 1.3  1994/09/06  20:26:57  ron
    Changed template source code #include from queue.cpp to ptrqueue.cpp

    Revision 1.2  1994/09/06  14:08:45  ron
    Adapted to changes in CSP framework

    Revision 1.1  1994/08/16  18:13:06  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

//----- System headers

#include <iostream.h>

//----- Project headers

#include <tlx\501\solve\csp.h>

/*---------------------------------------------------------------------------
    Template source code
---------------------------------------------------------------------------*/

#if defined(THINK_CPLUS)
    // Templates are in a separate file because of segment size
#elif defined(__BORLANDC__) || defined(_MSC_VER) || defined(__SC__) || defined(__WATCOMC__)
    #include <tlx\501\template\ptriter.cpp>
    #include <tlx\501\template\ptrqueue.cpp>
    #include <tlx\501\template\ptrseq.cpp>
#elif defined(__SUNPRO_CC)
    #include <tlx\501\template\ptriter.cpp>
    #include <tlx\501\template\ptrqueue.cpp>
    #include <tlx\501\template\ptrseq.cpp>
#elif defined(__IBMCPP__)
  #if __IBMCPP__ < 300
    #pragma implementation("tlx\\template\\ptriter.cpp")
    #pragma implementation("tlx\\template\\ptrqueue.cpp")
    #pragma implementation("tlx\\template\\ptrseq.cpp")
  #else
    #include <tlx\501\template\ptriter.cpp>
    #include <tlx\501\template\ptrqueue.cpp>
    #include <tlx\501\template\ptrseq.cpp>
  #endif
#else
    #error Unsupported compiler; contact Tarma Software Research
#endif

/*-------------------------------------------------------------------------*/
    TLPropagatorAC3::TLPropagatorAC3()

/*  Constructor. Initializes the propagator.
---------------------------------------------------------------------------*/
: TLPropagator(), mChanges(50, 50)
{
}

/*-------------------------------------------------------------------------*/
    bool TLPropagatorAC3::CheckAC3(TLVariable &aVar)

/*  Performs the arc consistency algorithm starting from the given variable.

    The function returns nonzero if the system is consistent, else zero.
---------------------------------------------------------------------------*/
{
    TLVariable *var = &aVar;
#ifdef __IBMCPP__
    TLPtrItemIter<TLVariable> it1(var);
    TLPtrIter<TLVariable> it2;
    return PerformAC3(it1, it2, true);
#else
    return PerformAC3(TLPtrItemIter<TLVariable>(var),
    		      TLPtrIter<TLVariable>(), true);
#endif
}

/*-------------------------------------------------------------------------*/
    bool TLPropagatorAC3::CheckAC3(TLPtrIter<TLVariable> &vars)

/*  Performs the arc consistency algorithm on the variables in 'vars'. It
    is an adaption of Mackworth's AC-3 algorithm, modified for operation
    within the general CSP framework.

    NOTE: In contrast to AC-3, we do not perform node consistency first. It
    is assumed that this has been done previously, and it doesn't make
    sense to execute a node consistency algorithm more than once.

    The function returns nonzero if the system is consistent, else zero.
---------------------------------------------------------------------------*/
{
#ifdef __IBMCPP__
    TLPtrIter<TLVariable> it;
    return PerformAC3(vars, it, true);
#else
    return PerformAC3(vars, TLPtrIter<TLVariable>(), true);
#endif
}

/*-------------------------------------------------------------------------*/
    bool TLPropagatorAC3::CheckAC3Set(
        TLPtrIter<TLVariable> &	seeds,	// Initially changed vars
        TLPtrIter<TLVariable> &	range   // Set of vars for propagation
    )

/*  Propagates the effects of constraints that originate with the given set
    of variables. Constraint propagation continues until an inconsistency
    is detected or no more changes occur. Only variables within the
    original set are considered for constraint propagation, although
    other variables may be modified as well.
---------------------------------------------------------------------------*/
{
    return PerformAC3(seeds, range, false);
}

/*-------------------------------------------------------------------------*/
    const char *TLPropagatorAC3::Description() const

/*  Returns a description of the propagation algorithm.
---------------------------------------------------------------------------*/
{
    return "AC-3 propagation";
}

/*-------------------------------------------------------------------------*/
    bool TLPropagatorAC3::PerformAC3(
        TLPtrIter<TLVariable> &	seeds,  // Initially changed vars
        TLPtrIter<TLVariable> &	range,  // Set of vars for propagation
        bool                    global	// true = progate all vars
    )

/*  Helper function that executes actual constraint propagation algorithm.
    It starts by propagating constraints that originate with the given
    set, but it will keep propagating until an inconsistency is detected
    or no more changes occur.

    The 'global' flag indicates whether or not changes to variables
    outside the original set should be propagated as well. If this flag
    is nonzero, they will.
---------------------------------------------------------------------------*/
{
    // 'evalq' will hold the variables that have changed. Initially, it
    // contains all variables passed along as a parameter to the function,
    // but during the execution of the algorithm variables are removed from
    // it. It may also happen that variables are added again.

    TLPtrQueue<TLVariable> evalq(seeds);
    TLX_ASSERT(evalq.Count() == seeds.Count());
    evalq.SetDelta(100);

    SetFailedConstraint(0);
    mChanges.RemoveAll();               // Just to be sure
    RegisterPropagator();		// Register current instance

    bool result = true;			// Assume success
    int32 ccnt = TLConstraint::CheckCount();

    while (!evalq.IsEmpty())
    {
        // Get the next variable
	TLVariable *curvar = evalq.Dequeue();

	// Ask the variable to propagate its changes
	if (!curvar->PropagateChanges())
	{
            // The domain has become empty ==> inconsistent
            TLX_TRACE(TLX, (60, "Inconsistency - Exiting evaluation loop"));
            result = false;
	    break;
	}

        // Check whether any changes have been made. If so, add the
        // variables to the evaluation queue.

        while (!mChanges.IsEmpty())
        {
            // Check if we must consider this variable
            TLVariable *var = mChanges.Dequeue();
            if (global || range.Contains(var))
		evalq.EnqueueUnique(var);
        }
    }

    mStats.mCheckCount += TLConstraint::CheckCount() - ccnt;
    DeregisterPropagator();

    return result;
}

/*-------------------------------------------------------------------------*/
    bool TLPropagatorAC3::Propagate(TLVariable *aVar)

/*  Standard propagation function for all propagators. It must propagate
    the effects of changes to 'aVar'.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aVar);

    return CheckAC3(*aVar);
}

/*-------------------------------------------------------------------------*/
    bool TLPropagatorAC3::Propagate(TLPtrIter<TLVariable> &aSeed)

/*  Standard propagation function for all propagators. It must propagate
    the effects of changes to all variables in 'aSeed'.
---------------------------------------------------------------------------*/
{
    return CheckAC3(aSeed);
}

/*-------------------------------------------------------------------------*/
    void TLPropagatorAC3::RegisterChanges(TLVariable *var)

/*  Makes changes to a variable permanent. This involves the creation of
    a (temporary) record of the variable and the changes made to it, in order
    to determine what constraints will have to be revised later on.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(var);
    mChanges.Enqueue(var);
    TLX_TRACE(TLX, (70, "Changes to variable %s registered", var->GetName()));
}

