/****************************************************************************
    $Id: log.cpp 501.0 1995/03/07 12:26:16 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLLog for message logging.

    $Log: log.cpp $
    Revision 501.0  1995/03/07 12:26:16  RON
    Updated for TLX 5.01
    Revision 1.7  1995/01/31 16:30:16  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.6  1995/01/06  15:58:02  ron
    Corrected Revision keyword

    Revision 1.5  1994/11/16  15:40:55  ron
    Added module info; rearranged #include directives

    Revision 1.4  1994/10/13  11:52:10  ron
    Corrected calculation of minutes in relative log time

    Revision 1.3  1994/10/10  16:51:51  ron
    Added initialization function
    Added entry formatting options

    Revision 1.2  1994/10/06  17:44:24  ron
    Made most mmber functions const

    Revision 1.1  1994/10/05  18:40:20  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <iostream.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include <tlx\501\log.h>

/*---------------------------------------------------------------------------
    Global objects
---------------------------------------------------------------------------*/

TLLog *		TLLog::sIt = 0;
static TLLog 	_gTheLog;

/*---------------------------------------------------------------------------
    Global functions
---------------------------------------------------------------------------*/

#ifdef _TLXDBG
TLLog &tlLog()
{
    TLX_ASSERT_PTR(TLLog::sIt);
    return *TLLog::sIt;
}
#endif

/*-------------------------------------------------------------------------*/
    TLLog::TLLog(bool aFlush)

/*  Default constructor. Connects to cout.
---------------------------------------------------------------------------*/
{
    Init(aFlush);
    mOStream = &cout;
}

/*-------------------------------------------------------------------------*/
    TLLog::TLLog(ostream &os, bool aFlush)

/*  Constructor that links to the given stream.
---------------------------------------------------------------------------*/
{
    Init(aFlush);
    mOStream = &os;
}

/*-------------------------------------------------------------------------*/
    TLLog::TLLog(const TLLog &aLog)

/*  Copy constructor. Links to the same output stream.
---------------------------------------------------------------------------*/
{
    Init(aLog.mFlushAll);
    mOStream = aLog.mOStream;
}

/*-------------------------------------------------------------------------*/
    TLLog::~TLLog()

/*  Destructor flushes the output stream and unlinks the instance from
    the stack of log instances.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mOStream);
    mOStream->flush();

    if (sIt == this)
	sIt = mPrev;
    else {
	TLX_ASSERT_PTR(sIt);
	TLLog *log;
	for (log = sIt; log; log = log->mPrev)
	    if (log->mPrev == this) {
		log->mPrev = mPrev;
		break;
	    }
	TLX_ASSERT_PTR(log);
    }
    mPrev = 0;
}

/*-------------------------------------------------------------------------*/
    ostream &TLLog::BeginEntry()

/*  Starts a new log entry by outputting a date/time stamp and some space.
---------------------------------------------------------------------------*/
{
    mEntryCount++;

    switch (mEntryFormat) {
	case etAbsTime: {
    	    time_t t;
    	    time(&t);
    	    struct tm *tmptr = localtime(&t);
    	    return Printf("%02d-%02d-%02d %02d:%02d:%02d  ",
    	   	tmptr->tm_year, tmptr->tm_mon+1, tmptr->tm_mday,
    	   	tmptr->tm_hour, tmptr->tm_min, tmptr->tm_sec);
	}
	case etRelTime: {
    	    time_t t;
    	    time(&t);
	    t -= mStartTime;
    	    return Printf("%03d:%02d:%02d  ", t / 3600, (t / 60) % 60, t % 60);
	}
	case etCount:
    	    return Printf("%05ld  ", mEntryCount);

	default:
	    TLX_ASSERT_UNREACHABLE;
    	    return Printf("***Log error***  ");
    }
}

/*-------------------------------------------------------------------------*/
    ostream &TLLog::EndEntry() const

/*  Completes an entry by appending a newline and optionally flushing the
    output stream.
---------------------------------------------------------------------------*/
{
    if (mFlushAll)
    	return operator ostream &() << endl;
    else
    	return operator ostream &() << "\n";
}

/*-------------------------------------------------------------------------*/
    void TLLog::Init(bool aFlush)

/*  Initialization routine, to be called only from constructors.
---------------------------------------------------------------------------*/
{
    // Link to previous entry on log stack (if any)
    mPrev = sIt;
    sIt   = this;

    time(&mStartTime);
    mEntryFormat = etRelTime;
    mEntryCount  = 0;
    mFlushAll 	 = aFlush;
}

#ifdef _TLXDBG
/*-------------------------------------------------------------------------*/
    TLLog::operator ostream &() const

/*  Conversion operator to ostream &.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mOStream);
    return *mOStream;
}
#endif

/*-------------------------------------------------------------------------*/
    TLLog &TLLog::operator =(ostream &os)

/*  Overloading of assignment operator that accepts ostream &. The
    previous ostream is flushed, then the log is linked to the new
    stream.
---------------------------------------------------------------------------*/
{
    if (mOStream != &os) {
	TLX_ASSERT_PTR(mOStream);
	mOStream->flush();
	mOStream = &os;
    }
    return *this;
}

/*-------------------------------------------------------------------------*/
    TLLog &TLLog::operator =(const TLLog &aLog)

/*  Overloading of assignment operator that will link to the same ostream
    as the other log. The previous log is flushed first.
---------------------------------------------------------------------------*/
{
    if (this != &aLog) {
	TLX_ASSERT_PTR(mOStream);
	mOStream->flush();
	mOStream = aLog.mOStream;
    }
    return *this;
}

/*-------------------------------------------------------------------------*/
    void __cdecl TLLog::PrintEntry(const char *aFmt, ...)

/*  Outputs a printf-style message on the log, preceded by a date/time
    stamp, and followed by a newline.
---------------------------------------------------------------------------*/
{
    va_list args;
    va_start(args, aFmt);

    BeginEntry();
    VPrintf(aFmt, args);
    EndEntry();

    va_end(args);
}

/*-------------------------------------------------------------------------*/
    ostream & __cdecl TLLog::Printf(const char *aFmt, ...) const

/*  Outputs a printf-style message on the log, preceded by a date/time
    stamp, and followed by a newline.
---------------------------------------------------------------------------*/
{
    va_list args;
    va_start(args, aFmt);
    VPrintf(aFmt, args);
    va_end(args);
    return Stream();
}

/*-------------------------------------------------------------------------*/
    int TLLog::SetEntryFormat(int aFmt)

/*  Sets a new entry format, while returning the previous one.
---------------------------------------------------------------------------*/
{
    int oldfmt 	 = mEntryFormat;
    mEntryFormat = aFmt;
    return oldfmt;
}

/*-------------------------------------------------------------------------*/
    void TLLog::VPrintf(const char *aFmt, va_list args) const

/*  Outputs a printf-style message on the log.
---------------------------------------------------------------------------*/
{
    static char _gBuffer[kMaxLogMessage];
    vsprintf(_gBuffer, aFmt, args);
    TLX_ASSERT(strlen(_gBuffer) < kMaxLogMessage);
    Stream() << _gBuffer;
}

ostream & _TLXFUNC operator <<(ostream &os, const TLLog &aLog)
{
    TLX_ASSERT(&os == &aLog.Stream());
    return aLog.EndEntry();
}
