/****************************************************************************
    $Id: dommonit.cpp 501.0 1995/03/07 12:26:14 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLDomainMonitor.

    $Log: dommonit.cpp $
    Revision 501.0  1995/03/07 12:26:14  RON
    Updated for TLX 5.01
    Revision 1.14  1995/02/28 15:11:36  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.13  1995/01/06  15:57:46  ron
    Corrected Revision keyword

    Revision 1.12  1994/11/16  15:39:14  ron
    Added module info; rearranged #include directives

    Revision 1.11  1994/10/12  10:04:10  ron
    Adapted to explicit TLDictionary<K,V> source code file
    Adapted to changes in dictionary iterator

    Revision 1.10  1994/10/11  19:03:11  ron
    Switched to a dictionary iterator instead of addressing the
    keys and values directly

    Revision 1.9  1994/10/10  16:50:05  ron
    Changed to <tlx\solve\csp.h>

    Revision 1.8  1994/10/07  17:02:08  ron
    Chnaged Unregister...() to Deregister...()

    Revision 1.7  1994/10/05  18:37:40  ron
    Added support for Watcom C++ templates

    Revision 1.6  1994/09/28  14:45:36  ron
    Removed Macintosh-style #include references

    Revision 1.5  1994/09/28  14:18:03  ron
    Renamed TLVarDomainMonitor to TLDomainMonitor

    Revision 1.4  1994/09/27  20:23:22  ron
    Changed path separator from / to \

    Revision 1.3  1994/09/26  15:50:21  ron
    Changed include file references

    Revision 1.2  1994/09/06  20:25:28  ron
    Renamed RegisterDomain() to CaptureDomain()

    Revision 1.1  1994/09/06  14:12:30  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

//----- System headers

#include <iostream.h>

//----- Project headers

#include <tlx\501\solve\csp.h>

/*---------------------------------------------------------------------------
    Template source code
---------------------------------------------------------------------------*/

#if defined (THINK_CPLUS)
    #include "diction.cpp"
    #include "seq.cpp"
    #pragma template_access public
    #pragma template TLDictionary<TLVariable *, TLDomain *>
    #pragma template TLSeq<TLVariable *>
    #pragma template TLSeq<TLDomain *>
#elif defined(__BORLANDC__) || defined(_MSC_VER) || defined(__SC__) || defined(__WATCOMC__)
    #include <tlx\501\template\diction.cpp>
    #include <tlx\501\template\seq.cpp>
#elif defined (__SUNPRO_CC)
    #include <tlx\501\template\diction.cpp>
    #include <tlx\501\template\seq.cpp>
#elif defined(__IBMCPP__)
  #if __IBMCPP__ < 300
    #pragma implementation("tlx\\template\\diction.cpp")
    #pragma implementation("tlx\\template\\seq.cpp")
  #else
    #include <tlx\501\template\diction.cpp>
    #include <tlx\501\template\seq.cpp>
  #endif
#else
    #error Unsupported compiler; contact Tarma Software Research
#endif

/*-------------------------------------------------------------------------*/
    TLDomainMonitor::TLDomainMonitor()

/*  Constructor. Initializes the instance.
---------------------------------------------------------------------------*/
: mDomains(50, 100)
{
    mPrevMonitor = 0;
}

/*-------------------------------------------------------------------------*/
    TLDomainMonitor::~TLDomainMonitor()

/*  Destructor. Ensures that the monitor is unregistered with the
    TLVariable class (if registered) and discards all domain copies.
---------------------------------------------------------------------------*/
{
    DeregisterMonitor();
    DiscardDomains();
}

/*-------------------------------------------------------------------------*/
    void TLDomainMonitor::CaptureDomain(TLVariable *var)

/*  Called to capture the domain of a variable, presumably because it is
    about to be modified. If there is currently no copy of the variable's
    domain stored in the mDomains dictionary, the variable is asked to
    produce a copy of its domain, which is then stored. If a domain copy
    is already stored, no further action is taken.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(var);

    if (!mDomains.Contains(var))
        mDomains.Insert(var, var->CopyDomain());
}

/*-------------------------------------------------------------------------*/
    void TLDomainMonitor::DiscardDomains()

/*  Deletes the domains of all variables whose domains have been captured
    by this monitor.
---------------------------------------------------------------------------*/
{
    for (TLDictIter<TLVariable*, TLDomain*> iter(mDomains); iter.Next(); )
    {
	delete iter.Peek();
	iter.Peek() = 0;
    }
    mDomains.RemoveAll();
}

/*-------------------------------------------------------------------------*/
    bool TLDomainMonitor::IsActive() const

/*  Returns true iff the current monitor is the active monitor, i.e. it
    is the monitor to which TLVariable::sDomainMonitor points.
---------------------------------------------------------------------------*/
{
    return this == TLVariable::sDomainMonitor;
}

/*-------------------------------------------------------------------------*/
    bool TLDomainMonitor::IsCapturing() const

/*  Returns true iff the current monitor is capturing, i.e. if it is part
    of the stack of domain monitors starting at TLVariable::sDomainMonitor.
    Note that this does not imply that the monitor does really capture
    any domains right now; use IsActiveMonitor() to check that.
---------------------------------------------------------------------------*/
{
    for (TLDomainMonitor *mon = TLVariable::sDomainMonitor; mon;
	 mon = mon->mPrevMonitor)
    {
	if (mon == this)
	    return true;
    }
    return false;
}

/*-------------------------------------------------------------------------*/
    void TLDomainMonitor::RestoreDomains()

/*  Restores the domains of all variables whose domains have been captured
    by this monitor.
---------------------------------------------------------------------------*/
{
    for (TLDictIter<TLVariable*, TLDomain*> iter(mDomains); iter.Next(); )
    {
	TLX_ASSERT_PTR(iter.PeekKey());
	iter.PeekKey()->RestoreDomain(iter.Peek());
    }
}

/*-------------------------------------------------------------------------*/
    void TLDomainMonitor::StartCapturing()

/*  Starts the capturing of changes to the domains of variables by
    registering the current monitor with the TLVariable class. It is
    an error to call this routine if the monitor is already capturing.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(!IsCapturing());
    mPrevMonitor = TLVariable::sDomainMonitor;	// Could be 0
    TLVariable::sDomainMonitor = this;
    TLX_ASSERT(IsCapturing());
}

/*-------------------------------------------------------------------------*/
    void TLDomainMonitor::StopCapturing()

/*  Stops the capturing of changes to the domains of variables by
    unregistering the current monitor with the TLVariable class. This
    routine may be called at any time while the monitor is capturing,
    regardless of whether or not it is the active monitor. However, it
    is an error to call this routine when the monitor is not capturing.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(IsCapturing());
    DeregisterMonitor();
    TLX_ASSERT(!IsCapturing());
}

/*-------------------------------------------------------------------------*/
    void TLDomainMonitor::DeregisterMonitor()

/*  Deregisters the current monitor by removing it from the monitor stack
    that is formed by TLVariable::sDomainMonitor and the mPrevMonitor
    fields in the monitors. It is *not* an error to call this routine
    if the current monitor is not capturing.
---------------------------------------------------------------------------*/
{
    if (this == TLVariable::sDomainMonitor)
    {
	// Topmost; just replace by previous monitor

	TLVariable::sDomainMonitor = mPrevMonitor;
	mPrevMonitor = 0;
    }
    else
    {
    	for (TLDomainMonitor *mon = TLVariable::sDomainMonitor; mon;
	     mon = mon->mPrevMonitor)
	{
	    if (mon->mPrevMonitor == this)
	    {
		mon->mPrevMonitor = mPrevMonitor;
		mPrevMonitor = 0;
		break;
	    }
	}
    }
}

