/****************************************************************************
    $Id: dlcbase.cpp 501.0 1995/03/07 12:26:12 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLDLCBase.

    $Log: dlcbase.cpp $
    Revision 501.0  1995/03/07 12:26:12  RON
    Updated for TLX 5.01
    Revision 1.7  1995/01/31 16:30:10  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.6  1995/01/06  15:57:37  ron
    Corrected Revision keyword

    Revision 1.5  1994/11/16  15:38:31  ron
    Added module info; rearranged #include directives

    Revision 1.4  1994/09/28  14:17:03  ron
    Removed Macintosh-style #include references

    Revision 1.3  1994/09/27  20:22:14  ron
    Changed path separator from / to \

    Revision 1.2  1994/09/26  15:41:42  ron
    Changed include file references

    Revision 1.1  1994/08/16  18:12:59  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <tlx\501\dlists.h>		// Class declaration
#include <tlx\501\except.h>		// Exception handling

/*-------------------------------------------------------------------------*/
    TLDLCBase::TLDLCBase(bool aOwner)

/*  Constructor, also doubles as default constructor. Creates an empty list.
---------------------------------------------------------------------------*/
: mHead(0), mCount(0), mOwner(aOwner)
{
}

/*-------------------------------------------------------------------------*/
    TLDLCBase::TLDLCBase(const TLDLCBase &)

/*  Copy constructor. Does not copy the list.
---------------------------------------------------------------------------*/
: mHead(0), mCount(0), mOwner(false)
{
}

/*-------------------------------------------------------------------------*/
    TLDLCBase::~TLDLCBase()

/*  Destructor. If the list is owner of its elements, they are deleted.
---------------------------------------------------------------------------*/
{
    if (mOwner) CleanUp();
}

/*-------------------------------------------------------------------------*/
    void TLDLCBase::Append(TLDLink *aLink)

/*  Adds an element at the end of the list. The element pointer must be
    nonzero, and the element must not currently be part of a list.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_NULL(aLink->mNext);
    TLX_ASSERT_NULL(aLink->mPrev);

    if (mHead)
	mHead->Prepend(aLink);
    else			// Create a single element list
    {
	TLX_ASSERT(mCount == 0);
	aLink->mNext = aLink;
	aLink->mPrev = aLink;
	mHead = aLink;
    }

    mCount++;
}

/*-------------------------------------------------------------------------*/
    void TLDLCBase::Append(TLDLink *aLink, TLDLink *aPos)

/*  Inserts a link after another link in the list. The link to insert must
    not be 0 and must not currently be part of a list.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_PTR(aPos);
    TLX_ASSERT_NULL(aLink->mNext);
    TLX_ASSERT_NULL(aLink->mPrev);
    TLX_ASSERT(Contains(aPos));

    aPos->Append(aLink);
    mCount++;
}

/*-------------------------------------------------------------------------*/
    bool TLDLCBase::BecomeOwner(bool aOwner)

/*  Changes the owner status, returning the previous status.
---------------------------------------------------------------------------*/
{
    bool oldstat = mOwner;
    mOwner = aOwner;
    return oldstat;
}

/*-------------------------------------------------------------------------*/
    void TLDLCBase::CleanUp()

/*  Deletes all elements in the list. The list should be owner for
    this function to be called.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(mOwner);
    if (!mHead) return;

    TLDLink *rover = mHead;
    do
    {
	TLX_ASSERT_PTR(rover);

	TLDLink *tmp = rover;
	rover = rover->Next();
	tmp->Unlink();
	delete tmp;
    }
    while (rover != mHead);
}

/*-------------------------------------------------------------------------*/
    bool TLDLCBase::Contains(TLDLink *aLink) const

/*  Tests whether a given element appears in the list, returning nonzero
    if it does.
---------------------------------------------------------------------------*/
{
    if (!aLink) return false;

    for (TLDLink *rover = mHead; rover; rover = rover->Next())
    {
	if (rover == aLink)
	    return true;
	if (rover->Next() == mHead)
	    break;
    }

    return false;
}

/*-------------------------------------------------------------------------*/
    void TLDLCBase::RemoveAll()

/*  Removes all elements from the list. If the list owns its items, they
    are deleted; else they are simply forgotten.
---------------------------------------------------------------------------*/
{
    if (mOwner) CleanUp();
    mHead  = 0;
    mCount = 0;
}

/*-------------------------------------------------------------------------*/
    TLDLink *TLDLCBase::Extract(TLDLink *aLink)

/*  Removes and returns a specific link from the list.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(mHead);
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT(Contains(aLink));

    if (aLink == mHead) mHead = aLink->Next();
    if (aLink == mHead) mHead = 0;	// List consisted of a single element

    aLink->Unlink();
    TLX_ASSERT(mCount > 0);
    mCount--;

    return aLink;
}

/*-------------------------------------------------------------------------*/
    TLDLink *TLDLCBase::ExtractHead()

/*  Removes and returns the first element of the list.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));

    return Extract(mHead);
}

/*-------------------------------------------------------------------------*/
    TLDLink *TLDLCBase::ExtractTail()

/*  Removes and returns the last element of the list.
---------------------------------------------------------------------------*/
{
    if (IsEmpty())
	THROW(TLXEmpty(LOCUS));

    TLX_ASSERT_PTR(mHead);
    TLX_ASSERT_PTR(mHead->mPrev);
    return Extract(mHead->mPrev);
}

/*-------------------------------------------------------------------------*/
    bool TLDLCBase::IsEmpty() const

/*  Checks if the list is currently empty, returning nonzero if it is.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(mCount > 0 || mHead == 0);
    return mCount == 0;
}

/*-------------------------------------------------------------------------*/
    TLDLCBase &TLDLCBase::operator =(const TLDLCBase &)

/*  Assignment operator, does not copy anything.
---------------------------------------------------------------------------*/
{
    return *this;
}

/*-------------------------------------------------------------------------*/
    void TLDLCBase::Prepend(TLDLink *aLink)

/*  Adds a new link at the start of the current list. The new link
    must not be 0 and must not be part of a list yet.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_NULL(aLink->mNext);
    TLX_ASSERT_NULL(aLink->mPrev);

    if (mHead)
	mHead->Prepend(aLink);
    else			// Make a single element list
    {
	aLink->mNext = aLink;
	aLink->mPrev = aLink;
    }

    mHead = aLink;
    mCount++;
}

/*-------------------------------------------------------------------------*/
    void TLDLCBase::Prepend(TLDLink *aLink, TLDLink *aPos)

/*  Inserts a link before another one. The link to insert must not be 0
    and must not be part of a list yet.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aLink);
    TLX_ASSERT_PTR(aPos);
    TLX_ASSERT_NULL(aLink->mNext);
    TLX_ASSERT_NULL(aLink->mPrev);
    TLX_ASSERT(Contains(aPos));

    aPos->Prepend(aLink);
    if (aPos == mHead) mHead = aLink;
    mCount++;
}

