/****************************************************************************
    $Id: bbsrchdf.cpp 501.0 1995/03/07 12:26:08 RON Exp $

    Copyright (c) 1994-95 Tarma Software Research. All rights reserved.

    Project:    Tarma Library for C++ V5.0
    Author:     Ron van der Wal

    Implementation of class TLBBSearcherDF, branch & bound searcher with
    depth-first searching only.

    $Log: bbsrchdf.cpp $
    Revision 501.0  1995/03/07 12:26:08  RON
    Updated for TLX 5.01
    Revision 1.19  1995/02/22 12:22:46  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.18  1995/01/16  12:24:10  ron
    Removed best gap statistics

    Revision 1.17  1995/01/13  15:29:25  ron
    Changed mBestProblem to mBestGap (due to change again)

    Revision 1.16  1995/01/12  13:37:32  ron
    Adapted to previous Searcher/ProblemRep model

    Revision 1.15  1995/01/10  16:31:29  ron
    Added ReportStats() and best lower bound memory

    Revision 1.14  1995/01/06  15:55:54  ron
    Adapted to new Searcher/Problem model

    Revision 1.13  1995/01/05  15:20:53  ron
    Added SetIncumbent()
    Formatting and naming changes

    Revision 1.12  1994/11/16  15:36:11  ron
    Added module info; rearranged #include directives

    Revision 1.11  1994/10/12  10:03:24  ron
    Switched to GetInfo() style of partial problem inclusion

    Revision 1.10  1994/10/11  19:01:46  ron
    Replaced DeadEnd() + SolveProblem() by ProcessProblem()

    Revision 1.9  1994/10/10  16:45:31  ron
    Added searcher description

    Revision 1.8  1994/10/05  18:34:02  ron
    Formatting changes

    Revision 1.7  1994/09/28  14:13:09  ron
    Added dead-end check to ContinueProblem()

    Revision 1.6  1994/09/27  20:22:00  ron
    Changed path separator from / to \

    Revision 1.5  1994/09/26  15:39:09  ron
    Changed include file references

    Revision 1.4  1994/09/13  10:11:26  ron
    Adapted to changes in the TLProblemRep hierarchy

    Revision 1.3  1994/09/12  14:53:07  ron
    Adapted ContinueProblem() implementation to changes in TLSearcher
    interface

    Revision 1.2  1994/09/06  14:07:56  ron
    Adapted to general changes in searcher framework

    Revision 1.1  1994/08/16  18:12:53  ron
    Initial revision

****************************************************************************/

#include <tlx\501\_build.h>

TLX_MODULE_INFO("$Revision: 501.0 $");

#include <float.h>
#include <tlx\501\log.h>
#include <tlx\501\solve\searcher.h>
#include <tlx\501\util.h>

/*-------------------------------------------------------------------------*/
    TLBBSearcherDF::TLBBSearcherDF()

/*  Default constructor. Initializes the searcher.
---------------------------------------------------------------------------*/
{
    mIncumbent = DBL_MAX;
}

/*-------------------------------------------------------------------------*/
    bool TLBBSearcherDF::ContinueProblem(TLProblemState *aProblem)

/*  Called to check if the current subproblem may be continued after it has
    been processed by the problem representation. In this implementation,
    we perform bounds checks etc. to determine if we must continue.

    The cut-off conditions are described in:

    T. Ibaraki: Enumerative Approaches to Combinatorial Optimization
                (Part I), pp. 88-94. JG Baltzer AG, 1988.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aProblem);
    TLX_ASSERT(aProblem->mIsProcessed);

    // The problem representation must have processed the subproblem already.
    // This must result in a calculation of the upperbound and the lowerbound
    // for the solution.

    TLBBInfo *info = DYNACAST(TLBBInfo *, aProblem->GetInfo());
    TLX_ASSERT_PTR(info);

    // Lower bound and dominance checks

    double lbound = info->LowerBound();

    if (lbound > mIncumbent || info->IsDominated())
        return false;

    // Check the upper bound to see if it improves on the current
    // best value. Because we store all optimal solutions, we
    // must distinguish between a strict improvement or a match of
    // the current best value.
    //
    // If the upperbound is less than infinity, it is assumed that a
    // feasible solution realising this upperbound is available; else
    // we assume that the upperbound was not calculated and that no
    // solution is available (the lowerbound and dominance are used,
    // though).

    double upbound = info->UpperBound();
    if (upbound <= mIncumbent && upbound < DBL_MAX)
    {
        if (upbound < mIncumbent)       // Strictly less?
        {
            ClearSolutions();
            mIncumbent = upbound;
        }

        // At least as good as previous solutions - store it

        CreateSolution(aProblem);
    }

    TLX_ASSERT(mIncumbent >= lbound);

    return true;
}

/*-------------------------------------------------------------------------*/
    bool TLBBSearcherDF::ContinueSearch()

/*  Called to determine whether or not the search process at large must be
    continued after finding a solution. This version returns true as long
    as not sufficient solutions are found and improvements in upper bound
    are possible.
---------------------------------------------------------------------------*/
{
    //return mBestGap > 0 && SolutionCount() < GetStats().mMaxSol;
    return true;
}

/*-------------------------------------------------------------------------*/
    const char *TLBBSearcherDF::Description() const

/*  Returns a description of the searcher.
---------------------------------------------------------------------------*/
{
    return "Depth-first branch & bound searcher";
}

/*-------------------------------------------------------------------------*/
    bool TLBBSearcherDF::PreProcess()

/*  Called to prepare for the search. It calls the inherited version, then
    discards any previous solutions and previous best value.
---------------------------------------------------------------------------*/
{
    if (!TLBTSearcher::PreProcess())
        return false;

    //mIncumbent = DBL_MAX;     // Do not initialize; may be set externally
    return true;
}

/*-------------------------------------------------------------------------*/
    void TLBBSearcherDF::ReportStats()

/*  Prints search statistics on stderr and the log output.
---------------------------------------------------------------------------*/
{
    TLSearcher::ReportStats();

    TLX_LOG_ENTRY("B&B: incumbent = %f", mIncumbent);
}

/*-------------------------------------------------------------------------*/
    void TLBBSearcherDF::SetIncumbent(double aIncumbent)

/*  Sets the incumbent value, possibly because the problem representation
    has some means to establish an initial lower bound before any
    subproblem is processed.
---------------------------------------------------------------------------*/
{
    if (aIncumbent > mIncumbent)
        TLX_TRACE_WARN(TLX, ("New incumbent %f greater than current %f",
                       aIncumbent, mIncumbent));
    mIncumbent = aIncumbent;
}

