/****************************************************************************
    $Id: stdcsp.h 501.0 1995/03/07 12:26:54 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for standard CSP classes. These classes build on the CSP
    base classes and provide templates and ordinary classes for common
    situations:

    TLNodeConOf<V>	- unary constraint on variables of type V
    TLBinConOf<V>	- binary constraint on variables of type V
    TLDomainOf<D>	- class template for domains of type D
    TLCSProblemState	- subproblem base class for CSPs
    TLCSProblemOf<V,L>	- subproblem template class for CSPs
    TLCSProblemRep	- C.S. problem representation with several strategies

    $Log: stdcsp.h $
    Revision 501.0  1995/03/07 12:26:54  RON
    Updated for TLX 5.01
    Revision 1.14  1995/02/22 12:40:00  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.13  1995/01/16  12:22:45  ron
    Added TLCSSubProblem parameter to varselectors

    Revision 1.12  1995/01/13  15:27:23  ron
    Added local variable processing to CSSubProblem
    Renamed ProcessVariable to Propagate in CSProblemRep

    Revision 1.11  1995/01/12  13:35:36  ron
    Adapted to previous Searcher/ProblemRep model

    Revision 1.10  1995/01/10  16:35:57  ron
    Small formatting change

    Revision 1.9  1995/01/08  11:53:05  ron
    Renamed TLCSProblem to TLCSSubProblem

    Revision 1.8  1995/01/06  16:03:49  ron
    Adapted to new Searcher/Problem model

    Revision 1.7  1995/01/05  15:38:26  ron
    Brought inline with thesis

    Revision 1.6  1994/11/16  15:32:27  ron
    Introduced TLVarSelector class

    Revision 1.5  1994/10/13  11:58:09  ron
    Renamed variable selection strategies
    Added |C|/(|D|^2) strategy
    Made strategies integers rather than enums

    Revision 1.4  1994/10/11  19:10:08  ron
    Replaced DeadEnd() by ProcessNode()

    Revision 1.3  1994/10/10  17:08:14  ron
    Renamed TLStdProblemRep to TLCSProblemRep
    Added several utility functions to TLCSProblemRep
    Added TLSearchWatchdog monitor class

    Revision 1.2  1994/10/07  17:11:58  ron
    Added TLSearchStats as a standard search monitor

    Revision 1.1  1994/10/06  17:53:10  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_STDCSP_H
#define _TLX_STDCSP_H

#ifndef _TLX_ITER_H
#include <tlx\501\iter.h>
#endif
#ifndef _TLX_CSP_H
#include <tlx\501\solve\csp.h>
#endif
#ifndef _TLX_SEARCHER_H
#include <tlx\501\solve\searcher.h>
#endif

class _TLXCLASS TLCSProblemRep;	// Below
class _TLXCLASS TLIntVar;	// Below

/*---------------------------------------------------------------------------
    TLNodeConOf<V> -

    Abstract base class for unary constraints that connect to variables of
    type V (V must be derived from TLVariable).
---------------------------------------------------------------------------*/

template<class V> class _TLXCLASS TLNodeConOf: public TLNodeCon
{
public:
    TLNodeConOf(V &);

    // The following functions hide the inherited ones for type safety

    V &			Var();
    const V &		Var() const;
};

/*---------------------------------------------------------------------------
    TLBinConOf<V> -

    Abstract base class template for binary constraints that connect
    variables of type V (V must be derived from TLVariable).
---------------------------------------------------------------------------*/

template<class V> class _TLXCLASS TLBinConOf: public TLBinCon
{
public:
    TLBinConOf(V &, V &);

    // The following functions hide the inherited ones for type safety

    V &			Var1();
    const V &		Var1() const;
    V &			Var2();
    const V &		Var2() const;
    V &			OppositeOf(V &);
    const V &		OppositeOf(const V &) const;
};

/*---------------------------------------------------------------------------
    TLDomainOf<D> -

    Class template for domains that contain a value of type D (D may be
    a collection of type L or something similar).
---------------------------------------------------------------------------*/

template<class D> class TLDomainOf: public TLDomain
{
    D			mValue;		// Domain value

public:
    // Constructor relies on copy constructor of type D; default destructor
    // relies on D's destructor.

    TLDomainOf(const D &);

    // Access to the domain value

    D &			Value();
    const D &		Value() const;

    // Implementation of output function relies on the existence of
    // operator <<(ostream &, const D &).

    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    TLCSProblemState -

    Subproblem class for use in CSPs. It contains a reference to a variable
    and a domain monitor to capture changes.
---------------------------------------------------------------------------*/

class TLCSProblemState: public TLProblemState
{
    TLVariable *	mVar;		// Variable being assigned
    TLDomainMonitor	mChanges;	// Changes to domains

public:
    virtual ~TLCSProblemState();

    // Access to the variable

    TLVariable *	Var();
    const TLVariable *	Var() const;

    // Overrideables for CSP subproblems:
    //
    // - AssignVariable(): the subproblem must perform the actual value
    //   assignment. Domain capturing etc. are taken care of externally.
    //
    // - ProcessVariable(): any processing that can be done locally should
    //   be done here. Domain capturing is active when the function is called.
    //
    // - DoneVariable(): called when processing on the variable is completed.
    //   Local restoration should be done here; global matters (e.g. domain
    //   restoration) is done externally.

    virtual void	AssignVariable() = 0;
    virtual bool	ProcessVariable() { return true; }
    virtual void	DoneVariable() {}

    // Helper functions to control the process of domain capturing:
    //
    // - SaveDomain() stores a copy of the domain of 'mVar'
    // - StartCapturing() starts capturing of other domains
    // - StopCapturing() stops capturing of other domains
    // - RestoreDomains() restores all captured domains (including 'mVar')

    void		SaveDomain();
    void		StartCapturing();
    void		StopCapturing();
    void		RestoreDomains();
    bool		IsCapturing() const;

protected:
    //TLCSProblemState(TLProblemState * = 0, TLVariable * = 0);
    TLCSProblemState(TLVariable * = 0);
};

/*---------------------------------------------------------------------------
    TLCSProblemStateOf<V,L> -

    Class template for CSP subproblems of variables of type V and values
    of type L. Type V must be TLVariable or derived thereof.
---------------------------------------------------------------------------*/

template<class V, class L> class TLCSProblemStateOf: public TLCSProblemState
{
    L			mValue;		// Value of the variable

public:
    TLCSProblemStateOf();
    //TLCSProblemStateOf(TLProblemState *, V *, const L &);
    TLCSProblemStateOf(V *, const L &);

    // Type-safe access to the variable (hides inherited version), and
    // access to its associated value.

    V *			Var();
    const V *		Var() const;
    L &			Value();
    const L &		Value() const;

    // Implementation of output function relies on operator <<(ostream &,
    // const L &).

    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    TLVarSelector -

    Variable selection strategy class. Its purpose is to select the next
    eligible variable from the remaining free variables.
---------------------------------------------------------------------------*/

class _TLXCLASS TLVarSelector
{
public:
    virtual ~TLVarSelector();

    virtual const char *Description() const = 0;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &) = 0;
};

//---------------------------------------------------------------------------
//  TLVSRandom: random selection strategy
//---------------------------------------------------------------------------

class _TLXCLASS TLVSRandom: public TLVarSelector
{
public:
    TLVSRandom();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

//---------------------------------------------------------------------------
//  TLVSSmallestDomain: selects variable with smallest domain
//---------------------------------------------------------------------------

class _TLXCLASS TLVSSmallestDomain: public TLVarSelector
{
public:
    TLVSSmallestDomain();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

//---------------------------------------------------------------------------
//  TLVSLargestDomain: selects variable with largest domain
//---------------------------------------------------------------------------

class _TLXCLASS TLVSLargestDomain: public TLVarSelector
{
public:
    TLVSLargestDomain();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

//---------------------------------------------------------------------------
//  TLVSMostConstrained: selects variable with most constraints
//---------------------------------------------------------------------------

class _TLXCLASS TLVSMostConstrained: public TLVarSelector
{
public:
    TLVSMostConstrained();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

//---------------------------------------------------------------------------
//  TLVSLeastConstrained: selects variable with least constraints
//---------------------------------------------------------------------------

class _TLXCLASS TLVSLeastConstrained: public TLVarSelector
{
public:
    TLVSLeastConstrained();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

//---------------------------------------------------------------------------
//  TLVSDomConMix: selects variable with smallest domain, most constraints
//---------------------------------------------------------------------------

class _TLXCLASS TLVSDomConMix: public TLVarSelector
{
public:
    TLVSDomConMix();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

/*---------------------------------------------------------------------------
    TLCSProblemRep -

    ProblemRep incorporating several commonly used selection strategies. The
    problem representation must be used with variable classes that are
    derived from TLCSProblemState.
---------------------------------------------------------------------------*/

class _TLXCLASS TLCSProblemRep: public TLProblemRep
{
    // Strategies are stored as pointers to strategy objects:

    TLPropagator *	mPropagator;	// Constraint propagation strategy
    TLVarSelector *	mVarSelector;	// Variable selection strategy

protected:
    TLVarList 		mVars;		// All variables
    TLVarList		mFreeVars;	// Free (i.e. unassigned) variables

public:
    // Strategy setting and inquiry. Changes are in effect from the next
    // opportunity onwards.

    TLPropagator *	GetPropagator() const;
    TLPropagator *	SetPropagator(TLPropagator *);
    TLVarSelector *	GetVarSelector() const;
    TLVarSelector *	SetVarSelector(TLVarSelector *);

    // Overridden versions of global problem representation functions

    virtual size_t 	Size() const;
    virtual bool  	PreProcess();
    virtual ostream &	PrintOn(ostream &) const;

    // Overridden versions of subproblem functions

    virtual void	EnterState(TLProblemState *);
    virtual bool	Process(TLProblemState *);
    virtual void	LeaveState(TLProblemState *);
    virtual size_t 	GenerateBranches(TLProblemState *, TLProblemList &);

protected:
    TLCSProblemRep();

    // Function to create the set of variables. They should be stored in
    // the mVars data member. Afterwards, they are all preprocessed.

    virtual bool	CreateVariables() = 0;
    virtual bool	PreProcessVariables();

    // Problem decomposition is divided into two steps: variable selection
    // and branch generation. This problem representation provides standard
    // strategies for variable selection, but they can be overridden. There
    // is no implementation for branch generation, so that should be provided
    // by a derived class.
    //
    // Note: SelectVariable() should *not* remove the variable from the
    // list of free variables (this should be done by Process()).

    virtual TLVariable *SelectVariable(TLCSProblemState *);
    virtual size_t	CreateBranches(TLCSProblemState *, TLVariable *,
				       TLProblemList &) = 0;

    // Subproblem processing is split into several steps:
    //
    // 1. Variable assignment, delegated to the subproblem
    // 2. Variable processing, delegated to the subproblem
    // 3. Constraint propagation, performed by the function below
    //
    // The function below uses the installed propagation strategy object,
    // but may be overridden in derived classes.

    virtual bool	Propagate(TLCSProblemState *);
};

/*---------------------------------------------------------------------------
    TLSolutionOf<V,L> -
---------------------------------------------------------------------------*/

template<class V, class L> class _TLXCLASS TLSolutionOf: public TLSolution
{
    TLPtrSeq<TLIntVar> 	mVars;		// Ordered list of variables
    TLSeq<int>		mValues;	// Parallel list of values

public:
    Solution(TLProblemState *);			// Creates entire solution

    // PrintOn() is overridden in order to print the solution

    virtual ostream &PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    TLCSPMonitor -

    Search monitor that collects statistics about the constraint
    propagation process. The following data are captured:

    - total, average, minimum, and maximum constraint checks
    - average, minimum, and maximum constraint checks at each search depth
---------------------------------------------------------------------------*/

class _TLXCLASS TLCSPMonitor: public TLSearchMonitor
{
    // Statistics that are kept by this monitor are all one-dimensional

    typedef TLStats1<int32> IntStats;

    IntStats		mChecks;	// Overall checking statistics
    TLArray<IntStats>	mLevelChecks;	// Checking stats per level
    int32		mPrevCount;	// Previous # of checks

public:
    TLCSPMonitor();

    // Overridden search monitor behavior.

    virtual int		DefaultEvents() const;
    virtual void	OnSearchEvent(TLSearcher *, const Event &);
    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    Inline functions
---------------------------------------------------------------------------*/

//-----	TLNodeConOf<V>

template<class V> TLNodeConOf<V>::TLNodeConOf(V &aVar)
    : TLNodeCon(aVar) {}

template<class V> inline V &TLNodeConOf<V>::Var()
    { return (V &)TLNodeCon::Var(); }

template<class V> inline const V &TLNodeConOf<V>::Var() const
    { return (const V &)TLNodeCon::Var(); }

//-----	TLBinConOf<V>

template<class V> TLBinConOf<V>::TLBinConOf(V &aVar1, V &aVar2)
    : TLBinCon(aVar1, aVar2) {}

template<class V> inline V &TLBinConOf<V>::Var1()
    { return (V &)TLBinCon::Var1(); }

template<class V> inline const V &TLBinConOf<V>::Var1() const
    { return (const V &)TLBinCon::Var1(); }

template<class V> inline V &TLBinConOf<V>::Var2()
    { return (V &)TLBinCon::Var2(); }

template<class V> inline const V &TLBinConOf<V>::Var2() const
    { return (const V &)TLBinCon::Var2(); }

template<class V> inline V &TLBinConOf<V>::OppositeOf(V &aVar)
    { return (V &)TLBinCon::OppositeOf(aVar); }

template<class V> inline
const V &TLBinConOf<V>::OppositeOf(const V &aVar) const
    { return (const V &)TLBinCon::OppositeOf(aVar); }

//-----	TLDomainOf<L>

template<class D> TLDomainOf<D>::TLDomainOf(const D &aValue)
    : mValue(aValue) {}

template<class D> inline D &TLDomainOf<D>::Value()
    { return mValue; }

template<class D> inline const D &TLDomainOf<D>::Value() const
    { return mValue; }

template<class D> ostream &TLDomainOf<D>::PrintOn(ostream &os) const
    { return os << mValue; }

//----- TLCSProblemState

inline TLVariable *TLCSProblemState::Var()
    { return mVar; }

inline const TLVariable *TLCSProblemState::Var() const
    { return mVar; }

inline void TLCSProblemState::SaveDomain()
    { mChanges.CaptureDomain(mVar); }

inline void TLCSProblemState::StartCapturing()
    { mChanges.StartCapturing(); }

inline void TLCSProblemState::StopCapturing()
    { mChanges.StopCapturing(); }

inline void TLCSProblemState::RestoreDomains()
    { mChanges.RestoreDomains(); }

inline bool TLCSProblemState::IsCapturing() const
    { return mChanges.IsCapturing(); }

//-----	TLCSProblemStateOf<V,L>

template<class V, class L>
TLCSProblemStateOf<V,L>::TLCSProblemStateOf()
    : TLCSProblemState() {}

template<class V, class L>
TLCSProblemStateOf<V,L>::TLCSProblemStateOf(
    //TLProblemState *	aParent,
    V *			aVar,
    const L &		aValue)
    //: TLCSProblemState(aParent, aVar), mValue(aValue) {}
    : TLCSProblemState(aVar), mValue(aValue) {}

template<class V, class L> inline V *TLCSProblemStateOf<V,L>::Var()
    { return (V *)TLCSProblemState::Var(); }

template<class V, class L> inline const V *TLCSProblemStateOf<V,L>::Var() const
    { return (const V *)TLCSProblemState::Var(); }

template<class V, class L> inline L &TLCSProblemStateOf<V,L>::Value()
    { return mValue; }

template<class V, class L> inline const L &TLCSProblemStateOf<V,L>::Value() const
    { return mValue; }

template<class V, class L>
ostream &TLCSProblemStateOf<V,L>::PrintOn(ostream &os) const
    { return os << *Var() << "=" << mValue; }

//-----	TLCSProblemRep

inline size_t TLCSProblemRep::Size() const
    { return mVars.Count(); }

inline TLPropagator *TLCSProblemRep::GetPropagator() const
    { return mPropagator; }

inline TLVarSelector *TLCSProblemRep::GetVarSelector() const
    { return mVarSelector; }

#endif	// _TLX_STDCSP_H
