/****************************************************************************
    $Id: searcher.h 501.0 1995/03/07 12:26:54 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for classes related to all forms of enumerative and
    branch & bound searching:

    TLSolution		- Generic solution wrapper
    TLProblemInfo	- Additional information about a subproblem
    TLBBInfo		- Problem info for branch & bound search
    TLProblemState	- Generic subproblem in the search tree
    TLProblemList	- Ordered list of subproblems
    TLProblemRep	- Generic problem representation interface
    TLSearchMonitor	- Base class for search monitors
    TLSearchStats	- Search monitors that collects statistics
    TLSearchWatchdog	- Search monitors that terminates after node count
    TLSearchBreaker	- Ctrl-Break handler for searchers
    TLSearcher		- Generic enumerative searcher
    TLBTSearcher	- Backtrack searcher algorithm (non-recursive)
    TLBTSearcherRcsv	- Backtrack searcher algorithm (recursive)
    TLBBSearcher	- General branch & bound searcher
    TLBBSearcherDF	- Depth-first branch & bound searcher

    $Log: searcher.h $
    Revision 501.0  1995/03/07 12:26:54  RON
    Updated for TLX 5.01
    Revision 1.16  1995/02/22 12:21:50  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.15  1995/01/18  10:36:05  ron
    Added TLSubProblem parameter to TLSearchMonitor::Event

    Revision 1.14  1995/01/16  12:22:14  ron
    Removed best gap; added GetIncumbent()

    Revision 1.13  1995/01/13  15:26:45  ron
    Added ContinueSearch() to BTSearcher and derived classes

    Revision 1.12  1995/01/12  13:34:56  ron
    Restored previous Searcher/ProblemRep design

    Revision 1.11  1995/01/10  16:35:27  ron
    Added ReportStats() and best lower bound to TLBBSearcherDF

    Revision 1.10  1995/01/08  11:52:21  ron
    Changed searcher depth type from int32 to int
    Renamed TLCSProblem to TLCSSubProblem

    Revision 1.9  1995/01/06  16:03:20  ron
    Adapted to new Searcher/Problem model

    Revision 1.8  1995/01/05  15:37:48  ron
    Brought inline with thesis

    Revision 1.7  1994/11/16  15:31:40  ron
    Added optional 'bool' variant to TLSearchMonitor::Event

    Revision 1.6  1994/10/13  11:57:16  ron
    Moved standard search monitors to this file
    Added AddFront() and AddRear() synonyms to TLProblemList

    Revision 1.5  1994/10/12  10:08:57  ron
    Added TLProblemInfo as an optional extension of the TLSubProblem
    interface
    Adapted TLBBInfo to the introduction of TLProblemInfo

    Revision 1.4  1994/10/11  19:09:19  ron
    Replaced DeadEnd()/SolveProblem() by ProcessProblem()
    Removed specialized B&B problem representation
    Removed several TLBTSearcher::Frame member functions

    Revision 1.3  1994/10/10  17:07:07  ron
    Moved searcher events to TLSearcher
    Added Terminate() function to the searcher interface
    Added Description() function to the searcher interface

    Revision 1.2  1994/10/07  17:11:33  ron
    Added search monitors and their support

    Revision 1.1  1994/10/06  17:53:00  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_SEARCHER_H
#define _TLX_SEARCHER_H

//-----	System headers

#include <time.h>

//-----	Project headers

#ifndef _TLX_ARRAYS_H
#include <tlx\501\arrays.h>
#endif
#ifndef _TLX_PTRARRAY_H
#include <tlx\501\ptrarray.h>
#endif
#ifndef _TLX_REFCNT_H
#include <tlx\501\refcnt.h>
#endif
#ifndef _TLX_SLISTS_H
#include <tlx\501\slists.h>
#endif
#ifndef _TLX_STATS_H
#include <tlx\501\stats.h>
#endif

class _RTLCLASS ostream;	// In iostream.h
class _TLXCLASS TLVariable;	// In solve\csp.h
class _TLXCLASS TLSearcher;	// Below

/*---------------------------------------------------------------------------
    TLSolution -

    Abstract base class representing a solution in the search process.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSolution: public TLRefCount
{
public:
    // Only required functionality is the ability to print itself.

    virtual ostream &	PrintOn(ostream &) const = 0;
    friend ostream & 	_TLXFUNC operator <<(ostream &, const TLSolution &);
};

typedef TLPtrSeq<TLSolution> TLSolutionSet;

/*---------------------------------------------------------------------------
    TLProblemInfo -

    Base class for classes that supply additional information about a
    subproblem. They are used as part of a TLProblemState, to introduce
    flexibility with respect to the interface composition mechanism:
    (multiple) inheritance, data member, or external data.
---------------------------------------------------------------------------*/

class _TLXCLASS TLProblemInfo
{
public:
    virtual ~TLProblemInfo();
};

/*---------------------------------------------------------------------------
    TLBBInfo -

    Subproblem information for use in branch & bound algorithms.
---------------------------------------------------------------------------*/

class _TLXCLASS TLBBInfo: public TLProblemInfo
{
    // Implementation hints: you may either store the values for the upper
    // and lower bound on first constructing an instance of the solution,
    // or you may have them recalculated each time one of the corresponding
    // functions is called. In the latter case, it may be necessary to
    // store a pointer or reference to the problem representation to which
    // this solution belongs.
public:
    // Derived classes must override and implement the following functions
    // that are used to direct the B & B searching process. The functions
    // are called after the problem representation has solved the subproblem.
    //
    // - IsDominated() must return nonzero if the problem is dominated
    //   by another one in the search tree.
    //
    // - UpperBound() must return the value of the upper bound for this
    //   problem. If the solution is exact, it is also the value of the
    //   solution.
    //
    // - LowerBound() must return the value of the lower bound for this
    //   problem if it's feasible.

    virtual double	UpperBound() = 0;
    virtual double	LowerBound() = 0;
    virtual bool 	IsDominated() { return false; }
};

/*---------------------------------------------------------------------------
    TLProblemState -

    Represents the information that describes a subproblem in the search
    tree. It is an abstract base class which should be derived from to
    create problem-specific subproblems.
---------------------------------------------------------------------------*/

class _TLXCLASS TLProblemState: public TLRefCount
{
  #ifdef _TLXDBG
public:
    bool		mIsProcessed;	// Set by Searcher if debugging
  #endif

public:
    // For extensibility, we define a function that returns a pointer to
    // additional node information. The default implementation has none,
    // but for other searching purposes (e.g. branch and bound), the
    // additional information must be added.

    virtual TLProblemInfo *GetInfo();

    // Derived classes may override GetRank() if they implement some sort
    // of best-first searching. The higher the rank value, the better the
    // node is considered to be.

    virtual double	GetRank() const;

    // PrintOn() should print some suitably formatted description of the
    // contents of current value of the selection; it is used for tracing
    // the execution of the search algorithm.

    virtual ostream &	PrintOn(ostream &) const = 0;
    friend ostream & 	_TLXFUNC operator <<(ostream &, const TLProblemState &);

protected:
    TLProblemState();
};

/*---------------------------------------------------------------------------
    TLProblemList -

    Specialized ordered list of TLProblemState instances that is used to keep
    track of active nodes during the search processes.
---------------------------------------------------------------------------*/

class _TLXCLASS TLProblemList: public TLPtrSeq<TLProblemState>
{
public:
    // Constructor sets the proper comparison function and makes the list
    // owner of the nodes stored in it.

    TLProblemList(TLProblemState *);
    TLProblemList(size_t, size_t);

    // The public interface of this class is very limited, since it is
    // exposed to problem-specific problem representation instances. They
    // should use one of the following functions to add new nodes to the list
    // when asked to expand an existing node.
    //
    // - AddDepthFirst() causes the search tree to be traversed depth-first,
    //   exploring the most recently added node first;
    //
    // - AddBreadthFirst() causes a breadth-first search, exploring the most
    //   recently added node last;
    //
    // - AddBestFirst() causes a best-first search, where nodes with higher
    //   rank values are considered better and are explored before nodes with
    //   lower rank values.
    //
    // AddFront() and AddRear() are synonyms for AddDepthFirst() and
    // AddBreadthFirst().

    void		AddDepthFirst(TLProblemState *);
    void		AddBreadthFirst(TLProblemState *);
    void		AddBestFirst(TLProblemState *);
    void		AddFront(TLProblemState *);
    void		AddRear(TLProblemState *);

    // Functions to be used to retrieve the next eligible subproblem.
    // The inherited IsEmpty() function should be used to check for
    // emptiness first.

    TLProblemState *	PeekFirst() const;
    TLProblemState *	ExtractFirst();

private:
    static int		CompareProblems(const void *, const void *);
};

/*---------------------------------------------------------------------------
    TLProblemRep -

    Manages the problem-specific data structures that are used during the
    various search processes. TLProblemRep is the abstract base class for
    more specialized problem representations that are fitted to different
    problem types.
---------------------------------------------------------------------------*/

class _TLXCLASS TLProblemRep
{
    friend class _TLXCLASS TLSearcher;

    TLSearcher *	mSearcher;	// Currently active searcher

public:
    virtual ~TLProblemRep();

    // Access to the current searcher (if any)

    TLSearcher *	GetSearcher() const;

    // Overridables that have a default implementation; derived classes need
    // only override them if they need different behavior. In those cases,
    // the original version should be called as part of the overridden
    // implementation.
    //
    // - PreProcess() is called prior to the search process and must be
    //   used to prepare the TLProblemRep for the search process.
    //
    // - PostProcess() is called when the search process is completed and
    //   may be used to do some kind of clean up.

    virtual bool  	PreProcess();
    virtual void 	PostProcess();

    // Overridables that must be implemented by derived classes. They are
    // used by all searchers as part of the searcher protocol.
    //
    // - CreateInitialProblem() must create the initial set of problems to
    //   explore (which may be a single node). The default implementation
    //   calls Decompose() with a 0 argument for the problem to decompose.
    //
    // - Decompose() should create the subproblems of the given problem.
    //   The preferred order of exploration is determined by the order in
    //   which they are inserted in the problem list.
    //
    // - Process() should accept a previously created subproblem and
    //   perform the associated value assignment. It should store sufficient
    //   information in the subproblem to undo the assignment later on. It
    //   should then perform whatever further processing is necessary.
    //
    // - DoneProcessing() is called with a processed subproblem when
    //   all processing by the searcher has been finished. It is the place
    //   to restore the effects of all processing since StartProcessing()
    //   was called on the subproblem.
    //
    // - CreateSolution() is called when a complete solution has been found,
    //   and should create and return a solution object that describes the
    //   solution.
    //
    // - Size() must return the "size" of the problem, i.e. the
    //   number of variables involved. This number must remain the same
    //   throughout the execution of the search process.

    virtual size_t 	CreateInitialProblem(TLProblemList &);
    virtual size_t 	GenerateBranches(TLProblemState *, TLProblemList &) = 0;
    virtual void  	EnterState(TLProblemState *) = 0;
    virtual bool  	Process(TLProblemState *) = 0;
    virtual void 	LeaveState(TLProblemState *);
    virtual TLSolution *CreateSolution(TLProblemState *) = 0;
    virtual size_t 	Size() const = 0;

    // Output functions:
    //
    // - ReportStats() is called to report on any statistics the problem
    //   representation might have collected;
    //
    // - PrintOn() is used to produce a print of the problem representation
    //   state on an output stream.

    virtual void	ReportStats(ostream &) const {}
    virtual ostream &	PrintOn(ostream &) const = 0;
    friend ostream & 	_TLXFUNC operator <<(ostream &, const TLProblemRep &);

protected:
    TLProblemRep();
};

/*---------------------------------------------------------------------------
    TLSearchMonitor -

    Base for classes that monitor the progress of a search, e.g. to capture
    statistics or to give an graphical representation of the search process.
    After registering a search monitor class with a searcher, the monitor
    will be informed of the relevant events within the search process.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSearchMonitor
{
public:
    // Events are sent to a search monitor in an Event structure. This is
    // done for efficiency and flexibility in the searcher class. It does
    // require the search monitor to decode the event, however.
    //
    // Events store basic information associated with them; for more details,
    // the search monitor should call back to the searcher that sent the
    // event.

    struct _TLXCLASS Event
    {
	int		mCode;		// Event code (see TLSearcher)
	TLProblemState *	mProblem;	// Subproblem for the event (optional)
	union
	{
	    bool	mBoolParm;	// (PreProcess, Processing)
	    int16	mInt16Parm;	// (reserved)
	    int32	mInt32Parm;	// (DoneProcess)
	    uint16	mUint16Parm;	// (reserved)
	    uint32	mUint32Parm;	// (Initial, Decompose, Solution)
	    double	mDoubleParm;	// (reserved)
	};

	Event(int, TLProblemState * = 0);
      #ifndef _NOBOOL
	Event(int, bool, TLProblemState * = 0);
      #endif
	Event(int, int16, TLProblemState * = 0);
	Event(int, int32, TLProblemState * = 0);
	Event(int, uint16, TLProblemState * = 0);
	Event(int, uint32, TLProblemState * = 0);
	Event(int, double, TLProblemState * = 0);
    };

public:
    virtual ~TLSearchMonitor();

    // The following function is called when an event occurs for which the
    // monitor has been registered with a searcher.

    virtual void	OnSearchEvent(TLSearcher *, const Event &);
    virtual int		DefaultEvents() const;

    // Output operators. PrintOn() is called by the searcher as part of its
    // own ReportStats() processing.

    virtual ostream &	PrintOn(ostream &) const;
    friend ostream &	_TLXFUNC operator <<(ostream &,
    					     const TLSearchMonitor &);
};

/*---------------------------------------------------------------------------
    TLSearchStats -

    Search monitor that collects statistics about the searching process.
    The following data are captured:

    - average, minimum, and maximum search depth
    - average, minimum, and maximum branching factor
    - average, minimum, and maximum branching factor at each search depth
---------------------------------------------------------------------------*/

class _TLXCLASS TLSearchStats: public TLSearchMonitor
{
    // Statistics that are kept by this monitor are all one-dimensional

    typedef TLStats1<int32> IntStats;

    IntStats		mDepth;		// Search depth statistics
    IntStats		mBranching;	// Overall branching statistics
    TLArray<IntStats>	mLevelBranch;	// Branching stats per level

public:
    TLSearchStats();

    // Overridden search monitor behavior.

    virtual int		DefaultEvents() const;
    virtual void	OnSearchEvent(TLSearcher *, const Event &);
    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    TLBBStats -

    Search monitor that collects statistics about a branch and bound
    searching process. The following data are captured:

    - average, minimum, and maximum lowerbounds during the search
    - average, minimum, and maximum lowerbounds at each search depth
---------------------------------------------------------------------------*/

class _TLXCLASS TLBBStats: public TLSearchMonitor
{
    // Statistics that are kept by this monitor are all one-dimensional

    typedef TLStats1<double> DoubleStats;

    DoubleStats		mLBounds;	// Overall lowerbound statistics
    TLArray<DoubleStats>  mLBDepth;	// Per-depth lowerbounds statistics

public:
    TLBBStats();

    // Overridden search monitor behavior.

    virtual int		DefaultEvents() const;
    virtual void	OnSearchEvent(TLSearcher *, const Event &);
    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    TLSearchWatchdog -

    Search monitor that terminates the search process if too many problems
    have been explored.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSearchWatchdog: public TLSearchMonitor
{
    int32		mProblemCount;	// Number of problems so far
    int32		mMaxProblems;	// Maximum allowed number of problems

public:
    TLSearchWatchdog(int32);

    // Overridden search monitor behavior.

    virtual int		DefaultEvents() const;
    virtual void	OnSearchEvent(TLSearcher *, const Event &);
    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    TLSearchBreaker -

    Search monitor that terminates the search process if Ctrl-Break or
    Ctrl-C have been detected.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSearchBreaker: public TLSearchMonitor
{
    static bool		sBreak;		// true if break detected
    static bool		sNotified;	// true if user has been notified
    static int		sBCount;	// Number of search breakers

public:
    TLSearchBreaker();
    virtual ~TLSearchBreaker();

    // Overridden search monitor behavior.

    virtual int		DefaultEvents() const;
    virtual void	OnSearchEvent(TLSearcher *, const Event &);
    virtual ostream &	PrintOn(ostream &) const;

private:
    // Signal handler for Ctrl-Break and Ctrl-C

    static void __cdecl BreakHandler(int);
};

/*---------------------------------------------------------------------------
    TLSearcher -

    Base class for all searcher classes. It is an abstract class, so users
    should instatiate one of its derivations.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSearcher
{
    friend class _TLXCLASS TLProblemRep;

public:
    // The following interface defines the events that search monitors will
    // receive. They may filter one or more events if they are not
    // interested in them at all; this will improve search performance.
    //
    // Classes derived from TLSearcher may add their own event codes.

    enum
    {
	seNone		= 0x0000,	// No events
	sePreProcess	= 0x0001,	// Preprocessing started
	sePostProcess	= 0x0002,	// Postprocessing started
	seInitial	= 0x0004,	// Initial problem(s) created
	seBranch	= 0x0008,	// Problem decomposed
	seEnter		= 0x0010,	// Problem state entered
	seProcess	= 0x0020,	// Problem processed
	seLeave		= 0x0040,       // Problem state left
	seSolution	= 0x0080,	// Solution found
	seAll		= 0xffff	// All events
    };

    // We keep a number of statistics about the search process that is
    // going on. They are used by this base class for report generation,
    // and should be filled in by derived classes as the search progresses.

    struct _TLXCLASS Stats
    {
    	size_t 		mMaxSol;	// Maximum number of solutions to find
    	uint32 	mBTCount;	// Number of backtracks performed
    	uint32		mProblemCount;	// Number of nodes visited
	int		mDepth;		// Current search depth
    	time_t 		mStart;		// Start time of solver
    	time_t 		mStartSearch;	// Start time of the search process
    	time_t		mPrevious;	// Time of the previous solution
    	time_t		mFinishSearch;	// Finish time of the search process
    	time_t		mFinish;	// Finish time of solver
    };

private:
    TLSearcher *	mPrevSearcher;	// Previous searcher if we're nested
    TLProblemRep *	mProblemRep;	// ProblemRep for the problem
    TLSolutionSet	mSolutions;	// Solutions
    Stats		mStats;		// Statistics

    // References to search monitors are stored along with their event mask

    struct _TLXCLASS MonitorEntry
    {
	TLSearchMonitor *mMonitor;	// The monitor
	int		mEventMask;	// Its event mask

	MonitorEntry();
	MonitorEntry(TLSearchMonitor *, int);
    };

    // Data members to keep track of the active monitors. We also store
    // a global event mask that is the union of all individual masks, so
    // there is a cheap check whether or not to broadcast an event.

    TLSeq<MonitorEntry>	mMonitors;	// Registered monitors
    int			mGlobalMask;	// Global event mask
    bool		mMonEnabled;	// Must be true to send events at all

public:
    TLSearcher();
    virtual ~TLSearcher();

    // - Solve() performs the actual search process, finding up to a given
    //   maximum number of solutions. It is really a wrapper that calls the
    //   algorithm-specific member functions of its derived classes.
    //
    // - Terminate() can be called during the search process to terminate it
    //   right away.

    size_t 		Solve(TLProblemRep *, size_t = 1);
    virtual void	Terminate() = 0;

    // Information about the search process. The solutions are bound to
    // change during the search.

    virtual const char *Description() const = 0;

    size_t		SolutionCount() const;
    const TLSolutionSet&GetSolutions() const;

    const Stats &	GetStats() const;
    int			GetLevel() const;

    // Functions to register and unregister a monitor, and to indicate the
    // events it will receive.

    void		RegisterMonitor(TLSearchMonitor *);
    void		DeregisterMonitor(TLSearchMonitor *);
    void		SetMonitorEvents(TLSearchMonitor *, int);
    bool		EnableMonitors(bool = true);

    // Access to the current problem representation

    TLProblemRep *	GetProblemRep() const;

protected:
    // The following interface must be implemented by derived classes.
    // The non-pure virtual functions have a reasonable default
    // implementation and need not always be overridden.

    virtual bool 	PreProcess();
    virtual void	PostProcess();
    virtual void	Search() = 0;

    // The following functions should be called to report the solution and
    // search statistics. They may be overridden if desired.

    virtual void 	ReportProgress();
    virtual void 	ReportSolutions();
    virtual void 	ReportStats();

    // The following functions mirror the problem representation interface.
    // They should be called by derived searcher classes rather than
    // addressing the problem representation directly, since this allows the
    // searcher to inform its monitors of the pertinent events.

    size_t 		CreateInitialProblem(TLProblemList &);
    size_t		GenerateBranches(TLProblemState *, TLProblemList &);
    void		EnterState(TLProblemState *);
    bool		Process(TLProblemState *);
    void		LeaveState(TLProblemState *, bool = true);
    void		CreateSolution(TLProblemState *);

    // The following functions manage the storage of solutions.

    void		AddSolution(TLSolution *);
    void		ClearSolutions();

    // Functions to register (with) an problem representation

    void		RegisterProblemRep(TLProblemRep *);
    void		DeregisterProblemRep();

private:
    // Implementation helper functions

    bool		MustSend(int);
    void 		SendEvent(const TLSearchMonitor::Event &);
    void		UpdateEventMask();
};

/*---------------------------------------------------------------------------
    TLBTSearcherRcsv -

    Searcher class that implements a recursive backtrack search algorithm.
---------------------------------------------------------------------------*/

class _TLXCLASS TLBTSearcherRcsv: public TLSearcher
{
    bool		mTerminate;	// Termination flag

public:
    TLBTSearcherRcsv();

    // Overridden version of the searcher functions

    virtual const char *Description() const;
    virtual void	Terminate();

protected:
    // Overridden version of the inherited search interface.

    virtual void	Search();

private:
    // Recursive searching step

    bool 		StepForward(TLProblemList &);
};

/*---------------------------------------------------------------------------
    TLBTSearcher -

    Non-recursive backtrack search algorithm.
---------------------------------------------------------------------------*/

class _TLXCLASS TLBTSearcher: public TLSearcher
{
protected:
    // The non-recursive search algorithm maintains an explicit stack to
    // keep track of the search progress. The stack consists of a linked
    // list of search frames.

    class _TLXCLASS Frame: public TLSLink
    {
	friend class _TLXCLASS TLBTSearcher;
	friend class _TLXCLASS TLBBSearcherDF;

	TLProblemList	mBranches;	// Branches at this level
	index_t		mCurProblem;	// Index of current problem

    private:
	Frame();
	virtual ~Frame();

	// Access to parent frame (if any)

	Frame *		GetParent() const;

	// Subproblem iteration functions

	TLProblemState *CurrentProblem() const;
	bool		NextProblem();
	void		PrevProblem();
	void		ResetProblem();
	size_t		ProblemCount() const;
    };

    enum {		// Searcher actions
	actStop,   	// Stop search
	actExpand,	// Expand the current branch
	actExplore,	// Explore the next branch of the current frame
	actBacktrack,	// Leave the current branch
	actBackjump,	// Backjump to a previous state
	actRestart	// Restart from a previous state
    };

    TLSLStack<Frame>	mStack;		// Search stack
    int			mNextAction;	// Next searcher action
    const Frame *	mMark;		// Target of backjump
    bool 		mUndoAll;	// Undo mode of backjump

public:
    // We allow our clients to mark a position during the search and
    // jump back to it later on.

    typedef const Frame *tMark;		// Marker type

public:
    TLBTSearcher();
    virtual ~TLBTSearcher();

    // Overridden versions of searcher functions

    virtual const char *Description() const;
    virtual void	Terminate();

    // This version of the backtrack searcher allows a different mode of
    // operation:
    //
    // 	   StartSearch(problem);
    //	   while (NextSolution())
    //	   	PeekSolution()...
    //	   StopSearch();

    bool		StartSearch(TLProblemRep *);
    bool		NextSolution();
    void		StopSearch();
    TLSolution *	PeekSolution() const;

    // Functions to influence the searcher's behavior. They may be called
    // at any time. BackTrack() causes the searcher to fail the current
    // variable/value assignment and continue with the next one; Terminate()
    // terminates the search process altogether.

    tMark		GetMark() const;
    void		BackTrack();

    // Functions to jump back to previous search levels. The back jump
    // functions have an optional parameter that indicates whether all
    // intermediate levels must be undone individually, or whether it
    // suffices to undo only the last level before the target level. true
    // indicates a complete undo of all levels.

    void		BackToLevel(int, bool = true);
    void		BackToMark(tMark, bool = true);
    void		BackToProblem(TLProblemState *, bool = true);
    void		BackToVariable(TLVariable *, bool = true);

    // Functions to restart the search at a previous level. They are similar
    // to the backjump functions, but they start node expansion all over
    // again at the target node, instead of continuing with its next branch.

    void		RestartAtLevel(int, bool = true);
    void		RestartAtMark(tMark, bool = true);
    void		RestartAtProblem(TLProblemState *, bool = true);
    void		RestartAtVariable(TLVariable *, bool = true);

protected:
    // Overridden versions of the inherited search interface.

    virtual bool 	PreProcess();
    virtual void	PostProcess();
    virtual void	Search();

    // The following functions may be overridden to modify 'fail' behavior.

    virtual bool        ContinueProblem(TLProblemState *);
    virtual bool	ContinueSearch();

private:
    // Implementation helper functions

    void		Push(Frame *);
    Frame *		Pop();
    Frame *		FindLevel(int);
    Frame *		FindProblem(TLProblemState *);
    Frame *		FindVariable(TLVariable *);
    Frame *		UnwindToMark();
};

/*---------------------------------------------------------------------------
    TLBBSearcher -

    Branch & bound searching class. This class supports all variations of
    search order: best-first, depth-first, breadth-first, or any
    combination thereof. It must be used in conjunction with a
    TLBBProblemRep-derived class.
---------------------------------------------------------------------------*/

class _TLXCLASS TLBBSearcher: public TLSearcher
{
    double		mIncumbent;	// Incumbent value
    bool		mTerminate;	// Termination flag

public:
    TLBBSearcher();

    // Overridden version of searcher functions

    virtual const char *Description() const;
    virtual void	Terminate();

    // Function to set the incumbent value, possibly to establish a good
    // lower bound before the search process proper starts.

    void		SetIncumbent(double);

protected:
    // Overridden versions of the inherited search interface.

    virtual bool 	PreProcess();
    virtual void	Search();
};

/*---------------------------------------------------------------------------
    TLBBSearcherDF -

    Branch & bound searching class supporting only depth-first searches.
---------------------------------------------------------------------------*/

class _TLXCLASS TLBBSearcherDF: public TLBTSearcher
{
    double		mIncumbent;	// Incumbent value

public:
    TLBBSearcherDF();

    // Overridden versions of searcher functions

    virtual const char *Description() const;

    // Function to set the incumbent value, possibly to estabish a good
    // lower bound before the search process proper starts.

    double		GetIncumbent() const;
    void		SetIncumbent(double);

protected:
    // Overridden versions of the inherited search interface.

    virtual bool 	PreProcess();
    virtual void 	ReportStats();

    // Overridden version of continuation functions - performs bounds checks

    virtual bool        ContinueProblem(TLProblemState *);
    virtual bool	ContinueSearch();
};

/*---------------------------------------------------------------------------
    Inline functions
---------------------------------------------------------------------------*/

//----- TLSolution

inline ostream & _TLXFUNC operator <<(ostream &os, const TLSolution &s)
    { return s.PrintOn(os); }

//----- TLProblemState

//inline TLProblemState *TLProblemState::GetParent() const
//    { return mParent; }

inline ostream & _TLXFUNC operator <<(ostream &os, const TLProblemState &s)
    { return s.PrintOn(os); }

//-----	TLProblemList

inline void TLProblemList::AddFront(TLProblemState *aProblem)
    { TLPtrSeq<TLProblemState>::Append(aProblem); }

inline void TLProblemList::AddRear(TLProblemState *aProblem)
    { TLPtrSeq<TLProblemState>::Prepend(aProblem); }

inline void TLProblemList::AddDepthFirst(TLProblemState *aProblem)
    { AddFront(aProblem); }

inline void TLProblemList::AddBreadthFirst(TLProblemState *aProblem)
    { AddRear(aProblem); }

inline void TLProblemList::AddBestFirst(TLProblemState *aProblem)
    { TLPtrSeq<TLProblemState>::Insert(aProblem); }

inline TLProblemState *TLProblemList::PeekFirst() const
    { return TLPtrSeq<TLProblemState>::PeekLast(); }

inline TLProblemState *TLProblemList::ExtractFirst()
    { return TLPtrSeq<TLProblemState>::ExtractLast(); }

//-----	TLProblemRep

inline TLSearcher *TLProblemRep::GetSearcher() const
    { return mSearcher; }

inline ostream & _TLXFUNC operator <<(ostream &os, const TLProblemRep &a)
    { return a.PrintOn(os); }

//-----	TLSearchMonitor

inline TLSearchMonitor::Event::Event(int aCode, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem) {}

#ifndef _NOBOOL
inline
TLSearchMonitor::Event::Event(int aCode, bool aP, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem), mBoolParm(aP) {}
#endif

inline
TLSearchMonitor::Event::Event(int aCode, int16 aP, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem), mInt16Parm(aP) {}

inline
TLSearchMonitor::Event::Event(int aCode, int32 aP, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem), mInt32Parm(aP) {}

inline
TLSearchMonitor::Event::Event(int aCode, uint16 aP, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem), mUint16Parm(aP) {}

inline
TLSearchMonitor::Event::Event(int aCode, uint32 aP, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem), mUint32Parm(aP) {}

inline
TLSearchMonitor::Event::Event(int aCode, double aP, TLProblemState *aProblem)
    : mCode(aCode), mProblem(aProblem), mDoubleParm(aP) {}

inline int TLSearchMonitor::DefaultEvents() const
    { return TLSearcher::seAll; }

inline ostream & _TLXFUNC operator <<(ostream &os, const TLSearchMonitor &m)
    { return m.PrintOn(os); }

//-----	TLSearcher

inline TLSearcher::MonitorEntry::MonitorEntry()
    : mMonitor(0), mEventMask(0) {}

inline TLSearcher::MonitorEntry::MonitorEntry(TLSearchMonitor *aMon, int aMask)
    : mMonitor(aMon), mEventMask(aMask) {}

inline const TLSearcher::Stats &TLSearcher::GetStats() const
    { return mStats; }

inline size_t TLSearcher::SolutionCount() const
    { return mSolutions.Count(); }

inline const TLSolutionSet &TLSearcher::GetSolutions() const
    { return mSolutions; }

inline TLProblemRep *TLSearcher::GetProblemRep() const
    { return mProblemRep; }

inline int TLSearcher::GetLevel() const
    { return mStats.mDepth; }

inline bool TLSearcher::MustSend(int aEvent)
    { return mMonEnabled && ((mGlobalMask & aEvent) == aEvent); }

//-----	TLBTSearcher

inline TLBTSearcher::Frame *TLBTSearcher::Frame::GetParent() const
    { return (Frame *)Next(); }

inline size_t TLBTSearcher::Frame::ProblemCount() const
    { return mBranches.Count(); }

//-----	TLBBSearcherDF

inline double TLBBSearcherDF::GetIncumbent() const
    { return mIncumbent; }

#endif	// _TLX_SEARCHER_H
