/****************************************************************************
    $Id: ac6.h 501.0 1995/03/07 12:26:52 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for classes involved in the implementation of the AC-6
    constraint propagation algorithm:

    VV_Assoc		- Variable/value association
    TLVarValueLink 	- Ditto, for use in linked lists
    TLSupportList 	- List of variable/value links
    TLSupportPtr	- Reference-counting pointer to the above
    TLDomainElement 	- Domain value + supported list
    TLDomainRemoval 	- Describes removal of a domain element
    TLWaitingList	- Queue of removals
    TLDomainAC6 	- Domain for AC-6 variables
    TLConstraintAC6	- Base class for AC-6 constraints
    TLVariableAC6	- Base class for AC-6 variables
    TLPropagatorAC6 	- AC-6 constraint propagator

    $Log: ac6.h $
    Revision 501.0  1995/03/07 12:26:52  RON
    Updated for TLX 5.01
    Revision 1.2  1995/01/31 16:29:04  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1994/10/06  17:52:31  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_AC6_H
#define _TLX_AC6_H

//-----	System headers

#include <stdarg.h>

//-----	Project headers

#ifndef _TLX_CSP_H
#include <tlx\501\solve\csp.h>
#endif
#ifndef _TLX_REFCNT_H
#include <tlx\501\refcnt.h>
#endif
#ifndef _TLX_SLISTS_H
#include <tlx\501\slists.h>
#endif

class _TLXCLASS TLVariableAC6;
class _TLXCLASS TLConstraintAC6;

/*---------------------------------------------------------------------------
    Auxiliary declarations and definitions
---------------------------------------------------------------------------*/

typedef int32  		tVarValue;	// Type for variable values

/*---------------------------------------------------------------------------
    Helper classes:

    - VV_Assoc is an association between TLVariableAC6 * and tVarValue that
      is used to store these pairs in various situations in the
      implementation of the AC-6 algorithm.
---------------------------------------------------------------------------*/

class _TLXCLASS VV_Assoc: public TLAssoc<TLVariableAC6 *, tVarValue>
{
public:
    VV_Assoc();
    VV_Assoc(TLVariableAC6 *, tVarValue);

    friend ostream & 	_TLXFUNC operator <<(ostream &, const VV_Assoc &);
};

/*---------------------------------------------------------------------------
    TLVarValueLink -

    Association between a variable and a value that is also a link field
    for a singly-linked list. Instances of this class are used to create
    support lists for use in the AC-6 algorithm.
---------------------------------------------------------------------------*/

class _TLXCLASS TLVarValueLink: public TLSLink, public VV_Assoc
{
public:
    TLVarValueLink(const VV_Assoc &);

    TLVarValueLink &	operator =(const VV_Assoc &);
    int			operator ==(const VV_Assoc &) const;

    TLVariableAC6 *     PeekVar() const { return Key(); }
    tVarValue		PeekValue() const { return Value(); }
};

/*---------------------------------------------------------------------------
    TLSupportList -

    Mix-in class that is both a TLRefCount (for reference counting) and a
    TLSList<TLVarValueLink> (to store a list of variable/value pairs).
    This class is used as the head of a support list; it needs reference
    counting because it must be passed on between domain elements without
    being physically copied or accidentally discarded.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSupportList: public TLRefCount, public TLSList<TLVarValueLink>
{
public:
    // Only a default constructor, which invokes the default constructors
    // of its base classes.

    TLSupportList();

    // All other functionality is inherited and not overridden.
};

// TLSupportPtr is an alias for a reference-counting pointer to the above

typedef TLCountedPtr<TLSupportList> TLSupportPtr;

/*---------------------------------------------------------------------------
    TLDomainElement -

    This class represents a domain element in the domain of variables used
    in the AC-6 algorithm. Each element contains the value proper, plus
    (the head of) a list that indicates which values in other variables
    depend on support from this value.

    The value part of this class is a reference counting pointer to support
    lists; the reference counting is used when domain elements are
    removed from a domain (TLDomainAC6) and placed on a waiting list
    (TLWaitingList).
---------------------------------------------------------------------------*/

class _TLXCLASS TLDomainElement: public TLAssoc<tVarValue, TLSupportPtr>
{
public:
    TLDomainElement(tVarValue = 0);

    TLDomainElement &	operator =(const TLDomainElement &);

    tVarValue		PeekValue() const { return Key(); }
    TLSupportPtr &	SupportList() { return Value(); }
    const TLSupportPtr &SupportList() const { return Value(); }

    // Functions to add and remove variable/value links from the support
    // list. ClearSupport() deletes the entire support list.

    void		AddSupport(const VV_Assoc &);
    void		RemoveSupport(const VV_Assoc &);
    void		ClearSupport();

    // Iteration through the support list of the domain element. You should
    // *not* try to modify the list during the iteration without resetting
    // the iterator, or havoc may result.

    bool 		ResetIter(iter_t &);
    bool 		NextIter(iter_t &);
    const TLVarValueLink *PeekIter(const iter_t &) const;
};

/*---------------------------------------------------------------------------
    TLDomainRemoval -

    Association between a domain element and the variable that contained
    it before it was removed.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDomainRemoval: public TLAssoc<TLVariableAC6 *,
						TLDomainElement>
{
public:
    TLDomainRemoval();
    TLDomainRemoval(TLVariableAC6 *, const TLDomainElement &);
};

// TLWaitingList is an alias for a queue of the above

typedef TLQueue<TLDomainRemoval> TLWaitingList;

/*---------------------------------------------------------------------------
    TLDomainAC6 -

    Represents the domain of variables that are used in the AC-6 algorithm.
    It is a sequence of TLDomainElement instances with some added
    functionality. The domain installs a comparison function that compares
    (and hence sorts) its elements by their element value.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDomainAC6: public TLSeq<TLDomainElement>
{
public:
    TLDomainAC6(size_t, size_t = 0);

    // Functions to find indices of elements

    index_t		IndexOfValue(tVarValue) const;
    index_t		EqualOrNextIndexOf(tVarValue) const;

    // Functions to find the element that stores a particular value

    TLDomainElement &	GetElement(tVarValue);
    const TLDomainElement &GetElement(tVarValue) const;

private:
    static int		DECompare(const TLDomainElement &,
    				  const TLDomainElement &);
};

/*---------------------------------------------------------------------------
    TLConstraintAC6 -

    Base class for constraints to be used in AC-6 constraint propagation
    algorithms. Constraints of this type are binary constraints that assume
    both their variables to be of type TLVariableAC6 or derived thereof.
---------------------------------------------------------------------------*/

class _TLXCLASS TLConstraintAC6: public TLBinCon
{
public:
    TLConstraintAC6(TLVariableAC6 &, TLVariableAC6 &);

    // Functions that hide TLBinCon member functions of the same name
    // in order to get some type safety.

    TLVariableAC6 &	Var1();
    const TLVariableAC6 &Var1() const;
    TLVariableAC6 &	Var2();
    const TLVariableAC6 &Var2() const;

    TLVariableAC6 &	OppositeOf(TLVariableAC6 &);
    const TLVariableAC6 &OppositeOf(const TLVariableAC6 &) const;

    // Functions to implement constraint-dependent support procedures.
    //
    // - FindNextSupport() attempts to find the next value in one variable
    //   that supports a particular value in the other variable. It returns
    //   nonzero if it succeeds, else zero.

    virtual bool        FindNextSupport(TLVariableAC6 &, tVarValue,
    					TLVariableAC6 &, tVarValue &) = 0;
};

/*---------------------------------------------------------------------------
    TLVariableAC6 -

    Variable class specifically for AC-6 constraint propagation. Variables
    of this class have domains that consist of tVarValue values.
---------------------------------------------------------------------------*/

class _TLXCLASS TLVariableAC6: public TLVariable
{
protected:
    // The domain of the variable should consist of a list of integral
    // values. It should be ordered by increasing value. This is accomplished
    // automatically by using mDomain.InsertSorted() or AddValue() to place
    // new values in the domain.

    TLDomainAC6		mDomain;	// Current domain of the variable

public:
    // Constructor allows creation of domain with a given size. Derived
    // classes may also resize the domain explicitly at a later stage.

    TLVariableAC6(const char *, size_t = 0);

    // Functions that hide TLVariable member functions of the same name
    // in order to get some type safety.

    void 		AddConstraint(TLConstraintAC6 *aCon)
			    { TLVariable::AddConstraint(aCon); }
    void 		RemoveConstraint(TLConstraintAC6 *aCon)
			    { TLVariable::RemoveConstraint(aCon); }
    TLConstraintAC6 *	FindConstraintTo(const TLVariableAC6 *) const;

    // Functions to add and remove values to/from the domain

    void		AddValue(tVarValue);
    void		RemoveValue(tVarValue);
    TLDomainAC6 &	Domain() { return mDomain; }
    const TLDomainAC6 &	Domain() const { return mDomain; }

    // Functions to manipulate the support lists in the domain

    void		AddSupport(tVarValue, const VV_Assoc &);
    void		RemoveSupport(tVarValue, const VV_Assoc &);
    void		ClearSupport();

    // The following functions implement the pure virtual functions inherited
    // from the base class. They use TLDomainAC6 instances to store and restore
    // the variable's domain.

    virtual TLDomain *	SaveDomain();
    virtual void 	RestoreDomain(const TLDomain *);
    virtual void 	DiscardDomain(TLDomain *);
};

/*---------------------------------------------------------------------------
    TLPropagatorAC6 -

    Class that implements the AC-6 constraint propagation algorithm.
---------------------------------------------------------------------------*/

class _TLXCLASS TLPropagatorAC6: public TLPropagator
{
public:
    TLPropagatorAC6();

    // Consistency checking and constraint propagation algorithms:
    //
    // - CheckAC6() propagates the effects of the constraints that
    //   originate with the given set of variables.

    bool 		CheckAC6(const TLPtrSeq<TLVariable> &);

private:
    bool  		InitSupport(TLConstraintAC6 *, TLVariableAC6 *,
				    TLVariableAC6 *, TLWaitingList &);
};

#endif	// _TLX_AC6_H
