/****************************************************************************
    $Id: slists.h 501.0 1995/03/07 12:26:46 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    All lists in this file are singly linked lists, based on TLSLink
    elements. The file also contains templates for lists based on classes
    derived from class TLSLink.

    TLSLink		- element in singly linked lists
    TLSLBase		- linear list of Links
    TLSLCBase		- circular list of Links
    TLSLQueueBase 	- queue based on a linked list
    TLSLStackBase 	- stack based on a linked list

    TLSList<T>    	- template for TLSLBase
    TLSLCList<T>  	- template for TLSLCBase
    TLSLQueue<T>  	- template for TLSLQueueBase
    TLSLStack<T>  	- template for TLSLStackBase

    $Log: slists.h $
    Revision 501.0  1995/03/07 12:26:46  RON
    Updated for TLX 5.01
    Revision 1.7  1995/01/31 16:29:24  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.6  1994/10/05  18:23:53  ron
    Added access adjustment for BecomeOwner in TLSLStack

    Revision 1.5  1994/09/28  14:28:41  ron
    Removed Macintosh-style #include references

    Revision 1.4  1994/09/27  20:25:39  ron
    Changed path separator from / to \

    Revision 1.3  1994/09/26  15:22:04  ron
    Changed include file references

    Revision 1.2  1994/09/06  13:59:00  ron
    Adapted to changes in tlx.h

    Revision 1.1  1994/08/16  18:06:52  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_SLISTS_H
#define _TLX_SLISTS_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

/*---------------------------------------------------------------------------
    TLSLink -

    Base class for all classes that may be stored in a singly linked list.
    It contains a forward pointer.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSLink
{
    friend class _TLXCLASS TLSLBase;
    friend class _TLXCLASS TLSLCBase;

    TLSLink *		mNext;		// Forward pointer

public:
    virtual ~TLSLink();

    // Functions to traverse through lists of links.

    TLSLink *		Next() const { return mNext; }
    bool 		IsLast() const { return mNext == 0; }

protected:
    // Constructors and destructor are only for use by derived classes;
    // copy constructor does *not* copy the link fields.

    TLSLink();
    TLSLink(const TLSLink &);

    // Assignment operator does *not* copy the link fields.

    TLSLink &		operator =(const TLSLink &);

private:
    // Helper functions for use by list classes.

    void		Append(TLSLink *);
    void		Unlink();
};

/*---------------------------------------------------------------------------
    TLSLBase -

    Manages linear singly linked lists based on TLSLinks. It serves as a base
    class for other list classes.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSLBase
{
    TLSLink *		mHead;		// First element in the list
    TLSLink *		mTail;		// Last element in the list
    size_t		mCount;		// Number of elements

    // If the list manager is owner of the elements stored in the list,
    // the elements will be deleted when the list manager itself is
    // destroyed. The owner status may be changed by everyone.

    bool 		mOwner;		// If nonzero, elements are discarded

public:
    // Constructor creates an empty list and sets owner status. Destructor
    // optionally deletes elements.

    TLSLBase(bool = false);
    virtual ~TLSLBase();

    bool 		IsOwner() const { return mOwner; }
    bool 		BecomeOwner(bool = true);

    // Operations that insert or remove elements or sublists.

    void		Append(TLSLink *);
    void		Append(TLSLink *, TLSLink *);
    void		RemoveAll();
    void		Concat(TLSLink *);
    void		Concat(TLSLBase &);
    void		Prepend(TLSLink *);
    void		Prepend(TLSLink *, TLSLink *);
    TLSLink *		Extract(TLSLink *);
    TLSLink *		ExtractHead();
    TLSLink *		ExtractTail();
    TLSLink *		Replace(TLSLink *, TLSLink *);
    TLSLink *		Split(TLSLink *);

    TLSLink *		PeekHead() const { return mHead; }
    TLSLink *		PeekTail() const { return mTail; }
    size_t		Count() const { return mCount; }
    bool 		IsEmpty() const;
    bool 		Contains(TLSLink *) const;

private:
    // Copying and assignment are prohibited.

    TLSLBase(const TLSLBase &);
    TLSLBase &		operator =(const TLSLBase &);

    // Implementation helper function.

    void		CleanUp();
};

/*---------------------------------------------------------------------------
    TLSLCBase -

    Manages a circular linked list consisting of TLSLink elements.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSLCBase
{
    // In the circular list, we only store a pointer to the *last* element.

    TLSLink *		mTail;
    size_t		mCount;

    // If the list manager is owner of the elements stored in the list,
    // the elements will be deleted when the list manager itself is
    // destroyed. The owner status may be changed by everyone.

    bool 		mOwner;

public:
    // Constructor creates an empty list and sets owner status. Destructor
    // optionally deletes elements.

    TLSLCBase(bool = false);
    virtual ~TLSLCBase();

    bool 		IsOwner() const { return mOwner; }
    bool 		BecomeOwner(bool = true);

    // Operations that insert or remove elements or sublists.

    void		Append(TLSLink *);
    void		Append(TLSLink *, TLSLink *);
    void		Prepend(TLSLink *);
    void		Prepend(TLSLink *, TLSLink *);
    TLSLink *		Extract(TLSLink *);
    TLSLink *		ExtractHead();
    TLSLink *		ExtractTail();
    void		RemoveAll();

    TLSLink *		PeekHead() const { return mTail ? mTail->Next() : 0; }
    TLSLink *		PeekTail() const { return mTail; }
    size_t		Count() const { return mCount; }
    bool 		IsEmpty() const;
    bool 		Contains(TLSLink *) const;

private:
    // Copying and assignment are prohibited.

    TLSLCBase(const TLSLCBase &);
    TLSLCBase &		operator =(const TLSLCBase &);

    // Implementation helper function.

    void		CleanUp();
};

/*---------------------------------------------------------------------------
    TLSLQueueBase -

    Represents a queue based on a singly-linked list. TLSLQueueBase is
    an intrusive queue.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSLQueueBase: private TLSLBase
{
public:
    TLSLQueueBase(bool = false);

    void		Enqueue(TLSLink *aLink) { Append(aLink); }
    TLSLink *		Dequeue() { return ExtractHead(); }

    TLSLBase::IsOwner;
    TLSLBase::RemoveAll;
    TLSLBase::Count;
    TLSLBase::IsEmpty;
    TLSLBase::PeekHead;
    TLSLBase::PeekTail;
};

/*---------------------------------------------------------------------------
    TLSLStackBase -

    Represents a stack based on a singly-linked list. TLSLStackBase is
    an intrusive stack.
---------------------------------------------------------------------------*/

class _TLXCLASS TLSLStackBase: private TLSLBase
{
public:
    TLSLStackBase(bool = false);

    void		Push(TLSLink *aLink) { Prepend(aLink); }
    TLSLink *		Pop() { return ExtractHead(); }
    TLSLink *		PeekTop() const { return PeekHead(); }

    TLSLBase::BecomeOwner;
    TLSLBase::IsOwner;
    TLSLBase::RemoveAll;
    TLSLBase::Count;
    TLSLBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLSList<T> -

    This class is a template for a type-safe derivation of TLSLBase for
    classes T that are derived from TLSLink. TLSList<T> is an intrusive
    list for items of type T.
---------------------------------------------------------------------------*/

template<class T> class TLSList: private TLSLBase
{
public:
    TLSList(bool = false);

    TLSList<T> &	operator =(const TLSList<T> &) { return  *this; }

    void		Append(T *aLink) { TLSLBase::Append(aLink); }
    void		Append(T *aLink, T *aPos)
			    { TLSLBase::Append(aLink, aPos); }
    void		Concat(T *aLink) { TLSLBase::Concat(aLink); }
    void		Concat(TLSList<T> &aList) { TLSLBase::Concat(aList); }
    void		Prepend(T *aLink) { TLSLBase::Prepend(aLink); }
    void		Prepend(T *aLink, T *aPos)
			    { TLSLBase::Prepend(aLink, aPos); }
    T *			ExtractHead() { return (T *) TLSLBase::ExtractHead(); }
    T *			ExtractTail() { return (T *) TLSLBase::ExtractTail(); }
    T *			Extract(T *aLink)
			    { return (T *) TLSLBase::Extract(aLink); }
    T *			Replace(T *aPos, T *aLink)
			    { return (T *) TLSLBase::Replace(aPos, aLink); }
    T *			Split(T *aLink) { return (T *) TLSLBase::Split(aLink); }

    T *			PeekHead() const { return (T *) TLSLBase::PeekHead(); }
    T *			PeekTail() const { return (T *) TLSLBase::PeekTail(); }
    bool 		Contains(T *t) const { return TLSLBase::Contains(t); }

    TLSLBase::BecomeOwner;
    TLSLBase::IsOwner;
    TLSLBase::RemoveAll;
    TLSLBase::Count;
    TLSLBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLSLCList<T> -

    A template for a type-safe derivation of TLSLCBase for classes T that
    are derived from TLSLink. TLSLCList<T> is an intrusive list for items of
    type T.
---------------------------------------------------------------------------*/

template<class T> class TLSLCList: private TLSLCBase
{
public:
    TLSLCList(bool = false);

    TLSLCList<T> &	operator =(const TLSLCList<T> &) { return *this; }

    void		Append(T *aLink) { TLSLCBase::Append(aLink); }
    void		Append(T *aLink, T *aPos)
			    { TLSLCBase::Append(aLink, aPos); }
    void		Prepend(T *aLink) { TLSLCBase::Prepend(aLink); }
    void		Prepend(T *aLink, T *aPos)
			    { TLSLCBase::Prepend(aLink, aPos); }
    T *			ExtractHead() { return (T *) TLSLCBase::ExtractHead(); }
    T *			ExtractTail() { return (T *) TLSLCBase::ExtractTail(); }
    T *			Extract(T *aLink)
			    { return (T *) TLSLCBase::Extract(aLink); }

    T *			PeekHead() const { return (T *) TLSLCBase::PeekHead(); }
    T *			PeekTail() const { return (T *) TLSLCBase::PeekTail(); }

    TLSLCBase::BecomeOwner;
    TLSLCBase::IsOwner;
    TLSLCBase::RemoveAll;
    TLSLCBase::Count;
    TLSLCBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLSLQueue<T> -

    A template for a type-safe derivation of TLSLQueueBase for classes T that
    are derived from TLSLink. TLSLQueue<T> is an intrusive queue for items of
    type T.
---------------------------------------------------------------------------*/

template<class T> class TLSLQueue: private TLSLQueueBase
{
public:
    TLSLQueue(bool = false);

    void		Enqueue(T *aLink) { TLSLQueueBase::Enqueue(aLink); }
    T *			Dequeue() { return (T *) TLSLQueueBase::Dequeue(); }

    T *			PeekHead() const
    			    { return (T *) TLSLQueueBase::PeekHead(); }
    T *			PeekTail() const
    			    { return (T *) TLSLQueueBase::PeekTail(); }

    TLSLQueueBase::BecomeOwner;
    TLSLQueueBase::IsOwner;
    TLSLQueueBase::RemoveAll;
    TLSLQueueBase::Count;
    TLSLQueueBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLSLStack<T> -

    A template for a type-safe derivation of TLSLStackBase for classes T that
    are derived from TLSLink. TLSLStack<T> is an intrusive stack for items of
    type T.
---------------------------------------------------------------------------*/

template<class T> class TLSLStack: private TLSLStackBase
{
public:
    TLSLStack(bool = false);

    void		Push(T *aLink) { TLSLStackBase::Push(aLink); }
    T *			Pop() { return (T *) TLSLStackBase::Pop(); }
    T *			PeekTop() const
    			    { return (T *) TLSLStackBase::PeekTop(); }

    TLSLStackBase::BecomeOwner;
    TLSLStackBase::IsOwner;
    TLSLStackBase::RemoveAll;
    TLSLStackBase::Count;
    TLSLStackBase::IsEmpty;
};

#endif	// _TLX_SLISTS_H
