/****************************************************************************
    $Id: select.cpp 501.0 1995/03/07 12:25:56 RON Exp $

    Copyright (c) 1995 Tarma Software Research. All rights reserved.

    Project:    Tarma Library for C++ V5.0
    Author:     Ron van der Wal
    Created:    09-01-95 11:39

    Implementation of the variable selection strategies in the CALMA example.

    $Log: select.cpp $
    Revision 501.0  1995/03/07 12:25:56  RON
    Updated for TLX 5.01
    Revision 1.2  1995/02/28 18:36:18  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/02/22 12:22:42  RON
    Initial revision
    Revision 1.2  1995/01/17  17:12:24  ron
    Added CLMaxConPenaltySelector

    Revision 1.1  1995/01/16  12:29:16  ron
    Initial revision

****************************************************************************/

//----- System headers

#include <float.h>
#include <iostream.h>

//----- Library headers

#include <tlx\501\debug.h>
#include <tlx\501\log.h>
#include <tlx\501\rtti.h>
#include <tlx\501\util.h>

//----- Project headers

#include "calma.h"

/*---------------------------------------------------------------------------
    CLMaxMinPenaltySelector
---------------------------------------------------------------------------*/

CLMaxMinPenaltySelector::CLMaxMinPenaltySelector()
//
//  Constructor. Provided for completeness only.
//
{
}

const char *CLMaxMinPenaltySelector::Description() const
//
//  Returns a description of the selection strategy.
//
{
    return "Largest remaining minimum penalty";
}

TLVariable *CLMaxMinPenaltySelector::SelectVariable(
    TLCSProblemState *,
    TLCSProblemRep *,
    const TLVarList &aList)
//
//  Selects a variable by picking the first one with the largest minimum
//  penalty.
//
{
    TLX_ASSERT(!aList.IsEmpty());

    CLVariable *bestvar = 0;
    int32 maxpenalty   = -1;

    for (index_t i = aList.Mini(); i <= aList.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, aList.PeekAt(i));
        TLX_ASSERT_PTR(var);

        if (var->IsSingleton())
            return var;

        TLX_ASSERT(var->MinPenalty() >= 0);

        if (var->MinPenalty() > maxpenalty)
        {
            bestvar    = var;
            maxpenalty = var->MinPenalty();
        }
    }

    TLX_ASSERT_PTR(bestvar);
    return bestvar;
}

/*---------------------------------------------------------------------------
    CLMinDomMaxPenaltySelector
---------------------------------------------------------------------------*/

CLMinDomMaxPenaltySelector::CLMinDomMaxPenaltySelector()
//
//  Constructor. Provided for completeness only.
//
{
}

const char *CLMinDomMaxPenaltySelector::Description() const
//
//  Returns a description of the selection strategy.
//
{
    return "Smallest domain, largest remaining minimum penalty";
}

TLVariable *CLMinDomMaxPenaltySelector::SelectVariable(
    TLCSProblemState *,
    TLCSProblemRep *,
    const TLVarList &aList)
//
//  Selects a variable by picking the first one with the smallest domain
//  and largest remaining minimum penalty.
//
{
    TLX_ASSERT(!aList.IsEmpty());

    CLVariable *bestvar = 0;
    int32 maxpenalty   = -1;
    size_t mindomain     = kMaxSize;

    for (index_t i = aList.Mini(); i <= aList.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, aList.PeekAt(i));
        TLX_ASSERT_PTR(var);

        if (var->IsSingleton())
            return var;

        TLX_ASSERT(var->MinPenalty() >= 0);

        if (var->Cardinality() < mindomain)
        {
            bestvar    = var;
            mindomain  = var->Cardinality();
            maxpenalty = var->MinPenalty();
        }
        else if (var->Cardinality() == mindomain)
        {
            if (var->MinPenalty() > maxpenalty)
            {
                bestvar    = var;
                maxpenalty = var->MinPenalty();
            }
        }
    }

    TLX_ASSERT_PTR(bestvar);
    return bestvar;
}

/*---------------------------------------------------------------------------
    CLMaxDiffSelector
---------------------------------------------------------------------------*/

CLMaxDiffSelector::CLMaxDiffSelector()
//
//  Constructor. Provided for completeness only.
//
{
}

const char *CLMaxDiffSelector::Description() const
//
//  Returns a description of the selection strategy.
//
{
    return "Largest difference between minimum penalties";
}

TLVariable *CLMaxDiffSelector::SelectVariable(
    TLCSProblemState *,
    TLCSProblemRep *,
    const TLVarList &aList)
//
//  Selects a variable by picking the first one with the largest difference
//  between the minimum and one-but-minimum penalty.
//
{
    TLX_ASSERT(!aList.IsEmpty());

    CLVariable *bestvar = 0;
    int32 maxdiff       = -1;

    for (index_t i = aList.Mini(); i <= aList.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, aList.PeekAt(i));
        TLX_ASSERT_PTR(var);

        if (var->IsSingleton())
            return var;

        TLX_ASSERT(var->MinDiff() >= 0);

        if (var->MinDiff() > maxdiff)
        {
            bestvar = var;
            maxdiff = var->MinDiff();
        }
    }

    TLX_ASSERT_PTR(bestvar);
    return bestvar;
}

/*---------------------------------------------------------------------------
    CLMaxDiffMinDomainSelector
---------------------------------------------------------------------------*/

CLMaxDiffMinDomainSelector::CLMaxDiffMinDomainSelector()
//
//  Constructor. Provided for completeness only.
//
{
}

const char *CLMaxDiffMinDomainSelector::Description() const
//
//  Returns a description of the selection strategy.
//
{
    return "Largest penalty difference, smallest domain";
}

TLVariable *CLMaxDiffMinDomainSelector::SelectVariable(
    TLCSProblemState *,
    TLCSProblemRep *,
    const TLVarList &aList)
//
//  Selects a variable by picking the first one with the largest difference
//  between the minimum and one-but-minimum penalty.
//
{
    TLX_ASSERT(!aList.IsEmpty());

    CLVariable *bestvar = 0;
    int32 maxdiff       = -1;
    size_t mindomain     = kMaxSize;

    for (index_t i = aList.Mini(); i <= aList.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, aList.PeekAt(i));
        TLX_ASSERT_PTR(var);

        if (var->IsSingleton())
            return var;

        TLX_ASSERT(var->MinDiff() >= 0);

        if (var->MinDiff() > maxdiff)
        {
            bestvar   = var;
            maxdiff   = var->MinDiff();
            mindomain = var->Cardinality();
        }
        else if (var->MinDiff() == maxdiff)
        {
            if (var->Cardinality() < mindomain)
            {
                bestvar   = var;
                mindomain = var->Cardinality();
            }
        }
    }

    TLX_ASSERT_PTR(bestvar);
    return bestvar;
}

/*---------------------------------------------------------------------------
    CLMaxConPenaltySelector
---------------------------------------------------------------------------*/

CLMaxConPenaltySelector::CLMaxConPenaltySelector()
//
//  Constructor. Provided for completeness only.
//
{
}

const char *CLMaxConPenaltySelector::Description() const
//
//  Returns a description of the selection strategy.
//
{
    return "Largest active constraints penalty";
}

TLVariable *CLMaxConPenaltySelector::SelectVariable(
    TLCSProblemState *,
    TLCSProblemRep *,
    const TLVarList &aList)
//
//  Selects a variable by picking the first one with the largest constraint
//  penalty.
//
{
    TLX_ASSERT(!aList.IsEmpty());

    CLVariable *bestvar = 0;
    int32 maxpenalty   = -1;

    for (index_t i = aList.Mini(); i <= aList.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, aList.PeekAt(i));
        TLX_ASSERT_PTR(var);

        if (var->IsSingleton())
            return var;

        int32 penalty = var->SumConstraintPenalties();
        TLX_ASSERT(penalty >= 0);

        if (penalty > maxpenalty)
        {
            bestvar    = var;
            maxpenalty = penalty;
        }
    }

    TLX_ASSERT_PTR(bestvar);
    return bestvar;
}

