/****************************************************************************
    $Id: main.cpp 501.0 1995/03/07 12:25:56 RON Exp $

    Copyright (c) 1995 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    main() function for the CALMA CSP example program.

    Syntax:
	CALMA  [options]

	/d	- make output DIFF-able
	/m	- Show all modules used (debug version only)
	/l<n>	- processing limit
	/u<n>	- initial upperbound
	/v<n>	- variable selection strategy
	/x	- extra lowerbound processing

    $Log: main.cpp $
    Revision 501.0  1995/03/07 12:25:56  RON
    Updated for TLX 5.01
    Revision 1.2  1995/02/28 18:36:16  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/01/18 10:41:50  RON
    Initial revision
    Revision 1.6  1995/01/18  10:41:51  ron
    Added external BB statistics monitor
    
    Revision 1.5  1995/01/17  17:11:50  ron
    Added switch for extra lowerbound processing
    Added support for new variable selection strategy

    Revision 1.4  1995/01/16  12:28:48  ron
    Added several options and varselection objects

    Revision 1.3  1995/01/13  15:41:14  ron
    Added new command line option to set upperbound

    Revision 1.2  1995/01/12  13:45:19  ron
    Adapted to previous Searcher/ProblemRep model

    Revision 1.1  1995/01/10  16:37:08  ron
    Initial revision

****************************************************************************/

//----- System headers

#if defined(__BORLANDC__)
#include <cstring.h>
#include <except.h>
#elif defined(_MSC_VER)
#include <eh.h>
#include <stdexcpt.h>
#elif defined(__SC__)
#include <eh.h>
#endif
#include <iostream.h>
#include <stdlib.h>

//----- Library headers

#include <tlx\501\debug.h>
#include <tlx\501\except.h>
#include <tlx\501\log.h>
#include <tlx\501\util.h>

//----- Project headers

#include "calma.h"

TLX_TRACE_DEFINE(Calma);

/*-------------------------------------------------------------------------*/
    int main(int argc, char *argv[])

/*  Entry point for the CALMA sample program.
---------------------------------------------------------------------------*/
{
    cerr << "CALMA cost minimization program\n\n";
    cerr << TLX::Description() << "\n";

    TLX_TRACE_MAX(TLX, 10);
    TLX_TRACE_MAX(Calma, 10);
    TLX_TRACE_TO(TLX, cerr);
    TLX_TRACE_TO(Calma, cerr);

    tlLog().SetFlush(false);		// To speed up the program

    int32 upbound   = LONG_MAX;		// Initial upperbound
    int32 plimit    = 10000L;           // Default # problems to process
    int select 	    = 5;		// Default varselector
    bool xbound	    = false;
    int result	    = 0;

    for (int i = 1; i < argc; i++)
    {
	if (argv[i][0] == '/')
	{
	    switch (argv[i][1])
	    {
		case 'd':
    		    tlLog().SetEntryFormat(TLLog::etCount);// To diff output
		    break;

		case 'm':
		    TLModuleInfo::PrintAll(cout);
		    return -1;

		case 'l':
		    plimit = atol(&argv[i][2]);
		    break;

		case 'u':
		    upbound = atol(&argv[i][2]);
		    break;

		case 'v':
		    select = atoi(&argv[i][2]);
		    break;

		case 'x':
		    xbound = true;
		    break;

		default:
		    cerr << "Unknown option " << argv[i] << "\n";
		    // FALLTHROUGH

		case '?':
		    cerr << "Valid options are:\n\n";
		    cerr << "/d    - make output DIFF-able\n";
		    cerr << "/m    - show all modules (debug version only)\n";
		    cerr << "/l<n> - set processing limit to <n>\n";
		    cerr << "/u<n> - set initial upperbound to <n>\n";
		    cerr << "/v<n> - set variable selection strategy to <n>\n";
		    cerr << "/x    - perform extra lowerbound processing\n";
		    return -1;
	    }
	}
    }

    try
    {
	CLProblemRep 	problem;	// The problem representation
	TLBBSearcherDF 	search;		// The searcher

	TLVSSmallestDomain		s1;
	CLMaxMinPenaltySelector		s2;
	CLMinDomMaxPenaltySelector	s3;
	CLMaxDiffSelector		s4;
	CLMaxDiffMinDomainSelector	s5;
	CLMaxConPenaltySelector		s6;

	TLVarSelector *selects[] = { 0, &s1, &s2, &s3, &s4, &s5, &s6 };

	select = tlMax(0, select);
	select = tlMin(TABLE_SIZE(selects) - 1, (size_t)select);

	problem.SetVarSelector(selects[select]);
	problem.SetXBound(xbound);

	TLSearchBreaker brk;
	TLSearchWatchdog wdog(plimit);
	TLSearchStats 	stats;
	TLBBStats	bbstats;
	//TLCSPMonitor 	cspmon;

	search.EnableMonitors();
	search.RegisterMonitor(&brk);
	search.RegisterMonitor(&wdog);
	search.RegisterMonitor(&stats);
	search.RegisterMonitor(&bbstats);
	//search.RegisterMonitor(&cspmon);

	cerr << search.Description() << "\n";
	cerr << "VarSelector: " << (problem.GetVarSelector() ?
		problem.GetVarSelector()->Description() : "None") << "\n";
	TLX_LOG_ENTRY("Searcher = %s", search.Description());
	TLX_LOG_ENTRY("VarSelector = %s", problem.GetVarSelector() ?
		problem.GetVarSelector()->Description() : "None");
	TLX_LOG_ENTRY("Initial upperbound = %ld", upbound);

	if (upbound < LONG_MAX) search.SetIncumbent(upbound);
 	if (search.Solve(&problem) < 1)
	    result = 1;
    }
    catch(const TLException &x)
    {
	cerr << "Exception: " << x << "\n";
	result = 10;
    }
#if defined(__BORLANDC__)
    catch(xmsg &x)
    {
	cerr << "Exception: " << x.why().c_str() << "\n";
	result = 10;
    }
#elif defined(_MSC_VER)
    catch(exception &x)
    {
        cerr << "TFC: Exception: " << x.what() << "\n";
	result = 10;
    }
#endif
    catch(...)
    {
	cerr << "Unknown exception...\n";
	result = 10;
    }
    return result;
}

