/****************************************************************************
    $Id: calma.h 501.0 1995/03/07 12:25:56 RON Exp $

    Copyright (c) 1995 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal
    Created:	09-01-95 09:49

    Declarations for the CALMA optimization program.

    $Log: calma.h $
    Revision 501.0  1995/03/07 12:25:56  RON
    Updated for TLX 5.01
    Revision 1.2  1995/02/28 18:36:16  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/02/22 13:07:10  RON
    Initial revision
    Revision 1.6  1995/01/18  10:41:27  ron
    Removed CLProblemRep statistics

    Revision 1.5  1995/01/17  17:11:19  ron
    Many changes; see calma.cpp for complete list

    Revision 1.4  1995/01/16  12:28:30  ron
    Added separate varselector objects

    Revision 1.3  1995/01/13  15:40:47  ron
    Many changes due to adaptation to modified framework

    Revision 1.2  1995/01/12  13:45:09  ron
    Adapted to previous Searcher/ProblemRep model

    Revision 1.1  1995/01/10  16:37:10  ron
    Initial revision

****************************************************************************/

#ifndef __CALMA_H
#define __CALMA_H

#ifndef _TLX_STDCSP_H
#include <tlx\501\solve\stdcsp.h>
#endif
#ifndef _TLX_TRACE_H
#include <tlx\501\debug\trace.h>
#endif

TLX_TRACE_DECLARE(Calma);

/*---------------------------------------------------------------------------
    CLDomElem -

    Domain element for the CALMA example. It consists of a frequency and
    the associated penalty if that frequency is assigned.
---------------------------------------------------------------------------*/

struct CLDomElem
{
    int			mFreq;		// Frequency
    int32		mPenalty;	// Penalty

public:
    // Constructors zero out the penalty and frequency, as applicable

    CLDomElem(): mFreq(0), mPenalty(0) {}
    CLDomElem(int aFreq): mFreq(aFreq), mPenalty(0) {}
    CLDomElem(int aFreq, int32 aPenal): mFreq(aFreq), mPenalty(aPenal) {}
};

/*---------------------------------------------------------------------------
    CLFreqDomain -
---------------------------------------------------------------------------*/

class CLFreqDomain: public TLSeq<CLDomElem>
{
public:
    CLFreqDomain();
    CLFreqDomain(size_t, size_t);

    CLFreqDomain &	operator =(const CLFreqDomain &aDomain)
			    { TLSeq<CLDomElem>::operator =(aDomain);
			      return *this; }
    CLFreqDomain &	operator =(const CLDomElem &aElem)
			    { TLSeq<CLDomElem>::operator =(aElem);
			      return *this; }

    friend ostream &	operator <<(ostream &, const CLFreqDomain &);
};

//-----	Auxiliary data types

typedef TLDomainOf<CLFreqDomain> CLDomain;	// Restorable domain info
typedef TLSeq<int>		CLFreqList;	// List of frequencies
typedef TLArray<CLFreqList>	CLFreqSets;	// List of frequency lists
typedef TLArray<int32>		CLPenalList;	// List of penalties

/*---------------------------------------------------------------------------
    CLVariable -

    Variable class for the problem. It represents a radio link with a
    domain of available frequencies. It also stores the mobility of the
    variable and the corresponding penalty and some intermediary data.
---------------------------------------------------------------------------*/

class CLVariable: public TLVariable
{
    int			mIndex;		// Index in the original description
    int			mPreference;	// Preferential frequency
    int			mMobility;	// Mobility category
    int32		mPenalty;	// Mobility penalty
    CLFreqDomain	mDomain;	// Current domain

    // Data members calculated as we go

    int32		mMinPenalty;	// Minimum penalty in domain
    int32		mMinDiff;	// Difference between least 2 penalties
    index_t		mMinIndex;	// Index of minimum penalty value
    index_t		mMin2Index;	// Index of next minimum penalty value

public:
    enum { eNoPref = -1 };		// Indicates no preferential freq.
    enum { eMobile = -1 };		// Indicates any mobility

public:
    // Constructors. Default constructor should be followed by initialization
    // through InitFromParameters().

    CLVariable();

    // Implementations of TLVariable behavior

    virtual ostream &	PrintOn(ostream &) const;

    // Access to parameter information

    int			Index() const { return mIndex; }
    int			Preference() const { return mPreference; }
    bool		HasPreference() const { return mPreference != eNoPref; }
    bool		IsImmobile() const { return mMobility == 0; }
    bool		IsMobile() const { return mMobility != 0; }
    int32		Penalty() const { return mPenalty; }

    // Domain information

    CLFreqDomain &	Domain() { return mDomain; }
    const CLFreqDomain &Domain() const { return mDomain; }
    bool		IsEmpty() const { return mDomain.IsEmpty(); }
    bool		IsSingleton() const { return mDomain.Count() == 1; }
    size_t		Cardinality() const { return mDomain.Count(); }

    // Functions to access the domain

    int			CurrentFreq() const;
    int32		CurrentPenalty() const;
    int	&		PeekFreq(index_t aIndex)
			    { return mDomain[aIndex].mFreq; }
    int			PeekFreq(index_t aIndex) const
			    { return mDomain[aIndex].mFreq; }
    int32 &		PeekPenalty(index_t aIndex)
			    { return mDomain[aIndex].mPenalty; }
    int32		PeekPenalty(index_t aIndex) const
			    { return mDomain[aIndex].mPenalty; }

    // Function to perform assignment

    void		AssignFreq(index_t);

    // Domain and constraint processing for selected variable

    void		CalcMinPenalty();
    int32		SumConstraintPenalties();

    // Volatile information about the variable

    index_t		MinIndex() const { return mMinIndex; }
    index_t		Min2Index() const { return mMin2Index; }
    int32		MinPenalty() const { return mMinPenalty; }
    int32		MinDiff() const { return mMinDiff; }

    // Function to initialize variable from parameter file information

    bool		InitFromParameters(const char *, CLFreqSets &,
    				CLPenalList &);

private:
    // Implementations of required CSP behavior

    virtual TLDomain *	CopyDomain();
    virtual void	RestoreDomain(const TLDomain *);
};

/*---------------------------------------------------------------------------
    CLVarList -
---------------------------------------------------------------------------*/

class CLVarList: public TLPtrArray<CLVariable>
{
public:
    CLVarList(index_t, index_t);

    friend ostream &	operator <<(ostream &, const CLVarList &);
};

/*---------------------------------------------------------------------------
    CLConstraint -

    Common base class for constraints in the CALMA problem. It contains a
    penalty and the distance, which may be used for either derived type.
---------------------------------------------------------------------------*/

class CLConstraint: public TLBinConOf<CLVariable>
{
    int			mDistance;	// Distance between linked variables
    int32		mPenalty;	// Penalty if constraint is violated
    bool		mCompulsory;	// May not be violated

public:
    // Access to the parameters

    int			Distance() const { return mDistance; }
    int32		Penalty() const { return mPenalty; }
    bool		IsCompulsory() const { return mCompulsory; }
    bool		IsViolated() const;

    // Required implementation

    virtual bool	IsValidPair(int, int) const = 0;

protected:
    CLConstraint(CLVariable &, CLVariable &, int, int, bool);

    // Implementation of propagation behavior

    virtual bool	Propagate(TLVariable *);
    bool		DoPropagate(CLVariable &, CLVariable &);
};

/*---------------------------------------------------------------------------
    CLGreaterThan -

    Constraint class representing a greater than constraint for the problem.
---------------------------------------------------------------------------*/

class CLGreaterThan: public CLConstraint
{
public:
    CLGreaterThan(CLVariable &, CLVariable &, int, int, bool);

    // Implementation of printing behavior

    virtual bool	IsValidPair(int, int) const;
    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    CLAbsDistance -

    Constraint class representing an absolute distance constraint for the
    problem.
---------------------------------------------------------------------------*/

class CLAbsDistance: public CLConstraint
{
public:
    CLAbsDistance(CLVariable &, CLVariable &, int, int, bool);

    // Implementation of printing behavior

    virtual bool	IsValidPair(int, int) const;
    virtual ostream &	PrintOn(ostream &) const;
};

/*---------------------------------------------------------------------------
    CLProblemState -

    Subproblem class for the CALMA example. It is derived both from the CSP
    subproblem class and from the branch and bound information class.
---------------------------------------------------------------------------*/

class CLProblemState: public TLCSProblemStateOf<CLVariable, index_t>,
		    public TLBBInfo
{
    friend class CLProblemRep;

    double		mRank;		// Rank (value ordering)
    double		mLowerBound;	// Lower bound
    double		mUpperBound;	// Upper bound
    int32		mFixedPenalty;	// Penalties of fixed variables

public:
    CLProblemState(CLProblemState *, CLVariable *, index_t);

    // Overridden general subproblem behavior

    virtual TLProblemInfo *GetInfo() { return this; }
    virtual double	GetRank() const { return mRank; }

    // Overridden CSP behavior

    virtual void	AssignVariable();
    virtual bool	ProcessVariable();

    // Overridden branch and bound behavior

    virtual double	LowerBound() { return mLowerBound; }
    virtual double	UpperBound() { return mUpperBound; }
};

/*---------------------------------------------------------------------------
    CLProblemRep -

    Problem instance class for the problem. It reads the parameters of the
    instance from a number of files, then constructs the problem
    representation.
---------------------------------------------------------------------------*/

class CLProblemRep: public TLCSProblemRep
{
    // In addition to the variables maintained by the base class, we keep a
    // separate list of variables that are immobile from the start, since
    // there is no point in ever selecting them.

    CLVarList		mOrigVars;	// Original variables
    TLPtrSeq<CLConstraint> mOrigCons;	// Original constraints
    TLVarList		mImmobiles;	// Immobile variables
    int32		mImmobilePen;	// Penalty due to immobile variables
    bool		mXBound;	// Extra lowerbound processing

    TLStats1<size_t> 	mPrunes;	// Pruning statistics

public:
    CLProblemRep();

    // Function to set lowerbound estimation strategy

    bool		DoesXBound() const { return mXBound; }
    void		SetXBound(bool aFlag = true) { mXBound = aFlag; }

    // Overridden TLProblemRep behavior

    virtual void	LeaveState(TLProblemState *);
    virtual TLSolution *CreateSolution(TLProblemState *);

protected:
    virtual void	ReportStats(ostream &) const;

    // Implemented required CSP behavior. Reads parameter files and creates
    // the problem representation.

    virtual bool	CreateVariables();
    virtual bool	PreProcessVariables();

    // Overridden optional CSP behavior

    virtual size_t	CreateBranches(TLCSProblemState *, TLVariable *,
				       TLProblemList &);
    virtual bool	Propagate(TLCSProblemState *);

private:
    // Implementation helper functions

    bool		ReadConstraints(CLPenalList &);
    bool		ReadDomains(CLFreqSets &);
    bool		ReadPenalties(CLPenalList &, CLPenalList &);
    bool		ReadVariables(CLFreqSets &, CLPenalList &);
};

/*---------------------------------------------------------------------------
    Variable selection strategies
---------------------------------------------------------------------------*/

class CLMaxMinPenaltySelector: public TLVarSelector
{
public:
    CLMaxMinPenaltySelector();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

class CLMinDomMaxPenaltySelector: public TLVarSelector
{
public:
    CLMinDomMaxPenaltySelector();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

class CLMaxDiffSelector: public TLVarSelector
{
public:
    CLMaxDiffSelector();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

class CLMaxDiffMinDomainSelector: public TLVarSelector
{
public:
    CLMaxDiffMinDomainSelector();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

class CLMaxConPenaltySelector: public TLVarSelector
{
public:
    CLMaxConPenaltySelector();

    virtual const char *Description() const;
    virtual TLVariable *SelectVariable(TLCSProblemState *, TLCSProblemRep *,
    				       const TLVarList &);
};

#endif	// __CALMA_H
