/****************************************************************************
    $Id: calma.cpp 501.0 1995/03/07 12:25:56 RON Exp $

    Copyright (c) 1995 Tarma Software Research. All rights reserved.

    Project:    Tarma Library for C++ V5.0
    Author:     Ron van der Wal
    Created:    09-01-95 11:39

    Implementation of the classes in the CALMA example.

    $Log: calma.cpp $
    Revision 501.0  1995/03/07 12:25:56  RON
    Updated for TLX 5.01
    Revision 1.2  1995/02/28 18:36:14  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/02/22 13:08:40  RON
    Initial revision
    Revision 1.6  1995/01/18  10:41:04  ron
    Removed statistics to newly introduced TLBBStats

    Revision 1.5  1995/01/17  17:08:28  ron
    Removed PenaltyOf() calculations
    Switched to frequency index instead of frequence value
    Removed PruneDomain(); pruning now done in CreateBranches()
    Moved constraint evaluation to common base class
    Corrected calculation of penalties in immobile variables
    Added penalty verification in CreateSolution()
    Extended the lowerbound calculations
    Added lowerbound statistics
    Added explicit list of constraints

    Revision 1.4  1995/01/16  12:27:57  ron
    Several changes; moved some code around
    Removed variable selection to separate objects

    Revision 1.3  1995/01/13  15:38:18  ron
    Many changes due to adaptation to the modified framework
    Several corrections:
    - Ranking uses minimum penalty with *negation*
    - Domain pruning occurs only during branch generation
    Changed variable selection strategy
    Added ReadPenalties()

    Revision 1.2  1995/01/12  13:44:54  ron
    Adapted to previous Searcher/ProblemRep model

    Revision 1.1  1995/01/10  16:37:03  ron
    Initial revision

****************************************************************************/

//----- System headers

#include <errno.h>
#include <float.h>
#include <fstream.h>
#include <iomanip.h>
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

//----- Library headers

#include <tlx\501\debug.h>
#include <tlx\501\log.h>
#include <tlx\501\rtti.h>
#include <tlx\501\util.h>

//----- Project headers

#include "calma.h"

/*---------------------------------------------------------------------------
    Template source code
---------------------------------------------------------------------------*/

#include <tlx\501\template\array.cpp>
#include <tlx\501\template\seq.cpp>
#include <tlx\501\template\ptrarray.cpp>
#include <tlx\501\template\ptrseq.cpp>
#include <tlx\501\template\stats1.cpp>

/*---------------------------------------------------------------------------
    CLFreqDomain
---------------------------------------------------------------------------*/

CLFreqDomain::CLFreqDomain()
//
// Default constructor. Uses default constructor of base class.
//
{
}

CLFreqDomain::CLFreqDomain(size_t aCap, size_t aDelta)
//
// Constructor. Forwards its parameters to the base class.
//
: TLSeq<CLDomElem>(aCap, aDelta)
{
}

ostream &operator <<(ostream &os, const CLFreqDomain &aDomain)
//
// Outputs the given domain.
//
{
    os << "[";
    for (index_t i = aDomain.Mini(); i <= aDomain.Maxi(); i++)
    {
        os << "(" << aDomain.PeekAt(i).mFreq << "," <<
           aDomain.PeekAt(i).mPenalty << ") ";
    }
    return os << "]";
}

/*---------------------------------------------------------------------------
    CLVariable
---------------------------------------------------------------------------*/

CLVariable::CLVariable()
//
// Default constructor. Initializes an empty variable; should be followed
// by a call to InitFromParameters() to complete the initialization.
//
: TLVariable("Not initialized")
{
    mIndex      = 0;
    mPreference = eNoPref;
    mMobility   = 0;
    mPenalty    = 0;
    mMinPenalty = 0;

    // The following must be initialized, as Purify found out...
    mMinDiff    = 0;
    mMinIndex   = 0;
    mMin2Index  = 0;
}

void CLVariable::AssignFreq(index_t aIndex)
//
// Performs value assignment for the variable and updates the current total
// penalty.
//
{
    // If the variable is immobile this function should only be called
    // to assign the single preferential frequency

    if (IsImmobile())
    {
        TLX_ASSERT(HasPreference());
        TLX_ASSERT(mPreference == PeekFreq(aIndex));
    }

    // Find the value in the domain to obtain its corresponding penalty.
    // The penalty corresponding to a deviation from the preferential
    // frequency (if any) has been added when the variable was initialized
    // by InitFromParameters(); the penalties resulting from violated
    // constraints with already instantiated variables was added along
    // the way.
    //
    // Next, perform the actual assignment.

    mDomain = CLDomElem(PeekFreq(aIndex), PeekPenalty(aIndex));
}

void CLVariable::CalcMinPenalty()
//
// Calculates and sets the minimum penalty and the difference between the
// smallest two penalties for the variable's current domain.
//
{
    TLX_ASSERT(!mDomain.IsEmpty());

    if (Cardinality() == 1)
    {
        mMinPenalty = CurrentPenalty();
        mMinDiff    = 0;
        mMinIndex   =
        mMin2Index  = mDomain.Mini();
    }
    else
    {
        TLX_ASSERT(mDomain.Mini() == 1);

        if (PeekPenalty(1) < PeekPenalty(2))
        {
            mMinIndex  = 1;
            mMin2Index = 2;
        }
        else
        {
            mMinIndex  = 2;
            mMin2Index = 1;
        }

        mMinPenalty = PeekPenalty(mMinIndex);
        int32 min2 = PeekPenalty(mMin2Index);

        for (index_t i = mDomain.Mini() + 2; i <= mDomain.Maxi(); ++i)
        {
            if (PeekPenalty(i) < mMinPenalty)
            {
                mMin2Index  = mMinIndex;
                mMinIndex   = i;
                min2        = mMinPenalty;
                mMinPenalty = PeekPenalty(i);
            }
            else if (PeekPenalty(i) < min2)
            {
                mMin2Index = i;
                min2       = PeekPenalty(i);
            }
        }

        mMinDiff = min2 - mMinPenalty;
    }

    TLX_ASSERT(mMinPenalty >= 0);
    TLX_ASSERT(mMinDiff >= 0);
}

TLDomain *CLVariable::CopyDomain()
//
// Called to create a copy of the current domain.
//
{
    return new CLDomain(mDomain);
}

int CLVariable::CurrentFreq() const
//
// Returns the single value of the variable's domain if the variable has
// a domain consisting of a single element.
//
{
    TLX_ASSERT(IsSingleton());
    return mDomain.PeekFirst().mFreq;
}

int32 CLVariable::CurrentPenalty() const
//
// Returns the single penalty of the variable's domain if the variable has
// a domain consisting of a single element.
//
{
    TLX_ASSERT(IsSingleton());
    return mDomain.PeekFirst().mPenalty;
}

bool CLVariable::InitFromParameters(
    const char *        aLine,
    CLFreqSets &        aSets,
    CLPenalList &       aPenalties)
//
// Initializes the variable from a line in the VAR.TXT file and some
// additional information. This line is assumed to have the following format:
//
//      no  dom#  [pref mob]
//
// where:
//      no      = number of the variable (not necessarily contiguous)
//      dom#    = index of the variable's domain
//      pref    = preferential frequency (optional)
//      mob     = frequency mobility (optional)
//
// The contents of the domain are taken from the aSets parameter, the
// penalty corresponding to the mobility is taken from aPenalties.
//
{
    TLX_ASSERT_PTR(aLine);

    int no, dom, pref, mob;
    int argcnt = sscanf(aLine, "%d %d %d %d", &no, &dom, &pref, &mob);

    if (argcnt < 2)
    {
        TLX_TRACE_ERROR(Calma, ("Invalid VAR line: %s", aLine));
        return false;
    }

    // At least the first two fields are read in

    char buf[20];
    SetName(itoa(no, buf, 10));
    mIndex = no;

    // Initialize domain

    if (dom < aSets.Mini() || dom > aSets.Maxi())
    {
        TLX_TRACE_ERROR(Calma, ("Invalid domain index %d", dom));
        return false;
    }

    mDomain.RemoveAll();
    mDomain.SetDelta(aSets[dom].Count());

    index_t i;
    for (i = aSets[dom].Mini(); i <= aSets[dom].Maxi(); ++i)
         mDomain.Append(CLDomElem(aSets[dom][i]));

    // Check if preferential frequency and mobility are present

    if (argcnt == 4)
    {
        mPreference = pref;
        mMobility   = mob;

        if (IsImmobile())               // No mobility; prune domain
        {
            for (i = mDomain.Mini(); i <= mDomain.Maxi(); ++i)
            {
                if (mDomain[i].mFreq != mPreference)
                {
                    mDomain.RemoveAt(i);
                    --i;                // To compensate for loop increment
                }
            }
            TLX_ASSERT(IsSingleton());
        }
        else                            // At least some mobility
        {
            if (mob < aPenalties.Mini() || mob > aPenalties.Maxi())
            {
                TLX_TRACE_ERROR(Calma, ("Invalid mobility %d", mob));
                return false;
            }
            mPenalty = aPenalties[mob];

            // Initialize penalties of domain values

            for (i = mDomain.Mini(); i <= mDomain.Maxi(); ++i)
                if (PeekFreq(i) != mPreference)
                    PeekPenalty(i) = mPenalty;
        }
    }
    else
    {
        mPreference = eNoPref;
        mMobility   = eMobile;
        mPenalty    = 0;
    }

    return true;
}

ostream &CLVariable::PrintOn(ostream &os) const
//
// Prints the contents of the variable on the given output stream.
//
{
    return os << GetName() << " = " << mDomain;
}

void CLVariable::RestoreDomain(const TLDomain *aDomain)
//
// Called to restore the variable's domain from a previously created copy.
//
{
#if defined(_MSC_VER)
    const CLDomain *dom = (const CLDomain *)(aDomain);
#else
    const CLDomain *dom = DYNACAST(const CLDomain *, aDomain);
#endif
    TLX_ASSERT_PTR(dom);

    mDomain = dom->Value();
}

int32 CLVariable::SumConstraintPenalties()
//
// Calculates the total penalty associated with the constraints that are
// still actively attached to the variable.
//
{
    int32 sum = 0;
    for (index_t i = Constraints().Mini(); i <= Constraints().Maxi(); ++i)
    {
        if (Constraints()[i]->IsActive())
        {
            CLConstraint *con = DYNACAST(CLConstraint *, Constraints()[i]);
            TLX_ASSERT_PTR(con);

            sum += con->Penalty();
        }
    }
    return sum;
}

/*---------------------------------------------------------------------------
    CLVarList
---------------------------------------------------------------------------*/

CLVarList::CLVarList(index_t aLower, index_t aUpper)
//
// Constructor. Forwards its arguments to the base class.
//
: TLPtrArray<CLVariable>(aLower, aUpper)
{
}

ostream &operator <<(ostream &os, const CLVarList &aList)
//
// Outputs the given list of variables
//
{
    for (index_t i = aList.Mini(); i <= aList.Maxi(); i++)
        if (aList.PeekAt(i)) os << *aList.PeekAt(i) << "\n";

    return os;
}

/*---------------------------------------------------------------------------
    CLConstraint
---------------------------------------------------------------------------*/

CLConstraint::CLConstraint(
    CLVariable &        aVar1,
    CLVariable &        aVar2,
    int                 aDistance,
    int                 aPenalty,
    bool                aCompulsory)
//
// Constructor. Initializes the instance in the obvious way.
//
: TLBinConOf<CLVariable>(aVar1, aVar2)
{
    mDistance   = aDistance;
    mPenalty    = aPenalty;
    mCompulsory = aCompulsory;

    WatchVars();
}

bool CLConstraint::DoPropagate(CLVariable &aVar1, CLVariable &aVar2)
//
// Performs constraint propagation on the two variables. This version assumes
// that the first variable has just been instantiated and that the second
// one is still free. It then proceeds to update the penalties in the
// domain of the second variable according to the penalty associated with
// the current constraint and the frequency distance required by the
// constraint.
//
// Normally, the function returns true.
//
{
    TLX_ASSERT(aVar1.IsSingleton());
    TLX_ASSERT(!aVar2.IsEmpty());
    TLX_ASSERT(Distance() > 0);

    // Check all remaining values for variable 2, updating those that
    // violate the constraint.

    int freq1    = aVar1.CurrentFreq();
    bool changed = false;

    for (index_t i = aVar2.Domain().Mini(); i <= aVar2.Domain().Maxi(); ++i)
    {
        if (!IsValidPair(freq1, aVar2.PeekFreq(i)))
        {
            if (!changed) aVar2.SaveDomain();
            changed = true;

            if (IsCompulsory())
            {
                aVar2.Domain().RemoveAt(i);
                --i;                    // To compensate loop increment

                if (aVar2.IsEmpty())
                {
                    TLX_TRACE_WARN(Calma,
                        ("var %s made domain of var %s empty",
                         aVar1.GetName(), aVar2.GetName()));
                    return false;
                }
            }
            else
                aVar2.PeekPenalty(i) += Penalty();
        }
    }

    return true;
}

bool CLConstraint::IsViolated() const
//
// Checks whether the constraint is violated by the assignments to its
// variables.
//
{
    if (!Var1().IsSingleton() || !Var2().IsSingleton())
        return false;

    return !IsValidPair(Var1().CurrentFreq(), Var2().CurrentFreq());
}

bool CLConstraint::Propagate(TLVariable *aVar)
//
// Called to propagate changes in the given variable. In this version, we
// assume that propagation is only used for variables that have been reduced
// to a single value.
//
{
    CLVariable *var = DYNACAST(CLVariable *, aVar);
    TLX_ASSERT_PTR(var);

    return DoPropagate(*var, OppositeOf(*var));
}

/*---------------------------------------------------------------------------
    CLGreaterThan
---------------------------------------------------------------------------*/

CLGreaterThan::CLGreaterThan(
    CLVariable &        aVar1,
    CLVariable &        aVar2,
    int                 aDistance,
    int                 aPenalty,
    bool                aCompulsory)
//
// Constructor. Initializes the instance in the obvious way.
//
: CLConstraint(aVar1, aVar2, aDistance, aPenalty, aCompulsory)
{
}

bool CLGreaterThan::IsValidPair(int freq1, int freq2) const
//
// Checks if the given pair of frequencies satisfies the constraint.
//
{
    return abs(freq1 - freq2) > Distance();
}

ostream &CLGreaterThan::PrintOn(ostream &os) const
//
// Prints a description of the constraint on the given stream.
//
{
    return os << "|" << Var1().GetName() << " - " << Var2().GetName() <<
        "| > " << Distance();
}

/*---------------------------------------------------------------------------
    CLAbsDistance
---------------------------------------------------------------------------*/

CLAbsDistance::CLAbsDistance(
    CLVariable &        aVar1,
    CLVariable &        aVar2,
    int                 aDistance,
    int                 aPenalty,
    bool                aCompulsory)
//
// Constructor. Initializes the instance in the obvious way.
//
: CLConstraint(aVar1, aVar2, aDistance, aPenalty, aCompulsory)
{
}

bool CLAbsDistance::IsValidPair(int freq1, int freq2) const
//
// Checks if the given pair of frequencies satisfies the constraint.
//
{
    return abs(freq1 - freq2) == Distance();
}

ostream &CLAbsDistance::PrintOn(ostream &os) const
//
// Prints a description of the constraint on the given stream.
//
{
    return os << "|" << Var1().GetName() << " - " << Var2().GetName() <<
        "| = " << Distance();
}

/*---------------------------------------------------------------------------
    CLProblemState
---------------------------------------------------------------------------*/

CLProblemState::CLProblemState(
    CLProblemState *    aParent,
    CLVariable *        aVar,
    index_t             aIndex
)
//
// Constructor for secondary subproblems with an associated variable.
//
: TLCSProblemStateOf<CLVariable, int>(aVar, aIndex)
{
    TLX_ASSERT_PTR(aVar);

    // The rank of the problem is determined by the penalty that is
    // associated with its value.

    mRank         = -aVar->PeekPenalty(aIndex);
    mLowerBound   = 0;
    mUpperBound   = DBL_MAX;
    mFixedPenalty = aParent ? aParent->mFixedPenalty : 0;
}

void CLProblemState::AssignVariable()
//
// Called to perform the assignment on the variable represented by the
// subproblem. It may be assumed that the current domain has been stored
// already.
//
{
    // Assignment is performed by the variable itself

    TLX_ASSERT_PTR(Var());
    Var()->AssignFreq(Value());

    // Update the fixed penalty associated with this problem

    mFixedPenalty += Var()->CurrentPenalty();
}

bool CLProblemState::ProcessVariable()
//
// Performs local processing after the assignment has taken place.
//
{
    // The current variable must have a single-value domain

    TLX_ASSERT_PTR(Var());
    TLX_ASSERT(Var()->IsSingleton());

    // The forward check is performed by the variable itself; this should
    // not normally lead to dead ends.

    if (!Var()->PropagateChanges())
    {
        TLX_TRACE_WARN(Calma, ("Unexpected propagation failure"));
        return false;
    }
    return true;
}

/*---------------------------------------------------------------------------
    CLSolution
---------------------------------------------------------------------------*/

struct CLAssignment
{
    int                 mVarIndex;
    int                 mFreq;

public:
    CLAssignment(): mVarIndex(0), mFreq(0) {}
    CLAssignment(int aIndex, int aFreq): mVarIndex(aIndex), mFreq(aFreq) {}
};

typedef TLSeq<CLAssignment> CLAssignmentList;

struct CLSolution: public TLSolution
{
    // Solution is stored as a sequence of variable indices with associated
    // frequencies, and the corresponding total penalty.

    CLAssignmentList    mAssignments;
    int32               mPenalty;

public:
    CLSolution(size_t);                 // Constructor requires # variables

    // Implementation of required TLSolution behavior

    virtual ostream &   PrintOn(ostream &) const;
};

CLSolution::CLSolution(size_t aSize)
//
// Constructor. Creates a sequence of the indicated size.
//
: mAssignments(aSize), mPenalty(0)
{
}

ostream &CLSolution::PrintOn(ostream &os) const
//
// Prints the solution on the given output stream.
//
{
    os << "Total costs = " << mPenalty << "\n";

#if 1
    for (index_t i = mAssignments.Mini(); i < mAssignments.Maxi(); i += 2)
    {
        TLX_ASSERT(mAssignments[i].mVarIndex+1 == mAssignments[i+1].mVarIndex);
        os << mAssignments[i].mVarIndex << "  " <<
           mAssignments[i+1].mVarIndex << "  " <<
           mAssignments[i].mFreq << "  " << mAssignments[i+1].mFreq << "\n";
    }
#else
    for (index_t i = mAssignments.Mini(); i <= mAssignments.Maxi(); ++i)
    {
        os << "Var " << mAssignments[i].mVarIndex << " = " <<
           mAssignments[i].mFreq << "\n";
    }
#endif
    return os;
}

/*---------------------------------------------------------------------------
    CLProblemRep
---------------------------------------------------------------------------*/

CLProblemRep::CLProblemRep()
//
// Default constructor.
//
: mOrigVars(1, 1000), mOrigCons(1000u, 1000u)
{
    mImmobilePen = 0;
}

size_t CLProblemRep::CreateBranches(
    TLCSProblemState *  aProblem,
    TLVariable *        aVar,
    TLProblemList &     aList)
//
// Called to create the branches for the given newly selected variable. The
// branches are subproblems of the given subproblem.
//
{
    CLProblemState *problem = DYNACAST(CLProblemState *, aProblem);
    TLX_ASSERT(problem || !aProblem);

    CLVariable *var = DYNACAST(CLVariable *, aVar);
    TLX_ASSERT_PTR(var);

    // Calculate the largest useful penalty of any value, in order to
    // prune useless domain values from the variable and take advantage
    // of dominance.

    int32 maxallowed = var->MinPenalty() + var->SumConstraintPenalties();
    size_t count       = 0;

    for (index_t i = var->Domain().Mini(); i <= var->Domain().Maxi(); ++i)
    {
        if (var->PeekPenalty(i) > maxallowed)
            continue;

        CLProblemState *sub = new CLProblemState(problem, var, i);
        TLX_ASSERT_PTR(sub);

        // If the problem has no parent, initialize its fixed penalty to
        // the penalties due to the immobile variables.

        if (!problem)
        {
            TLX_ASSERT(sub->mFixedPenalty == 0);
            sub->mFixedPenalty = mImmobilePen;
        }

        aList.AddBestFirst(sub);
        ++count;
    }

    mPrunes.Add(var->Cardinality() - count);
    return count;
}

TLSolution *CLProblemRep::CreateSolution(TLProblemState *aProblem)
//
// Called to create a solution representation for the current state of
// the global problem instance.
//
{
    TLX_ASSERT_PTR(aProblem);
    TLX_ASSERT(mFreeVars.IsEmpty());

    CLSolution *sol = new CLSolution(mImmobiles.Count() + mVars.Count());
    TLX_ASSERT_PTR(sol);

    // We calculate the total penalties in three different ways: incremental
    // per subproblem, incremental while constructing the soltuion, and
    // afresh from all constraints and variables.

    TLX_DEBUG_CODE(int32 total_penalty = 0;)    // Incremental
    TLX_DEBUG_CODE(int32 alt_penalty   = 0;)    // Alternative method

    // Construct the solution by making a pass throught the variables
    index_t i;
    for (i = mOrigVars.Mini(); i <= mOrigVars.Maxi(); ++i)
    {
        CLVariable *var = mOrigVars[i];
        if (!var) continue;

        TLX_ASSERT(var->IsSingleton());

        sol->mAssignments.Append(CLAssignment(var->Index(),
                                              var->CurrentFreq()));
      #ifndef NDEBUG
        total_penalty += var->CurrentPenalty();
        if (var->HasPreference() && var->Preference() != var->CurrentFreq())
            alt_penalty += var->Penalty();
      #endif
    }

  #ifndef NDEBUG
    // For the alternative penalty calculation, we also make a pass through
    // all constraints to calculate the penalties induced by their violations.

    for (i = mOrigCons.Mini(); i <= mOrigCons.Maxi(); ++i)
    {
        CLConstraint *con = mOrigCons[i];
        TLX_ASSERT_PTR(con);

        if (con->IsViolated())
        {
            if (con->IsCompulsory())
                TLX_TRACE_ERROR(Calma, ("Compulsory constraint violated"));
            alt_penalty += con->Penalty();
        }
    }
  #endif

    CLProblemState *problem = DYNACAST(CLProblemState *, aProblem);
    TLX_ASSERT_PTR(problem);

  #ifndef NDEBUG
    // Check that the different penalty calculations reached the same result

    TLX_ASSERT(total_penalty == problem->mFixedPenalty);
    if (total_penalty != alt_penalty)
        TLX_TRACE_ERROR(Calma, ("Solution penalty error: cum=%ld, alt=%ld",
                        total_penalty, alt_penalty));
  #endif

    sol->mPenalty = problem->mFixedPenalty;
    return sol;
}

bool CLProblemRep::CreateVariables()
//
// Called to create the variables representing the problem. This version
// reads the problem description from 4 files:
//
//      CST.TXT - cost (penalty) descriptions
//      DOM.TXT - domain descriptions
//      VAR.TXT - variable descriptions
//      CTR.TXT - constraint descriptions
//
// Variables that have mobility 0 are stored in the mImmobiles list, the
// others in the inherited mVars.
{
    CLPenalList varpenalties(1, 4);
    CLPenalList conpenalties(1, 4);
    CLFreqSets domains;

    // Read penalties

    TLX_LOG_ENTRY("Reading penalties file");
    if (!ReadPenalties(varpenalties, conpenalties))
        return false;

    // Read and create domains

    TLX_LOG_ENTRY("Reading domains file");
    if (!ReadDomains(domains))
        return false;

    // Read and create variables

    TLX_LOG_ENTRY("Reading variables file");
    if (!ReadVariables(domains, varpenalties))
        return false;

    // Read and create constraints

    TLX_LOG_ENTRY("Reading constraints file");
    if (!ReadConstraints(conpenalties))
        return false;

    // Store the variables, divided into immobiles and normal ones

    for (index_t i = mOrigVars.Mini(); i <= mOrigVars.Maxi(); ++i)
    {
        CLVariable *var = mOrigVars.PeekAt(i);
        if (!var) continue;

        if (var->IsImmobile())
            mImmobiles.Append(var);
        else
            mVars.Append(var);
    }

    TLX_LOG_ENTRY("%u variables total, %u immobile",
                  mVars.Count() + mImmobiles.Count(), mImmobiles.Count());
    TLX_LOG_ENTRY("%u constraints total", mOrigCons.Count());

    //cout << "Domains:\n" << domains << "\n";
    //cout << "Variables:\n" << mOrigVars << "\n";
    //cout << "Constraints"\n" << mOrigCons << "\n";

    return true;
}

void CLProblemRep::LeaveState(TLProblemState *aProblem)
//
// Called to clean up after processing has been completed on the subproblem.
// In this case, we must perform an extra deactivation of the constraints
// involved in the subproblem's variable.
//
{
    CLProblemState *problem = DYNACAST(CLProblemState *, aProblem);
    TLX_ASSERT_PTR(problem);
    TLX_ASSERT_PTR(problem->Var());

    problem->Var()->ActivateConstraints();

    // The remainder is handled by the base class

    TLCSProblemRep::LeaveState(aProblem);
}

bool CLProblemRep::PreProcessVariables()
//
// Called to preprocess the variables that have just been created. This
// involves calculating the penalties that are due to the immobile variables.
//
// The penalty incurred by the immobile variables is assigned to the
// current subproblem.
//
{
    // For all immobile variables, assign their preferential values. We
    // do this in a separate loop before propagation proper starts in
    // order to reduce subsequent processing on domain values that are
    // to be discarded anyway.

    mImmobilePen = 0;

    index_t i;
    for (i = mImmobiles.Mini(); i <= mImmobiles.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, mImmobiles.PeekAt(i));
        TLX_ASSERT_PTR(var);

        // Assignment should have taken place during InitFromParameters()
        //var->AssignFreq(var->Domain().IndexOf(var->Preference()));

        TLX_ASSERT(var->IsImmobile());
        TLX_ASSERT(var->IsSingleton());
        TLX_ASSERT(var->CurrentFreq() == var->Preference());

        // The forward check is performed by the variable itself; this should
        // not lead to dead ends.

        if (!var->PropagateChanges())
        {
            TLX_TRACE_WARN(Calma, ("Unexpected propagation failure"));
            return false;
        }

        // Deactivate the variable's constraints

        var->DeactivateConstraints();
    }

    // Calculate the total resulting penalties; this includes the constraint
    // penalties among immobile variables.

    for (i = mImmobiles.Mini(); i <= mImmobiles.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, mImmobiles.PeekAt(i));
        TLX_ASSERT_PTR(var);

        mImmobilePen += var->CurrentPenalty();
    }

    TLX_LOG_ENTRY("Penalties in immobiles = %ld", mImmobilePen);
    return true;
}

bool CLProblemRep::Propagate(TLCSProblemState *aProblem)
//
// Called to perform global propagation, after local (i.e. subproblem)
// processing has been successful.
//
// In this case, we proceed to calculate the minimum penalties for the
// remaining free variables. This routine also calculates and sets the
// lower and upperbounds for the given subproblem.
//
// Normally, the processing should always succeed.
//
{
    CLProblemState *problem = DYNACAST(CLProblemState *, aProblem);
    TLX_ASSERT_PTR(problem);
    TLX_ASSERT_PTR(problem->Var());

    // Lower bound is calculated as the sum of the penalties of the assigned
    // variables, plus the sum of the minimum penalties of the remaining
    // free variables. Each subproblem stores the sum of the penalties of
    // the assigned variables up to and including that problem, which
    // therefore serves as the starting point.

    int32 lowerbound = problem->mFixedPenalty;

    // The remainder of the lowerbound calculation depends on the strategy.
    // In the short version, we just add the minimum remaining penalties
    // of all remaining free variables; in the extended version, we take
    // interaction with constraints into account as well.
    //
    // In any case, we deactivate the constraints on the newly assigned
    // variable right here, instead of waiting for default behavior.

    problem->Var()->DeactivateConstraints();

    index_t i;
    for (i = mFreeVars.Mini(); i <= mFreeVars.Maxi(); ++i)
    {
        CLVariable *var = DYNACAST(CLVariable *, mFreeVars.PeekAt(i));
        TLX_ASSERT_PTR(var);

        // Let the variable calculate its minimum remaining penalty,
        // which is then used to increase the lowerbound for the problem.

        var->CalcMinPenalty();
        lowerbound += var->MinPenalty();
    }

    if (DoesXBound())
    {
        // Make the second pass through all active constraints and check
        // the penalty with respect to violated constraints. The activation/
        // deactivation mechanism ensures that we will only look at
        // constraints that are connected to two free variables.

        int32 inc = 0;

        for (i = mOrigCons.Mini(); i <= mOrigCons.Maxi(); ++i)
        {
            CLConstraint *con = mOrigCons[i];
            TLX_ASSERT_PTR(con);

            if (!con->IsActive()) continue;

            // Check if the minimum penalty values of both variables violate
            // the constraint. If not, there is no extra penalty; if they do,
            // we can increase the lowerbound.

            CLVariable &var1 = con->Var1();
            CLVariable &var2 = con->Var2();

            if (!con->IsValidPair(var1.PeekFreq(var1.MinIndex()),
                                  var2.PeekFreq(var2.MinIndex())))
            {
                // Constraint violated. The lowerbound can be increased by
                // the minimum of the constraint penalty, and the differences
                // between the minimum and least but one penalties of the
                // two variables. In case one or both of the variables has
                // only one value left, that variable's mindiff will be 0
                // and we disregard it.

                if (var1.IsSingleton())
                {
                    if (var2.IsSingleton())     // Always add constraint
                        inc += con->Penalty();
                    else                        // Minimum of (con, var2)
                        inc += tlMin(con->Penalty(), var2.MinDiff());
                }
                else if (var2.IsSingleton())    // Minimum of (con, var1)
                    inc += tlMin(con->Penalty(), var1.MinDiff());
                else                            // Minimum of three values
                    inc += tlMin(con->Penalty(),
                                tlMin(var1.MinDiff(), var2.MinDiff()));
            }
        }

        lowerbound += inc;
    }

    // Update subproblem's information. Upper bound is only given if the
    // problem has been solved completely.

    problem->mLowerBound = lowerbound;

    if (mFreeVars.IsEmpty())
        problem->mUpperBound = lowerbound;
    else
        problem->mUpperBound = DBL_MAX;

    return true;
}

bool CLProblemRep::ReadConstraints(CLPenalList &aPenalties)
//
// Reads the CTR.TXT file and creates the corresponding constraints. The
// expected format of each line in this file is:
//
//      v1  v2  l  type  dist  pen
//
// where:
//      v1      - index of first variable
//      v2      - index of second variable
//      l       - letter (C, D, F, L) whose meaning is ignored
//      type    - type of the constraint, '=' or '>'
//      dist    - associated distance
//      pen     - penalty category, 0 = may not be violated
//
{
    const char fname[] = "CTR.TXT";

    ifstream f(fname);
    if (!f)
    {
        TLX_TRACE_ERROR(Calma, ("%s: %s", fname, strerror(errno)));
        return false;
    }

    int lineno  = 0;

    const BUFSIZE = 1024;
    char buf[BUFSIZE];

    while (f.getline(buf, BUFSIZE))
    {
        // We have a line. Parse it to see what constraint is required.

        ++lineno;

        int v1, v2, dist, pen;
        char l[2], type[2];

        if (sscanf(buf, "%d %d %1s %1s %d %d", &v1, &v2, l, type, &dist, &pen)
            != 6)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Invalid format in line %d", fname,
                            lineno));
            return false;
        }

        // Find the variables involved in the constraint

        CLVariable *var1 = mOrigVars[v1];
        if (!var1)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Unknown first variable %d in line %d",
                            fname, v1, lineno));
            return false;
        }

        CLVariable *var2 = mOrigVars[v2];
        if (!var2)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Unknown second variable %d in line %d",
                            fname, v2, lineno));
            return false;
        }

        // Find the correct penalty category

        int penalty = 0;
        if (pen != 0)
        {
            // Non-compulsory constraint

            if (pen < aPenalties.Mini() || pen > aPenalties.Maxi())
            {
                TLX_TRACE_ERROR(Calma, ("%s: Invalid penalty in line %d",
                                fname, lineno));
                return false;
            }
            penalty = aPenalties[pen];
        }

        // Create the correct constraint type

        switch (type[0])
        {
            case '=':
                mOrigCons.Append(new CLAbsDistance(*var1, *var2, dist,
                                               penalty, pen == 0));
                break;

            case '>':
                mOrigCons.Append(new CLGreaterThan(*var1, *var2, dist,
                                               penalty, pen == 0));
                break;

            default:
                TLX_TRACE_ERROR(Calma,
                        ("%s: Invalid constraint type %s in line %d",
                         fname, type, lineno));
                return false;
        }
    }
    return true;
}

bool CLProblemRep::ReadDomains(CLFreqSets &aDomains)
//
// Reads the DOM.TXT file and creates the corresponding domains. Each line
// in the file has the following format:
//
//      no  cnt  freq*
//
// where:
//
//      no      - domain number
//      cnt     - number of frequencies in the domain
//      freq    - cnt frequencies
//
// The first line has domain number 0 and represents the union of all
// domains; this is not treated specially here, however.
//
{
    const char fname[] = "DOM.TXT";

    ifstream f(fname);
    if (!f)
    {
        TLX_TRACE_ERROR(Calma, ("%s: %s", fname, strerror(errno)));
        return false;
    }

    int lineno  = 0;

    const BUFSIZE = 1024;
    char buf[BUFSIZE];

    while (f.getline(buf, BUFSIZE))
    {
        // We have a line. Parse its contents.

        ++lineno;
        const char delim[] = " \t";     // White space delimiters
        char *curtok;

        // 1. Get domain number

        if ((curtok = strtok(buf, delim)) == 0)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Missing domain index in line %d",
                            fname, lineno));
            return false;
        }
        int no = atoi(curtok);
        TLX_ASSERT(no == lineno - 1);

        // Check the array of domains to see if it must be extended

        if (lineno == 1)
            aDomains.SetLimits(0, 10);
        else if (no > aDomains.Maxi())
            aDomains.SetLimits(aDomains.Mini(), no);

        TLX_ASSERT(no >= aDomains.Mini());
        TLX_ASSERT(no <= aDomains.Maxi());

        // 2. Get domain size

        if ((curtok = strtok(0, delim)) == 0)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Missing domain size in line %d",
                            fname, lineno));
            return false;
        }
        int size = atoi(curtok);

        // 3. While frequencies remain, get the frequencies

        aDomains[no].SetDelta(size);

        while (size > 0 && ((curtok = strtok(0, delim)) != 0))
        {
            int freq = atoi(curtok);
            aDomains[no].Append(freq);
            --size;
        }

        if (size > 0)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Too few frequencies in line %d",
                            fname, lineno));
            return false;
        }
    }

    // Adjust limits of domains array

    aDomains.SetLimits(0, lineno - 1);
    return true;
}

bool CLProblemRep::ReadPenalties(CLPenalList &aVarPen, CLPenalList &aConPen)
//
// Reads the CST.TXT file and initializes the corresponding penalties. The
// expected format of the file is:
//
// - 5 lines of descriptive text, or blank (skipped)
//
// - 4 lines with constraint penalties, formatted as follows:
//      a[1-4] = pen
//
// - 4 lines with mobility penalties, formatted as follows:
//      b[1-4] = pen
//
{
    const char fname[] = "CST.TXT";

    ifstream f(fname);
    if (!f)
    {
        TLX_TRACE_ERROR(Calma, ("%s: %s", fname, strerror(errno)));
        return false;
    }

    int lineno    = 0;
    const BUFSIZE = 1024;
    char buf[BUFSIZE];

    // Skip first five lines

    while (f.getline(buf, BUFSIZE) && ++lineno < 5)
        ;

    if (lineno < 5)
    {
        TLX_TRACE_ERROR(Calma, ("%s: Unexpected end of file", fname));
        return false;
    }

    // Read four lines of constraint penalties

    int index;
    for (index = 1; index <= 4 && f.getline(buf, BUFSIZE); ++index)
    {
        // We have a line. Parse it to obtain the penalty.

        ++lineno;

        int pen;
        char fbuf[40];
        sprintf(fbuf, " a%d = %%d", index);

        if (sscanf(buf, fbuf, &pen) != 1)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Invalid format in line %d", fname,
                            lineno));
            return false;
        }
        aConPen[index] = pen;
    }

    // Read four lines of variable penalties

    for (index = 1; index <= 4 && f.getline(buf, BUFSIZE); ++index)
    {
        // We have a line. Parse it to obtain the penalty.

        ++lineno;

        int pen;
        char fbuf[40];
        sprintf(fbuf, " b%d = %%d", index);

        if (sscanf(buf, fbuf, &pen) != 1)
        {
            TLX_TRACE_ERROR(Calma, ("%s: Invalid format in line %d", fname,
                            lineno));
            return false;
        }
        aVarPen[index] = pen;
    }

    if (lineno < 13)
    {
        TLX_TRACE_ERROR(Calma, ("%s: Too few lines (%d)", lineno));
        return false;
    }
    return true;
}

bool CLProblemRep::ReadVariables(
    CLFreqSets &        aDomains,
    CLPenalList &       aPenalties)
//
// Reads the VAR.TXT file and creates the corresponding variables. The
// expected format of each line is determined by
// CLVariable::InitFromParameters().
//
{
    const char fname[] = "VAR.TXT";

    ifstream f(fname);
    if (!f)
    {
        TLX_TRACE_ERROR(Calma, ("%s: %s", fname, strerror(errno)));
        return false;
    }

    int lineno  = 0;

    const BUFSIZE = 1024;
    char buf[BUFSIZE];

    while (f.getline(buf, BUFSIZE))
    {
        // We have a line. Create a variable and let it parse the line.

        ++lineno;

        CLVariable *var = new CLVariable;
        TLX_ASSERT_PTR(var);

        if (!var->InitFromParameters(buf, aDomains, aPenalties))
        {
            TLX_TRACE_ERROR(Calma, ("%s: Variable init failure in line %d",
                            fname, lineno));
            delete var;
            return false;
        }

        if (mOrigVars[var->Index()])
        {
            TLX_TRACE_ERROR(Calma, ("%s: Duplicate variable %d in line %d",
                            fname, var->Index(), lineno));
            delete var;
            return false;
        }
        else
            mOrigVars[var->Index()] = var;
    }
    return true;
}

void CLProblemRep::ReportStats(ostream &os) const
//
// Called to report any statistics that the representation may have collected.
//
{
    os << "Pruning statistics: " << mPrunes;
}

