/****************************************************************************
    $Id: refcnt.h 501.0 1995/03/07 12:26:46 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations of smart pointer classes. 'Smart' pointers can be used to
    perform automatic memory management. They are class objects that mimic
    pointer semantics, while keeping track of the pointed to object.

    In general, they cannot reclaim storage in circular data structures.

    $Log: refcnt.h $
    Revision 501.0  1995/03/07 12:26:46  RON
    Updated for TLX 5.01
    Revision 1.6  1995/01/31 16:29:24  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.5  1994/11/16  15:22:48  ron
    Changed TLRefPtr from private to public

    Revision 1.4  1994/09/27  20:25:28  ron
    Changed path separator from / to \

    Revision 1.3  1994/09/26  15:20:28  ron
    Moved inline functions to the bottom of the file
    Removed handle/safe pointer classes
    Renamed TLPointer<T> to TLCountedPtr<T>

    Revision 1.2  1994/09/06  14:01:31  ron
    Changed TLRef to TLRefCount

    Revision 1.1  1994/08/16  18:06:50  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_REFCNT_H
#define _TLX_REFCNT_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

/*---------------------------------------------------------------------------
    TLRefCount -

    Base class for all classes that must be managed by smart pointers.
    It holds a reference count that is used by the smart pointers to
    regulate memory management.
---------------------------------------------------------------------------*/

class _TLXCLASS TLRefCount
{
    friend class _TLXCLASS TLRefPtr;	// Uses reference count

    int			mRefCnt;	// The reference count

public:
    // Constructors and destructor. Constructors make sure that the
    // reference count is initialized to 0; destructor is made virtual
    // to provide for derivation.

    TLRefCount();
    TLRefCount(const TLRefCount &);
    virtual ~TLRefCount();

    // Assignment operator must leave reference count alone

    TLRefCount &	operator =(const TLRefCount &);

    // Utility functions

    bool 		IsProtected() const;
    void		Protect();
    void		Unprotect();
    int			RefCount() const;
};

/*---------------------------------------------------------------------------
    TLRefPtr -

    Smart pointer to TLRefCount instances, destroys a pointed to object when
    the pointer is detached from it and there are no other references
    to the object.
---------------------------------------------------------------------------*/

class _TLXCLASS TLRefPtr
{
    friend class _TLXCLASS TLRefHnd;

    TLRefCount *	mPtr;		// Actual pointer

public:
    // Constructors set up a possible link to the pointed to object;
    // destructor deletes the object if this pointer was the last one
    // pointing to it. One of the constructors provides for conversion
    // from ordinary TLRefCount pointers.
    //
    // The destructor is NOT made virtual so as to keep the size of
    // a TLRefPtr instance the same as an ordinary pointer. For the same
    // reason, there are no other virtual functions either.

    TLRefPtr();
    TLRefPtr(TLRefCount *);
    TLRefPtr(const TLRefPtr &);
    ~TLRefPtr();

    // Assignment operators manage reference counts as well, and delete
    // the object previously pointed to if this pointer was the last
    // pointer to it. To mimick ordinary pointers, both assignment operators
    // return a pointer to the pointed to object, rather than a reference
    // to the smart pointer.

    TLRefPtr & 		operator =(TLRefCount *);
    TLRefPtr & 		operator =(const TLRefPtr &);

    // Operators to mimick normal pointer behavior. They enable the
    // user to write expressions with TLRefPtr instances as if they
    // were genuine pointers, and to convert from smart pointers
    // to ordinary TLRefCount pointers.

			operator TLRefCount *() const;
    TLRefCount *	operator ->() const;
    TLRefCount &	operator *() const;
    int			operator !() const;

    // Helper functions available to general public. They allow benign
    // detachment of the smart pointer from the pointed to object, or
    // delete the pointed to object through the smart pointer.

    TLRefCount *	Detach();
    void		Delete();
    TLRefCount *	GetRef() const;

protected:
    // Helper function only for derived classes. It attaches a smart
    // pointer to an object for management.

    TLRefCount *	Attach(TLRefCount *);
};

/*---------------------------------------------------------------------------
    TLCountedPtr<T> -

    Template derivation of TLRefPtr to manage objects of classes derived
    from class TLRefCount.
---------------------------------------------------------------------------*/

template<class T> class TLCountedPtr: public TLRefPtr
{
public:
    // Declarations for the constructors we cannot leave to the compiler.

    TLCountedPtr();
    TLCountedPtr(T *);

    // Type-safe overloadings of TLRefPtr functions. They hide the functions
    // inherited from class TLRefPtr.

    TLCountedPtr<T> &	operator =(T *);
    TLCountedPtr<T> &	operator =(const TLCountedPtr<T> &);
			operator T *() const;
    T *			operator ->() const;
    T &			operator *() const;
    T *			Detach();
    T *			GetRef() const;
};

/*---------------------------------------------------------------------------
    Inline functions
---------------------------------------------------------------------------*/

//-----	TLRefCount

inline TLRefCount &TLRefCount::operator =(const TLRefCount &)
    { return *this; }

inline bool TLRefCount::IsProtected() const
    { return mRefCnt > 0; }

inline int TLRefCount::RefCount() const
    { return mRefCnt; }

//-----	TLRefPtr

inline TLRefPtr::operator TLRefCount *() const
    { return mPtr; }

inline TLRefCount *TLRefPtr::operator ->() const
    { return mPtr; }

inline int TLRefPtr::operator !() const
    { return mPtr == 0; }

inline TLRefCount *TLRefPtr::GetRef() const
    { return mPtr; }

//-----	TLCountedPtr<T>

template<class T> inline TLCountedPtr<T> &TLCountedPtr<T>::operator =(T *aPtr)
    { TLRefPtr::operator =(aPtr); return *this; }

template<class T> inline
TLCountedPtr<T> &TLCountedPtr<T>::operator =(const TLCountedPtr<T> &aPtr)
    { TLRefPtr::operator =(aPtr); return *this; }

template<class T> inline TLCountedPtr<T>::operator T *() const
    { return (T *) TLRefPtr::operator TLRefCount *(); }

template<class T> inline T *TLCountedPtr<T>::operator ->() const
    { return (T *) TLRefPtr::operator ->(); }

template<class T> inline T &TLCountedPtr<T>::operator *() const
    { return (T &) TLRefPtr::operator *(); }

template<class T> inline T *TLCountedPtr<T>::Detach()
    { return (T *) TLRefPtr::Detach(); }

template<class T> inline T *TLCountedPtr<T>::GetRef() const
    { return (T *)TLRefPtr::GetRef(); }

#endif	// _TLX_REFCNT_H
