/****************************************************************************
    $Id: ptrarray.h 501.0 1995/03/07 12:26:44 RON Exp $

    Copyright (c) 1993-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++
    Author:	Ron van der Wal

    Declarations of classes that implement array-based collections of (T *):

    - TLPtrVector<T>
    - TLPtrArray<T>
    - TLPtrSeq<T>
    - TLPtrQueue<T>
    - TLPtrStack<T>
    - TLPtrSet<T>

    All templates are type-safe wrappers around the (void *) versions of
    the same collections.

    NOTE: By default, all pointed to objects of type 'T' are deleted by
    means of the operator delete. This implies that they should have been
    allocated with the operator new.

    $Log: ptrarray.h $
    Revision 501.0  1995/03/07 12:26:44  RON
    Updated for TLX 5.01
    Revision 1.5  1995/01/31 16:32:12  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.4  1994/10/05  18:23:18  ron
    Added more functions that accept iterators

    Revision 1.3  1994/09/27  20:25:26  ron
    Changed path separator from / to \

    Revision 1.2  1994/09/26  15:19:39  ron
    Moved inline functions to separate file
    Added iterator classes

    Revision 1.1  1994/08/16  18:06:49  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_PTRARRAY_H
#define _TLX_PTRARRAY_H

#ifndef _TLX_ITER_H
#include <tlx\501\iter.h>
#endif
#ifndef _TLX_VPARRAYS_H
#include <tlx\501\vparrays.h>
#endif

/*---------------------------------------------------------------------------
    TLPtrVector<T> -

    Class template for vectors of (T *). It is implemented as a type-safe
    wrapper around class TLVPVector.
---------------------------------------------------------------------------*/

template<class T> class TLPtrVector: public TLVPVector
{
public:
    static void		DeleteT(void *);

public:
    // Constructors to cater for the most common situations.

    TLPtrVector(size_t = 0);		// Creates sized vector; default = 0
    TLPtrVector(T *);			// Creates single element vector
    TLPtrVector(T **, size_t);		// Creates copy of C-style vector
    TLPtrVector(const TLPtrVector<T> &);	// Copies other vector

    // Assignment operators for common situations.

    TLPtrVector<T> &	operator =(const TLPtrVector<T> &);
    TLPtrVector<T> &	operator =(T *);

    // Access to vector elements. We provide both range checked and
    // direct access versions (for speed). The operator [] versions have
    // range checking; the PeekAt() versions don't.

    T *&		operator [](index_t);
    T *			operator [](index_t) const;
    T *&		PeekAt(index_t);
    T *			PeekAt(index_t) const;
    index_t		IndexOf(const T *) const;

    // The following function gives access to the implementation vector.
    // It should be used with care.

    T **		StorageVector() const;
};

/*---------------------------------------------------------------------------
    TLPtrArray<T> -

    Class template for arrays of (T *) Indexing may start at any
    base; the array is resizable and reindexable. Indexed access is
    range checked by default.
---------------------------------------------------------------------------*/

template<class T> class TLPtrArray: public TLVPArray
{
public:
    static void		DeleteT(void *);

public:
    // Constructors to cater for the most common situations.

    TLPtrArray(size_t = 0);		// Creates 0-based array; default = 0
    TLPtrArray(index_t, index_t);		// Creates array [from, to]
    TLPtrArray(T *);			// Creates single element array
    TLPtrArray(T **, size_t);		// Creates copy of C-style vector
    TLPtrArray(const TLPtrArray<T> &);	// Copies other array

    // Assignment operators for common situations.

    TLPtrArray<T> &	operator =(const TLPtrArray<T> &);
    TLPtrArray<T> &	operator =(T *);

    // Access to array elements. We provide both range checked and
    // direct access versions (for speed). The operator [] versions have
    // range checking; the PeekAt() versions don't and are implemented inline.

    T *&		operator [](index_t);
    T *			operator [](index_t) const;
    T *&		PeekAt(index_t);
    T *			PeekAt(index_t) const;
    index_t		IndexOf(const T *) const;
};

/*---------------------------------------------------------------------------
    TLPtrSeq<T> -

    Class that implements a resizable sequence of (void *). Elements can be
    added at any place; access is by index (1-based), by position, or by
    value.
---------------------------------------------------------------------------*/

const size_t kDefaultDelta = 100;

template<class T> class TLPtrSeq: public TLVPSeq
{
public:
    static void		DeleteT(void *);

public:
    TLPtrSeq(size_t = 0, size_t = kDefaultDelta);
    TLPtrSeq(T *);			// Single element sequence
    TLPtrSeq(T **, size_t);		// From C-style vector
    TLPtrSeq(TLPtrIter<T> &);		// From iterator
    TLPtrSeq(const TLPtrSeq<T> &);	// Copy constructor

    // Assignment operators.

    TLPtrSeq<T> &	operator =(const TLPtrSeq<T> &);
    TLPtrSeq<T> &	operator =(TLPtrIter<T> &);
    TLPtrSeq<T> &	operator =(T *);
    TLPtrSeq<T> &	operator +=(const TLPtrSeq<T> &);
    TLPtrSeq<T> &	operator +=(TLPtrIter<T> &);
    TLPtrSeq<T> &	operator +=(T *);

    // Insertion and removal of elements. Append() functions add the elements
    // at the end of the sequence; Prepend() functions doe so at the start;
    // the Insert() versions allow the user to specify the index for insertion.
    // With the Replace() functions it is possible to replace parts of the
    // sequence.

    void		Append(T *);
    void		Append(T **, size_t);
    void		Append(TLPtrIter<T> &);
    void		Append(const TLPtrSeq<T> &);
    void		Prepend(T *);
    void		Prepend(T **, size_t);
    void		Prepend(TLPtrIter<T> &);
    void		Prepend(const TLPtrSeq<T> &);
    void		Insert(T *);
    void		InsertAt(index_t, T *);
    void		InsertAt(index_t, T **, size_t);
    void		InsertAt(index_t, TLPtrIter<T> &);
    void		InsertAt(index_t, const TLPtrSeq<T> &);
    void		Replace(index_t, T *);
    void		Replace(index_t, T **, size_t);
    void		Replace(index_t, TLPtrIter<T> &);
    void		Replace(index_t, const TLPtrSeq<T> &);
    T *			Extract(T *);
    T *			Extract(index_t);
    TLPtrSeq<T>		Extract(index_t, size_t);
    T *			ExtractFirst();
    TLPtrSeq<T>		ExtractFirst(size_t);
    T *			ExtractLast();
    TLPtrSeq<T>		ExtractLast(size_t);

    // Indexed access to the elements. The operator [] versions are range
    // checked, while the PeekAt() functions are not and are implemented
    // inline for speed.

    T *&		operator [](index_t);
    T *			operator [](index_t) const;
    T *&		PeekAt(index_t);
    T *			PeekAt(index_t) const;
    T *&		PeekReverse(index_t);
    T *			PeekReverse(index_t) const;
    T *&		PeekFirst();
    T *			PeekFirst() const;
    T *&		PeekLast();
    T *			PeekLast() const;

    bool 		Contains(T *) const;
    index_t		IndexOf(const T *) const;
    bool 		Search(const T *, index_t &) const;
};

//-----	Iterator class for non-const sequences

template<class T> class TLPtrSeqIter: public TLPtrIter<T>
{
    TLPtrSeq<T> &	mSeq;
    index_t		mPos;

public:
    TLPtrSeqIter(TLPtrSeq<T> &);

    // Overridden iterator functions

    virtual T *&	Peek() const;
    virtual size_t	Count() const;

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

//-----	Iterator class for const sets

template<class T> class TLPtrSeqIterConst: public TLPtrIterConst<T>
{
    const TLPtrSeq<T> &	mSeq;
    index_t		mPos;

public:
    TLPtrSeqIterConst(const TLPtrSeq<T> &);

    // Overridden iterator functions

    virtual T *		Peek() const;
    virtual size_t	Count() const;

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLPtrQueue -

    Class that implements a resizable queue of (T *). The queue can be
    used as an ordinary FIFO queue, or as a double-ended one.
---------------------------------------------------------------------------*/

template<class T> class TLPtrQueue: public TLVPQueue
{
public:
    static void		DeleteT(void *);

public:
    TLPtrQueue(size_t = 0, size_t = 0);	// Creates initial size & expansion
    TLPtrQueue(T *);			// Single element sequence
    TLPtrQueue(T **, size_t);		// Copies C-style vector
    TLPtrQueue(const TLPtrQueue<T> &);	// Copy constructor
    TLPtrQueue(TLPtrIter<T> &);		// From iterator
    TLPtrQueue(TLPtrIterConst<T> &);	// From iterator

    // Assignment operators.

    TLPtrQueue<T> &	operator =(const TLPtrQueue<T> &);
    TLPtrQueue<T> &	operator =(T *);

    // Insertion and removal of elements. For a double-ended queue:
    //
    // - AddLeft() and AddRight() add elements at either end
    // - ExtractLeft() and ExtractRight() remove and return them
    // - RemoveLeft() and RemoveRight() remove them
    // - PeekLeft() and PeekRight() allow access to those elements

    void		AddLeft(T *);
    void		AddRight(T *);
    T *			ExtractLeft();
    T *			ExtractRight();
    T *&		PeekLeft();
    T *			PeekLeft() const;
    T *&		PeekRight();
    T *			PeekRight() const;

    // For FIFO queue treatment:
    //
    // - Enqueue() adds elements at the back end
    // - Dequeue() removes and returns elements from the front end
    // - PeekHead() and PeekTail() allow inspection

    void		Enqueue(T *);
    void		EnqueueUnique(T *);
    T *			Dequeue();
    T *&		PeekHead();
    T *			PeekHead() const;
    T *&		PeekTail();
    T *			PeekTail() const;

    // For additional flexibility, it is also possible to check for
    // the presence of an element and remove it if necessary:
    //
    // - RemoveAll() discards all elements
    // - Contains() checks for presence
    // - Extract() removes and returns an element
    // - Remove() removes an element

    bool 		Contains(T *) const;
    T *			Extract(T *);
    void		Remove(T *);
};

/*---------------------------------------------------------------------------
    TLPtrStack -

    Class that implements a resizable stack of (T *).
---------------------------------------------------------------------------*/

template<class T> class TLPtrStack: public TLVPStack
{
public:
    static void		DeleteT(void *);

public:
    TLPtrStack(size_t = 0, size_t = 0);	// Creates initial size & expansion
    TLPtrStack(T *);			// Single element sequence
    TLPtrStack(T **, size_t);		// Copies C-style vector
    TLPtrStack(const TLPtrStack<T> &);	// Copy constructor

    // Assignment operators.

    TLPtrStack<T> &	operator =(const TLPtrStack<T> &);
    TLPtrStack<T> &	operator =(T *);

    // Insertion and removal of elements is on a LIFO basis.

    void		Push(T *);
    void		Push(T **, size_t);
    T *			Pop();
    T *			Pop(T *);
    void		Remove(T *);
    T *&		PeekTop();
    T *			PeekTop() const;

    // A few extra member function for additional flexibility.

    bool 		Contains(T *);
};

/*---------------------------------------------------------------------------
    TLPtrSet<T> -

    Class that implements a set of (T *). Elements can be inserted and
    removed.
---------------------------------------------------------------------------*/

template<class T>  class TLPtrSetIter;
template<class T>  class TLPtrSetIterConst;

template<class T> class TLPtrSet: public TLVPSet
{
    friend class TLPtrSetIter<T>;
    friend class TLPtrSetIterConst<T>;

public:
    static void		DeleteT(void *);

public:
    TLPtrSet(size_t = 0, size_t = 0);	// Creates initial size & expansion
    TLPtrSet(T *);			// Single element set
    TLPtrSet(const TLPtrSet<T> &);	// Copy constructor
    TLPtrSet(TLPtrIterConst<T> &);	// From iterator

    // Assignment operators.

    TLPtrSet<T> &	operator =(T *);
    TLPtrSet<T> &	operator =(const TLPtrSet<T> &);
    TLPtrSet<T> &	operator =(TLPtrIterConst<T> &);
    TLPtrSet<T> &	operator +=(T *);
    TLPtrSet<T> &	operator +=(const TLPtrSet<T> &);
    TLPtrSet<T> &	operator +=(TLPtrIterConst<T> &);

    // Insertion and removal of elements. Insert() adds an element; Remove()
    // removes it.

    void		Insert(T *);
    void		Insert(TLPtrIterConst<T> &);
    T *			Extract(T *);
    void		Remove(T *);
    bool 		Contains(const T *) const;

private:
    T *&		PeekAt(index_t);
    T *			PeekAt(index_t) const;
};

//-----	Iterator class for non-const sets

template<class T> class TLPtrSetIter: public TLPtrIter<T>
{
    TLPtrSet<T> &	mSet;
    index_t		mPos;

public:
    TLPtrSetIter(TLPtrSet<T> &);

    // Overridden iterator functions

    virtual T *&	Peek() const;
    virtual size_t	Count() const;

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

//-----	Iterator class for const sets

template<class T> class TLPtrSetIterConst: public TLPtrIterConst<T>
{
    const TLPtrSet<T> &	mSet;
    index_t		mPos;

public:
    TLPtrSetIterConst(const TLPtrSet<T> &);

    // Overridden iterator functions

    virtual T *		Peek() const;
    virtual size_t	Count() const;

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    Inline functions
---------------------------------------------------------------------------*/

#include <tlx\501\ptrarray.inl>

#endif	// _TLX_PTRARRAY_H

