/****************************************************************************
    $Id: log.h 501.0 1995/03/07 12:26:44 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for message logging support.

    $Log: log.h $
    Revision 501.0  1995/03/07 12:26:44  RON
    Updated for TLX 5.01
    Revision 1.4  1995/01/31 16:29:22  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.3  1994/10/10  17:02:20  ron
    Added entry formatting options

    Revision 1.2  1994/10/06  17:48:23  ron
    Made most member functions public

    Revision 1.1  1994/10/05  18:23:00  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_LOG_H
#define _TLX_LOG_H

#include <stdarg.h>
#include <time.h>

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

class _RTLCLASS ostream;

/*---------------------------------------------------------------------------
    TLLog -

    Message log class. It provides the means for message logging with
    date/time stamps. It can also be used wherever an ostream & argument
    is required.

    There is normally only a single instance of this class in a given
    program.
---------------------------------------------------------------------------*/

const size_t kMaxLogMessage = 255;	// Maximum size of single log message

class _TLXCLASS TLLog {
    ostream *		mOStream;	// Current output stream
    TLLog *		mPrev;		// Previously active message log
    bool		mFlushAll;	// Flush output after each entry
    time_t		mStartTime;	// Time of log creation
    uint32		mEntryCount;	// Number of entries
    int			mEntryFormat;	// Entry prefix type (see below)

public:
    // Log entries may be prefixed by a full date/time stamp, a relative
    // time, or an entry count.

    enum {
	etAbsTime,			// Absolute date + time
	etRelTime,			// Relative time
	etCount				// Count of the entry
    };

    // The (presumably) only log instance is accessible to all

    static TLLog *	sIt;

public:
    // Default constructor links to cout; other constructor allows
    // specification of initial ostream. Destructor flushes log stream
    // and disconnects from log stack.

    TLLog(bool = true);
    TLLog(ostream &, bool = true);
    TLLog(const TLLog &);
    ~TLLog();

    // Conversion from and to ostream &

			operator ostream &() const;
    ostream &		Stream() const;
    TLLog &		operator =(ostream &);
    TLLog &		operator =(const TLLog &);

    // Ways to set the entry format

    int			GetEntryFormat() const;
    int			SetEntryFormat(int);
    void		SetFlush(bool = true);

    // Message output: can be ostream-style or printf-style. When using
    // ostream-style, log entries must be explicitly started and terminated:
    //
    // 	  BeginEntry();
    //	  ...use ostream functions and/or Printf()...
    //	  EndEntry();
    //
    // or:
    //
    //    PrintEntry();

    ostream &		BeginEntry();
    ostream &		EndEntry() const;
    ostream & __cdecl	Printf(const char *, ...) const;
    void __cdecl	PrintEntry(const char *, ...);

    // Dummy output operator which will terminate the log

    friend ostream &	_TLXFUNC operator <<(ostream &, const TLLog &);

private:
    void		Init(bool);
    void		VPrintf(const char *, va_list) const;
};

/*---------------------------------------------------------------------------
    Helper functions and macros.

    Log()		- returns reference to the current log instance

    The following macros all return an (ostream &):

    TLX_LOG		- accesses current log stream
    TLX_LOG_BEGIN	- starts a log entry
    TLX_LOG_END		- ends a log entry

    TLX_LOG_ENTRY()	- use as TLX_LOG_ENTRY("format", arg1, arg2, ...)
    TLX_LOG_PRINT()	- use as TLX_LOG_PRINT("format", arg1, arg2, ...)
---------------------------------------------------------------------------*/

TLLog &			tlLog();

#define TLX_LOG		tlLog().Stream()
#define TLX_LOG_BEGIN	tlLog().BeginEntry()
#define TLX_LOG_END	tlLog()
#define TLX_LOG_PRINT	tlLog().Printf
#define TLX_LOG_ENTRY	tlLog().PrintEntry

/*---------------------------------------------------------------------------
    Inline definitions
---------------------------------------------------------------------------*/

//-----	TLLog

#ifndef _TLXDBG
inline TLLog::operator ostream &() const
    { return *mOStream; }
#endif

inline ostream &TLLog::Stream() const
    { return operator ostream &(); }

inline void TLLog::SetFlush(bool aFlag)
    { mFlushAll = aFlag; }

inline int TLLog::GetEntryFormat() const
    { return mEntryFormat; }

//-----	Helper functions

#ifndef _TLXDBG
inline TLLog &tlLog()
    { return *TLLog::sIt; }
#endif

#endif	// _TLX_LOG_H
