/****************************************************************************
    $Id: iter.h 501.0 1995/03/07 12:26:44 RON Exp $

    Copyright (c) 1994-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declaration of iterator base classes and templates:

    TLPrintFormat	- wrapper for formatting info
    TLIter		- base for all iterators
    TLValueIter<T>	- base for iterators of non-const value containers
    TLValueIterConst<T> - base for iterators of const value containers
    TLPtrIter<T>	- base for iterators of non-const pointer containers
    TLPtrIterConst<T>	- base for iterators of const pointer containers

    $Log: iter.h $
    Revision 501.0  1995/03/07 12:26:44  RON
    Updated for TLX 5.01
    Revision 1.7  1995/01/31 16:28:02  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.6  1994/11/16  15:22:13  ron
    Added 'Const' postfix to some constructors

    Revision 1.5  1994/10/06  17:48:05  ron
    Small formatting changes

    Revision 1.4  1994/10/05  18:22:34	ron
    Added TLPtrItemIter and TLPtrItemIterConst

    Revision 1.3  1994/09/28  14:27:03	ron
    Small formatting changes

    Revision 1.2  1994/09/27  20:25:18	ron
    Changed path separator from / to \

    Revision 1.1  1994/09/26  15:18:28	ron
    Initial revision

****************************************************************************/

#ifndef _TLX_ITER_H
#define _TLX_ITER_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

/*---------------------------------------------------------------------------
    TLPrintFormat -

    Wrapper for information that is used to format print-outs by the
    various iterator classes.
---------------------------------------------------------------------------*/

struct _TLXCLASS TLPrintFormat
{
    const char *	mLeader;	// Leader text
    const char *	mPresep;	// Pre-separation text
    const char *	mPostsep;	// Post-separation text
    const char *	mTrailer;	// Trailer text

    TLPrintFormat();
    TLPrintFormat(const char *, const char *, const char *, const char *);
};

/*---------------------------------------------------------------------------
    TLIter -

    Abstract base class for all iterator types. It defines the iterator
    mechanism that is shared by all iterators.
---------------------------------------------------------------------------*/

class _TLXCLASS TLIter
{
    // Iterators can be in one of several states.

    enum State
    {
	stReset,			// Reset, not advanced yet
	stValid,			// Valid position in the collection
	stEnd				// Beyond the end of the collection
    };
    State		mState; 	// Current state

public:
    virtual ~TLIter();

    // Iteration functions. They can be used as follows:
    //
    //	iter.Reset();	       *or*
    //	while (iter.Next())		for (iter.Reset(); iter.Next(); )
    //	    iter.Peek()...		    iter.Peek()...
    //
    // Immediately after construction, an iterator is in the reset state
    // and may be used without intervening Reset() operation.
    //
    // Derived classes must define suitable Peek() functions, for example:
    //
    //	  T &Peek() const;
    //
    // This function should check that it is called in a valid state (use
    // IsValid() for the check).

    void		Reset();
    bool		Next();

    // Functions to interrogate the state of the iterator. In addition,
    // Count() returns the number of elements in the collection that is
    // being iterated.

    bool		IsReset() const;
    bool		IsValid() const;
    bool		IsAtEnd() const;
    virtual size_t	Count() const = 0;

protected:
    TLIter();		// Constructor resets state

private:
    // The following functions are used by the implementation of Next().
    //
    // - FirstPos() is called by Next() when the iterator is currently
    //	 in the reset state. It should set up the iterator to point
    //	 at the first element of the collection and return true if it
    //	 succeeds. If the iterator cannot be started (e.g. collection empty),
    //	 the false must be returned.
    //
    // - NextPos() is called by Next() when the iterator is in a valid state
    //	 and must be advanced to the next element. It should advance the
    //	 iterator position, and return true if it is still valid, else
    //	 (end of collection) false.

    virtual bool	FirstPos() = 0;
    virtual bool	NextPos() = 0;
};

/*---------------------------------------------------------------------------
    TLValueIter -

    Abstract base class template for iterators that operate on non-const
    value containers of type T.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLValueIter: public TLIter
{
public:
    typedef void	(*tApplyFunc)(T &, void *);
    typedef bool	(*tTestFunc)(T &, void *);

public:
    virtual T & 	Peek() const = 0;
    virtual bool	Contains(T &);

    // Functions to apply an operation to all or a selected number of
    // elements.

    void		ForAll(tApplyFunc, void *);
    bool		FirstThat(tTestFunc, void *, T *&);
    bool		FirstThatNot(tTestFunc, void *, T *&);

    // Output operations

    ostream &		PrintAll(ostream &);
    ostream &		PrintFormat(ostream &, const TLPrintFormat &);
};

/*---------------------------------------------------------------------------
    TLValueIterConst -

    Abstract base class template for all iterators that operate on const
    value containers of type T.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLValueIterConst: public TLIter
{
public:
    typedef void	(*tApplyFunc)(const T &, void *);
    typedef bool	(*tTestFunc)(const T &, void *);

public:
    virtual const T &	Peek() const = 0;
    virtual bool	Contains(const T &);

    // Functions to apply an operation to all or a selected number of
    // elements.

    void		ForAll(tApplyFunc, void *);
    bool		FirstThat(tTestFunc, void *, const T *&);
    bool		FirstThatNot(tTestFunc, void *, const T *&);

    // Output operations

    ostream &		PrintAll(ostream &);
    ostream &		PrintFormat(ostream &, const TLPrintFormat &);
};

/*---------------------------------------------------------------------------
    TLItemIter -

    Iterator template for 'iteration' over non-const items of type T.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLItemIter: public TLValueIter<T>
{
    T & 		mItem;		// Single item

public:
    TLItemIter(T &);

    // Overridden iterator functions

    virtual T & 	Peek() const;
    virtual size_t	Count() const;
    virtual bool	Contains(T &);

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLItemIterConst -

    Iterator template for 'iteration' over const items of type T.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLItemIterConst: public TLValueIterConst<T>
{
    const T &		mItem;		// Single item

public:
    TLItemIterConst(const T &);

    // Overridden iterator functions

    virtual const T &	Peek() const;
    virtual size_t	Count() const;
    virtual bool	Contains(const T &);

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLVectorIter -

    Iterator template for non-const C-style vectors of type T elements.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLVectorIter: public TLValueIter<T>
{
    T * 		mVector;	// C-style vector
    size_t		mSize;		// Size of the vector
    size_t		mPos;		// Iterator position

public:
    TLVectorIter(T [], size_t);

    // Overridden iterator functions

    virtual T & 	Peek() const;
    virtual size_t	Count() const;

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLVectorIterConst -

    Iterator template for const C-style vectors of type T elements.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLVectorIterConst: public TLValueIterConst<T>
{
    const T *		mVector;	// C-style vector
    size_t		mSize;		// Size of the vector
    size_t		mPos;		// Iterator position

public:
    TLVectorIterConst(const T [], size_t);

    // Overridden iterator functions

    virtual const T &	Peek() const;
    virtual size_t	Count() const;

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLPtrIter -

    Base class template for iterators that operate on non-const containers
    of (T *). It can be used as-is, in which case it acts as an iterator
    on an empty collection.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLPtrIter: public TLIter
{
public:
    typedef void	(*tApplyFunc)(T *&, void *);
    typedef bool	(*tTestFunc)(T *&, void *);

public:
    TLPtrIter();

    // Default implementations of iterator access: mimic empty collection

    virtual T *&	Peek() const;
    virtual bool	Contains(T *&);
    virtual size_t	Count() const;

    // Functions to apply an operation to all or a selected number of
    // elements.

    void		ForAll(tApplyFunc, void *);
    bool		FirstThat(tTestFunc, void *, T **&);
    bool		FirstThatNot(tTestFunc, void *, T **&);

    // Output operations

    ostream &		PrintAll(ostream &);
    ostream &		PrintFormat(ostream &, const TLPrintFormat &);

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLPtrIterConst -

    Base class template for iterators that operate on const containers
    of (T *). It can be used as-is, in which case it acts as an iterator
    on an empty collection.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLPtrIterConst: public TLIter
{
public:
    typedef void	(*tApplyFunc)(T *, void *);
    typedef bool	(*tTestFunc)(T *, void *);

public:
    TLPtrIterConst();

    // Default implementations of iterator access: mimic empty collection

    virtual T * 	Peek() const;
    virtual size_t	Count() const;
    virtual bool	Contains(T *);

    // Functions to apply an operation to all or a selected number of
    // elements.

    void		ForAll(tApplyFunc, void *);
    bool		FirstThat(tTestFunc, void *, T *&);
    bool		FirstThatNot(tTestFunc, void *, T *&);

    // Output operations

    ostream &		PrintAll(ostream &);
    ostream &		PrintFormat(ostream &, const TLPrintFormat &);

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLPtrItemIter -

    Iterator template for 'iteration' over non-const items of type T *.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLPtrItemIter: public TLPtrIter<T>
{
    T *&		mItem;		// Single item

public:
    TLPtrItemIter(T *&);

    // Overridden iterator functions

    virtual T *&	Peek() const;
    virtual size_t	Count() const;
    virtual bool	Contains(T *&);

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    TLPtrItemIterConst -

    Iterator template for 'iteration' over const items of type T *.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLPtrItemIterConst: public TLPtrIterConst<T>
{
    T * 		mItem;		// Single item

public:
    TLPtrItemIterConst(T *);

    // Overridden iterator functions

    virtual T * 	Peek() const;
    virtual size_t	Count() const;
    virtual bool	Contains(T *);

private:
    virtual bool	FirstPos();
    virtual bool	NextPos();
};

/*---------------------------------------------------------------------------
    Inline functions
---------------------------------------------------------------------------*/

//----- TLIter

inline bool TLIter::IsReset() const
    { return mState == stReset; }

inline bool TLIter::IsValid() const
    { return mState == stValid; }

inline bool TLIter::IsAtEnd() const
    { return mState == stEnd; }

//----- TLItemIter<T>

template<class T> inline size_t TLItemIter<T>::Count() const
    { return 1; }

//----- TLItemIterConst<T>

template<class T> inline size_t TLItemIterConst<T>::Count() const
    { return 1; }

//----- TLVectorIter<T>

template<class T> inline size_t TLVectorIter<T>::Count() const
    { return mSize; }

//----- TLVectorIterConst<T>

template<class T> inline size_t TLVectorIterConst<T>::Count() const
    { return mSize; }

#endif	// _TLX_ITER_H
