/****************************************************************************
    $Id: except.h 501.0 1995/03/07 12:26:42 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for exceptions.

    $Log: except.h $
    Revision 501.0  1995/03/07 12:26:42  RON
    Updated for TLX 5.01
    Revision 1.10  1995/01/31 16:29:20  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.9  1994/11/16  15:21:02  ron
    Made TLXLocus constructors outline

    Revision 1.8  1994/10/05  18:21:26  ron
    Renamed TLx...() functions to tl...()

    Revision 1.7  1994/09/28  14:26:51  ron
    Removed Macintosh-style #include references

    Revision 1.6  1994/09/27  20:25:05  ron
    Changed path separator from / to \

    Revision 1.5  1994/09/26  15:17:56  ron
    Changed include file references

    Revision 1.4  1994/09/12  14:49:41  ron
    Added PrintOn() and operator << overloading tot TLException

    Revision 1.3  1994/09/07  15:32:42  ron
    Added TLXResize exception

    Revision 1.2  1994/09/06  13:58:22  ron
    Added TLXAssert exception class

    Revision 1.1  1994/08/16  18:06:47  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_EXCEPT_H
#define _TLX_EXCEPT_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

class _RTLCLASS ios;
class _RTLCLASS ostream;

/*---------------------------------------------------------------------------
    TLXLocus -

    Class describing the locus of an exception or similar diagnostic.
---------------------------------------------------------------------------*/

struct _TLXCLASS TLXLocus
{
    const char *	mFile;
    int			mLine;

public:
    TLXLocus();
    TLXLocus(const char *, int);
    TLXLocus(const TLXLocus &);
};

/*---------------------------------------------------------------------------
    TLException -

    Base class for all exceptions thrown by TLX, apart from xalloc. Use
    these exceptions as follows:

    THROW(TLException(LOCUS, ...))
---------------------------------------------------------------------------*/

class _TLXCLASS TLException
{
    TLXLocus		mLocus;

public:
    // Static storage for the construction of exception messages. Can be
    // used by only one exception at a time.

    static const size_t	sBufferSize;
    static char		sBuffer[];

public:
    TLException(const TLXLocus &);
    virtual ~TLException();

    // All exceptions must be able to describe themselves

    virtual const char *Description() const;
    const TLXLocus &	Locus() const { return mLocus; }

    // Output functions:
    //
    // - PrintOn() describes the exception on the given stream;
    // - Report() describes the exception on the diagnostic output;
    // - operator << is also overloaded.

    void		Report() const;
    virtual ostream &	PrintOn(ostream &) const;
    friend ostream & _TLXFUNC operator <<(ostream &, const TLException &);
};

/*---------------------------------------------------------------------------
    TLXAssert -

    Exception class for assertion failures. Use it as follows:

    THROW(TLXAssert(TLXLocus(file, line)))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXAssert: public TLException
{
public:
    TLXAssert(const TLXLocus &);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXNullPtr -

    Exception class for NULL pointer exceptions. Use it as follows:

    THROW(TLXNullPtr(LOCUS))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXNullPtr: public TLException
{
public:
    TLXNullPtr(const TLXLocus &);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXIndex -

    Exception class for out-of-range indices. Use it as follows:

    THROW(TLXIndex(LOCUS, index))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXIndex: public TLException
{
    index_t		mIndex;

public:
    TLXIndex(const TLXLocus &, index_t);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXNotFound -

    Exception class for 'element not found' exceptions. Use it as follows:

    THROW(TLXNotFound(LOCUS))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXNotFound: public TLException
{
public:
    TLXNotFound(const TLXLocus &);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXEmpty -

    Exception class for 'collection empty' exceptions. Use it as follows:

    THROW(TLXEmpty(LOCUS))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXEmpty: public TLException
{
public:
    TLXEmpty(const TLXLocus &);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXFull -

    Exception class for 'collection full' exceptions. Use it as follows:

    THROW(TLXFull(LOCUS))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXFull: public TLException
{
public:
    TLXFull(const TLXLocus &);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXAlloc -

    Exception class for memory allocations exceptions. Use it as follows:

    THROW(TLXAlloc(LOCUS, size))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXAlloc: public TLException
{
    size_t		mSize;

public:
    TLXAlloc(const TLXLocus &, size_t);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXResize -

    Exception class for invalid resize exceptions. Use it as follows:

    THROW(TLXResize(LOCUS, newsize, minsize))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXResize: public TLException
{
    size_t		mNewSize;
    size_t		mMinSize;

public:
    TLXResize(const TLXLocus &, size_t, size_t);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXStdio -

    Exception class for stdio errors. Use it as follows:

    THROW(TLXStdio(LOCUS, filename))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXStdio: public TLException
{
    int			mErrNo;
    const char *	mFileName;

public:
    TLXStdio(const TLXLocus &, const char *);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    TLXIos -

    Exception class for iostream errors. Use it as follows:

    THROW(TLXIos(LOCUS, ios-ref))
---------------------------------------------------------------------------*/

class _TLXCLASS TLXIos: public TLException
{
    int			mErrState;

public:
    TLXIos(const TLXLocus &, const ios &);
    virtual const char *Description() const;
};

/*---------------------------------------------------------------------------
    Helper functions and macros for exception handling.
---------------------------------------------------------------------------*/

#define	LOCUS		TLXLocus(__FILE__, __LINE__)
void _TLXFUNC 		_tlThrow(const TLException &);

/*---------------------------------------------------------------------------
    Macros to fake exception handling for those compilers that don't yet
    implement them.
---------------------------------------------------------------------------*/

#ifdef _NOEXCEPT
    #define THROW(x)	_tlThrow(x)
#else
    #define THROW(x)	throw x
#endif

#endif	// _TLX_EXCEPT_H
