/****************************************************************************
    $Id: dlists.h 501.0 1995/03/07 12:26:42 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    All lists in this file are doubly linked lists, based on TLDLink
    elements. The file also contains templates for lists based on classes
    derived from class TLDLink.

    TLDLink		- element in doubly linked lists
    TLDLBase		- linear list of Links
    TLDLCBase		- circular list of Links
    TLDLQueueBase 	- queue based on a linked list
    TLDLStackBase 	- stack based on a linked list

    TLDList<T>    	- template for TLDLBase
    TLDLCList<T>  	- template for TLDLCBase
    TLDLQueue<T>  	- template for TLDLQueueBase
    TLDLStack<T>  	- template for TLDLStackBase

    $Log: dlists.h $
    Revision 501.0  1995/03/07 12:26:42  RON
    Updated for TLX 5.01
    Revision 1.6  1995/01/31 16:29:20  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.5  1994/09/28  14:26:43  ron
    Removed Macintosh-style #include references

    Revision 1.4  1994/09/27  20:25:03  ron
    Changed path separator from / to \

    Revision 1.3  1994/09/26  15:17:39  ron
    Changed include file references

    Revision 1.2  1994/09/06  13:58:09  ron
    Adapted to changes in tlx.h

    Revision 1.1  1994/08/16  18:06:46  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_DLISTS_H
#define _TLX_DLISTS_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

/*---------------------------------------------------------------------------
    TLDLink -

    Base class for all classes that may be stored in a doubly linked list.
    It contains forward and backwards pointers.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDLink
{
    friend class _TLXCLASS TLDLBase;
    friend class _TLXCLASS TLDLCBase;

    TLDLink *		mNext;		// Forward pointer
    TLDLink *		mPrev;		// Backward pointer

public:
    virtual ~TLDLink();

    // Functions to traverse through lists of DLinks.

    TLDLink *		Next() const { return mNext; }
    TLDLink *		Prev() const { return mPrev; }
    bool 		IsLast() const { return mNext == 0; }

protected:
    // Constructors and destructor are only for use by derived classes;
    // copy constructor does *not* copy the link fields.

    TLDLink();
    TLDLink(const TLDLink &);

    // Assignment operator does *not* copy the link fields.

    TLDLink &		operator =(const TLDLink &);

private:
    // Helper functions for use by list classes.

    void		Append(TLDLink *);
    void		Prepend(TLDLink *);
    void		Unlink();
};

/*---------------------------------------------------------------------------
    TLDLBase -

    Manages linear doubly linked lists based on TLDLinks. It serves as a base
    class for other list classes.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDLBase
{
    TLDLink *		mHead;		// First element in the list
    TLDLink *		mTail;		// Last element in the list
    size_t		mCount;		// Number of elements

    // If the list manager is owner of the elements stored in the list,
    // the elements will be deleted when the list manager itself is
    // destroyed. The owner status may be changed by everyone.

    bool 		mOwner;		// If nonzero, elements are discarded

public:
    // Constructor creates an empty list and sets owner status. Destructor
    // optionally deletes elements.

    TLDLBase(bool = false);
    virtual ~TLDLBase();

    bool 		IsOwner() const { return mOwner; }
    bool 		BecomeOwner(bool = true);

    // Operations that insert or remove elements or sublists.

    void		Append(TLDLink *);
    void		Append(TLDLink *, TLDLink *);
    void		RemoveAll();
    void		Concat(TLDLink *);
    void		Concat(TLDLBase &);
    void		Prepend(TLDLink *);
    void		Prepend(TLDLink *, TLDLink *);
    TLDLink *		Extract(TLDLink *);
    TLDLink *		ExtractHead();
    TLDLink *		ExtractTail();
    TLDLink *		Replace(TLDLink *, TLDLink *);
    TLDLink *		Split(TLDLink *);

    TLDLink *		PeekHead() const { return mHead; }
    TLDLink *		PeekTail() const { return mTail; }
    size_t		Count() const { return mCount; }
    bool 		IsEmpty() const;
    bool 		Contains(TLDLink *) const;

private:
    // Copying and assignment are prohibited.

    TLDLBase(const TLDLBase &);
    TLDLBase &		operator =(const TLDLBase &);

    // Implementation helper function.

    void		CleanUp();
};

/*---------------------------------------------------------------------------
    TLDLCBase -

    Manages a circular linked list consisting of TLDLink elements.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDLCBase
{
    // In the circular list, we only store a pointer to the first element.

    TLDLink *		mHead;
    size_t		mCount;

    // If the list manager is owner of the elements stored in the list,
    // the elements will be deleted when the list manager itself is
    // destroyed. The owner status may be changed by everyone.

    bool 		mOwner;

public:
    // Constructor creates an empty list and sets owner status. Destructor
    // optionally deletes elements.

    TLDLCBase(bool = false);
    virtual ~TLDLCBase();

    bool 		IsOwner() const { return mOwner; }
    bool 		BecomeOwner(bool = true);

    // Operations that insert or remove elements or sublists.

    void		Append(TLDLink *);
    void		Append(TLDLink *, TLDLink *);
    void		Prepend(TLDLink *);
    void		Prepend(TLDLink *, TLDLink *);
    TLDLink *		Extract(TLDLink *);
    TLDLink *		ExtractHead();
    TLDLink *		ExtractTail();
    void		RemoveAll();

    TLDLink *		PeekHead() const { return mHead; }
    TLDLink *		PeekTail() const { return mHead ? mHead->mPrev : 0; }
    size_t		Count() const { return mCount; }
    bool 		IsEmpty() const;
    bool 		Contains(TLDLink *) const;

private:
    // Copying and assignment are prohibited.

    TLDLCBase(const TLDLCBase &);
    TLDLCBase &		operator =(const TLDLCBase &);

    // Implementation helper function.

    void		CleanUp();
};

/*---------------------------------------------------------------------------
    TLDLQueueBase -

    Represents a queue based on a doubly-linked list. TLDLQueueBase is
    an intrusive queue.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDLQueueBase: private TLDLBase
{
public:
    TLDLQueueBase(bool = false);

    void		Enqueue(TLDLink *aLink) { Append(aLink); }
    TLDLink *		Dequeue() { return ExtractHead(); }

    TLDLBase::IsOwner;
    TLDLBase::RemoveAll;
    TLDLBase::Count;
    TLDLBase::IsEmpty;
    TLDLBase::PeekHead;
    TLDLBase::PeekTail;
};

/*---------------------------------------------------------------------------
    TLDLStackBase -

    Represents a stack based on a doubly-linked list. TLDLStackBase is
    an intrusive stack.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDLStackBase: private TLDLBase
{
public:
    TLDLStackBase(bool = false);

    void		Push(TLDLink *aLink) { Prepend(aLink); }
    TLDLink *		Pop() { return ExtractHead(); }
    TLDLink *		PeekTop() const { return PeekHead(); }

    TLDLBase::IsOwner;
    TLDLBase::RemoveAll;
    TLDLBase::Count;
    TLDLBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLDList<T> -

    This class is a template for a type-safe derivation of TLDLBase for
    classes T that are derived from TLDLink. TLDList<T> is an intrusive
    list for items of type T.
---------------------------------------------------------------------------*/

template<class T> class TLDList: private TLDLBase
{
public:
    TLDList(bool = false);

    TLDList<T> &	operator =(const TLDList<T> &) { return  *this; }

    void		Append(T *aLink) { TLDLBase::Append(aLink); }
    void		Append(T *aLink, T *aPos)
			    { TLDLBase::Append(aLink, aPos); }
    void		Concat(T *aLink) { TLDLBase::Concat(aLink); }
    void		Concat(TLDList<T> &aList) { TLDLBase::Concat(aList); }
    void		Prepend(T *aLink) { TLDLBase::Prepend(aLink); }
    void		Prepend(T *aLink, T *aPos)
			    { TLDLBase::Prepend(aLink, aPos); }
    T *			ExtractHead() { return (T *) TLDLBase::ExtractHead(); }
    T *			ExtractTail() { return (T *) TLDLBase::ExtractTail(); }
    T *			Extract(T *aLink)
			    { return (T *) TLDLBase::Extract(aLink); }
    T *			Replace(T *aPos, T *aLink)
			    { return (T *) TLDLBase::Replace(aPos, aLink); }
    T *			Split(T *aLink) { return (T *) TLDLBase::Split(aLink); }

    T *			PeekHead() const { return (T *) TLDLBase::PeekHead(); }
    T *			PeekTail() const { return (T *) TLDLBase::PeekTail(); }
    bool 		Contains(T *t) const { return TLDLBase::Contains(t); }

    TLDLBase::IsOwner;
    TLDLBase::RemoveAll;
    TLDLBase::Count;
    TLDLBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLDLCList<T> -

    A template for a type-safe derivation of TLDLCBase for classes T that
    are derived from TLDLink. TLDLCList<T> is an intrusive list for items of
    type T.
---------------------------------------------------------------------------*/

template<class T> class TLDLCList: private TLDLCBase
{
public:
    TLDLCList(bool = false);

    TLDLCList<T> &	operator =(const TLDLCList<T> &) { return *this; }

    void		Append(T *aLink) { TLDLCBase::Append(aLink); }
    void		Append(T *aLink, T *aPos)
			    { TLDLCBase::Append(aLink, aPos); }
    void		Prepend(T *aLink) { TLDLCBase::Prepend(aLink); }
    void		Prepend(T *aLink, T *aPos)
			    { TLDLCBase::Prepend(aLink, aPos); }
    T *			ExtractHead() { return (T *) TLDLCBase::ExtractHead(); }
    T *			ExtractTail() { return (T *) TLDLCBase::ExtractTail(); }
    T *			Extract(T *aLink)
			    { return (T *) TLDLCBase::Extract(aLink); }

    T *			PeekHead() const { return (T *) TLDLCBase::PeekHead(); }
    T *			PeekTail() const { return (T *) TLDLCBase::PeekTail(); }

    TLDLCBase::IsOwner;
    TLDLCBase::RemoveAll;
    TLDLCBase::Count;
    TLDLCBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLDLQueue<T> -

    A template for a type-safe derivation of TLDLQueueBase for classes T that
    are derived from TLDLink. TLDLQueue<T> is an intrusive queue for items of
    type T.
---------------------------------------------------------------------------*/

template<class T> class TLDLQueue: private TLDLQueueBase
{
public:
    TLDLQueue(bool = false);

    void		Enqueue(T *aLink) { TLDLQueueBase::Enqueue(aLink); }
    T *			Dequeue() { return (T *) TLDLQueueBase::Dequeue(); }

    T *			PeekHead() const
    			    { return (T *) TLDLQueueBase::PeekHead(); }
    T *			PeekTail() const
    			    { return (T *) TLDLQueueBase::PeekTail(); }

    TLDLQueueBase::IsOwner;
    TLDLQueueBase::RemoveAll;
    TLDLQueueBase::Count;
    TLDLQueueBase::IsEmpty;
};

/*---------------------------------------------------------------------------
    TLDLStack<T> -

    A template for a type-safe derivation of TLDLStackBase for classes T that
    are derived from TLDLink. TLDLStack<T> is an intrusive stack for items of
    type T.
---------------------------------------------------------------------------*/

template<class T> class TLDLStack: private TLDLStackBase
{
public:
    TLDLStack(bool = false);

    void		Push(T *aLink) { TLDLStackBase::Push(aLink); }
    T *			Pop() { return (T *) TLDLStackBase::Pop(); }
    T *			PeekTop() const
    			    { return (T *) TLDLStackBase::PeekTop(); }

    TLDLStackBase::IsOwner;
    TLDLStackBase::RemoveAll;
    TLDLStackBase::Count;
    TLDLStackBase::IsEmpty;
};

#endif	// _TLX_DLISTS_H
