/****************************************************************************
    $Id: dispose.h 501.0 1995/03/07 12:26:42 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for the Disposable pattern.

    $Log: dispose.h $
    Revision 501.0  1995/03/07 12:26:42  RON
    Updated for TLX 5.01
    Revision 1.2  1995/01/31 16:29:18  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/01/05  15:33:17  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_DISPOSE_H
#define _TLX_DISPOSE_H

/*-----	Disposable pattern --------------------------------------------------

    Intent:

	This pattern models the situation where one or more objects must be
	disposed, but only at controlled moments. A primary example is a
	GUI application with a central event dispatching loop, where during
	the dispatch of a single event one or more objects might become
	ripe for disposal. However, it is often not safe to do so
	immediately, because other objects might still need access to
	the doomed objects. As a special case, the objects might 'commit
	suicide' as the (indirect) result of the event, but cannot safely
	do so at once. In those cases, the disposable object is marked as
	such, and a specialized garbage manager will take care of the
	actual clean-up at a more appropriate moment, e.g. before the next
	event is dispatched.

    Participants:

	class TLDisposable	- (virtual) base class for disposable objects
	    MarkForDisposal()	- marks the object for garbage collection
	    ReclaimObject()	- removes the object from garbage collection
	    CollectGarbage()	- destroys all objects marked for disposal

	class TLCDChecker	- (virtual base class to check con/destruction
	    AssertValid()	- ensures that the object is valid

    Usage:

	Derive the class(es) that must be disposable from the TLDisposable
	base class, possibly with virtual derivation to allow multiple
	inheritance. When an instance of the class must be disposed, call
	its MarkForDisposal() member function. The instance will be marked
	for destruction, but not be destroyed until the static member
	function TLDisposable::CollectGarbage() is called.

---------------------------------------------------------------------------*/

#ifndef _TLX_TLX_H
#include <tlx\501\tlx\h>
#endif

/*---------------------------------------------------------------------------
    TLCDChecker -

    Class for objects that need a check on construction and destruction.
    It can be used a a member or a (virtual) base class; using it as a
    member allows easier use of the containing class with respect to
    derivation from other bases.

    In the non-debug version, the class is empty.
---------------------------------------------------------------------------*/

// Helper macro that can be used to include a member of class TLCDChecker,
// This macro is only useful when combined with the debug version of the
// library.

#ifdef NDEBUG
#define INSERT_CHECKER
#define TLX_ASSERT_VALID()	((void)0)
#else
#define INSERT_CHECKER		TLCDChecker _mChecker;
#define TLX_ASSERT_VALID()	_mChecker.AssertValid(__FILE__,__LINE__)
#endif

class _TLXCLASS TLCDChecker
{
    // During debugging only, we insert extra checks as to the validity of
    // the object with respect to initialization and destruction.

    enum {
	osConstructed	= 0x4B4F,	// Marker for constructed object
	osDestructed	= 0x3F21	// Marker for destructed object
    };
  #ifdef _TLXDBG
    int16		mState;		// Current state
  #endif

public:
    TLCDChecker();
    ~TLCDChecker();

    // Copying and assignment do not copy the check field

    TLCDChecker(const TLCDChecker &);
    TLCDChecker &	operator =(const TLCDChecker &);

    // Function to check the validity of the object

    void		AssertValid(const char *, int) const;
};

/*---------------------------------------------------------------------------
    TLDisposable -

    (Virtual) base class for disposable objects. In the debug version, it
    also contains a construction/destruction check.
---------------------------------------------------------------------------*/

class _TLXCLASS TLDisposable
{
    // Objects marked for disposal are put in a linked list with a static
    // data member as its head.

    static TLDisposable *sGarbage;	// Chain of garbage objects
    TLDisposable *	mNext;		// Next disposable object

    // During debugging only, we insert extra checks as to the validity of
    // the object with respect to initialization and destruction.

  #ifdef _TLXDBG
    TLCDChecker		_mChecker;
  #endif

public:
    TLDisposable();
    virtual ~TLDisposable();

    // Copying and assignment do not copy the pointer fields

    TLDisposable(const TLDisposable &);
    TLDisposable &	operator =(const TLDisposable &);

    // Functions to manage the disposal/reclamation process per instance

    void		MarkForDisposal();
    void		ReclaimObject();
    bool		IsMarkedForDisposal() const;

    // Static function to perform garbage collection on all disposed objects.

    static void		CollectGarbage();

private:
    // Helper functions to add and remove instances from the linked list
    // of disposable objects.

    void		AddToGarbage();
    void		ExtractFromGarbage();
};

/*---------------------------------------------------------------------------
    Inline function definitions
---------------------------------------------------------------------------*/

//-----	TLCDChecker

#ifndef _TLXDBG
inline void TLCDChecker::AssertValid(const char *, int) const
    {}
#endif

#endif	// _TLX_DISPOSE_H
