/****************************************************************************
    $Id: trace.h 501.0 1995/03/07 12:26:50 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:    Tarma Library for C++ V5.0
    Author:     Ron van der Wal

    Tracing macros. As with the other debug macros, #defining NDEBUG
    before #including this file turns them into null code (with a few
    exceptions, noted below).

    Trace output may go through one of several trace groups, which allows you
    to separate traces that pertain to different topics. In addition, each
    trace message has a trace level, and trace output may be restricted to
    the messages below a maximum.

    The name of a tracing group must conform to C++ identifier syntax, and
    may include any mixture of: '_', letters, and digits. Each trace message
    is prefixed by its group name, so it's probably best to keep the name
    short and descriptive. Usage is as follows:

    TLX_TRACE_DECLARE(group);

        - Declares the existence of the tracing group; place it in every file
          that produces trace output to the given group. For convenience, the
          declaration may be put in a shared header file.

    TLX_TRACE_DEFINE(group);

        - Defines the tracing group; use it once for each tracing group.

    TLX_TRACE_ON(group);

        - Restores tracing for the group at its current maximum level.
          Tracing is on by default.

    TLX_TRACE_OFF(group);

        - Disables all tracing for the group (it is on by default).

    TLX_TRACE_MAX(group,level);

        - Sets the maximum tracing level for the group. It will not change
          the ON/OFF setting for the group.

    TLX_TRACE_TO(group,os);

        - Sets the tracing output stream for the group. By default this is
          the predefined 'cerr' stream.

    TLX_TRACE(group, (level, message, parameters...));

        - Produces a printf-style formatted trace message through the
          given trace group, presuming 'level' is less than or equal to
          the current maximum tracing level for the group.

    TLX_TRACE_INFO(group, (message, parameters...));

        - Produces a printf-style formatted trace message with an 'Info'
          prefix through the given trace group.

    TLX_TRACE_WARN(group, (message, parameters...));

        - Produces a printf-style formatted trace message with a 'Warning'
          prefix through the given trace group.

    TLX_TRACE_ERROR(group, (message, parameters...));

        - Produces a printf-style formatted trace message with an 'Error'
          prefix through the given trace group.

    TLX_TRACE_FATAL(group, (message, parameters...));

        - Produces a printf-style formatted trace message with a 'Fatal'
          prefix through the given trace group.

    For convenience, the 'TLX' tracing group is predefined and always
    available. Also, the following macros provide for manipulation of all
    tracing groups at once:

    TLX_TRACE_ALL();

        - Restores tracing for all tracing groups, subject to their
          individual tracing ON/OFF status.

    TLX_TRACE_NONE();

        - Disables tracing for all groups.

    $Log: trace.h $
    Revision 501.0  1995/03/07 12:26:50  RON
    Updated for TLX 5.01
    Revision 1.2  1995/01/31 16:29:02  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/01/05  15:36:04  ron
    Initial revision

    Revision 1.1  1994/11/16  15:26:09  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_TRACE_H
#define _TLX_TRACE_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif
#include <stdarg.h>

#ifdef NDEBUG
/*---------------------------------------------------------------------------
    Tracing macros when not debugging - resolve to null code
---------------------------------------------------------------------------*/

#define TLX_TRACE_ALL()         ((void)0)
#define TLX_TRACE_NONE()        ((void)0)

#define TLX_TRACE_DECLARE(grp)
#define TLX_TRACE_DEFINE(grp)

#define TLX_TRACE_ON(grp)       ((void)0)
#define TLX_TRACE_OFF(grp)      ((void)0)
#define TLX_TRACE_MAX(grp,lvl)  ((void)0)
#define TLX_TRACE_TO(grp,os)    ((void)0)

#define TLX_TRACE(grp,msg)      ((void)0)
#define TLX_TRACE_INFO(grp,msg) ((void)0)
#define TLX_TRACE_WARN(g,m)     ((void)0)
#define TLX_TRACE_ERROR(g,m)    ((void)0)

#define TLX_TRACE_SCOPE(g,n)    ((void)0)

#else
/*---------------------------------------------------------------------------
    Tracing macros when debugging - resolve to actual code
---------------------------------------------------------------------------*/

#define TLX_TRACE_ALL()         TLTracer::EnableAll()
#define TLX_TRACE_NONE()        TLTracer::DisableAll()

#define TLX_TRACE_DECLARE(grp)  extern TLTracer gTraceGrp ## grp
#define TLX_TRACE_DEFINE(grp)   TLTracer gTraceGrp ## grp(#grp)

#define TLX_TRACE_ON(grp)       gTraceGrp ## grp.Enable()
#define TLX_TRACE_OFF(grp)      gTraceGrp ## grp.Disable()
#define TLX_TRACE_MAX(grp,lvl)  gTraceGrp ## grp.SetMaxLevel(lvl)
#define TLX_TRACE_TO(grp,os)    gTraceGrp ## grp.SetStream(os)

#define TLX_TRACE(grp,arg)      gTraceGrp ## grp. \
                                SetLocus(__FILE__,__LINE__).TraceFmt arg
#define TLX_TRACE_INFO(grp,arg) gTraceGrp ## grp. \
                                SetLocus(__FILE__,__LINE__).InfoFmt arg
#define TLX_TRACE_WARN(grp,arg) gTraceGrp ## grp. \
                                SetLocus(__FILE__,__LINE__).WarningFmt arg
#define TLX_TRACE_ERROR(grp,arg) gTraceGrp ## grp. \
                                SetLocus(__FILE__,__LINE__).ErrorFmt arg
#define TLX_TRACE_FATAL(grp,arg) gTraceGrp ## grp. \
                                SetLocus(__FILE__,__LINE__).FatalFmt arg

#define TLX_TRACE_SCOPE(grp,n)  TLScopeTracer _ ## __LINE__(gTraceGrp ## grp, \
                                n, __FILE__, __LINE__)

#endif  // NDEBUG

/*---------------------------------------------------------------------------
    Auxiliary declarations
---------------------------------------------------------------------------*/

class _TLXCLASS TLTracer
{
    // To allow global disabling/enabling of trace output, we have a
    // static flag.

    static bool         sEnabled;       // Tracing globally enabled?

    // For identification, we store the group name. We also keep track of
    // the current maximum tracing level for the group and its output
    // stream. Next to the maximum tracing level, we remember the on/off
    // status of the group as well.

    const char *        mGroupName;     // Name of the tracing group
    int                 mMaxLevel;      // Maximum tracing level
    bool                mEnabled;       // Group enabled?
    ostream *           mOStream;       // Output stream

    // To allow display of the filename and line number of the origin of the
    // trace message and still use variable-length argument lists (and ditto
    // macros), we store the message locus just prior to the call to the
    // actual trace function.
    //
    // The locus is local to the group to reduce (but not eliminate) the
    // chances of a conflict in multi-threaded environments.

    const char *        mFileName;      // Filename of the trace message
    int                 mLineNo;        // Line number in the file

public:
    TLTracer(const char *);
    ~TLTracer();

    // Global and local trace enabling and level setting.

    static void         EnableAll();
    static void         DisableAll();
    void                Enable();
    void                Disable();
    void                SetMaxLevel(int);
    void                SetStream(ostream &);

    // Formatted trace output functions

    void __cdecl	TraceFmt(int, const char *, ...);
    void __cdecl	InfoFmt(const char *, ...);
    void __cdecl	WarningFmt(const char *, ...);
    void __cdecl	ErrorFmt(const char *, ...);
    void __cdecl	FatalFmt(const char *, ...);

    // Helper function to store the locus of the message that follows

    TLTracer &          SetLocus(const char *, int);

private:
    // Implementation helper functions

    bool                CanTrace(int) const;
    void                VPrintf(const char *, const char *, va_list);
};

class _TLXCLASS TLScopeTracer
{
    TLTracer &          mTracer;
    const char *        mName;
    const char *        mFile;
    int                 mLine;

public:
    TLScopeTracer(TLTracer &, const char *, const char *, int);
    ~TLScopeTracer();
};

inline TLScopeTracer::TLScopeTracer(
    TLTracer &  aTracer,
    const char *aName,
    const char *aFile,
    int         aLine)
: mTracer(aTracer), mName(aName), mFile(aFile), mLine(aLine)
{
    mTracer.SetLocus(mFile, mLine);
    mTracer.InfoFmt("Entering scope %s...", mName);
}

inline TLScopeTracer::~TLScopeTracer()
{
    mTracer.SetLocus(mFile, mLine);
    mTracer.InfoFmt("Leaving scope %s...", mName);
}

/*---------------------------------------------------------------------------
    Inline and other definitions
---------------------------------------------------------------------------*/

//----- Predefined tracing group 'TLX'

extern _TLXDATA TLTracer        gTraceGrpTLX;

//----- TLTracer

inline bool TLTracer::CanTrace(int aLevel) const
    { return sEnabled && mEnabled && aLevel <= mMaxLevel; }

#endif  // _TLX_TRACE_H
