/****************************************************************************
    $Id: asserts.h 501.0 1995/03/07 12:26:50 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:    Tarma Library for C++ V5.0
    Author:     Ron van der Wal

    The macros in this file supersede the traditional assert() macro and
    should be used instead. As with the traditional macros, #defining NDEBUG
    before #including this file turns them into null code (with a few
    exceptions, noted below).

    Assertion macros
    ----------------
    In general, these macros are intended to test or verify assumptions,
    (pre/post) conditions, invariants, etc. During debugging they perform
    their test and terminate the program with some kind of error message if
    the corresponding assumption is violated. In the retail version of
    the program, they are converted to null expressions (with a few
    exceptions, see below).

    To switch between the debug and the retail versions of these macros,
    either leave NDEBUG #undefined (debugging code included), or #define
    it to any value or null (debugging code excluded).

    These macros revert to void expressions if NDEBUG is #defined:

    TLX_ASSERT(c)       - verifies that 'c' evaluates to nonzero
    TLX_ASSERT_PTR(p)   - verifies that 'p' evaluates to non-NULL pointer
    TLX_ASSERT_NULL(p)  - verifies that 'p' evaluates to NULL pointer
    TLX_ASSERT_UNREACHABLE	- verifies that it is never executed
    TLX_ASSERT_TODO            	- prints a "To do" message when executed
    TLX_ASSERT_NOTIMPLEMENTED  	- prints a "Not implemented" message
    TLX_ASSERT_MSG(m,a1,a2)	- prints an assertion message when executed

    The next macros revert to their argument expression if NDEBUG is #defined,
    so they should be used if the argument must be evaluated (and the result
    of the evaluation used) whether or not debugging is enabled:

    TLX_CHECK(c)        - verifies that 'c' evaluates to nonzero
    TLX_CHECK_PTR(p)    - verifies that 'p' evaluates to non-NULL pointer
    TLX_CHECK_NULL(p)   - verifies that 'p' evaluates to NULL pointer

    Environment variables
    ---------------------

    This feature is not functional on platforms that do not have an
    environment, such as Macintosh.

    $Log: asserts.h $
    Revision 501.0  1995/03/07 12:26:50  RON
    Updated for TLX 5.01
    Revision 1.2  1995/01/31 16:28:58  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.1  1995/01/05  15:35:48  ron
    Initial revision

    Revision 1.1  1994/11/16  15:25:46  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_ASSERTS_H
#define _TLX_ASSERTS_H

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

#ifdef NDEBUG
/*---------------------------------------------------------------------------
    Assertion macros when not debugging - resolve to null code
---------------------------------------------------------------------------*/

#define TLX_ASSERT(c)                   ((void)0)
#define TLX_ASSERT_PTR(p)               ((void)0)
#define TLX_ASSERT_NULL(p)              ((void)0)
#define TLX_ASSERT_UNREACHABLE		((void)0)

#define TLX_REQUIRE(c)                  ((void)0)
#define TLX_REQUIRE_PTR(p)              ((void)0)
#define TLX_REQUIRE_NULL(p)             ((void)0)

#define TLX_ENSURE(c)                   ((void)0)
#define TLX_ENSURE_PTR(p)               ((void)0)
#define TLX_ENSURE_NULL(p)              ((void)0)

#define TLX_CHECK(c)                    (c)
#define TLX_CHECK_PTR(p)                (p)
#define TLX_CHECK_NULL(p)               (!(p))

#define TLX_TODO(msg)			((void)0)
#define TLX_NOTIMPLEMENTED(msg)         ((void)0)
#define TLX_DIAGNOSTIC(l,msg,p1,p2)	((void)0)

#else
/*---------------------------------------------------------------------------
    Assertion macros when debugging - resolve to actual code
---------------------------------------------------------------------------*/

#define TLX_ASSERT(c)           ((c) ? (void)0 : \
                                TLAssert::AssertFailed(__FILE__, __LINE__))
#define TLX_ASSERT_PTR(p)       (((p) != 0) ? (void)0 : \
                                TLAssert::PtrFailed(__FILE__, __LINE__))
#define TLX_ASSERT_NULL(p)      (((p) == 0) ? (void)0 : \
                                TLAssert::NullPtrFailed(__FILE__, __LINE__))
#define TLX_ASSERT_UNREACHABLE  TLAssert::Reached(__FILE__, __LINE__)

#define TLX_REQUIRE(c)          ((c) ? (void)0 : \
                                TLAssert::RequireFailed(__FILE__, __LINE__))
#define TLX_REQUIRE_PTR(p)      (((p) != 0) ? (void)0 : \
                                TLAssert::RequirePtrFailed(__FILE__, __LINE__))
#define TLX_REQUIRE_NULL(p)     (((p) == 0) ? (void)0 : \
                                TLAssert::RequireNullPtrFailed(__FILE__, __LINE__))

#define TLX_ENSURE(c)           ((c) ? (void)0 : \
                                TLAssert::EnsureFailed(__FILE__, __LINE__))
#define TLX_ENSURE_PTR(p)       (((p) != 0) ? (void)0 : \
                                TLAssert::EnsurePtrFailed(__FILE__, __LINE__))
#define TLX_ENSURE_NULL(p)      (((p) == 0) ? (void)0 : \
                                TLAssert::EnsureNullPtrFailed(__FILE__, __LINE__))

#define TLX_CHECK(c)            ((c) ? true : \
                                (TLAssert::CheckFailed(__FILE__, __LINE__), \
				false))
#define TLX_CHECK_PTR(p)        (((p) != 0) ? true : \
                                (TLAssert::PtrCheck(__FILE__, __LINE__), \
				false))
#define TLX_CHECK_NULL(p)       (((p) == 0) ? true : \
                                (TLAssert::NullPtrCheck(__FILE__, __LINE__),\
				false))

#define TLX_TODO(msg)    	TLAssert::ToDo(#msg, __FILE__, __LINE__)
#define TLX_NOTIMPLEMENTED(msg) TLAssert::NotImplemented(#msg, __FILE__, \
				__LINE__)

#define TLX_DIAGNOSTIC(lvl,msg,p1,p2) TLAssert::Diagnostic(lvl, \
				__FILE__, __LINE__, msg, p1, p2)

#endif	// NDEBUG

/*---------------------------------------------------------------------------
    Auxiliary definitions
---------------------------------------------------------------------------*/

class _TLXCLASS TLAssert
{
public:
    enum {
    	dfFatal,        // Diagnostic is fatal; program will be terminated
    	dfError,        // Diagnostic is an error; proceed at your risk
    	dfWarning,      // Diagnostic is a warning; may continue
    	dfInfo		// Diagnostic is only informational
    };

public:
    static void		AssertFailed(const char *, int);
    static void		CheckFailed(const char *, int);
    static void	__cdecl	Diagnostic(int, const char *,int,const char *, ...);
    static void		EnsureFailed(const char *, int);
    static void		EnsureNullPtrFailed(const char *, int);
    static void		EnsurePtrFailed(const char *, int);
    static void		NotImplemented(const char *, const char *, int);
    static void		NullPtrCheck(const char *, int);
    static void		NullPtrFailed(const char *, int);
    static void		PtrCheck(const char *, int);
    static void		PtrFailed(const char *, int);
    static void		Reached(const char *, int);
    static void		RequireFailed(const char *, int);
    static void		RequireNullPtrFailed(const char *, int);
    static void		RequirePtrFailed(const char *, int);
    static void		ToDo(const char *, const char *, int);
};

#endif	// _TLX_ASSERTS_H
