/****************************************************************************
    $Id: arrays.h 501.0 1995/03/07 12:26:40 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations of collection templates based on vectors:

    TLVBase<T>	- template for vector base class
    TLVector<T>	- template for 0-based vectors of type T
    TLArray<T>	- template for arrays of type T
    TLSeqBase<T>- template for sequence-like collections
    TLSeq<T> 	- template for vector-based sequences of type T
    TLQueue<T>	- template for vector-based queues of type T
    TLStack<T>	- template for vector-based stacks of type T

    $Log: arrays.h $
    Revision 501.0  1995/03/07 12:26:40  RON
    Updated for TLX 5.01
    Revision 1.9  1995/01/31 16:29:14  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.8  1994/11/16  15:19:04  ron
    Added lint directive

    Revision 1.7  1994/10/07  17:07:14  ron
    Added Array<T>::SetLimits()

    Revision 1.6  1994/09/28  14:24:56  ron
    Removed Macintosh-style #include references

    Revision 1.5  1994/09/27  20:24:46  ron
    Changed path separator from / to \

    Revision 1.4  1994/09/26  15:14:10  ron
    Moved inline functions to bottom of the file

    Revision 1.3  1994/09/12  14:49:14  ron
    Small formatting changes

    Revision 1.2  1994/09/06  13:57:14  ron
    Adapted to changes in tlx.h

    Revision 1.1  1994/08/16  18:06:43  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_ARRAYS_H
#define _TLX_ARRAYS_H

//-----	System headers

#include <stdarg.h>

//-----	Library headers

#ifndef _TLX_TLX_H
#include <tlx\501\tlx.h>
#endif

class _RTLCLASS ostream;	// In iostream.h

/*----------------------------------------------------------------------------
    TLVBase<T> -

    Base class template for all vector-like collections
*----------------------------------------------------------------------------*/

template<class T> class TLVBase
{
public:
    //lint -esym(578,tCompareFunc,tOutputFunc)
    typedef int 	(*tCompareFunc)(const T &, const T &);
    typedef void 	(*tOutputFunc)(ostream &, const T &);

    static int		DefaultCompare(const T &, const T &);
    static void		DefaultOutput(ostream &, const T &);

protected:
    T *			mVect;		// TLVector of T
    size_t 		mSize;		// TLVector size
    tCompareFunc 	mCompare;       // Element comparison
    tOutputFunc 	mOutF;		// Element output

public:
  #if defined (__BORLANDC__) //&& defined (__OS2__)
    // Borland C++ for OS/2 bug fix
    static char *	sPre;		// TLVector prefix for print()
    static char *	sPreSep;	// Element prefix for print()
    static char *	sPostSep;	// Element postfix for print()
    static char *	sPost;		// TLVector postfix for print()
  #elif defined(__SC__)
    // Symantec C++ bug fix -- members are global
  #else
    static const char *	sPre;		// TLVector prefix for print()
    static const char *	sPreSep;	// Element prefix for print()
    static const char *	sPostSep;	// Element postfix for print()
    static const char *	sPost;		// TLVector postfix for print()
  #endif

public:
    // Indexed access to vector items, either by index or by iterator

    T &			operator [](index_t);
    const T &		operator [](index_t) const;
    T &			operator [](iter_t i) { return mVect[i.ix]; }
    const T &		operator [](iter_t i) const { return mVect[i.ix]; }

    // Index boundaries

    virtual index_t 	Mini() const { return 0; }
    virtual index_t 	Maxi() const { return mSize - 1; }
    bool  		IsValidIndex(index_t) const;

    // TLVBase size information

    static size_t 	MaxSize() { return kMaxAlloc / sizeof(T); }
    size_t 		Size() const { return mSize; }
    virtual size_t 	Count() const { return mSize; }

    // TLVBase class invariant

    virtual bool  	OK() const;

    // Membership tests, perform a linear search

    int			Compare(const TLVBase<T> &) const;
    bool  		Contains(const T &) const;
    virtual index_t 	IndexOf(const T &) const;

    // Filling the vector up to its current size

    virtual void 	Fill(const T &);

    // Printing a vector

    virtual ostream & 	PrintOn(ostream &) const;

    // Built-in iteration

    void __cdecl	DoAll(void (*)(T &, va_list), ...);
    void __cdecl	DoAll(void (*)(const T &, va_list), ...) const;
    index_t __cdecl	DoWhile(bool (*)(T &, va_list), ...);
    index_t __cdecl	DoWhile(bool (*)(const T &, va_list), ...) const;
    index_t __cdecl	DoUntil(bool (*)(T &, va_list), ...);
    index_t __cdecl	DoUntil(bool (*)(const T &, va_list), ...) const;

    // External iterators

    virtual bool  	IterFirst(iter_t &) const;
    virtual bool  	IterLast(iter_t &) const;
    virtual bool  	IterNext(iter_t &) const;
    virtual bool  	IterPrev(iter_t &) const;

    // Element comparison and output functions

    tCompareFunc 	GetCompare() const { return mCompare; }
    tCompareFunc 	SetCompare(tCompareFunc f)
    			{
			    tCompareFunc oldf = mCompare;
			    if (f) mCompare = f;
			    return oldf;
    			}
    tOutputFunc 	GetOutput() const { return mOutF; }
    tOutputFunc 	SetOutput(tOutputFunc f)
    			{
    		  	    tOutputFunc oldf = mOutF;
			    if (f) mOutF = f;
			    return oldf;
    			}

    friend int 		operator ==(const TLVBase<T> &v1, const TLVBase<T> &v2)
			    { return v1.Compare(v2) == 0; }
    friend int 		operator !=(const TLVBase<T> &v1, const TLVBase<T> &v2)
			    { return v1.Compare(v2) != 0; }
    friend int 		operator < (const TLVBase<T> &v1, const TLVBase<T> &v2)
			    { return v1.Compare(v2) < 0; }
    friend int 		operator <=(const TLVBase<T> &v1, const TLVBase<T> &v2)
			    { return v1.Compare(v2) <= 0; }
    friend int 		operator > (const TLVBase<T> &v1, const TLVBase<T> &v2)
			    { return v1.Compare(v2) > 0; }
    friend int 		operator >=(const TLVBase<T> &v1, const TLVBase<T> &v2)
			    { return v1.Compare(v2) >= 0; }
    friend ostream &	operator <<(ostream &os, const TLVBase<T> &v)
			    { return v.PrintOn(os); }

//protected:
    // Constructors and destructor

    TLVBase(size_t = 0);
    TLVBase(const T &);
    TLVBase(const T *, size_t);
    TLVBase(const TLVBase<T> &);
    virtual ~TLVBase();

protected:
    // Assignment operators

    TLVBase<T> &	operator =(const T &);
    TLVBase<T> &	operator =(const TLVBase<T> &);

    // Resize operations

    void 		ExpandBy(size_t);
    void 		ShrinkBy(size_t);
    void 		Resize(size_t);

    virtual index_t 	MapIndex(index_t i) const { return i; }
    virtual index_t 	UnmapIndex(index_t i) const { return i; }

    void		AssertIndex(index_t) const;

private:
    void 		Duplicate(const T *, size_t);
};

#if defined(__SC__)
    // Symantec C++ bug fix -- members are global
    extern const char *	sPre;		// TLVector prefix for print()
    extern const char *	sPreSep;	// Element prefix for print()
    extern const char *	sPostSep;	// Element postfix for print()
    extern const char *	sPost;		// TLVector postfix for print()
#endif

/*---------------------------------------------------------------------------
	TLVector<T>
---------------------------------------------------------------------------*/

template<class T> class TLVector: public TLVBase<T>
{
public:
    TLVector(size_t = 0);
    TLVector(const T &);
    TLVector(const T *, size_t);

    // Assignment operators
    TLVector<T> &	operator =(const TLVector<T> &t);

    T &			PeekAt(index_t i) { return mVect[i]; }
    const T &		PeekAt(index_t i) const { return mVect[i]; }

    void 		ExpandBy(size_t sz) { TLVBase<T>::ExpandBy(sz); }
    void 		ShrinkBy(size_t sz) { TLVBase<T>::ShrinkBy(sz); }
    void 		Resize(size_t sz) { TLVBase<T>::Resize(sz); }
};

/*---------------------------------------------------------------------------
	TLArray<T>
---------------------------------------------------------------------------*/

template<class T> class TLArray: public TLVBase<T>
{
    int 		mLower; 	// Index base

public:
    TLArray(size_t = 0);
    TLArray(index_t, index_t);
    TLArray(const T &);
    TLArray(const T *, size_t);

    TLArray<T> &	operator =(const TLArray<T> &);

    T &			PeekAt(index_t i) { return mVect[i - mLower]; }
    const T &		PeekAt(index_t i) const { return mVect[i - mLower]; }

    void 		ExpandBy(size_t sz) { TLVBase<T>::ExpandBy(sz); }
    void 		ShrinkBy(size_t sz) { TLVBase<T>::ShrinkBy(sz); }
    void 		Resize(size_t sz) { TLVBase<T>::Resize(sz); }

    // Index bounds

    virtual index_t 	Maxi() const { return mLower + Size() - 1; }
    virtual index_t 	Mini() const { return mLower; }
    index_t 		SetMini(index_t);
    void		SetLimits(index_t, index_t);

protected:
    virtual index_t 	MapIndex(index_t i) const { return i - mLower; }
    virtual index_t 	UnmapIndex(index_t i) const { return i + mLower; }
};

/*---------------------------------------------------------------------------
	TLSeqBase<T>
---------------------------------------------------------------------------*/

template<class T> class TLSeqBase: public TLVBase<T>
{
protected:
    size_t 		mCount;
    size_t 		mDelta;

public:
    TLSeqBase<T> &	operator =(const TLSeqBase<T> &);

    size_t 		Count() const { return mCount; }
    size_t 		Available() const { return mSize - mCount; }
    virtual void	RemoveAll() { mCount = 0; }
    bool  		IsEmpty() const { return mCount == 0; }
    bool  		IsFull() const { return mCount == mSize; }
    bool  		IsFlexible() const { return mDelta > 0; }
    size_t 		SetDelta(size_t);
    void 		Freeze() { mDelta = 0; }

    virtual index_t 	Mini() const { return 1; }
    virtual index_t 	Maxi() const { return mCount; }

    // Class invariant

    virtual bool  	OK() const { return 1; }

protected:
    TLSeqBase(size_t = 0, size_t = 0);
    TLSeqBase(const T &);
    TLSeqBase(const T *, size_t);

    virtual void 	Expand();
    virtual void 	CompactStorage();
    virtual index_t 	MapIndex(index_t i) const { return i - 1; }
    virtual index_t 	UnmapIndex(index_t i) const { return i + 1; }

    void		AssertNonEmpty() const;
    void		AssertNonFull();
};

/*---------------------------------------------------------------------------
	TLSeq<T>
---------------------------------------------------------------------------*/

template<class T> class TLSeq: public TLSeqBase<T>
{
    index_t		mLower;		// Index base

public:
    TLSeq(size_t = 0, size_t = 0);
    TLSeq(const T &);
    TLSeq(const T *, size_t);

    // Assignment operators

    TLSeq<T> &		operator =(const T &);
    TLSeq<T> &		operator =(const TLSeq<T> &);
    TLSeq<T> &		operator +=(const T &t) { Append(t); return *this; }
    TLSeq<T> &		operator +=(const TLSeq<T> &s) { Append(s); return *this; }
    TLSeq<T> &		operator -=(const T &t) { Extract(t); return *this; }

    // Additions of new items

    void 		Append(const T &t) { InsertAt(mCount + 1, t); }
    void 		Append(const T &, const T &);
    void 		Append(const TLSeq<T> &);
    void 		Prepend(const T &t) { InsertAt(1, t); }
    void 		Prepend(const T &, const T &);
    void 		Prepend(const TLSeq<T> &);
    void 		Insert(const T &);
    void 		InsertAt(index_t, const T &);

    // Removal of items

    T 			Extract(const T &);
    T 			ExtractAt(index_t);
    T 			ExtractFirst();
    T 			ExtractLast();
    void		Remove(const T &);
    void		RemoveAt(index_t);
    void		RemoveFirst();
    void		RemoveLast();

    // Access to items in the sequence

    T &			PeekFirst();
    const T &		PeekFirst() const;
    T &			PeekLast();
    const T &		PeekLast() const;
    T &			PeekValue(const T &);
    const T &		PeekValue(const T &) const;
    T &			PeekAt(index_t);
    const T &		PeekAt(index_t) const;

    // Membership tests

    index_t 		Search(const T &) const;
    void 		Sort();

    virtual bool  	IterFirst(iter_t &) const;
    virtual bool  	IterLast(iter_t &) const;
    virtual bool  	IterNext(iter_t &) const;
    virtual bool  	IterPrev(iter_t &) const;

    virtual index_t 	Mini() const { return 1; }
    virtual index_t 	Maxi() const { return mCount; }

    void		Reserve(size_t aSize) { Resize(aSize); mCount = aSize; }
private:
    bool  		FindInsertPos(const T &, index_t &) const;
    index_t		QPart(index_t, index_t);
    void		QSort(index_t, index_t);
};

/*---------------------------------------------------------------------------
	TLQueue<T>
---------------------------------------------------------------------------*/

template<class T> class TLQueue: public TLSeqBase<T>
{
    index_t 		mLeft, mRight;

public:
    TLQueue(size_t = 0, size_t = 0);
    TLQueue(const T &);

    // Assignment operators
    TLQueue<T> &	operator =(const T &);
    TLQueue<T> &	operator =(const TLQueue<T> &);

    void 		RemoveAll();

    // Single-ended queue operations

    void 		EnqueueUnique(const T &);
    void 		Enqueue(const T &t) { AddRight(t); }
    T 			Dequeue() { return ExtractLeft(); }
    void		RemoveHead() { RemoveLeft(); }
    T &			PeekHead() { return PeekLeft(); }
    const T &		PeekHead() const { return PeekLeft(); }
    T &			PeekTail() { return PeekRight(); }
    const T &		PeekTail() const { return PeekRight(); }

    // Double-ended queue operations

    void 		AddLeft(const T &);
    void 		AddRight(const T &);
    T 			ExtractLeft();
    T 			ExtractRight();
    void		RemoveLeft();
    void		RemoveRight();
    T &			PeekLeft();
    const T &		PeekLeft() const;
    T &			PeekRight();
    const T &		PeekRight() const;

    // Class invariant

    virtual bool  	OK() const;

    virtual bool  	IterFirst(iter_t &) const;
    virtual bool  	IterLast(iter_t &) const;
    virtual bool  	IterNext(iter_t &) const;
    virtual bool  	IterPrev(iter_t &) const;

protected:
    void 		Expand();
    void 		CompactStorage();
    virtual index_t 	MapIndex(index_t) const;
    virtual index_t 	UnmapIndex(index_t) const;
};

/*---------------------------------------------------------------------------
	TLStack<T>
---------------------------------------------------------------------------*/

template<class T> class TLStack: public TLSeqBase<T>
{
public:
    TLStack(size_t = 0, size_t = 0);
    TLStack(const T &);

    // Assignment operators

    TLStack<T> &	operator =(const T &);
    TLStack<T> &	operator =(const TLStack<T> &);

    // Adding and removing items

    void 		Push(const T &);
    T 			Pop();
    void		RemoveTop();

    // Access to stack items

    T &			PeekTop();
    const T &		PeekTop() const;

    virtual bool  	IterFirst(iter_t &) const;
    virtual bool  	IterLast(iter_t &) const;
    virtual bool  	IterNext(iter_t &) const;
    virtual bool  	IterPrev(iter_t &) const;
};

#endif	// _TLX_ARRAYS_H
