/* $Id: picture.c,v 1.1 2000/03/27 04:52:57 ktk Exp $ */

/*
 * Dynamic Windows:
 *          A GTK like implementation of the PM GUI
 *          Bitmap window class
 *
 * (C) 2000 Achim Hasenmueller <achimha@innotek.de>
 * (C) 2000 Brian Smith <dbsoft@technologist.com>
 *
 */
#define INCL_DOS
#define INCL_WIN
#define INCL_GPI

#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <io.h>
#include "picture.h"
#include "dw.h"

/*
 * Registers the bitmap window class
 * Parameters:
 *
 */
HWND dw_bitmapfile_new(ULONG id)
{
	/* static variable to check if we need to register the window class */
	static BOOL classRegistered = FALSE;

	if (!classRegistered)
	{
		/* register class and reserve user data storage for bitmap handle */
		WinRegisterClass(dwhab, "DWBITMAPFILE", dw_bitmapclass_wndproc,
		                 0, sizeof(HBITMAP));
		classRegistered = TRUE;
	}
	return WinCreateWindow(HWND_OBJECT,
	                       "DWBITMAPFILE",
	                       NULL,
	                       WS_VISIBLE,
	                       0, 0, 2000, 1000,
	                       NULLHANDLE,
	                       HWND_TOP,
	                       id,
	                       NULL,
	                       NULL);
}

/*
 * Sets the bitmap used for a given bitmap window, loaded from a bitmap file
 * Parameters:
 *       handle: Handle to the window
 *       filename: the file name that contains the bitmap file
 *
 */
BOOL dw_bitmapfile_set_bitmap(HWND handle, char *filename)
{
	HFILE BitmapFileHandle = NULLHANDLE; /* handle for the file */
	ULONG OpenAction = 0;
	PBYTE BitmapFileBegin; /* pointer to the first byte of bitmap data  */
	FILESTATUS BitmapStatus;
	ULONG cbRead;
	PBITMAPFILEHEADER2 pBitmapFileHeader;
	PBITMAPINFOHEADER2 pBitmapInfoHeader;
	ULONG ScanLines;
	HBITMAP hbm;
	HPS hps;

	/* check if we can read from this file (it exists and read permission) */
	if (access(filename, 04) != 0)
		return NULLHANDLE;
	/* open bitmap file */
	DosOpen(filename, &BitmapFileHandle, &OpenAction, 0L,
	        FILE_ARCHIVED | FILE_NORMAL | FILE_READONLY,
	        OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
	        OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY |
	        OPEN_FLAGS_NOINHERIT, 0L);
	if (BitmapFileHandle == NULLHANDLE)
		return NULLHANDLE;
	/* find out how big the file is  */
	DosQueryFileInfo(BitmapFileHandle, 1, &BitmapStatus,
	                 sizeof(BitmapStatus));
	/* allocate memory to load the bitmap */
	DosAllocMem((PPVOID)&BitmapFileBegin, (ULONG)BitmapStatus.cbFile,
	            PAG_READ | PAG_WRITE | PAG_COMMIT);
	/* read bitmap file into memory buffer */
	DosRead(BitmapFileHandle, (PVOID)BitmapFileBegin,
	        BitmapStatus.cbFile, &cbRead);
	/* access first bytes as bitmap header */
	pBitmapFileHeader = (PBITMAPFILEHEADER2)BitmapFileBegin;
	/* check if it's a valid bitmap data file */
	if ((pBitmapFileHeader->usType != BFT_BITMAPARRAY) &&
	    (pBitmapFileHeader->usType != BFT_BMAP))
	{
		/* free memory of bitmap file buffer */
		DosFreeMem(BitmapFileBegin);
		/* close the bitmap file */
		DosClose(BitmapFileHandle);
		return NULLHANDLE;
	}
	/* check if it's a file with multiple bitmaps */
	if (pBitmapFileHeader->usType == BFT_BITMAPARRAY)
	{
		/* we'll just use the first bitmap and ignore the others */
		pBitmapFileHeader = &(((PBITMAPARRAYFILEHEADER2)BitmapFileBegin)->bfh2);
	}
	/* set pointer to bitmap information block */
	pBitmapInfoHeader = &pBitmapFileHeader->bmp2;
	/* find out if it's the new 2.0 format or the old format */
	/* and query number of lines */
	if (pBitmapInfoHeader->cbFix == sizeof(BITMAPINFOHEADER))
		ScanLines = (ULONG)((PBITMAPINFOHEADER)pBitmapInfoHeader)->cy;
	else
		ScanLines = pBitmapInfoHeader->cy;
	/* now we need a presentation space, get it from static control */
	hps = WinGetPS(handle);
	/* create bitmap now using the parameters from the info block */
	hbm = GpiCreateBitmap(hps, pBitmapInfoHeader, 0L, NULL, NULL);
	/* select the new bitmap into presentation space */
	GpiSetBitmap(hps, hbm);
	/* now copy the bitmap data into the bitmap */
	GpiSetBitmapBits(hps, 0L, ScanLines,
	                 BitmapFileBegin + pBitmapFileHeader->offBits,
	                 (PBITMAPINFO2)pBitmapInfoHeader);
	/* free memory of bitmap file buffer */
	DosFreeMem(BitmapFileBegin);
	/* close the bitmap file */
	DosClose(BitmapFileHandle);

	/* tell static control to use that bitmap, make sure it is an icon control */
	WinSendMsg(handle, SM_SETHANDLE, MPFROMP(hbm), NULL);
	return TRUE;
} 


/*
 * window class procedure
 * Parameters:
 *       hwnd - window handle of the window instance
 *       msg - window message to handle
 *       mp1 - first window message parameter
 *       mp2 - second window message parameter
 * Returns:
 *       message specific return value
 *
 */
MRESULT EXPENTRY dw_bitmapclass_wndproc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
	switch (msg)
	{
		case WM_CREATE:
		{
			WinDefWindowProc(hwnd, msg, mp1, mp2);
			/* reset bitmap handle to NULLHANDLE */
			WinSetWindowULong(hwnd, QWL_USER, NULLHANDLE);
			return (MRESULT)FALSE;
		}

		case WM_DESTROY:
		{
			HBITMAP bitmapHandle;
			bitmapHandle = (HBITMAP)WinQueryWindowULong(hwnd, QWL_USER);
			/* check if we have a bitmap associated */
			if (bitmapHandle != NULLHANDLE)
				GpiDeleteBitmap(bitmapHandle);
			return (MRESULT)TRUE;
		}

		case WM_PAINT:
		{
			HPS hps;
			HBITMAP bitmapHandle;
			RECTL UpdtRegion;
			hps = WinBeginPaint(hwnd, NULLHANDLE, &UpdtRegion);
			/* get handle of bitmap from window words */
			bitmapHandle = (HBITMAP)WinQueryWindowULong(hwnd, QWL_USER);
			if (bitmapHandle != NULLHANDLE)
			{
				RECTL DestRectl;
				/* get the window size into the destination rectangle */
				WinQueryWindowRect(hwnd, &DestRectl);
				/* draw bitmap stretched to window size */
				WinDrawBitmap(hps, bitmapHandle, NULL,
				              (PPOINTL)&DestRectl, 0, 0, DBM_STRETCH);
			} else
			{
				/* we have no bitmap, paint pale gray */
				WinFillRect(hps, &UpdtRegion, CLR_PALEGRAY);
			}
			WinEndPaint(hps);
			return (MRESULT)FALSE;
		}

		case SM_SETHANDLE:
		{
			/* store handle of new bitmap in window words */
			WinSetWindowULong(hwnd, QWL_USER, (ULONG)mp1);
			return (MRESULT)TRUE;
		}

		default:
			return WinDefWindowProc(hwnd, msg, mp1, mp2);
	}
}
