/* kill.c -- kill ring management. */

/* Copyright (C) 1994 Free Software Foundation, Inc.

   This file is part of the GNU Readline Library, a library for
   reading lines of text with interactive input and history editing.

   The GNU Readline Library is free software; you can redistribute it
   and/or modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2, or
   (at your option) any later version.

   The GNU Readline Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */
#define READLINE_LIBRARY

#if defined (HAVE_CONFIG_H)
#  include <config.h>
#endif

#include <sys/types.h>

#if defined (HAVE_UNISTD_H)
#  include <unistd.h>           /* for _POSIX_VERSION */
#endif /* HAVE_UNISTD_H */

#if defined (HAVE_STDLIB_H)
#  include <stdlib.h>
#else
#  include "ansi_stdlib.h"
#endif /* HAVE_STDLIB_H */

#include <stdio.h>

/* System-specific feature definitions and include files. */
#include "rldefs.h"

/* Some standard library routines. */
#include "readline.h"
#include "history.h"

#include "rlprivate.h"
#include "xmalloc.h"

/* **************************************************************** */
/*								    */
/*			Killing Mechanism			    */
/*								    */
/* **************************************************************** */

/* What we assume for a max number of kills. */
#define DEFAULT_MAX_KILLS 10

/* The real variable to look at to find out when to flush kills. */
static int rl_max_kills =  DEFAULT_MAX_KILLS;

/* Where to store killed text. */
static char **rl_kill_ring = (char **)NULL;

/* Where we are in the kill ring. */
static int rl_kill_index;

/* How many slots we have in the kill ring. */
static int rl_kill_ring_length;

/* How to say that you only want to save a certain amount
   of kill material. */
int
rl_set_retained_kills (num)
     int num;
{
  return 0;
}

/* Add TEXT to the kill ring, allocating a new kill ring slot as necessary.
   This uses TEXT directly, so the caller must not free it.  If APPEND is
   non-zero, and the last command was a kill, the text is appended to the
   current kill ring slot, otherwise prepended. */
static int
_rl_copy_to_kill_ring (text, append)
     char *text;
     int append;
{
  char *old, *new;
  int slot;

  /* First, find the slot to work with. */
  if (_rl_last_command_was_kill == 0)
    {
      /* Get a new slot.  */
      if (rl_kill_ring == 0)
	{
	  /* If we don't have any defined, then make one. */
	  rl_kill_ring = (char **)
	    xmalloc (((rl_kill_ring_length = 1) + 1) * sizeof (char *));
	  rl_kill_ring[slot = 0] = (char *)NULL;
	}
      else
	{
	  /* We have to add a new slot on the end, unless we have
	     exceeded the max limit for remembering kills. */
	  slot = rl_kill_ring_length;
	  if (slot == rl_max_kills)
	    {
	      register int i;
	      free (rl_kill_ring[0]);
	      for (i = 0; i < slot; i++)
		rl_kill_ring[i] = rl_kill_ring[i + 1];
	    }
	  else
	    {
	      slot = rl_kill_ring_length += 1;
	      rl_kill_ring = (char **)xrealloc (rl_kill_ring, slot * sizeof (char *));
	    }
	  rl_kill_ring[--slot] = (char *)NULL;
	}
    }
  else
    slot = rl_kill_ring_length - 1;

  /* If the last command was a kill, prepend or append. */
  if (_rl_last_command_was_kill && rl_editing_mode != vi_mode)
    {
      old = rl_kill_ring[slot];
      new = xmalloc (1 + strlen (old) + strlen (text));

      if (append)
	{
	  strcpy (new, old);
	  strcat (new, text);
	}
      else
	{
	  strcpy (new, text);
	  strcat (new, old);
	}
      free (old);
      free (text);
      rl_kill_ring[slot] = new;
    }
  else
    rl_kill_ring[slot] = text;

  rl_kill_index = slot;
  return 0;
}

/* The way to kill something.  This appends or prepends to the last
   kill, if the last command was a kill command.  if FROM is less
   than TO, then the text is appended, otherwise prepended.  If the
   last command was not a kill command, then a new slot is made for
   this kill. */
int
rl_kill_text (from, to)
     int from, to;
{
  char *text;

  /* Is there anything to kill? */
  if (from == to)
    {
      _rl_last_command_was_kill++;
      return 0;
    }

  text = rl_copy_text (from, to);

  /* Delete the copied text from the line. */
  rl_delete_text (from, to);

  _rl_copy_to_kill_ring (text, from < to);

  _rl_last_command_was_kill++;
  return 0;
}

/* Now REMEMBER!  In order to do prepending or appending correctly, kill
   commands always make rl_point's original position be the FROM argument,
   and rl_point's extent be the TO argument. */

/* **************************************************************** */
/*								    */
/*			Killing Commands			    */
/*								    */
/* **************************************************************** */

/* Delete the word at point, saving the text in the kill ring. */
int
rl_kill_word (count, key)
     int count, key;
{
  int orig_point = rl_point;

  if (count < 0)
    return (rl_backward_kill_word (-count, key));
  else
    {
      rl_forward_word (count, key);

      if (rl_point != orig_point)
	rl_kill_text (orig_point, rl_point);

      rl_point = orig_point;
    }
  return 0;
}

/* Rubout the word before point, placing it on the kill ring. */
int
rl_backward_kill_word (count, ignore)
     int count, ignore;
{
  int orig_point = rl_point;

  if (count < 0)
    return (rl_kill_word (-count, ignore));
  else
    {
      rl_backward_word (count, ignore);

      if (rl_point != orig_point)
	rl_kill_text (orig_point, rl_point);
    }
  return 0;
}

/* Kill from here to the end of the line.  If DIRECTION is negative, kill
   back to the line start instead. */
int
rl_kill_line (direction, ignore)
     int direction, ignore;
{
  int orig_point = rl_point;

  if (direction < 0)
    return (rl_backward_kill_line (1, ignore));
  else
    {
      rl_end_of_line (1, ignore);
      if (orig_point != rl_point)
	rl_kill_text (orig_point, rl_point);
      rl_point = orig_point;
    }
  return 0;
}

/* Kill backwards to the start of the line.  If DIRECTION is negative, kill
   forwards to the line end instead. */
int
rl_backward_kill_line (direction, ignore)
     int direction, ignore;
{
  int orig_point = rl_point;

  if (direction < 0)
    return (rl_kill_line (1, ignore));
  else
    {
      if (!rl_point)
	ding ();
      else
	{
	  rl_beg_of_line (1, ignore);
	  rl_kill_text (orig_point, rl_point);
	}
    }
  return 0;
}

/* Kill the whole line, no matter where point is. */
int
rl_kill_full_line (count, ignore)
     int count, ignore;
{
  rl_begin_undo_group ();
  rl_point = 0;
  rl_kill_text (rl_point, rl_end);
  rl_end_undo_group ();
  return 0;
}

/* The next two functions mimic unix line editing behaviour, except they
   save the deleted text on the kill ring.  This is safer than not saving
   it, and since we have a ring, nobody should get screwed. */

/* This does what C-w does in Unix.  We can't prevent people from
   using behaviour that they expect. */
int
rl_unix_word_rubout (count, key)
     int count, key;
{
  int orig_point;

  if (rl_point == 0)
    ding ();
  else
    {
      orig_point = rl_point;
      if (count <= 0)
	count = 1;

      while (count--)
	{
	  while (rl_point && whitespace (rl_line_buffer[rl_point - 1]))
	    rl_point--;

	  while (rl_point && (whitespace (rl_line_buffer[rl_point - 1]) == 0))
	    rl_point--;
	}

      rl_kill_text (orig_point, rl_point);
    }
  return 0;
}

/* Here is C-u doing what Unix does.  You don't *have* to use these
   key-bindings.  We have a choice of killing the entire line, or
   killing from where we are to the start of the line.  We choose the
   latter, because if you are a Unix weenie, then you haven't backspaced
   into the line at all, and if you aren't, then you know what you are
   doing. */
int
rl_unix_line_discard (count, key)
     int count, key;
{
  if (rl_point == 0)
    ding ();
  else
    {
      rl_kill_text (rl_point, 0);
      rl_point = 0;
    }
  return 0;
}

/* Copy the text in the `region' to the kill ring.  If DELETE is non-zero,
   delete the text from the line as well. */
static int
region_kill_internal (delete)
     int delete;
{
  char *text;

  if (rl_mark == rl_point)
    {
      _rl_last_command_was_kill++;
      return 0;
    }

  text = rl_copy_text (rl_point, rl_mark);
  if (delete)
    rl_delete_text (rl_point, rl_mark);
  _rl_copy_to_kill_ring (text, rl_point < rl_mark);

  _rl_last_command_was_kill++;
  return 0;
}

/* Copy the text in the region to the kill ring. */
int
rl_copy_region_to_kill (count, ignore)
     int count, ignore;
{
  return (region_kill_internal (0));
}

/* Kill the text between the point and mark. */
int
rl_kill_region (count, ignore)
     int count, ignore;
{
  int r, npoint;

  npoint = (rl_point < rl_mark) ? rl_point : rl_mark;
  r = region_kill_internal (1);
  _rl_fix_point (1);
  rl_point = npoint;
  return r;
}

/* Copy COUNT words to the kill ring.  DIR says which direction we look
   to find the words. */
static int
_rl_copy_word_as_kill (count, dir)
     int count, dir;
{
  int om, op, r;

  om = rl_mark;
  op = rl_point;

  if (dir > 0)
    rl_forward_word (count, 0);
  else
    rl_backward_word (count, 0);

  rl_mark = rl_point;

  if (dir > 0)
    rl_backward_word (count, 0);
  else
    rl_forward_word (count, 0);

  r = region_kill_internal (0);

  rl_mark = om;
  rl_point = op;

  return r;
}

int
rl_copy_forward_word (count, key)
     int count, key;
{
  if (count < 0)
    return (rl_copy_backward_word (-count, key));

  return (_rl_copy_word_as_kill (count, 1));
}

int
rl_copy_backward_word (count, key)
     int count, key;
{
  if (count < 0)
    return (rl_copy_forward_word (-count, key));

  return (_rl_copy_word_as_kill (count, -1));
}
  
/* Yank back the last killed text.  This ignores arguments. */
int
rl_yank (count, ignore)
     int count, ignore;
{
  if (rl_kill_ring == 0)
    {
      _rl_abort_internal ();
      return -1;
    }

  _rl_set_mark_at_pos (rl_point);
  rl_insert_text (rl_kill_ring[rl_kill_index]);
  return 0;
}

/* If the last command was yank, or yank_pop, and the text just
   before point is identical to the current kill item, then
   delete that text from the line, rotate the index down, and
   yank back some other text. */
int
rl_yank_pop (count, key)
     int count, key;
{
  int l, n;

  if (((rl_last_func != rl_yank_pop) && (rl_last_func != rl_yank)) ||
      !rl_kill_ring)
    {
      _rl_abort_internal ();
      return -1;
    }

  l = strlen (rl_kill_ring[rl_kill_index]);
  n = rl_point - l;
  if (n >= 0 && STREQN (rl_line_buffer + n, rl_kill_ring[rl_kill_index], l))
    {
      rl_delete_text (n, rl_point);
      rl_point = n;
      rl_kill_index--;
      if (rl_kill_index < 0)
	rl_kill_index = rl_kill_ring_length - 1;
      rl_yank (1, 0);
      return 0;
    }
  else
    {
      _rl_abort_internal ();
      return -1;
    }
}

/* Yank the COUNTh argument from the previous history line, skipping
   HISTORY_SKIP lines before looking for the `previous line'. */
static int
rl_yank_nth_arg_internal (count, ignore, history_skip)
     int count, ignore, history_skip;
{
  register HIST_ENTRY *entry;
  char *arg;
  int i, pos;

  pos = where_history ();

  if (history_skip)
    {
      for (i = 0; i < history_skip; i++)
	entry = previous_history ();
    }

  entry = previous_history ();

  history_set_pos (pos);

  if (entry == 0)
    {
      ding ();
      return -1;
    }

  arg = history_arg_extract (count, count, entry->line);
  if (!arg || !*arg)
    {
      ding ();
      return -1;
    }

  rl_begin_undo_group ();

#if defined (VI_MODE)
  /* Vi mode always inserts a space before yanking the argument, and it
     inserts it right *after* rl_point. */
  if (rl_editing_mode == vi_mode)
    {
      rl_vi_append_mode (1, ignore);
      rl_insert_text (" ");
    }
#endif /* VI_MODE */

  rl_insert_text (arg);
  free (arg);

  rl_end_undo_group ();
  return 0;
}

/* Yank the COUNTth argument from the previous history line. */
int
rl_yank_nth_arg (count, ignore)
     int count, ignore;
{
  return (rl_yank_nth_arg_internal (count, ignore, 0));
}

/* Yank the last argument from the previous history line.  This `knows'
   how rl_yank_nth_arg treats a count of `$'.  With an argument, this
   behaves the same as rl_yank_nth_arg. */
int
rl_yank_last_arg (count, key)
     int count, key;
{
  static int history_skip = 0;
  static int explicit_arg_p = 0;
  static int count_passed = 1;
  static int direction = 1;
  static int undo_needed = 0;
  int retval;

  if (rl_last_func != rl_yank_last_arg)
    {
      history_skip = 0;
      explicit_arg_p = rl_explicit_arg;
      count_passed = count;
      direction = 1;
    }
  else
    {
      if (undo_needed)
	rl_do_undo ();
      if (count < 1)
        direction = -direction;
      history_skip += direction;
      if (history_skip < 0)
	history_skip = 0;
    }
 
  if (explicit_arg_p)
    retval = rl_yank_nth_arg_internal (count_passed, key, history_skip);
  else
    retval = rl_yank_nth_arg_internal ('$', key, history_skip);

  undo_needed = retval == 0;
  return retval;
}

/* A special paste command for users of some PC's. */
#if defined (__CYGWIN32__)
#  include <windows.h>
#else
# ifdef __EMX__

#  define INCL_PM			/* I want some PM functions.. */
#  define INCL_DOSPROCESS		/* TIB PIB. */
#  include <os2.h>

static int proc_type = -1;
static PPIB pib;
HAB hab;
HMQ hmq;

BOOL (APIENTRY *_WinCancelShutdown)(HMQ hmq, BOOL fCancelAlways);
BOOL (APIENTRY *_WinCloseClipbrd)(HAB);
HMQ  (APIENTRY *_WinCreateMsgQueue)(HAB, LONG);
BOOL (APIENTRY *_WinDestroyMsgQueue)(HMQ);
BOOL (APIENTRY *_WinEmptyClipbrd)(HAB);
HAB  (APIENTRY *_WinInitialize)(ULONG);
BOOL (APIENTRY *_WinOpenClipbrd)(HAB);
ULONG(APIENTRY *_WinQueryClipbrdData)(HAB, ULONG);
BOOL (APIENTRY *_WinQueryClipbrdFmtInfo)(HAB, ULONG, PULONG);
BOOL (APIENTRY *_WinSetClipbrdData)(HAB, ULONG, ULONG, ULONG);
BOOL (APIENTRY *_inTerminate)(HAB);

static void
unmorph_PM ()
{
    if (hmq)
	_WinDestroyMsgQueue(hmq);
    hmq = 0;
    pib->pib_ultype = proc_type;
}

static void
morph_PM ()
{
    PTIB tib;
    int first = 0;

    if (proc_type == -1) {
	APIRET rc;
	HMODULE hMte = 0;
	char loadErr[260];			/* MAXPATHLEN */

	DosGetInfoBlocks(&tib, &pib);
	proc_type = pib->pib_ultype;
	first = 1;
        if (proc_type < 2) { /* full screen session */
            HMTX hMtx = 0;

            if (DosOpenMutexSem("\\SEM32\\PMDRAG.SEM", &hMtx) == 0)
                DosCloseMutexSem(hMtx);
	    else			/* No PM present */
		return;
	}
        if (DosLoadModule(loadErr, 256, "PMWIN", &hMte) != 0)
            return;

        rc = DosQueryProcAddr(hMte, 705, 0, (PFN*)&_WinCancelShutdown);
        rc = DosQueryProcAddr(hMte, 707, 0, (PFN*)&_WinCloseClipbrd);
        rc = DosQueryProcAddr(hMte, 716, 0, (PFN*)&_WinCreateMsgQueue);
        rc = DosQueryProcAddr(hMte, 726, 0, (PFN*)&_WinDestroyMsgQueue);
        rc = DosQueryProcAddr(hMte, 733, 0, (PFN*)&_WinEmptyClipbrd);
        rc = DosQueryProcAddr(hMte, 763, 0, (PFN*)&_WinInitialize);
        rc = DosQueryProcAddr(hMte, 793, 0, (PFN*)&_WinOpenClipbrd);
        rc = DosQueryProcAddr(hMte, 806, 0, (PFN*)&_WinQueryClipbrdData);
        rc = DosQueryProcAddr(hMte, 807, 0, (PFN*)&_WinQueryClipbrdFmtInfo);
        rc = DosQueryProcAddr(hMte, 854, 0, (PFN*)&_WinSetClipbrdData);
#if 0
        rc = DosQueryProcAddr(hMte, 888, 0, (PFN*)&_inTerminate);
        rc = DosQueryProcAddr(hMte, 841, 0, (PFN*)&_inQueryWindowText);
        rc = DosQueryProcAddr(hMte, 877, 0, (PFN*)&_inSetWindowText);
#endif
    }
    if (pib->pib_ultype != 3)		/* 2 is VIO */
	pib->pib_ultype = 3;		/* 3 is PM */
    if (first)
	hab = _WinInitialize(0);
    /* 64 messages if before OS/2 3.0, ignored otherwise */
    hmq = _WinCreateMsgQueue(hab, 64);
    if (!hmq)
      {
	unmorph_PM ();
	return;
      }
    _WinCancelShutdown(hmq, 1);	/* Do not inform us on shutdown */
}

static int clip_open;

static int
put_clip (s)
     char * s;
{
    int sz = strlen(s) + 1;
    int ret = EOF, nl = 0;
    char *pByte = 0, *s1 = s, c, *t;

    while ((c = *s1++)) {
	if (c == '\r' && *s1 == '\n')
	    s1++;
	else if (c == '\n')
	    nl++;
    }
    if (DosAllocSharedMem((PPVOID)&pByte, 0, sz + nl,
			  PAG_WRITE | PAG_COMMIT | OBJ_GIVEABLE | OBJ_GETTABLE))
	return ret;

    if (!nl)
	memcpy(pByte, s, sz);
    else {
	t = pByte;
	while ((c = *t++ = *s++))
	    if (c == '\n' && (t == pByte + 1 || t[-2] != '\r'))
		t[-1] = '\r', *t++ = '\n';
    }

    morph_PM();
    if(!hmq)
        goto fail;

    _WinOpenClipbrd(hab);
    _WinEmptyClipbrd(hab);
    if (_WinSetClipbrdData(hab, (ULONG) pByte, CF_TEXT, CFI_POINTER))
	ret = 0;
    _WinCloseClipbrd(hab);
    unmorph_PM();
    if (ret == 0)
	return 0;
  fail:
    DosFreeMem((PPVOID)&pByte);
    return -1;
}

static void
CloseClipboard ()
{
    if (!clip_open)
	return;
    _WinCloseClipbrd (hab);
    clip_open = 0;
    unmorph_PM();
}

static char*
GetClipboardData (unused)
     int unused;
{
    char *ClipData;
    ULONG ulFormat;
    int sz;

    morph_PM();
    if(!hmq)
        return 0;
    if (clip_open)
	CloseClipboard ();

    _WinQueryClipbrdFmtInfo (hab, CF_TEXT, &ulFormat);
    if(ulFormat != CFI_POINTER) {
	unmorph_PM();
	return 0;
    }
    _WinOpenClipbrd(hab);
    clip_open = 1;
    ClipData = (char *)_WinQueryClipbrdData(hab, CF_TEXT);
    sz = strlen(ClipData);
    if(!ClipData || !sz) {
        CloseClipboard();
	return 0;
    }
    return ClipData;
}

#define OpenClipboard(ignore)	1

# else /* !__EMX__ */

static FILE* paste_handle = 0;
static char *paste_buf = NULL;

static void
CloseClipboard ()
{
  if (paste_handle >= 0)
    pclose(paste_handle);
  if (paste_buf)
    xfree (paste_buf);
}

static int
OpenClipboard(ignore)
     int ignore;
{
  char *cmd = getenv ("RL_PASTE_CMD");

  if (paste_handle)
    pclose(paste_handle);
  if (!cmd)
    return 0;

  paste_handle = popen(cmd, "rt");
  if (!paste_handle)
    return 0;
  return 1;
}

#define PASTE_BUFFER 1008
#define CF_TEXT 0			/* Not used */

static char*
GetClipboardData (unused)
     int unused;
{
  char *ClipData;
  int len, size = PASTE_BUFFER;
  int off = 0;

  if (!paste_handle)
    return NULL;  
  if (paste_buf)
    xfree (paste_buf);
  paste_buf = (char*)xmalloc (PASTE_BUFFER);
  while (1)
    {
      len = fread (paste_buf + off, 1, PASTE_BUFFER - 1, paste_handle);
      paste_buf[off + len] = '\0';
      if (len < PASTE_BUFFER - 1)
	break;
      if (strchr (paste_buf + off, '\r') || strchr (paste_buf + off, '\n'))
	break;
      paste_buf = xrealloc (paste_buf, size += PASTE_BUFFER - 1);
      off += len;
    } 
  return paste_buf;
}

static int
put_clip (s)
     char * s;
{
  char *cmd = getenv ("RL_CLCOPY_CMD");
  FILE *fh;
  int l = strlen(s), res;

  if (!cmd)
    return -1;

  fh = popen (cmd, "wt");
  if (!fh)
    return -1;
  res = fwrite (s, 1, l, fh);
  if (pclose (fh) != 0 || res != l)
    return -1;
  return 0;
}

# endif /* !__EMX__ */
#endif /* __CYGWIN32__ */

#ifdef CAN_PASTE_CLIPBOARD

int
rl_paste_from_clipboard (count, key)
     int count, key;
{
  char *data, *ptr;
  int len;

  if (OpenClipboard (NULL) == 0)
    return (-1);

  data = (char *)GetClipboardData (CF_TEXT);
  if (data)
    {
      ptr = strchr (data, '\r');
      if (!ptr)
	ptr = strchr (data, '\n');
      if (ptr)
	{
	  len = ptr - data;
	  ptr = xmalloc (len + 1);
	  ptr[len] = '\0';
	  strncpy (ptr, data, len);
	}
      else
        ptr = data;
      rl_insert_text (ptr);
      if (ptr != data)
	free (ptr);
      CloseClipboard ();
    }
  return (0);
}
#endif

#ifdef CAN_COPY_CLIPBOARD

int
rl_last_kill_to_clipboard (count, key)
     int count, key;
{
  char *data, *ptr;
  int len;

  if (rl_kill_ring == 0)
    {
      _rl_abort_internal ();
      return -1;
    }

  if (put_clip (rl_kill_ring[rl_kill_index]) < 0)
      return -1;
  return 0;
}


int
rl_copy_to_clipboard (count, key)
     int count, key;
{
  if (region_kill_internal (0) >= 0
      && rl_last_kill_to_clipboard (count, key) >= 0)
    return 0;
  return -1;
}

int
rl_cut_to_clipboard (count, key)
     int count, key;
{
  if (rl_kill_region (count, key) >= 0
      && rl_last_kill_to_clipboard (count, key) >= 0)
    return 0;
  return -1;
}
#endif

