######################################################################
#               REXX API Extension Module for Python
#             by Jeff Rush <jrush@summit-research.com>
#
# Example of Usage:
#
#     import string
#     from rexx import REXXModule, EnterREXX, rexxvar
#
#     def main():
#         rxutil = REXXModule('rexxutil')
#         print "You are running OS/2 %s" % rxutil.SYSOS2VER()
#
#         # Demonstrate Access to REXX Stem Variables
#         rxutil.SYSQUERYCLASSLIST('wpsclasses.')
#         numclasses = string.atoi( rexxvar['wpsclasses.0'] )
#
#         for i in range(numclasses):
#             print rexxvar['wpsclasses.%d' % i]
#
#     if __name__ == "__main__":
#         EnterREXX(main)  # Run main Function Inside REXX Environment
#
# Known REXX DLLs for OS/2: (Some APIs Duplicate Existing Python Modules)
#
#  Library and API Reference  Functionality
#
#  \os2\dll\rexxutil.dll      System API and Some Workplace Shell Stuff
#  \os2\books\rexx.inf        (Also Access to Extended Attributes on Files)
#
#  \mmos2\dll\mciapi.dll      Multimedia API
#  \mmos2\mcirexx.inf
#
#  \tcpip\dll\rxftp.dll       TCP/IP FTP API
#  \tcpip\help\rxftp.inf
#
#  \tcpip\dll\rxsocket.dll    TCP/IP Sockets API
#  \tcpip\help\rxsocket.inf
#
#  \db2\sqllib\dll\db2ar.dll  IBM DB2/2 Database (SQL Commands and Such)
#  \db2\sqllib\book\*.inf
#
#         There are numerous other freeware DLLs available.
#          (serial I/O, NetBIOS, Lan Server, POP3, etc.)
#
# Notes:
#   REXX always returns strings.  You must convert values to other
#   Python types yourself, according to the meaning of the result.
#
#   REXX also requires strings for all arguments, but we handle any
#   necessary conversions to strings for you in the wrapper classes.
#
#   REXX variable names are case-INsensitive.  Internally they are
#   always converted to uppercase.  REXX function names are case
#   SENSITIVE because of a few DLLs I've found that used mixed cases.
#   Most DLLs export the names as all UPPERCASE so try that first.
#   You can dump the DLLs and see the names in their exported form.
#
#   I have not provided the ability to construct REXX source code
#   under Python and invoke it, nor any ability for REXX scripts to
#   call back into Python.  Such would be relatively easy to add but
#   I could think of no reason to do so as Python is powerful enough.
#
# ----
# Modification History (Latest at Top Please):
#
# 24 Nov 97 jrr Original
#
# ----
# Contributor Key:
#   jrr Jeff Rush <jrush@summit-research.com>
#
# ----
# (C) Copyright 1997 Tau Productions Inc.
#
# This source, binary and documentation are hereby placed into the
# Public Domain in honor of the people who created/maintain the
# Python language and then made it available to all.  Thanks.
#
######################################################################

import string
from types import StringType

import PyREXX

##########
#
# This function tricks REXX into thinking it is executing a REXX script
# so that we can regain control on the inside and issue calls to REXX
# to manipulate that script's variable pool.  Such a pool only exists
# in a script's execution context and since many REXX DLLs take
# parameters/return results using REXX stem variables, pool
# manipulation is necessary.
#
# I have tried to make this Python <-> REXX barrier as transparent as
# possible by allowing you to (a) pass in arguments, (b) get back a
# return value and (c) raise Python exceptions from inside and have
# them propagate outward through the barrier (unless caught earlier).
#
##########

# Execute a Python Function Inside the REXX Interpreter Environment
def EnterREXX(funcobject, *args):               # Convert Args Tail to Tuple
    return PyREXX.EnterREXX(funcobject, args)  # And Pass On to C Extension

##########
#
# This class is a wrapper around the per-thread REXX variable pool,
# which only exists as long as a REXX script is executing.  In our
# case, this means it only exists in the Python functions inside of
# a called to EnterREXX().
#
# We make this variable pool look like a dictionary, such that the
# subscript operator fetches and stores elements, all of this must
# be strings.
#
# If you are running Python with multiple threads, each doing REXX
# operations, you will have to instantiate a pool in each thread.
# We provide a default pool at the bottom of this file, named 'rexxvar'.
#
##########

class REXXVariablePool:

    # self['name'] --> value
    def __getitem__(self, key):
        return PyREXX.getvar( string.upper(key) )

    # self['name'] = value
    def __setitem__(self, key, value):
        if type(value) != StringType:
            value = `value` # REXX Requires All Values to Be Strings
        return PyREXX.setvar( string.upper(key), value )

    # del self['name']
    def __delitem__(self, key):
        return PyREXX.setvar( string.upper(key) )

# Provide Dictionary-Like Access to the Thread-Global REXX Variable Pool
rexxvar = REXXVariablePool()

##########
#
# This class is a wrapper around a DLL that contains functions callable
# by a REXX script or 'a REXX DLL', for short.  The class loads the
# DLL at __init__ time, and dynamically hands back instances of the
# REXXMethod Python class whenever a REXX function is invoked on the
# DLL in attribute format:
#
#     rxutil = REXXModule('rexxutil')
#     print "You are running OS/2 %s" % rxutil.SysOS2Ver()
#
# A cache of previously instantiated REXXMethods is kept for performance
# reasons, because the creation of each REXXMethod requires an OS/2
# kernel call, which may need to read part of the REXX DLL from disk.
#
# Notes:
#   As long as any REXXMethod instances exist, the reference count on
#   the REXX DLL will be non-zero and the DLL will be kept loaded by OS/2.
#
##########

class REXXModule:

    def __init__(self, dllname):
	self.rexxdll   = PyREXX.REXXlibrary(dllname)
        self.__name__  = dllname
        self.methcache = {}

    def __repr__(self):
        return "<REXXModule '%s'>" % self.__name__

    # Look Up REXX Method and Return REXX Method Object if Found
    def __getattr__(self, name):
        try:
            return self.methcache[name]
        except:
            if self.rexxdll.has_function(name):
                meth = REXXMethod(self.__name__, name)
                self.methcache[name] = meth
                return meth
            else:
                raise AttributeError, name

##########
#
# This class is a hidden wrapper around a single function within a
# REXX DLL.  It is hidden in that you don't instantiate it directly
# but rather via an attribute call on an instance of a REXXModule.
#
# It exists to convert calls on the instances into calls into the
# REXX DLL, using the Python magic method name __call__.
#
##########

class REXXMethod:

    def __init__(self, dllname, funcname):
        self.rexxfunc = PyREXX.REXXfunction(dllname, funcname)
        self.dllname  = dllname
        self.__name__ = funcname

    def __repr__(self):
        return "<REXXMethod '%s.%s'>" % (self.dllname, self.__name__)

    def __call__(self, *args):
        rxargs = [] # REXX Requires All Arguments to Be Strings
        for arg in args:
            if type(arg) == StringType: rxargs.append(arg)
            else:                       rxargs.append(`arg`)
        return apply(self.rexxfunc.invoke, tuple(rxargs))

