/*@file***********************************************************************
 *
 *              (C) Copyright 1993 Measurex Automation Systems
 *              This notice does not waive trade secret rights.
 *
 *****************************************************************************
 *
 *          $Header$
 *
 *          $Source$
 *        $Revision$
 *            $Date$
 *           $State$
 *
 *          $Author$  Bill Cowden
 *          $Locker$
 *
 *****************************************************************************
 *
 * DESCRIPTION:
 *  Command file which will perform PlantWorks installation
 *  checking.  Currently, the checking is done on the
 *  following areas:
 *     I/O LIMITS   (IOLIMCHK.EXE)   arrays    = 320 points per array
 *                                   packets   = 512 points per packet
 *
 *     ALARM LIMITS (ALMLIMCK.EXE)   templates = 507 templates
 *
 * OUTPUT:
 *  stdout - redirected to [d:]\FF\INSTCHK.LOG
 *           Contains a list of arrays/packets/alarm templates
 *           that do not meet the limit criteria.
 *
 *? HELP SECTION
 *  Command file that will perform PlantWorks installation
 *  verification.  The checking is done on the following area(s):
 *     I/O LIMITS      arrays    = 320 points per array
 *                     packets   = 512 points per packet
 *
 *     ALARM LIMITS    templates = 507 templates
 *
 *  Output from this command will be written to
 *  [d:]\FF\INSTCHK.LOG.
 *
 *  Warning messages will indicate if any limits have been
 *  exceeded or if any run-time errors have been encountered.
 *?
 ****************************************************************************/

arg help obnoxious debug .

if ( obnoxious <> '' ) then do
    if ( obnoxious <> 0 ) then
        obnoxious = 1;
end
else
    obnoxious = 0;

if ( debug <> '' ) then do
    if ( debug <> 0 ) then
        debug = 1;
end
else
    debug = 0;


/*
 *  INITIALIZE PROGRAM VARIABLES.
 */
call Init;


/*
 *  DISPLAY BANNER MESSAGE FOR USER.
 */
call Greeting;


/*
 *  IF USER ENTERED '?', THEN DISPLAY HELP DESCRIPTION.
 */
if ( help == '?' ) then
    call showHelp;


/*
 *  LET USER KNOW WHAT PROGRAM IS GOING TO DO.  ASK IF HE/SHE
 *  WANTS TO CONTINUE.
 */
if ( help <> '?' ) then
    say 'This program will validate the PlantWorks installation';

say;

if ( getResponse("Do you wish to continue?", "Y/N", 1, "Y") == "N" ) then
    call eoj 'PlantWorks Install Verification cancelled'


/*
 *  IS PLANTWORKS RUNNING?  IF NOT, THEN EXIT.
 */
if ( getResponse("Is PlantWorks currently running?", "Y/N", 1) == "N" ) then
    call eoj "Start PlantWorks then re-run" fn


/*
 *  LOOK FOR PLANTWORKS DRIVE.  IF NOT FOUND, THEN EXIT; OTHERWISE,
 *  ASK USER IF THE DRIVE LETTER IS CORRECT.
 */
say;
say 'Searching for PlantWorks drive'
if ( gotFFdrive(FFdir, drives) == FALSE ) then
    call eoj "** Error ** NO PlantWorks drive found!"


/*
 *  USER RESPONSE LOOP.  ASK USER IF DRIVE LETTER IS CORRECT,
 *  IF NOT, THEN ASK FOR NEW DRIVE LETTER.  CHECK TO SEE THAT
 *  DRIVE LETTER IS A VALID PW DRIVE.
 */
say;
prompt = 'PlantWorks drive' FFdrive||':  Is this correct?';
if ( getResponse(prompt, 'Y/N', 1, 'Y') == 'N' ) then do
    got_drive = 0;

    do while ( got_drive=0 )
        drive = getResponse('Enter PlantWorks drive letter', drives, 1, FFdrive);

        if ( gotFFdrive(FFdir, drive) == TRUE ) then do
            prompt = 'PlantWorks drive' drive||':  Is this correct?';
            if ( getResponse(prompt, 'Y/N', 1, 'Y') == 'Y' ) then do
                FFdrive   = drive;
                got_drive = 1;
            end
        end
        else
            say drive||': is NOT a PlantWorks drive!';
    end
end

FFdrive = FFdrive||':';


/*
 *  SET FFDRIVE LETTER AND OUTPUT FILENAME FOR THE ERROR LOG.
 */
if ( FFdrive <> fnDrive ) then do
    parse value filespec("path", fnPath) with fnPath .
    fnDrive = FFdrive;
    call setNames;
end


/*
 *  TELL USER WHAT DIRECTORIES/FILES WILL BE USED
 *  IN THE VERIFICATION.
 */
say;
say 'The following information will be used for the PlantWorks'
say 'installation verification:'
say right('PlantWorks directory:', 25) FFdrive||FFdir
say right(fn 'logfile:', 25) fn_out
say;
prompt = 'This may take a few minutes.  Do you wish to continue?';
if ( getResponse(prompt, 'Y/N', 1, 'Y') == 'N' ) then
    call eoj 'PlantWorks Install Verification cancelled'


/*
 *  CHECK TO SEE IF OUTPUT FILE EXISTS.  IF IT DOES, THEN
 *  ASK THE USER IF THEY WISH TO DELETE THE FILE.  IF THEY
 *  DON'T, THEN EXIT WITH A MESSAGE; OTHERWISE, DELETE THE
 *  OUTPUT LOG.
 */
if ( fileExists(fn_out) == TRUE ) then do
    prompt = fn_out 'exists. Delete this file?'
    if ( getResponse(prompt, "Y/N", 1, "Y") == "Y" ) then do
        say 'Deleting' fn_out
        '@DEL' fn_out
    end
    else
        call eoj "Rename" fn_out "then re-run" fn
end


/*
 *  OPEN THE OUTPUT FILE.
 */
if ( openFile(fn_out) <> 0 ) then
    call eoj "cannot open" fn_out


/*
 *  INSERT A DATE AND TIMESTAMP PLUS THE NAME OF THIS UTILITY
 *  INTO THE FILE AND THEN CLOSE THE FILE.
 */
line = date('USA') time('Normal') "PlantWorks Install Verification";
call lineout fn_out, line, 1;


/*
 *  CHECK ALARM TEMPLATE LIMITS.
 */
say;
say 'Checking ALARM TEMPLATE limits';

call lineout fn_out, 'ALARM LIMIT CHECK';
call lineout fn_out

if ( fileExists(alarmExe) == FALSE ) then do
    '@DEL' fn_out
    call warnMsg alarmExe 'not found!'
    runRc = -1;
end
else do
    '@'alarmExe '>>' fn_out;
    alarmRc = rc;
end


/*
 *  CHECK I/O POINT LIMITS.
 */
say;
say 'Checking I/O point limits';

call lineout fn_out, 'I/O LIMIT CHECK';
call lineout fn_out

if ( fileExists(ioExe) == FALSE ) then do
    if ( fileExists(fn_out) == TRUE ) then
        '@DEL' fn_out
    call warnMsg ioExe 'not found!'
    runRc = -1;
end
else do
    '@'ioExe '>>' fn_out;
    ioRc = rc;
end

if ( ioRc <> 0 | alarmRc <> 0 | runRc <> 0 ) then do
    call errMsg;
    if ( debug ) then do
        say '*debug* ioRc   ' ioRc;
        say '*debug* alarmRc' alarmRc;
        say '*debug* runRc  ' runRc;
    end

    if ( runRc <> 0 ) then do
        say "Run-time errors encountered."
        say "Could not complete PlantWorks Install Verification."
    end

    if ( ioRc <> 0 ) then
        say 'Warnings found when checking I/O LIMITS'

    if ( alarmRc <> 0 ) then
        say 'Warnings found when checking ALARM TEMPLATE LIMITS'

    say 'Warning messages have been logged to:' fn_out
    say 'Call PlantWorks Support personnel.'
end
else do
    say;
    say '-->  PlantWorks Install Verification OK  <--'
    say;
    if ( fileExists(fn_out) == TRUE ) then
        '@DEL' fn_out
end

if ( runRc == 0 ) then
    call eoj 'PlantWorks Install Verification complete.'
else
    call eoj;



/*
 *  PROCEDURE Init
 */
Init:
    do queued()
        pull;
    end

    parse upper source . . fn .
    parse value filespec( "drive", fn ) with fnDrive '.'
    parse value filespec( "path", fn ) with fnPath '.'
    parse value filespec( "name", fn ) with fn '.'

    parse version . . . . year .
    if ( year == '1992' ) then
        verNum = '4.00'
    else
        verNum = '3.00'

    if ( verNum == '4.00' ) then do
        call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
        call SysLoadFuncs
        drives   = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    end
    else
        drives   = 'CDEFGHIJKLMNOPQRSTUVWXYZ';

    TRUE     = 1;
    FALSE    = 0;
setNames:
    fnPath   = fnDrive||fnPath;
    FFdir    = '\FF\'
    fn_out   = fnDrive||FFdir||fn||'.LOG'
    alarmExe = fnPath||'ALMLIMCK.EXE'
    ioExe    = fnPath||'IOLIMCHK.EXE'
    alarmRc  = 0;
    ioRc     = 0;
    runRc    = 0;
    n.1=262
    n.2=294
    n.3=330
    n.4=349
    n.5=392
    n.6=440
    n.7=494
    n.8=524
return;


/*
 *  PROCEDURE errMsg
 */
errMsg:
    call beep 100, 500
    say;
    say '**********************************************************************'
    say '*   W A R N I N G   W A R N I N G   W A R N I N G   W A R N I N G    *'
    say '**********************************************************************'
    say;
return;



/*
 *  PROCEDURE Greeting
 */
Greeting:
    'CLS'
    do 2
        say;
    end
    say '**********************************************************************'
    if ( obnoxious ) then call beep n.1, 250
    if ( obnoxious ) then call beep n.2, 250
    say '*                       P L A N T W O R K S                          *'
    if ( obnoxious ) then call beep n.3, 250
    say '*                                                                    *'
    if ( obnoxious ) then call beep n.4, 250
    say '*                    INSTALLATION VERIFICATION                       *'
    if ( obnoxious ) then call beep n.5, 250
    if ( obnoxious ) then call beep n.6, 250
    say '**********************************************************************'
    if ( obnoxious ) then call beep n.7, 250
    say;
    if ( obnoxious ) then call beep n.8, 250
return;


/*
 *  PROCEDURE getResponse
 */
getResponse: procedure
    parse arg prompt, valid_responses, length, default .

    valid_responses = translate( valid_responses );
    bad_response    = 1;

    if ( valid_responses <> '' ) then
        prompt = prompt "["valid_responses"]";

    if ( default <> '' ) then
        prompt = prompt "("default")";

    do while ( bad_response )
        say prompt;

        pull response .

        if ( length > 0 ) then
            response = substr( response, 1, length );

        if ( (response='') & (default<>'') ) then
            response = default;

        if ( pos(response, valid_responses) == 0 ) then
            say "["response"] is not a valid response."
        else
            bad_response = 0;
    end
return response;


/*
 *  PROCEDURE gotFFdrive
 */
gotFFdrive: procedure expose FFdrive TRUE FALSE
    arg dir, drives .

    rc = FALSE;

    do i = 1 to length( drives )
        drive = substr( drives, i, 1 );
        fspec = drive":"dir;

if ( verNum = '4.00' ) then do
        if ( SysDriveInfo(drive) <> '' ) then do

            call SysFileTree fspec, stem, 'D';

            if ( stem.0 > 0 ) then do
                FFdrive = drive;
                rc      = TRUE;
                leave;
            end     /* if */
        end     /* if */
end /* version 4.00 */
else do
        if ( fileExists( fspec||'BIN\*.EXE' ) == TRUE ) then do
            FFdrive = drive;
            rc      = TRUE;
            leave;
        end
end
    end     /* do */

return rc;


/*
 *  PROCEDURE fileExists
 */
fileExists: procedure expose TRUE FALSE
    arg fspec .

    rc = FALSE;

    if ( stream(fspec, 'C', 'QUERY EXIST' ) <> '' ) then
        rc = TRUE;
return rc;


/*
 *  PROCEDURE openFile
 */
openFile: procedure
    arg fspec .

    if ( lineout(fspec,,1) <> 0 ) then
        return -1;

    if ( lineout(fspec) <> 0 ) then
        return -2;
return 0;


/*
 *  PROCEDURE showHelp
 */
showHelp: procedure
    totalLines = sourceline();
    got_help   = 0;

    do i = 1 to totalLines
        line = sourceline(i);

        if ( pos('*?', line) <> 0 ) then do
            if ( pos('<> 0', line) <> 0 ) then
                leave;

            if ( got_help ) then
                leave;
            else do
                got_help = 1;
                iterate;
            end
        end
        if ( got_help ) then
            say line;
    end

    if ( got_help = 0 ) then
        say 'Help not available'
return;


/*
 *  PROCEDURE eoj
 */
eoj: procedure expose fn
    parse arg msg
    if ( msg <> '' ) then
        say fn":" msg

    exit;
return;


/*
 *  PROCEDURE warnMsg
 */
warnMsg:
    parse arg msg
    call beep 100, 500
    say fn": ** warning **" msg;
return;
