/*---------------------------------------------------------------------
Copyright (c) 2000-2001, Vadim Yegorov
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
THE POSSIBILITY OF SUCH DAMAGE.
---------------------------------------------------------------------*/

#define  INCL_ERRORS
#define  INCL_REXXSAA
#include <os2.h>
#include <rexxsaa.h>
#include <stdio.h>
#include <string.h>
#include <pcre.h>
#include <locale.h>

static PSZ  fncTable[] = {
  "PCRELoad",
  "PCREDrop",
  "PCREInit",
  "PCREVersion",
  "PCRERxVersion",  
  "PCREAuthor",
  "PCREPortAuthor",
  "PCRECompile",
  "PCREExec"
};

static int  fncTableEntries =
  sizeof(fncTable)/sizeof(PSZ)
;


static PUCHAR locTable = NULL;

/*********************************************************************/
/* Numeric Error Return Strings                                      */

#define OK              "0"
#define BAD_OPTS        "1"
#define BAD_MASK        "2"
#define BAD_VAR         "3"
#define CMP_ERR         "4"
#define NOT_CMP_MASK    "5"
#define BAD_POS         "6"
#define VAR_ERR         "100"

#define RX_PCRE_VER     "1.0.1"
#define RX_LIB          "REXXPCRE"

#define MAX_ENTRIES     512
#define MAX_ELEMS       3*MAX_ENTRIES

/* Raise Rexx error  */
#define  INVALID_ROUTINE 40

/* Successful completion */
#define  VALID_ROUTINE    0

/* Some useful macros */

#define BUILDRXSTRING(t, s) { \
  strcpy((t)->strptr,(s));\
  (t)->strlength = strlen((s)); \
}

#define RXFUNC(rname) \
  RexxFunctionHandler rname; \
  ULONG rname##( \
    PUCHAR name, \
    ULONG numargs, RXSTRING args[], \
    PSZ queuename, \
    RXSTRING *retstr \
  ) {

#define ENDFUNC(rc) \
  return rc; }

/************************************************************************/

/***** useful funcs *****************************************************/
static int DropRexxVariable(PSZ  name)  {
   SHVBLOCK   block;
   block.shvcode = RXSHV_SYDRO;
   block.shvret=(UCHAR)0;
   block.shvnext=(PSHVBLOCK)0;
   MAKERXSTRING(block.shvname, name, strlen(name));
   return RexxVariablePool(&block);
}

static int SetRexxVariable(PSZ  name, PSZ  value)  {
  int rc;
   SHVBLOCK   block;                    /* variable pool control block*/
   block.shvcode = RXSHV_SYSET;         /* do a symbolic set operation*/
   block.shvret=(UCHAR)0;               /* clear return code field    */
   block.shvnext=(PSHVBLOCK)0;          /* no next block              */
                                        /* set variable name string   */
   MAKERXSTRING(block.shvname, name, strlen(name));
                                        /* set value string           */
   MAKERXSTRING(block.shvvalue, value, strlen(value));
   block.shvvaluelen=strlen(value);     /* set value length           */
   return  RexxVariablePool(&block)   ;
}


static int SetStemVar(PSZ name, int stemNum, PSZ val) {
  char buf[512];
  int rc;
  sprintf(buf,"%s.%u", name, stemNum);
  return SetRexxVariable(buf,val) != RXSHV_BADN;
}

static int SetStemVarInt(PSZ name, int stemNum, int val) {
  char vb[32];
  itoa(val, vb, 10);
  return SetStemVar(name,stemNum,vb);
}

static void resetLocTable(PUCHAR t) {
  if (locTable != NULL) {
    pcre_free(locTable);
  }
  locTable = t;
}

static int validateCMask(pcre **crx, pcre_extra **erx, char* cmask) {
  int  l;
  PSZ  emask;
  PSZ  emsg;
  char buf[128];
  PSZ  p = (PSZ)buf;
  if (cmask == NULL || cmask[0] == 0)
    return 0;
  strcpy(p,cmask);
  //pfx
  if (memcmp(p, "PCRE[", 5))
    return 0;
  l = strlen(p)-1;
  if (p[l] != ']')
    return 0;
  p[l] = 0;
  p += 5;
  emask = strchr(p, ':');
  if (emask != NULL) {
    *emask++ = 0;
    *erx = (pcre_extra *)strtol(emask, &emsg, 16);
    if (emsg[0] != 0)
      return 0;
  } else
    *erx = NULL;
  *crx = (pcre*)strtol(p, &emsg, 16);
  return emsg[0] == 0;
}

static int validateRetVal(char nm[], char* n) {
  int l;
  if (n == NULL || n[0] == 0)
    return 0;
  l = strlen(n);
  if (l > 250) { /* trim */
    l = 250;
    n[l] = 0;
  }
  l--;
  strcpy(nm,n);
  if (n[l] == '.')
    n[l] = 0;
  return 1;
}

/*********************************************************************/
/**
 * Return Port Library Author string
 *
 * rc = PCREPortAuthor()
 *
 */
RXFUNC(PCREPortAuthor)
  if (numargs != 0)
    return INVALID_ROUTINE;
  BUILDRXSTRING(retstr, "Vadim Yegorov <vy@vy.vsu.ru>");
ENDFUNC(VALID_ROUTINE)

/**
 * Return PCRE Library Author string
 *
 * rc = PCREAuthor()
 *
 */
RXFUNC(PCREAuthor)
  if (numargs != 0)
    return INVALID_ROUTINE;
  BUILDRXSTRING(retstr, "Philip Hazel <ph10@cam.ac.uk>");
ENDFUNC(VALID_ROUTINE)

/**
 * Return PCRE Library version string
 *
 * rc = PCREVersion()
 *
 */
RXFUNC(PCREVersion)
  if (numargs != 0)
    return INVALID_ROUTINE;
  BUILDRXSTRING(retstr,pcre_version());
ENDFUNC(VALID_ROUTINE)

/**
 * Return PCRE REXX Library version string
 *
 * rc = PCRERxVersion()
 *
 */
RXFUNC(PCRERxVersion)
  if (numargs != 0)
    return INVALID_ROUTINE;
  BUILDRXSTRING(retstr,RX_PCRE_VER);
ENDFUNC(VALID_ROUTINE)

/**
 * Initializing PCRE with required locale
 * (by default, PCRE use C locale)
 *
 * rc = PCREInit([locale])
 *
 * locale such as "C", "EN_US", "RU_ru", etc
 * empty string mean current system locale
 * omit parameter mean default 'C' locale
 *
 * return using locale string
 *
 */
RXFUNC(PCREInit)

  /*0 or 1 parameter*/
  switch (numargs) {
    default:
      return INVALID_ROUTINE;
    case 0: /* to default */
      BUILDRXSTRING(retstr, "C");
      resetLocTable(NULL);
      break;
    case 1: /* to user*/
      {
        char * loc;
        char * nlstr = args[0].strptr;
        if (nlstr == NULL)
          nlstr = "";
        if (
            !strcmp(nlstr,"C") ||
            !strcmp(nlstr,"c") ||
            !strcmp(nlstr,"default")
           )
           nlstr = "";

        loc = setlocale(LC_ALL, nlstr);
        if (loc == NULL ){
          /*errors. return empty string */
          retstr->strlength = 0;
        } else {
          /*set ret. string */
          BUILDRXSTRING(retstr, loc);
          resetLocTable((PUCHAR)pcre_maketables());
        }
      }
      break;
  }
ENDFUNC(VALID_ROUTINE)

/**
 * Compile mask
 *
 * Syntax: rc = PCRECompile( 'pcre', mask )
 *
 * 'pcre' -- var name for result or error string
 * mask : '/mask/[8|i|m|s|x|g|A|U|E]'
 *
 * return:
 *   0 - ok
 *   1 - bad opts
 *   2 - bad mask
 *   3 - bad result var name (i.e. empty string)
 *   4 - comp. error, errstr -> 'pcre'
 */
RXFUNC(PCRECompile)
  pcre_extra  *erx = NULL;
  pcre        *crx = NULL;
  PSZ   cmsk_name;
  PSZ   mask;
  PSZ   o;
  PSZ   o1;
  PSZ   errm;

  int   eofs;
  char  ch, locbuf[128];
  int   opts = 0; /*no opts by default*/

  if (numargs != 2)
    return INVALID_ROUTINE;

  /* checkup params*/
  cmsk_name = args[0].strptr;
  mask      = args[1].strptr;

  if (mask == NULL || mask[0] == 0) {
    BUILDRXSTRING(retstr, BAD_MASK);
    return  VALID_ROUTINE;
  }

  if (cmsk_name == NULL || cmsk_name[0] == 0) {
    BUILDRXSTRING(retstr, BAD_VAR);
    return  VALID_ROUTINE;
  }

  ch = mask[0];/*1st char -- delimiter */
  mask++; /*skip*/
  if ((o = strrchr(mask,ch)) == NULL) {
    BUILDRXSTRING(retstr, BAD_MASK);
    return  VALID_ROUTINE;
  }
  o1 = o;
  *o++ = 0; /*split on 2 string: 'pure' mask & opts*/
  /*set options bits*/
  while(*o) {
    switch(*o++) {
      case '8' :
        opts |= PCRE_UTF8;
        break;
      case 'i' :
        opts |= PCRE_CASELESS;
        break;
      case 'm' :
        opts |= PCRE_MULTILINE;
        break;
      case 's' :
        opts |= PCRE_DOTALL;
        break;
      case 'x' : case 'X' :
        opts |= PCRE_EXTENDED;
        break;
      case 'g' : case 'G' :
        opts |= PCRE_NOTEMPTY;
        break;
      case 'A' :
        opts |= PCRE_ANCHORED;
        break;
      case 'U' :
        opts |= PCRE_UNGREEDY;
        break;
      case 'E' :
        opts |= PCRE_DOLLAR_ENDONLY;
        break;
      default:
        // restore mask
        *o1 = ch;
        BUILDRXSTRING(retstr, BAD_OPTS);
        return VALID_ROUTINE;
    }
  }
  /*compile*/
  crx = pcre_compile(mask, opts, (const char**)&errm, &eofs, locTable);
  if (crx == NULL) {
    BUILDRXSTRING(retstr, CMP_ERR);
    SetRexxVariable(cmsk_name, errm);
    *o1 = ch;
    return VALID_ROUTINE;
  }
  /* study */
  erx = pcre_study(crx, opts & PCRE_CASELESS, (const char**)&errm );
  /* return */
  sprintf(locbuf, "PCRE[%08X:%08X]", (ULONG)crx,(ULONG)erx);
  SetRexxVariable(cmsk_name, locbuf);
  BUILDRXSTRING(retstr, OK);
  *o1 = ch;
ENDFUNC(VALID_ROUTINE)

/**
 * Exec expression
 *
 * rc = PCREExec('pos_stem.',pcre,string,[pos,[,opts])
 *
 * Params:
 *   'pos_stem.' - stem var name for output
 *      pos_stem.0 -- count
 *   pcre        - compiled expression
 *   string      - user string
 *   pos         - start pos, 1 based; (1 by default)
 *   opts        -
 *      A -  Anchored
 *      b -  Not begin of line
 *      e -  Not end of line
 *      g -  Not empty
 *
 * Return:
 *  -1 - not found
 *   0 - Ok
 *   1 - bad opts
 *   3 - bad result var name (i.e. empty string)
 *   5 - pcre not contains compiled mask
 *   6 - bad position
 *
 */
RXFUNC(PCREExec)
  pcre_extra  *erx = NULL;
  pcre        *crx = NULL;

  PSZ   retval;
  PSZ   cmask;
  PSZ   o;

  PSZ   errm;
  int   pos = 0;
  int   N, idx;

  int   opts = 0;       /* no opts by default */
  int   ofs[MAX_ELEMS]; /* max 512 entries    */

  char  namebuf [256];

  /* params checkup*/
  if (numargs < 3 || numargs > 5)
    return INVALID_ROUTINE;

  /* 'pos_stem.',pcre,string, pos */
  if (numargs > 3) {
    /* start pos */
    pos = (int)strtol(args[3].strptr, &errm, 10);

    if (errm[0] != 0 || pos < 1) {
      BUILDRXSTRING(retstr, BAD_POS);
      return VALID_ROUTINE;
    } else
      --pos;
  }

  /* 'pos_stem.',pcre,string, pos, opts */
  if (numargs == 5) {
    o = args[4].strptr;
    while(*o) {
      switch(*o++) {
        case 'A':
          opts |= PCRE_ANCHORED;
          break;
        case 'b':
          opts |= PCRE_NOTBOL;
          break;
        case 'e':
          opts |= PCRE_NOTEOL;
          break;
        case 'g':
          opts |= PCRE_NOTEMPTY;
          break;
        default:
          BUILDRXSTRING(retstr, BAD_OPTS);
          return VALID_ROUTINE;
      }
    }
  }

  /* checkup params*/
  retval = args[0].strptr;
  cmask  = args[1].strptr;

  if (!validateCMask(&crx, &erx, cmask)) {
    BUILDRXSTRING(retstr, NOT_CMP_MASK);
    return  VALID_ROUTINE;
  }

  if (!validateRetVal(namebuf, retval)) {
    puts(retval);
    BUILDRXSTRING(retstr, BAD_VAR);
    return  VALID_ROUTINE;
  }

  N = pcre_exec(
        crx, erx,
        args[2].strptr, args[2].strlength,
        pos,
        opts,
        ofs, MAX_ELEMS
      );

  /* not found */
  if (N < 0) {
    BUILDRXSTRING(retstr, "-1");
    return VALID_ROUTINE;
  }

  if (N == 0)
    N = MAX_ENTRIES;
  else
    N *= 2;
  /* store count*/
  if (!SetStemVarInt(namebuf,0,N)) {
    BUILDRXSTRING(retstr, VAR_ERR);
    return VALID_ROUTINE;
  }
  /* entries */
  for (idx = 0; idx < N; idx++) {
    if (!SetStemVarInt(namebuf,idx + 1,ofs[idx] + 1)) {
      BUILDRXSTRING(retstr, VAR_ERR);
      return VALID_ROUTINE;
    }
  }
  BUILDRXSTRING(retstr, OK);

ENDFUNC(VALID_ROUTINE)
/*******************************************************************/
/**
 * Register all funcs
 *
 * syntax: call PCRELoad
 *.
 */
RXFUNC(PCRELoad)
  if (numargs != 0)
    return INVALID_ROUTINE;

  BUILDRXSTRING(retstr, "0");

  /* good old hack ;) */
  for (numargs = 0; numargs < fncTableEntries; numargs++) {
    RexxRegisterFunctionDll(fncTable[numargs],  RX_LIB, fncTable[numargs]);
  }
ENDFUNC(VALID_ROUTINE)

/**
 * Deregister all funcs
 *
 * syntax: call DropPCRE
 */
RXFUNC(PCREDrop)
  if (numargs != 0)
    return INVALID_ROUTINE;

  BUILDRXSTRING(retstr, "0");

  /* good old hack ;) */
  for (numargs = 0; numargs < fncTableEntries; numargs++)
    RexxDeregisterFunction(fncTable[numargs]);

  if (locTable != NULL) {
    pcre_free(locTable);
    locTable = NULL;
  }
ENDFUNC(VALID_ROUTINE)

