/*************************************************************************
*
* filename		: mlefile.cpp
*
* description	: move text between MLE and filesystem
*
* methods		: [none]
*
* functions		: LoadFile		- load a file to the MLE
*				  SaveFile		- copy the MLE-contents to the filesystem
*				  ValidateFilename	- check if a file already exists
*				  CopyTextToBuffer	- copy MLE-contents to a buffer
*
* APIs			: DosOpen					DosClose
*         		  DosQueryFileInfo			DosAllocMem
*         		  DosFreeMem				WinSendMsg
*		  		  DosRead					DosWrite
*				  WinMessageBox
*
* Used Classes	: [none]
*
* copyright (C) 1993 Jrg Caumanns (caumanns@cs.tu-berlin.de)
*
*************************************************************************/#define INCL_DOSMEMMGR
#define INCL_WINDIALOGS
#define INCL_WINPOINTERS
#define INCL_WINMLE
#define INCL_DOSFILEMGR
#include <os2.h>
#include <stdio.h>
#include <string.h>

extern VOID FatalError(HWND, CHAR*);

/*************************************************************************
*
* Name	: LoadFile
*
* Descr.: Load a file into the MLE
*
* APIs	: DosOpen					DosClose
*         DosQueryFileInfo			DosAllocMem
*         DosFreeMem				WinSendMsg
*		  DosRead
*
* Param.: HWND	hwnd		- parent window handle
*		  HWND  hwndMLE		- MLE window handle
*		  CHAR  *pszFile	- file name
*
* Return: BOOL fSuccess
*
*************************************************************************/
BOOL LoadFile(HWND hwnd, HWND hwndMLE, CHAR *pszFile)	{
	VOID *pvBuf;
	HFILE hf;
	IPT   lOffset = 0;
	ULONG cbRead, cbCopied, ulAction;

	/*
	* open requested file
	*/
	if(DosOpen(pszFile,
			   &hf,
			   &ulAction, 0,
			   FILE_NORMAL,
			   FILE_OPEN | FILE_CREATE,
			   OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE,
			   NULL))	{
		FatalError(hwnd, "can't open requested file");
		return FALSE;
		}

	/*
	* Get the length of the file
	*/
	FILESTATUS fileStatus;
	if (DosQueryFileInfo(hf, 1, (PVOID)&fileStatus,	sizeof(FILESTATUS)))	{
		DosClose(hf);
		FatalError(hwnd, "can't read requested file!");
		return FALSE;
		}

	if(fileStatus.cbFileAlloc == 0)	{
		return TRUE;
		}

	WinSetPointer(HWND_DESKTOP,
					WinQuerySysPointer(HWND_DESKTOP, SPTR_WAIT, FALSE));
	/*
	* Allocate a buffer for the file
	*/
	if(DosAllocMem((PPVOID)&pvBuf, (ULONG)fileStatus.cbFileAlloc,
										PAG_READ | PAG_WRITE | PAG_COMMIT))	{
		DosClose(hf);
		FatalError(hwnd, "can't allocate buffer to read requested file!");
		return FALSE;
		}

	/*
	* Read in the file
	*/
	if(DosRead(hf, (char*)pvBuf, fileStatus.cbFileAlloc, &cbRead))	{
		DosClose(hf);
		FatalError(hwnd, "can't read requested file!");
		return FALSE;
		}

	/*
	* Set the file into the MLE
	*/
	WinSendMsg(hwndMLE, MLM_SETIMPORTEXPORT, MPFROMP((PBYTE)pvBuf),
										  MPFROMSHORT(fileStatus.cbFileAlloc));

	lOffset = 0;
	WinSendMsg(hwndMLE, MLM_IMPORT, MPFROMP(&lOffset),
										MPFROMSHORT(fileStatus.cbFileAlloc));

	/*
	* Reset the changed flag
	*/
	WinSendMsg(hwndMLE, MLM_SETCHANGED, MPFROMSHORT((BOOL)FALSE), NULL);

	DosFreeMem(pvBuf);
	DosClose(hf);

	WinSetPointer(HWND_DESKTOP,
					WinQuerySysPointer(HWND_DESKTOP, SPTR_ARROW, FALSE));

	return TRUE;
	}


/*************************************************************************
*
* Name	: SaveFile
*
* Descr.: Save the contents of a MLE to a file
*
* APIs	: DosOpen					DosClose
*         DosWrite					DosAllocMem
*         DosFreeMem				WinSendMsg
*
* Param.: HWND	hwnd		- parent window handle
*		  HWND  hwndMLE		- MLE window handle
*		  CHAR  *pszFile	- file name
*
* Return: BOOL fSuccess
*
*************************************************************************/
BOOL SaveFile(HWND hwnd, HWND hwndMLE, CHAR *pszFile)	{
	VOID *pvBuf;
	HFILE hf;
	IPT   lOffset = 0;
	ULONG cbWritten, cbCopied, ulAction;
	ULONG cbExport, ulFileLen;

	/*
	* open the file for writing
	*/
	if(DosOpen(pszFile,
			   &hf,
			   &ulAction, 0,
			   FILE_NORMAL,
			   FILE_OPEN | FILE_CREATE,
			   OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYNONE,
			   NULL))	{
		FatalError(hwnd, "can't open requested file");
		return FALSE;
		}

	/*
	* Get the length of the file
	*/
	ulFileLen = (ULONG)WinSendMsg(hwndMLE, MLM_QUERYTEXTLENGTH, MPVOID, MPVOID);
	if(!ulFileLen)	{
		DosClose(hf);
		return TRUE;
		}

	/*
	* Allocate a buffer for the file
	*/
	if(DosAllocMem((PPVOID)&pvBuf, ulFileLen,
										PAG_READ | PAG_WRITE | PAG_COMMIT))	{
		DosClose(hf);
		FatalError(hwnd, "can't allocate buffer to write requested file!");
		return FALSE;
		}

	/*
	* Get the file from the MLE
	*/
	cbExport = ulFileLen;
	WinSendMsg(hwndMLE, MLM_SETIMPORTEXPORT, MPFROMP((PBYTE)pvBuf),
											 MPFROMLONG(cbExport));

	/*
	* Export MLE starting at offset 0
	*/
	lOffset = 0;
	WinSendMsg(hwndMLE, MLM_EXPORT, MPFROMP(&lOffset), MPFROMLONG(&cbExport));

	/*
	* Write the file
	*/
	if(DosWrite(hf, pvBuf, ulFileLen, &cbWritten))	{
		DosClose(hf);
		FatalError(hwnd, "can't write requested file!");
		return FALSE;
		}

	/*
	* Reset the changed flag
	*/
	WinSendMsg(hwndMLE, MLM_SETCHANGED, MPFROMSHORT((BOOL)FALSE), NULL);

	DosFreeMem(pvBuf);	DosClose(hf);
	return TRUE;
	}


/*************************************************************************
*
* Name	: ValidateFilename
*
* Descr.: Check if a file exists
*
* APIs	: DosOpen					DosClose
*		  WinMessageBox
*
* Param.: HWND	hwnd		- parent window handle
*		  CHAR  *pszFile	- file name
*
* Return: BOOL fValid
*
*************************************************************************/
BOOL ValidateFilename(HWND hwnd, CHAR *pszFile)	{
	HFILE hf;
	ULONG ulAction;
	SHORT sResponce;

	if(DosOpen(pszFile,
			   &hf,
			   &ulAction, 0,
			   FILE_NORMAL,
			   FILE_OPEN | FILE_CREATE,
			   OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYNONE,
			   NULL))	{
		FatalError(hwnd, "can't open requested file");
		return FALSE;
		}

	DosClose(hf);

	/*
	* If file exists, ask if we want to overwrite it
	*/
	CHAR szQuery[256];
	if(ulAction == FILE_EXISTED)	{
		sprintf(szQuery, "%s already exists! Do you want to overwrite it?", pszFile);
		sResponce = WinMessageBox(HWND_DESKTOP, hwnd, szQuery, "File Exists",
													0, MB_QUERY | MB_YESNO);

	   if(sResponce == MBID_NO)
		  return FALSE;
		}

	WinSetWindowText(hwnd, pszFile);
	return TRUE;
	}


/*************************************************************************
*
* Name	: CopyTextToBuffer
*
* Descr.: Copy MLE-contents to a buffer
*
* APIs	: WinSendMsg			DosAllocMem
*
* Param.: HWND	hwnd		- parent window handle
*		  HWND  hwnd		- MLE window handle
*		  CHAR  **ppszBuffer- buffer's address
*
* Return: BOOL fValid
*
*************************************************************************/
BOOL CopyTextToBuffer(HWND hwnd, HWND hwndMLE, CHAR **ppszBuffer)	{

	/*
	* Get the length of the text
	*/
	ULONG ulFileLen = (ULONG)WinSendMsg(hwndMLE,
										MLM_QUERYTEXTLENGTH,
										MPVOID, MPVOID);
	if(!ulFileLen)	{
		*ppszBuffer = NULL;
		return TRUE;
		}

	/*
	* Allocate a buffer for the file
	*/
	if(DosAllocMem((PPVOID)ppszBuffer, ulFileLen,
				   PAG_READ | PAG_WRITE | PAG_COMMIT))	{
		FatalError(hwnd, "can't allocate buffer to write requested file!");
		return FALSE;
		}

	/*
	* Set export buffer
	*/
	ULONG cbExport = ulFileLen;
	WinSendMsg(hwndMLE, MLM_SETIMPORTEXPORT, MPFROMP((PBYTE)*ppszBuffer),
											 MPFROMLONG(cbExport));

	/*
	* Export MLE starting at offset 0
	*/
	IPT lOffset = 0;
	WinSendMsg(hwndMLE, MLM_EXPORT, MPFROMP(&lOffset), MPFROMLONG(&cbExport));
	return TRUE;
	}
