/*************************************************************************
*
* filename		: editor.cpp	(implementation of class EDITOR)
*
* description	: manage a list of Multi-Line Entryfields
*				  (e.g for "file"-menus)
*
* methods		: EDITOR::WindowFromFile	- get MLE from filename
*				  EDITOR::ActiveWindow		- get handle of active MLE
*				  EDITOR::NewFile			- create an empty MLE
*				  EDITOR::OpenFile			- load a file to a MLE
*				  EDITOR::SaveFile			- save the contents of an MLE
*				  EDITOR::SaveFileAs		- save the contents of an MLE
*				  EDITOR::CloseFile			- close an MLE
*				  EDITOR::SaveAllFiles		- save the contents of all MLEs
*				  EDITOR::CloseAllFiles		- close all MLEs
*				  EDITOR::Cut				- move text to the clipboard
*				  EDITOR::Copy				- copy text to the clipboard
*				  EDITOR::Paste				- insert text from the clipboard
*				  EDITOR::Clear				- remove text
*				  EDITOR::ChangeColor		- change all MLE's colors
*
* APIs			: DosEnterCritSec
*				  DosExitCritSec
*
* Used Classes	: class MLE
*
* copyright (C) 1993 Jrg Caumanns (caumanns@cs.tu-berlin.de)
*
*************************************************************************/
#define INCL_DOSPROCESS
#include <os2.h>
#include <string.h>

#include "editor.h"
#include "mle.h"


/*************************************************************************
*
* Method: WindowFromFile (private method)
*
* Descr.: Get the handle of the MLE where the specified file was loaded to.
*
* Impl. : Search all all open MLEs for the file
*
* APIs	: [none]
*
* Param.: CHAR *pszFile		- filename
*
* Return: MLE* MLE-handle (NULL if file is not loaded to any MLE)
*
*************************************************************************/
MLE *EDITOR::WindowFromFile(CHAR *pszFile)	{
	MLE_LIST *plst = pmlelist;

	if(pszFile)
	  for(; plst; plst = plst->next)
		if(plst->pmle && (plst->pmle)->LoadedFile() &&
								!strcmp(pszFile, (plst->pmle)->LoadedFile()))
		  return plst->pmle;
	return (MLE*)0;
	}


/*************************************************************************
*
* Method: ActiveWindow (private method)
*
* Descr.: Get the handle of the MLE that has the focus
*
* Impl. : Get the handle of a valid MLE and call its 'ActiveMLE'-method
*
* APIs	: [none]
*
* Param.: [none]
*
* Return: MLE* MLE-handle (NULL if no MLE has the focus)
*
*************************************************************************/
MLE *EDITOR::ActiveWindow(VOID)	{
	MLE_LIST *plst = pmlelist;

	for(; plst; plst = plst->next)
		if(plst->pmle)
			return (pmlelist->pmle)->ActiveMLE();

	return (MLE *)0;
	}


/*************************************************************************
*
* Method: NewFile
*
* Descr.: Create an empty MLE
*
* Impl. : An empty MLE is created and put to the list of open MLEs
*
* APIs	: DosEnterCritSec
*		  DosExitCritSec
*
* Param.: [none]
*
* Return: BOOL fSuccess (TRUE if MLE was created successfully)
*
*************************************************************************/
BOOL EDITOR::NewFile(VOID)	{
	MLE *pmle = new MLE;
	MLE_LIST *plst = new MLE_LIST;

	if(pmle->OpenFile(NULL, MLE_NEWFILE))	{
		DosEnterCritSec();
		plst->pmle 		= pmle;
		plst->next		= pmlelist;
		pmlelist		= plst;
		DosExitCritSec();
		return TRUE;
		}
	return FALSE;
	}


/*************************************************************************
*
* Method: OpenFile
*
* Descr.: Create an empty MLE and load a file to it
*
* Impl. : A file is loaded to an empty MLE. The MLE is put to the list
*		  of open MLEs
*
* APIs	: DosEnterCritSec
*		  DosExitCritSec
*
* Param.: CHAR *pszFile		- name of the file to be loaded (optional)
*							  If no parameter is given, a dialogbox
*                             is used to ask the user for a filename.
*
* Return: BOOL fSuccess (TRUE if MLE was created successfully)
*
*************************************************************************/
BOOL EDITOR::OpenFile(CHAR *pszFile)	{
	MLE *pmle = new MLE;
	MLE_LIST *plst = new MLE_LIST;

	if(pmle->OpenFile(pszFile))	{
		DosEnterCritSec();
		plst->pmle 		= pmle;
		plst->next		= pmlelist;
		pmlelist		= plst;
		DosExitCritSec();
		return TRUE;
		}
	return FALSE;
	}


/*************************************************************************
*
* Method: SaveFile
*
* Descr.: Save the contents of an MLE
*
* Impl. : If an argument is given the specified file is saved.
*		  If no argument is given, the contents of the active MLE is
*		  saved. No saving is done, if the file hasn't changed.
*
* APIs	: [none]
*
* Param.: CHAR *pszFile		- name of the file to be saved (optional)
*
* Return: BOOL fSuccess (TRUE if file was saved)
*
*************************************************************************/
BOOL EDITOR::SaveFile(CHAR *pszFile)	{
	MLE *pmle;

	if((pmle = (pszFile)? WindowFromFile(pszFile) : ActiveWindow()) ==(MLE *)0)
		return FALSE;

	if(pmle->Touched())
		return pmle->SaveFile();

	return TRUE;
	}


/*************************************************************************
*
* Method: SaveFileAs
*
* Descr.: Save the contents of an MLE after asking the user for
*		  a new filename ("save as"-dialogbox)
*
* Impl. : If an argument is given the specified file is saved.
*		  If no argument is given, the contents of the active MLE is
*		  saved.
*
* APIs	: [none]
*
* Param.: CHAR *pszFile		- name of the file to be saved (optional)
*
* Return: BOOL fSuccess (TRUE if file was saved)
*
*************************************************************************/
BOOL EDITOR::SaveFileAs(CHAR *pszFile)	{
	MLE *pmle;

	if((pmle = (pszFile)? WindowFromFile(pszFile) : ActiveWindow()) ==(MLE *)0)
		return FALSE;

	return pmle->SaveFileAs();
	}


/*************************************************************************
*
* Method: CloseFile
*
* Descr.: Save the contents of an MLE and close the MLE
*
* Impl. : If an argument is given the specified file is saved and closed.
*		  If no argument is given, the contents of the active MLE is
*		  saved and the MLE closed. No saving is done, if the file
*		  hasn't changed.
*
* APIs	: [none]
*
* Param.: CHAR *pszFile		- name of the file to be closed (optional)
*
* Return: BOOL fSuccess (TRUE if file was closed successfully)
*
*************************************************************************/
BOOL EDITOR::CloseFile(CHAR *pszFile)	{
	MLE *pmle;

	if((pmle = (pszFile)? WindowFromFile(pszFile) : ActiveWindow()) ==(MLE *)0)
		return FALSE;

	return pmle->CloseFile();
	}


/*************************************************************************
*
* Method: SaveAllFiles
*
* Descr.: Save the contents of all MLEs
*
* Impl. : Check all MLEs one after another and save their contents if
*		  it has changed
*
* APIs	: [none]
*
* Param.: [none]
*
* Return: BOOL fSuccess (TRUE if all files have been saved)
*
*************************************************************************/
BOOL EDITOR::SaveAllFiles(VOID)	{
	BOOL ret = TRUE;

	for(MLE_LIST *plst = pmlelist; plst; plst = plst->next)
		if(plst->pmle && (plst->pmle)->Touched())
			ret &= (plst->pmle)->SaveFile();

	return ret;
	}


/*************************************************************************
*
* Method: CloseAllFiles
*
* Descr.: Close all MLEs and save their contents
*
* Impl. : Close an MLE after another
*
* APIs	: [none]
*
* Param.: [none]
*
* Return: BOOL fSuccess (TRUE if all files have been closed)
*
*************************************************************************/
BOOL EDITOR::CloseAllFiles(VOID)	{
	BOOL ret = TRUE;

	for(MLE_LIST *plst = pmlelist; plst; plst = plst->next)
		if(plst->pmle)
			ret &= (plst->pmle)->CloseFile();

	return ret;
	}


/*************************************************************************
*
* Method: Cut, Copy, Paste, Clear
*
* Descr.: Move text between an MLE and the clipboard
*
* Impl. : Call the appropriate method of the active MLE
*
* APIs	: [none]
*
* Param.: [none]
*
* Return: -
*
*************************************************************************/
VOID EDITOR::Cut(VOID)	{
	MLE *pmle;
	if((pmle = ActiveWindow()) != (MLE *)0)
		pmle->Cut();
	}

VOID EDITOR::Copy(VOID)	{
	MLE *pmle;
	if((pmle = ActiveWindow()) != (MLE *)0)
		pmle->Copy();
	}

VOID EDITOR::Paste(VOID)	{
	MLE *pmle;
	if((pmle = ActiveWindow()) != (MLE *)0)
		pmle->Paste();
	}

VOID EDITOR::Clear(VOID)	{
	MLE *pmle;
	if((pmle = ActiveWindow()) != (MLE *)0)
		pmle->Clear();
	}


/*************************************************************************
*
* Method: SetColor
*
* Descr.: Change the foreground- and background-color of all MLEs
*
* Impl. : Change the colors of one MLE after another
*
* APIs	: [none]
*
* Param.: LONG clrForeground	- new foreground-color
* 		  LONG clrBackground	- new background-color
*
* Return: -
*
*************************************************************************/
VOID EDITOR::SetColor(LONG clrForeground, LONG clrBackground)	{

	for(MLE_LIST *plst = pmlelist; plst; plst = plst->next)
		if(plst->pmle)
			(plst->pmle)->SetColor(clrForeground, clrBackground);
	}


/*************************************************************************
*
* Method: destructor
*
* Descr.: do some cleanup
*
* Impl. : Close all MLEs and free all resources
*
* APIs	: [none]
*
* Param.: [none]
*
* Return: -
*
*************************************************************************/
EDITOR::~EDITOR()	{

	CloseAllFiles();

	MLE_LIST *plst = pmlelist;
	MLE_LIST *tmp = (plst)? plst->next : (MLE_LIST*)0;
	for(; plst; plst = tmp, tmp = (plst)? plst->next : (MLE_LIST*)0)	{
		delete plst->pmle;
		delete plst;
		}
	}



