#ifndef _XLISTBOX_
#define _XLISTBOX_
/*******************************************************************************
* FILE NAME: xlistbox.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the class(es):                                              *
*   ListBox32 - 32-Bit list box control.                                       *
*                                                                              *
*******************************************************************************/
#ifndef _ILISTBOX_
  #include <ilistbox.hpp>
#endif


/*----------------------------------------------------------------------------*/
/* Align classes on four byte boundary.                                       */
/*----------------------------------------------------------------------------*/
#pragma pack(4)

class ListBox32 : public IListBox {
typedef IListBox
  Inherited;
/*******************************************************************************
*                                                                              *
*******************************************************************************/
public:
/*------------------------------ Extended Style -------------------------------
  The following functions provide a means to set and query the 32-Bit
  replacement list box extended styles:

    ExtendedStyle - Nested class that provides static members that define
                    the set of valid 32-Bit list box extended styles.
                    For example, you could define an instance of the
                    ListBox32::ExtendedStyle class and initialize it like:
                        ListBox32::ExtendedStyle extendedStyle =
                                                         ListBox32::checkBox;
                    An object of this type is provided when the 32-Bit list
                    box is created.  A customizable default is used if no
                    styles are specified.  Once the object is constructed,
                    ListBox32 member functions can be used to set or query
                    the object's extended style.

                    The declaration of the ListBox32::ExtendedStyle nested
                    class is generated by the INESTEDBITFLAGCLASSDEF0 macro.

  The valid list box styles are:
    classDefaultExtendedStyle - This specifies the original default style
                                for this class, which is nothing.
    checkBox                  - This style prefixes the list item with a
                                check box.

  The following functions provide a means of getting and setting the default
  extended style for this class:
    defaultExtendedStyle    - Returns the current default extended style.
                              This is the same as classDefaultExtendedStyle
                              unless setDefaultExtendedStyle has been called.
    setDefaultExtendedStyle - Sets the default extended style for all
                              subsequent 32-Bit list boxes.


*******************************************************************************
*****************************  I M P O R T A N T  *****************************
*******************************************************************************
  The base list box styles (i.e. IListBox::horizontalScroll) are
  controlled by the Style nested class which is defined in ILISTBOX.HPP.
-----------------------------------------------------------------------------*/
INESTEDBITFLAGCLASSDEF0(ExtendedStyle, ListBox32);

static const ExtendedStyle
  classDefaultExtendedStyle,
  checkBox;

static ExtendedStyle
  defaultExtendedStyle     ( );

static void
  setDefaultExtendedStyle  ( ExtendedStyle extendedStyle );


/*-------------------------- Constructors/Destructor ---------------------------
| You can construct generic objects of this class by providing one of the      |
| following items:                                                             |
|    o From a control ID, parent and owner windows, rectangle, style, and      |
|      extended style.  This creates the specified 32-Bit list box control     |
|      and an object for it.                                                   |
|    o From the ID of a 32-Bit list box on a dialog window.  This creates      |
|      the object for the 32-Bit list box.                                     |
|    o From the window handle of an existing 32-Bit list box control.  This    |
|      creates the object for the specified 32-Bit list box control.           |
------------------------------------------------------------------------------*/
  ListBox32   ( unsigned long       id,
                IWindow             *parent,
                IWindow             *owner,
                const IRectangle    &initial= IRectangle(),
                const Style         &style = defaultStyle(),
                const ExtendedStyle &extendedStyle = defaultExtendedStyle() );

  ListBox32   ( unsigned long       id,
                IWindow             *parentDialog );

  ListBox32   ( const IWindowHandle &handle );

virtual
  ~ListBox32  ( );

/*----------------------------- Implementation ---------------------------------
| createListBox - Static function that creates the 32-Bit list box.            |
| validate      - Static function that validates the following:                |
|                    1) The window handle is valid.                            |
|                    2) The window's class is "ListBoxWindow".                 |
------------------------------------------------------------------------------*/
static IWindowHandle
  createListBox  ( unsigned long       id,
                   IWindow             *parent,
                   IWindow             *owner,
                   const IRectangle    &initial,
                   const Style         &style,
                   const ExtendedStyle &extendedStyle ),
  validate       ( const IWindowHandle &handle );

/*---------------------------- Check-state Operations --------------------------
| Set or return the check state operations.                                    |
|   check         - Sets the check state of an item based on the               |
|                   Boolean value that is passed in.  If true is passed        |
|                   in and the list box is a single-selection list box,        |
|                   the item is checked and any item that was                  |
|                   previously checked is unchecked.  If true is               |
|                   passed in and the list box is a multiple-selection         |
|                   or extended-selection list box, the item is checked.       |
|                   In all cases, if false is passed in, the item is           |
|                   unchecked.                                                 |
|   unCheck       - Removes the check state from an item.                      |
|   checkAll      - Sets the check state for all items in the list box.        |
|   unCheckAll    - Removes the check state from all items in the list box.    |
|   isChecked     - Returns the check state of an item.                        |
|   numberChecked - Returns 0 if no item in a single-selection list box        |
|                   is checked or 1 if an item is checked.  Returns            |
|                   the number of checked items in a multiple-selection        |
|                   or extended-selection list box.                            |
|   checkedItem   - Returns the 0-based index of the checked item in a         |
|                   single-selection list box, or the 0-based index of         |
|                   the first checked item in a multiple-selection or          |
|                   extended-selection list box.  In all cases, if no          |
|                   item is checked, -1 is returned.                           |
------------------------------------------------------------------------------*/
virtual ListBox32
 &check          ( unsigned long index, Boolean check = true ),
 &unCheck        ( unsigned long index ),
 &checkAll       ( ),
 &unCheckAll     ( );

virtual Boolean
  isChecked      ( unsigned long index ) const;

virtual unsigned long
  numberChecked  ( ) const;

virtual long
  checkedItem    ( ) const;

/*---------------------------- Enumerations ------------------------------------
| The following enumerations are defined:                                      |
|   ClickEvent - Used to indicate which mouse click event to query or set:     |
|                singleClick - mouse button 1 single click.                    |
|                doubleClick - mouse button 2 double click.                    |
------------------------------------------------------------------------------*/
enum ClickEvent {
  singleClick,
  doubleClick
};

/*------------------------------- Sound Support --------------------------------
| These functions provide a means of getting and setting the sound support     |
| attributes of instances of this class:                                       |
| isSound       - Returns true if the sound is supported on this system.       |
| soundEvent    - Returns a wave file name for the event if it is set.         |
| setSoundEvent - Sets a wave file name for the event.                         |
------------------------------------------------------------------------------*/
virtual Boolean
  isSound        ( ) const;

virtual IString
  soundEvent     ( ClickEvent value ) const;

virtual ListBox32
 &setSoundEvent  ( ClickEvent value, const char *waveFile );

/*------------------------- Extended Style Functions ---------------------------
| These functions provide a means of getting and setting the default extended  |
| style attributes of instances of this class:                                 |
| isCheckBox      - Returns true if the checkBox extended style is set.        |
| enableCheckBox  - Enables or disables the extended check box style.          |
| disableCheckBox - Disables the extended check box style.                     |
------------------------------------------------------------------------------*/
virtual Boolean
  isCheckBox       ( ) const;

virtual ListBox32
 &enableCheckBox   ( Boolean enable = true ),
 &disableCheckBox  ( );


protected:
/*----------------------------- Layout Size ------------------------------------
| calcMinimumSize - Returns the recommended minimum size of the 32-Bit list    |
|                   box control.  The size is based on the text string length  |
|                   of the longest string (if not using                        |
|                   IListBox::horizontalScroll style) and the current font.    |
------------------------------------------------------------------------------*/
virtual ISize
  calcMinimumSize  ( ) const;

/*----------------------------- Implementation ---------------------------------
| initialize - Perform initialization tasks that are common across the ctors.  |
------------------------------------------------------------------------------*/
void
  initialize  ( );

/*------------------------------- Extended Styles ------------------------------
| extendedStyle    - Returns the extended style for the list box.              |
| setExtendedStyle - Sets the extended style for the list box.                 |
------------------------------------------------------------------------------*/
virtual unsigned long
  extendedStyle  ( ) const;

virtual ListBox32
 &setExtendedStyle  ( const unsigned long extStyle );


private:
/*--------------------------------- PRIVATE ----------------------------------*/

  ListBox32  ( const ListBox32 & );  //copy ctor
  operator = ( const ListBox32 & );  //assignment ctor

static ExtendedStyle
  currentDefaultExtendedStyle;

unsigned long
  ulExtendedStyle;

Boolean
  bSoundSupported;

};  // class ListBox32

/*----------------------------------------------------------------------------*/
/* Resume compiler default packing.                                           */
/*----------------------------------------------------------------------------*/
#pragma pack()

#endif  //_XLISTBOX_
