
#ifndef VORONOI_DIAGRAM_C
#define VORONOI_DIAGRAM_C

#include "avd.h"

//------------------------------------------------------------------------------
// clear()
// empties the Voronoi Diagram (=GRAPH) and trashes the containers.
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
clear()
{
  node v;
  edge e;
  Mod_item.init(*this);
  H_node.init(*this);
  E_t_item.init(*this);

     ASSERT(E_t.empty(),"Voronoi_Diagram: E_t not empty!");
     ASSERT(V_mod.empty(),"Voronoi_Diagram: V_mod not empty!");
     ASSERT(V_new.empty(),"Voronoi_Diagram: V_new not empty!");
     ASSERT(V_del.empty(),"Voronoi_Diagram: V_del not empty!");
     ASSERT(T_edges.empty(),"Voronoi_Diagram: T_edges not empty!");
     ASSERT(Aff_edges.empty(),"Voronoi_Diagram: Aff_edges not empty!");

  GRAPH<P_VNODE,P_SITE>::clear();
}  

//------------------------------------------------------------------------------
// new_bi_edge()
// inserts two reversal edges into the Voronoi Diagram. 
// parameters:
// node v1, v2  the source and target of the edge in case of append.
// edge e1, e2  the orientation edges; we insert edges between source(e1)
//               and source(e2)
// P_SITE p1,p2 the sites left of (v1,v2) and left of (v2,v1)
// return value:
// edge         the master edge just inserted
//------------------------------------------------------------------------------
      
template <class P_VNODE, class P_SITE>
edge  Voronoi_Diagram<P_VNODE,P_SITE>::
new_bi_edge(node v1, node v2, P_SITE p1, P_SITE p2, edge e1, edge e2)
{ 
    TRACE("  V.new_bi_edge: ");AVDD(v1);AVDD(v2);AVDDS(p1);AVDDS(p2);
    AVDD(e1);AVDD(e2);TRACEN("");
    ASSERT((e1==nil||v1==source(e1)) && (e2==nil||v2==source(e2)), 
           new_bi_edge - source condition);
  edge e,r;
  if (e1 != nil)
    e = new_edge(e1,v2,p1,after);
  else
    e = new_edge(v1,v2,p1);
  if (e2 != nil)
    r = new_edge(e2,v1,p2,after);
  else
    r = new_edge(v2,v1,p2);
  set_reversal(e,r);
  return e;
}


//------------------------------------------------------------------------------
// secure_links_and_master_correspondence()
// creates the cross links between a leaf vh in H and a bidirected edge 
// (e,rev(e)) in V. The description in vh corresponds to the half edge which
// has the same site description in V. 
// This correscpondence is important for the topology information in case of 
// intersection.
// parameters:
// edge     e           the edge which is examined
// h_node   vh          the corresponding h_node
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
secure_links_and_master_correspondence(edge e, h_node vh)
{ 
    TRACE(" AV.secure_links_and_master_correspondence: ");
    AVDD(e);AVDDD(pAVD->H,vh);TRACEN("");

  hist_node(e) = vh;
  hist_node(reversal(e)) = vh;
  if (left_site(e) == pAVD->H.left_site(vh)) {
    pAVD->H.master_edge(vh) = e;
  } 
  else if (left_site(e) == pAVD->H.right_site(vh)) {
    pAVD->H.master_edge(vh) = reversal(e);
  }
  else
    error_handler(1,"secure_links_and_master_correspondence: site mismatch!");
  ASSERT(left_site(pAVD->H.master_edge(vh))==pAVD->H.left_site(vh) && 
         right_site(pAVD->H.master_edge(vh))==pAVD->H.right_site(vh),
         master site correspondence check);
  ASSERT(hist_node(e) && hist_node(reversal(e)), link existence);
}

      
//------------------------------------------------------------------------------
// clipped_at_source()
// checks if edge e is clipped at its source.
// parameters:
// edge e       the edge which is (not) clipped
// return value:
// true         if e is clipped at source
// false        else
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
bool  Voronoi_Diagram<P_VNODE,P_SITE>::
clipped_at_source(edge e)
{ 
    TRACE("      V.clipped_at_source: ");AVDD(e);TRACEN("");
  e_t_list_item e_it = get_e_t_list_item(e);
  if (e_it == nil) return false;  // edge not intersected
  e_t_list_info& e_inf = E_t[e_it];

  switch (e_inf.topol) {
  case SOURCE_SEGMENT:
    if (e == e_inf.e) return true; // is master edge
    break;
  case TARGET_SEGMENT:
    if (e == reversal(e_inf.e)) return true; 
    break;
  case TWO_SEGMENTS:
  case TOTAL:
    return true;                     // edge is clipped anyway
  case INTERN:
    break;
  }
  return false;  
  // this is the case for INTERN and v is not at the clipped end of edge e
}

//------------------------------------------------------------------------------
// first_not_clipped_edge()
// searches counterclockwise the edges adjacent to source(e) for a first not 
// clipped edge.
// precondition:
// * source(e) is in V_mod
// * e as start edge is clipped at its source
// parameters:
// edge e       the edge from which starts the search
// return value:
// edge         the edge which is the first not clipped edge
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
edge Voronoi_Diagram<P_VNODE,P_SITE>::
first_not_clipped_edge(edge e)
{
    TRACE("      V.first_not_clipped_edge: ");AVDD(e);TRACEN("");
    ASSERT(clipped_at_source(e), first_not_clipped_edge1);
    ASSERT(Mod_item[source(e)], first_not_clipped_edge2);
  do {
    e = cyclic_adj_pred(e);
  } while (clipped_at_source(e));       
  return e;
}


//------------------------------------------------------------------------------
// edge_basic_intersection_operation()
// calculates if the edge e is intersected by the new site t and returns a
// container which holds the calculated result. It is based on the basic
// operation edge_intersection_topology().
// parameters:
// edge e       the examined edge
// P_SITE t     the new site
// bool& error  an error flag in case of object restrictions
// return value:
// e_t_list_info        if the edge is intersected
// nil                  else
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
e_t_list_info  Voronoi_Diagram<P_VNODE,P_SITE>::
edge_basic_intersection_operation(edge e, P_SITE t, bool& error)
{
    TRACE("\n  **V.edge_basic_intersection_operation ");AVDD(e);TRACEN("");

  P_SITE p = left_site(e);
  P_SITE q = right_site(e);
  P_SITE r = source_site(e);
  P_SITE s = target_site(e);

  P_VNODE mr = (*this)[source(e)];
  P_VNODE ms = (*this)[target(e)];
  
  topology top_res = pAVD->edge_intersection_topology(p,q,r,s,t,mr,ms,error);
  return e_t_list_info(e,top_res,source(e),target(e));
}

//------------------------------------------------------------------------------
// fill_in_top_info()
// fills in the missing topology information for the update of the Voronoi 
// Diagram. This means it calculates the coordinates of the new nodes which lie
// on the edge and creates the referring nodes in the Voronoi Diagram. These 
// are kept in the e_t_list_info container until these nodes are connected 
// lateron. Only edges which are totally intersected are not treated this way, 
// as they're deleted anyway and don't produce new nodes. Note that two edges
// reference the topology box but only for the master edge p_et_info->e the
// topology information is correct without symmetric considerations. 
// preconditions:
// * all containers referred to by et_info contain references to edges which
//   are intersected in the topology 0 to 4.
// parameters: 
// e_t_list_item e_it   the container with the topology information
// P_SITE t             the new site
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
fill_in_top_info(e_t_list_item e_it, P_SITE t)
{
  e_t_list_info& et_info = E_t[e_it];
    TRACE("    V.fill_in_top_info: ");AVDD(et_info);TRACEN("");
    ASSERT(get_e_t_list_item(et_info.e) && get_e_t_list_item(reversal(et_info.e)),
           fill_in_top_info - check e_it master edge and rev)

  P_SITE p, q;
  switch (et_info.topol) {
  case SOURCE_SEGMENT:
  // calculates the coords of the new node <p,q,t> close to the source node 
  // of the edge and puts a new node into the Voronoi Diagram V.
    put_into_V_new(e_it);
    p = left_site(et_info.e);
    q = right_site(et_info.e);
    et_info.vn[SOURCE] = new_node(pAVD->get_node_coords(p,q,t));
    break;
  case TARGET_SEGMENT:
  // calculates the coords of the new node <p,q,t> close to the target node 
  // of the edge and puts a new node into the Voronoi Diagram V.
    put_into_V_new(e_it);
    p = right_site(et_info.e);
    q = left_site(et_info.e);
    et_info.vn[TARGET] = new_node(pAVD->get_node_coords(p,q,t));
    break;
  case INTERN:
  // calculates the coords of the new nodes <q,p,t> (close to the source 
  // node) and <p,q,t> (close to the target node) of the edge and puts two 
  // new nodes into the Voronoi Diagram V.
    put_into_V_new(e_it);
    p = left_site(et_info.e);
    q = right_site(et_info.e);
    et_info.vn[SOURCE] = new_node(pAVD->get_node_coords(p,t,q));
    et_info.vn[TARGET] = new_node(pAVD->get_node_coords(p,q,t));
    break;
  case TWO_SEGMENTS:
  // calculates the coords of the new nodes <p,q,t> (close to the source 
  // node) and <q,p,t> (close to the target node) of the edge and puts two 
  // new nodes into the Voronoi Diagram V.
    put_into_V_new(e_it);
    p = left_site(et_info.e);
    q = right_site(et_info.e);
    et_info.vn[SOURCE] = new_node(pAVD->get_node_coords(p,q,t));
    et_info.vn[TARGET] = new_node(pAVD->get_node_coords(p,t,q));
    break;
  case TOTAL:
    break;
  }
  TRACE("    V.fill_in_top_info created: ");
  AVDD(et_info.vn[SOURCE]);AVDD(et_info.vn[TARGET]);TRACEN("");
}

//------------------------------------------------------------------------------
// put_shortened_edge_into_T_edges()
// gets from the topology information of any intersected edge the description
// of the shortened edge and updates the history graph respectively.
// Note that two edges reference the topology box but only for the master 
// edge p_et_info->e the topology information is correct without symmetric 
// considerations. 
// parameters: 
// e_t_list_item e_it   the container with the topology information
// P_SITE t             the new site
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
put_shortened_edge_into_T_edges(e_t_list_item e_it, P_SITE t)
{
  e_t_list_info& et_info = E_t[e_it]; 

    TRACE("  V.put_shortened_edge_into_T_edges ");AVDD(et_info);TRACEN("");
    //error_handler(0,"fill_in_top_info: can it be clear who is master edge?");

  node v_new, v_new1, v_new2;
  h_node vh, vh1, vh2;
  edge e = et_info.e;
  P_SITE p = left_site(e);
  P_SITE q = right_site(e);

  switch (et_info.topol) {
  case SOURCE_SEGMENT:
    v_new = et_info.vn[SOURCE];
    vh = pAVD->H.new_description(hist_node(e));
    if (e == pAVD->H.master_edge(vh))
      pAVD->H.source_site(vh) = t; 
    else {
      ASSERT(reversal(e)==pAVD->H.master_edge(vh), 
             put_shortened_edge_into_T_edges - master link crash);
      pAVD->H.target_site(vh) = t;
    }
        TRACE("  update shortened: ");AVDDD(pAVD->H,vh);TRACEN("");
    put_into_T_edges(v_new,target(e),p,q,nil,reversal(e),vh);
    break;
  case TARGET_SEGMENT:
    v_new = et_info.vn[TARGET];
    vh = pAVD->H.new_description(hist_node(e));
    if (e == pAVD->H.master_edge(vh))
      pAVD->H.target_site(vh) = t; 
    else {
      ASSERT(reversal(e)==pAVD->H.master_edge(vh), 
             put_shortened_edge_into_T_edges - master link crash);
      pAVD->H.source_site(vh) = t;
    }
        TRACE("  update shortened: ");AVDDD(pAVD->H,vh);TRACEN("");
    put_into_T_edges(source(e),v_new,p,q,e,nil,vh);
    break;
  case INTERN:
    v_new1 = et_info.vn[SOURCE];
    v_new2 = et_info.vn[TARGET];
    vh1 = pAVD->H.new_description(hist_node(e));
    vh2 = pAVD->H.new_description(hist_node(e));
    if (e == pAVD->H.master_edge(vh1)) {
      pAVD->H.target_site(vh1) = t; 
      pAVD->H.source_site(vh2) = t; 
    } else {
      ASSERT(reversal(e)==pAVD->H.master_edge(vh1), 
             put_shortened_edge_into_T_edges - master link crash);
      pAVD->H.target_site(vh2) = t; 
      pAVD->H.source_site(vh1) = t; 
    }
      // at the target end of vh1 and the source end of vh2 
      // the desciption is changing
        TRACE("  update shortened1: ");AVDDD(pAVD->H,vh1);TRACEN("");
        TRACE("  update shortened2: ");AVDDD(pAVD->H,vh2);TRACEN("");
    put_into_T_edges(source(e),v_new1,p,q,e,nil,vh1);
    put_into_T_edges(v_new2,target(e),p,q,nil,reversal(e),vh2);
    break;
  case TWO_SEGMENTS:
    v_new1 = et_info.vn[SOURCE];
    v_new2 = et_info.vn[TARGET];
    vh = pAVD->H.new_description(hist_node(e));
    pAVD->H.source_site(vh) = t;
    pAVD->H.target_site(vh) = t; 
      // at the source and target end of vh the desciption is changing
        TRACE("  update shortened: ");AVDDD(pAVD->H,vh);TRACEN("");
    put_into_T_edges(v_new1,v_new2,p,q,nil,nil,vh);
    break;
  case TOTAL:
    break;
  }
}

//------------------------------------------------------------------------------
// plane_search_for_intersected_edges()
// collects all intersected edges of the Voronoi Diagram which are intersected 
// by the new site t. It's using a plane search algorithm from the start edge.
// From there it localizes all the intersected edges of V. As the cut out part
// of the Voronoi Diagram is one connected component this search finds all edges
// terminates then. After finding all intersected edges the topology information
// in the e_t_list_info container is completed with the new nodes of the Voronoi
// Diagram.
// The concerned nodes of the examined edges are categorized into the three node
// lists by the procedures put_into_V_mod()/V_del()/V_new().
// parameters:
// P_SITE t             the new site to be inserted
// bool& error          an error flag in case of object restrictions
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
plane_search_for_intersected_edges(edge e_start, P_SITE t, bool& error)
{
    TRACEN("V.plane_search_for_intersected_edges");

  mod_list_item ml_it;
  node vs,v,v_dummy;
  edge es,e;

  e_t_list_info et_info = edge_basic_intersection_operation(e_start,t,error);
  ASSERT((et_info.topol!=NO_INTERSECTION),"psfie: start edge not intersected!");

  put_into_E_t(e_start, et_info);

  switch (et_info.topol) {
    case SOURCE_SEGMENT:
      put_into_V_mod(e_start);
      break;
    case TARGET_SEGMENT:
      put_into_V_mod(reversal(e_start));
      break;
    case TWO_SEGMENTS:
    case TOTAL:
      put_into_V_mod(e_start);
      put_into_V_mod(reversal(e_start));
      break;
    case INTERN:
      // nothing to search, only one edge
      fill_in_top_info(get_e_t_list_item(e_start),t);
      return;
  }  

  ml_it = V_mod.first();
  do { // we cycle over all nodes in V_mod
    es = V_mod[ml_it];
    vs = source(es);

      TRACE("iteration start: ");AVDD(vs);AVDD(es);TRACEN("");

    e = cyclic_adj_succ(es);
    while (e != es) { // we cycle at current node |vs| clockwise

        TRACE("examined edge: ");AVDD(e);TRACEN("");

      if (!get_e_t_list_item(e)) { // supposing edge wasn't examined:
        et_info = edge_basic_intersection_operation(e,t,error);
        if (et_info.topol != NO_INTERSECTION) {
          // now the edge is intersected: 
          put_into_E_t(e, et_info);
          v = target(e);

          if (et_info.topol == TOTAL && !get_mod_list_item(v))
            put_into_V_mod(reversal(e));
          // only with TOTAL topology the area T is spread out among the nodes
          // and this only if the other node isn't already in V_mod or V_del

        } 
        else // no intersection of edge e or error means leaving T
          break; // leave while iteration before e == es
      }
      // we can expect a topology container for e here!
      if (get_topology(e,v_dummy,v_dummy) == TARGET_SEGMENT)
         // a topology of SOURCE_SEGMENT, TWO_SEGMENTS, TOTAL does not leave T
         // INTERN => does not get here, stops at the beginning!
        break; // leave first inner while iteration before e == es
      e = cyclic_adj_succ(e);
    }

    if (error) break;
    // leave outer do/while iteration because of error
    
    if (e == es) {
      ml_it = V_mod.succ(ml_it);
      move_into_V_del(vs);
      // transfer v from V_mod to V_del
    } else {
      // now the other way around from es counterclockwise:
      e = cyclic_adj_pred(es);
      while (e != es) { // we cycle at current node |vs| counterclockwise
        if (!get_e_t_list_item(e)) { // supposing edge wasn't examined:
          et_info = edge_basic_intersection_operation(e,t,error);
          if (et_info.topol != NO_INTERSECTION) {
            // now the edge is intersected:
            put_into_E_t(e, et_info);
            v = target(e);

            if (et_info.topol == TOTAL && !get_mod_list_item(v))
              put_into_V_mod(reversal(e));
            // only with TOTAL topology the area T is spread out among the nodes
            // and this only if the other node isn't already in V_mod/V_del
          } 
          else // no intersection of edge e or error means leaving T
            break;  // leave while iteration before e == es
        }
        if (get_topology(e,v_dummy,v_dummy) == TARGET_SEGMENT) 
           // a topology of SOURCE_SEGMENT, TWO_SEGMENTS, TOTAL does not leave T
           // INTERN => does not get here, stops at the beginning!
          break; // leave second inner while iteration before e == es
        e = cyclic_adj_pred(e);
      }
      if (error) break;
      // leave outer do/while iteration because of error
      ml_it = V_mod.succ(ml_it);
    }
  } while (ml_it != nil);

    TRACEN("  after plane search");

  if (error) { 
      TRACEN("  plane_search: ERROR!");
    clear_E_t();
    clear_mod_lists();
    V_new.clear();
    // clean up S, E_t, V_new, V_del, V_mod
    return;
  }

    TRACEN("  filling topology E_t:");

  e_t_list_item e_it;
  forall_items(e_it,E_t)
    fill_in_top_info(e_it,t);

    TRACEN("END of plane search");
} 

//------------------------------------------------------------------------------
// next_node_on_bd_T()
// calculates the next node which lies on the boundary of the intersection area
// T and is reached by a traversal of all edges which limit the actual examined 
// region of site p. The procedure starts at node v, along edge e, on the left 
// hand side there's the site p. The referenced parameters v and e deliver the 
// target node and the edge e which was traversed last. Note that v needs not
// to be source(e) but can also be a new node close to the source.
// In addition the witnesses for the new edge representation in H are entered 
// into H.
// precondition:
// * left_site(e) == p
// parameters:
// node& v      the start/end node 
// edge& e      the start/end edge
// P_SITE p     p is the site on the left
// h_node vh    the node which will represent the new edge in the updated V
// return value:
// true         if v_end is in V_new
// false        if v_end is in V_mod
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
bool  Voronoi_Diagram<P_VNODE,P_SITE>::
next_node_on_bd_T(node& v, edge& e, P_SITE p, h_node vh)
{
    TRACE("->V.next_node_on_bd_T: ");AVDD(v);AVDD(e);TRACEN("");
    ASSERT((left_site(e) == p),next_node_on_bd_T preconds)
    ASSERT(get_e_t_list_item(e), 
           "V.next_node_on_bd_T: edge has no topology information!");

  // first the special cases:
  // the edge directly leaves the intersection region T

  e_t_list_info& et_info = get_topology_box(e);
  node v1,v2;
  if (get_topology(e,v1,v2) == SOURCE_SEGMENT &&
      v == source(e)) {
    // the start node is the source node of e
    // the component of e between source and the clipping point is in T

    pAVD->H.witness_entry(e,vh);         // e is witness for the new edge
    v = v1;                      // endpoint = clipping point
                                 // e remains
      TRACE("      direct new exit via SOURCE_SEGMENT");
    return true;                 // v_end is in V_new
  }

  if (get_topology(e,v1,v2) == TWO_SEGMENTS &&
      v == source(e)) {
    // the component of e between source/target and the 
    // clipping point is in T
    pAVD->H.witness_entry(e,vh);  // e is witness for the new edge
    v = v1;                 // endpoint = clipping point
                            // e remains
      TRACE("      direct new exit via TWO_SEGMENTS");
    return true;            // v_end is in V_new
  }

  if (get_topology(e,v1,v2) == INTERN) {
    // the component of e between the two new points on e lies in T
    ASSERT((v == v1), next_node_on_bd_T);
    pAVD->H.witness_entry(e,vh);  // e is witness for new edge
    v = v2;                 // new node is at target end
                            // e remains
      TRACE("      direct new exit via INTERN");
    return true;            // v_end is in V_new        
  }

  // now the iterative search for the endpoint

  bool in_T = true;
  node w;
  edge en;

  while (in_T) {
    pAVD->H.witness_entry(e,vh);
    en = cyclic_adj_succ(reversal(e));  //face_cycle_succ(e);

    if (!get_e_t_list_item(en)) { // en is not intersected at all
      v = target(e);    // target(e)==source(en) is v_end
                        // e stays the same
        TRACE("      mod exit via "); AVDD(en);TRACEN("");
      return false;     // v_end is in V_mod
      in_T = false;      
    }

    // now the edge has topology information via the link to an e_t_list_item

    node v_e,v_dummy;
    switch (get_topology(en,v_e,v_dummy)) {

    case SOURCE_SEGMENT:
      pAVD->H.witness_entry_if_not_existing(en,vh); 
      // en is witness but maybe already inserted into H
      v = v_e;             // the new node close to source is v_end
      e = en;              // the last examined edge is en
        TRACE("      new exit via "); AVDD(e);TRACEN("");
      return true;         // v_end is in V_new
      in_T = false;
      break;
    case TARGET_SEGMENT:
      v = target(e);    // v is target(e)
                        // e stays the same
        TRACE("      mod exit via "); AVDD(e);TRACEN("");
      return false;     // v_end is in V_mod
      in_T = false;
      break;
    case TWO_SEGMENTS:
      pAVD->H.witness_entry_if_not_existing(en,vh); 
      // en is witness but maybe already inserted into H
      v = v_e;          // the new node close to source is v_end
      e = en;           // the last examined edge is en
        TRACE("      new exit via "); AVDD(e);TRACEN("");
      return true;      // v_end is in V_new
      in_T = false;
      break;
    default:
      // this is the case for TOTAL topology => e lies totally inside T
      // stay in iteration but examine next edge:
      e = en;
        TRACE("      iterate via "); AVDD(e);TRACEN("");
    }
  }
  return true; 
  // just for this bloody compiler who always warns about 
  // reaching the end of a non-void function
}


//------------------------------------------------------------------------------
// initialize_iteration_around_T()
// initializes the 4 start values for the iteration along the boundary of T = 
// the Voronoi region of t. The start node v_start is from V_mod or V_new. The 
// start edge e_start is accessible via the start node or the e_t_list_item. The 
// site p is determined as the site "left of" e_start starting from v_start. The 
// edge e is orientation edge in case of a start node v_mod for the insertion of 
// the new edge along the boundary of T and it's a witness for the new edge.
// precondition:
// * V_mod U V_new not empty
// parameters:
// node& v_start        the start node
// edge& e_start        the start edge
// P_SITE p             the site left of e
// edge& e              the orientation edge and witness in case of v_mod
// History_Graph& H
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
initialize_iteration_around_T(node& v_start, edge& e_start, P_SITE& p, edge& e)
{
    TRACEN("->V.initialize_iteration_around_T: ");
    ASSERT(!V_mod.empty()||!V_new.empty(),
           "V.initialize_iteration_around_T: both node_lists empty!");
 
  if (!V_mod.empty()) {
    edge e_mod = V_mod.head();
      TRACE("   V_mod not empty");AVDD(e);TRACEN("");
    e = first_not_clipped_edge(e_mod);
    v_start = source(e);
    e_start = cyclic_adj_succ(e);
    p = left_site(e_start);
  } 
  else { // V_mod is empty, but V_new is not
    e_t_list_info& et_inf = E_t[V_new.head()];
      TRACE("   V_new not empty with item:");AVDD(et_inf);TRACEN("");

    switch (et_inf.topol) {
    case SOURCE_SEGMENT:
      v_start = et_inf.vn[SOURCE];
      e_start = reversal(et_inf.e);
      break;
    case TARGET_SEGMENT:
      v_start = et_inf.vn[TARGET];
      e_start = et_inf.e;
      break;
    case INTERN:
      v_start = et_inf.vn[SOURCE];
      e_start = et_inf.e;
      break;
    case TWO_SEGMENTS:
      v_start = et_inf.vn[SOURCE];
      e_start = reversal(et_inf.e);
      break;
    default:
      error_handler(1,"  V.initialize_iteration_around_T: topology error!");
    }
    p = left_site(e_start);
    e = nil; // there's no orientation edge at a new node
      TRACE("   init: ");AVDD(v_start);AVDD(e_start);AVDD(p);TRACEN("");
  }
  // now v_start, e_start, p, e are determined
}

//------------------------------------------------------------------------------
// iterate_around_T_and_build_T_edges()
// basically fills the data structure T_edges with containers that describe the 
// edges which have to be inserted into the Voronoi Diagram, when t is inserted. 
// These edges are determined by a clockwise traversal around the boundary of T
// (the area which is cut out of the Voronoi Diagram by insertion of t). 
// First the start values are initialized: most important a node from 
// V_mod U V_new. Then the edges are iteratively determined by a search along 
// the boundary of an intersected Voronoi region which is partly intersected by 
// T. The new edge is kept in an item in T_edges. By a call of the procedures 
// insert_description_at... the needed update of the History Graph concerning 
// new, affected and shortened edges is triggered.
// precondition:
// * E_t != empty
// parameters:
// P_SITE t                     the new site to be inserted
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
iterate_around_T_and_build_T_edges(P_SITE t)
{
    TRACEN("V.iterate_around_T_and_build_T_edges ");
    ASSERT(!E_t.empty(), iterate_around_T_and_build_T_edges);

  node v_start, v_start_edge, v_current;
  edge e_current, e_witness, e1, e2, e3, e4;
  h_node vh;
  P_SITE p, p1, p2;
  P_SITE p_store;
  // in this pointer are the sites stored which determine the source end 
  // points of the edges along T.
  // for the first edge this site is filled in at the end of the iteration.

  e_t_list_item e_it;
  forall_items(e_it,E_t)
    put_shortened_edge_into_T_edges(e_it,t);
  // now all shortened edges are put into T_edges at the beginning
  // and also the History_Graph ist updated with the equivalent 
  // descriptions of the shortened edges

  t_edges_list_item   te_it = nil;
  if (!T_edges.empty())
    te_it = T_edges.last();
  // now te_it points to the last element of T_edges if not empty

  initialize_iteration_around_T(v_current,e_current,p,e_witness);
  // now all 4 start values are initialized: 
  // v_current, e_current, p, e_witness

  v_start = v_current;      // keep absolute start node
  v_start_edge = v_current; // keep current start node for new edge

  // main iteration along the boundary of the intersection area T:
  do {
    vh = pAVD->H.new_witness_entry(e_witness,p_store);
    // now there's a new node vh1 in H
    if (next_node_on_bd_T(v_current,e_current,p,vh)) {
      // now v_current is in V_new as the endpoint of the just found 
      // new edge along the boundary of T:
      TRACE(" end on bd T at v_new next edge: ");AVDD(e_current);TRACEN("");

      put_into_T_edges(v_start_edge,v_current,p,t,e_witness,nil,vh);

      // now the new item in T_edges has all necessary information of the new
      // edge: start&end nodes, the two sites defining it, two edges for the 
      // insertion position at the nodes, and the corresponding h_node in H

      p1 = p;                     // left of boundary of T left of e_current
      p2 = right_site(e_current); // left of boundary of T right of e_current
      p_store = pAVD->H.insert_descriptions_at_v_new(vh,p1,p2,t);

      // the new tupel to start next search:
      v_start_edge = v_current;  
        // keep the node as the start node of the next edge
      e_current = reversal(e_current);
        // the edge along which the search is started is the reverted one
      p = p2;
        // change side and site
      e_witness = nil;           
        // no new witness at v_new outside T
    }
    else {      
      // now v_current is in V_mod
      TRACE("end on bd T at v_mod next edge: ");AVDD(e_current);TRACEN("");

      e1 = reversal(e_current);
      e2 = cyclic_adj_succ(e1);
      e3 = first_not_clipped_edge(e1);
      e4 = cyclic_adj_succ(e3);
      p1 = p;
      p2 = left_site(e4);
      pAVD->H.witness_entry_if_not_existing(e2,vh);   
      // e2 is a witness for the new edge if it was not already used as one

      put_into_T_edges(v_start_edge,v_current,p,t,e_witness,e1,vh);
      // now the new item in T_edges has all necessary information of the new 
      // edge: start&end nodes, the two sites defining it, two edges for the 
      // insertion position at the nodes, and the corresponding h_node in H

      p_store = pAVD->H.insert_descriptions_at_v_mod(e2,e3,vh,p1,p2,t);

      // the new tupel to start next search:
      v_start_edge = v_current;  
        // keep the node as the start node of the next edge
      p = p2;
        // the site changes to that between e3 and e4
      e_current = e4;
        // this is the edge along which the search starts
      e_witness = e3;
        // this is the first witness and orientation edge after which 
        // the new edge is to be inserted
    }
  } while (v_current != v_start); // iterate until arrived at v_start

  // now at the end only update the start node of the first new edge:
  if (te_it)
    te_it = T_edges.succ(te_it);
  else // te_it = nil
    te_it = T_edges.first();
  pAVD->H[T_edges[te_it].vh].p_s[0] = p_store;
}

//------------------------------------------------------------------------------
// insert_new_edges_and_clean_up()
// empties the 5 help data structures and updates the Voronoi Diagramm with the 
// obtained information. T_edges contains the new to be inserted edges. E_t 
// contains the to be deleted edges. V_del contains the to be deleted nodes. 
// V_mod contains the modified nodes, which have to be reinitialized. V_new had
// just references to new node information via E_t_edges_items.
// preconditions:
// * T_edges != empty
// parameters:
// return value:
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
insert_new_edges_and_clean_up()
{
    TRACEN("V.insert_new_edges_and_clean_up");
    ASSERT(!T_edges.empty(),"V.insert_new_edges_and_clean_up: T_edges empty!");

  edge e;
  e_t_list_info et_inf;

  // there are affected edges which still have links
  // to previous leafs and thus have to be updated:
  while (!Aff_edges.empty())
  {
    aff_edges_list_info aff_inf = Aff_edges.pop();            
    // get container
    secure_links_and_master_correspondence(aff_inf.e, aff_inf.vh);
    // secure correspondence between source and target of e and vh
  }

  // now insert the new edges and then the shortened edges into the
  // diagram, as the later are all at the beginning; this is done in 
  // reverted order from T_edges
  // by doing so the adjacency list of new nodes get the right cyclic
  // order
  edge e_last = nil;
  bool last_edge_correction = (T_edges[T_edges.last()].e[1]==nil);
    // in this case the target node of the last edge is a new node;
    // if you consider the edge insertion at this node you notice
    // that the adjacency list of this node is wrongly ordered as
    // we first insert the last edge of the iteration around T
    // then the first edge of the iteration around T and finally
    // the shortened edge ending at the new node. This gives a
    // wrong embedding which we have to correct at the end
  
  while (!T_edges.empty()) {
    t_edges_list_info<P_SITE> t_inf = T_edges.Pop();
    // get last T_edges container in list

    e = new_bi_edge(t_inf.v[0],   t_inf.v[1], 
                    t_inf.p_s[0], t_inf.p_s[1],
                    t_inf.e[0],   t_inf.e[1]);
      // put edge into Voronoi Diagram
    secure_links_and_master_correspondence(e,t_inf.vh);
      // set cross structure links

    if (last_edge_correction) {
      e_last = e;
      last_edge_correction = false;
    }
  }

  node v_s,v_t;
  if (e_last) {
    // now the incident edges of the target node of this edge 
    // have to be reordered

    e_last = reversal(e_last); 
    // we switch to the one out of the target node
    move_edge(e_last, cyclic_adj_succ(e_last), target(e_last));
  }

  // now delete all intersected edges:
  while (!E_t.empty()) {
    et_inf = E_t.pop();            
    // get container from E_t
    ASSERT((pAVD->H.master_edge(hist_node(et_inf.e)) == et_inf.e ||
            pAVD->H.master_edge(hist_node(et_inf.e)) == reversal(et_inf.e)),
           insert_new_edges_and_clean_up);
    pAVD->H.master_edge(hist_node(et_inf.e)) = nil;
    del_edge(et_inf.e);
    del_edge(reversal(et_inf.e));
    // delete the concerned edge
  }

  // now delete all nodes which have to be deleted
  while (!V_del.empty()) {
    edge e_del = V_del.pop(); // get container from V_del
    del_node(source(e_del));  // delete v_del
  }

  // now reinitialize all modification information and
  // clear the references to the now empty E_t list
  Mod_item.init(*this,nil);
  E_t_item.init(*this,nil);
  V_mod.clear();
  V_new.clear();

    // some debugging which shows everything's alright
    debug_nodes();
    debug_node_lists();
}

//------------------------------------------------------------------------------
// draw_edge()
// draws the edge e in the Window W. It calls the basic procedure 
// draw_bisector. 
// parameters:
// edge e       the edge to be drawn
// window& W    the window in which is drawn
// color c      the color in which is drawn
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
draw_edge(edge e, window& W, color c)
{
    TRACE("V.draw_edge: "); AVDD(e);AVDD(source(e));AVDD(target(e));TRACEN("");
  P_SITE p = left_site(e);
  P_SITE q = right_site(e);
  if ((p == pAVD->infinity()) || (q == pAVD->infinity())) return;
  pAVD->draw_bisector(p,q,(*this)[source(e)],(*this)[target(e)],W,c);
}


//------------------------------------------------------------------------------
// draw_diagram()
// we only draw the edges of the Voronoi Diagram of V into the window W.
// Note that we only draw the master edge to avoid double drawing of 
// bidirected edges.
// parameters:
// window& W    the window drawn in
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
draw_diagram(window& W, color c)
{
  edge e;
  forall_edges(e,*this)
    if (pAVD->H.master_edge(hist_node(e)) == e) draw_edge(e,W,c); 
}


//------------------------------------------------------------------------------
// draw_update_region()
// draws the edges of the Voronoi Diagram of V into the window W which are 
// concerned by the incrementation step. 
// parameters:
// window& W    the window drawn in
// color c1,c2  the colors the intersected and new edges are drawn with
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Voronoi_Diagram<P_VNODE,P_SITE>::
draw_update_region(P_SITE t, window& W, color c1, color c2)
{
  //W.start_buffering();
  pAVD->draw_site(t,W,c2);
  e_t_list_item eit;
  forall_items(eit,E_t) { // intersected edges
    draw_edge(E_t[eit].e,W,c1);
  }
  t_edges_list_item ptit;
  P_SITE p,q;
  P_VNODE vn1, vn2;
  forall_items(ptit,T_edges) { // new edges
    p = T_edges[ptit].p_s[0]; q = T_edges[ptit].p_s[1];
    vn1 = inf(T_edges[ptit].v[0]); 
    vn2 = inf(T_edges[ptit].v[1]); 
    if (p != pAVD->infinity() && q != pAVD->infinity())
      pAVD->draw_bisector(p,q,vn1,vn2,W,c2);
  }
  //W.flush_buffer();
  //W.stop_buffering();
}

#endif // VORONOI_DIAGRAM_C
//----------------------------------- EOF --------------------------------------

