
#ifndef HISTORY_GRAPH_C
#define HISTORY_GRAPH_C

#include "avd.h"
 
//------------------------------------------------------------------------------
// procedure definition:
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// description_is_intersected()
// calculates if the edge which is described by the h_node vh is intersected 
// by the new site t. It calls our basic operation edge_is_intersected().
// parameters:
// h_node vh    the node containing the edge information
// P_SITE t     the new site
// bool& error  an error flag in case of object restrictions
// return value:
// true         if the edge is intersected
// false        else
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
bool  History_Graph<P_VNODE,P_SITE>::
description_is_intersected(h_node vh, P_SITE t, bool& error)
{
    TRACE("\n->H.description_is_intersected: ");
    debug(vh);
    TRACEN("");

  h_node_info<P_SITE>& hi = (*this)[vh];
  P_SITE p = hi.p_s[1],
         q = hi.p_s[2],
         r = hi.p_s[0],
         s = hi.p_s[3];

  P_VNODE mr = pAVD->get_node_coords(p,q,r);
  P_VNODE ms = pAVD->get_node_coords(q,p,s);

  bool is_intersected = pAVD->edge_is_intersected(p,q,r,s,t,mr,ms,error);
  
    TRACE("is intersected: ");TRACEN(int(is_intersected));

  return is_intersected;
}

//------------------------------------------------------------------------------
// dfs_for_intersected_leaf()
// searches for a leaf in the History Graph which is intersected by the new
// site t. It's using the recursive dfs procedure go_for_intersected_leaf from
// the source of the HG.
// parameters:
// P_SITE t             the new site to be inserted
// edge& e_int          the intersected edge corresponding to the leaf
// bool& error          an error flag in case of object restrictions
// return value:
// true         if an intersected leaf is found
// false        else
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
bool  History_Graph<P_VNODE,P_SITE>::
dfs_for_intersected_leaf(P_SITE t, edge& e_int, bool& error)
{
    TRACE("H.dfs_for_intersected_leaf ");TRACEN("<source>");

  h_node wh;
  mark_visited(vh_source);  
  forall_adj_nodes(wh,vh_source) {
    if (go_for_intersected_leaf(wh,t,e_int,error)) {
      re_init(vh_source);
      return true;
    }
  }
  re_init(vh_source);
  return false;
}

//------------------------------------------------------------------------------
// go_for_intersected_leaf()
// the recursive procedure searching the History Graph for an intersected
// leaf.
// parameters:
// h_node vh            the root of the subgraph to be searched
// P_SITE t             the new site to be inserted
// edge& e_int          the intersected edge corresponding to the leaf
// bool& error          an error flag in case of object restrictions
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
bool  History_Graph<P_VNODE,P_SITE>::
go_for_intersected_leaf(h_node vh, P_SITE t, edge& e_int, bool& error)
{
    TRACE("H.go_for_intersected_leaf "); AVDD(vh); TRACEN("");

  if (visited(vh) || error) 
    return false;
  else 
    mark_visited(vh);
  
  if (!description_is_intersected(vh,t,error))
    return false;

  // now the corresponding edge of vh is intersected:

  if (outdeg(vh) == 0) {
  // vh is leaf and the corresponding edge is intersected
    e_int = master_edge(vh);
    return true;
  }
  
  // now vh is no leaf thus all sons should be examined:

  h_node wh;
  forall_adj_nodes(wh,vh) {
    if (go_for_intersected_leaf(wh,t,e_int,error))
      return true;
  }
  return false;
}

//------------------------------------------------------------------------------
// re_init()
// sets the visited flags back to false.
// parameters:
// h_node vh    the actual start node of the subtree
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  History_Graph<P_VNODE,P_SITE>::
re_init(h_node vh)
{
  if (!visited(vh))
    return;
  mark_unvisited(vh);
  h_node wh;
  forall_adj_nodes(wh,vh) {
    re_init(wh);
  }
}

//------------------------------------------------------------------------------
// new_witness_entry()
// creates a new node and a directed edge to this node from the node which holds
// the description of e in the History Graph. Moreover the h_node_info container
// is filled with the node description information given to the procedure.
// parameters:
// edge e                 the edge which is the witness for the new node vh
// P_SITE p_store         the stored start node determining site of the edge 
// return value:
// h_node                 the new created node in H
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
h_node  History_Graph<P_VNODE,P_SITE>::
new_witness_entry(edge e, P_SITE p_store)
{
    TRACE("  H.new_witness_entry: witness: ");AVDDD(pAVD->V,e); TRACEN("");
    TRACE("   stored site: ");AVDD(p_store);TRACEN("");
 
  P_SITE dummy;
  h_node vh = new_node(h_node_info<P_SITE>(p_store,dummy,dummy,dummy));  
    // create new node in H
  if (e != nil) { // if e is provided enter history
    new_edge(pAVD->V.hist_node(e),vh);                  
  }

    TRACE("   new node: "); AVDD(vh);TRACEN(" END");

  return vh;
}


//------------------------------------------------------------------------------
// witness_entry()
// creates an directed edge in the History Graph between the node which holds 
// the description of e and the node vh.
// parameters:
// edge e       the edge which is the witness for the new node vh
// h_node vh    the node to which the edge is directed
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  History_Graph<P_VNODE,P_SITE>::
witness_entry(edge e, h_node vh)
{
    TRACE("  H.witness_entry: ");AVDDD(pAVD->V,e);TRACEN(" END");
  new_edge(pAVD->V.hist_node(e),vh);
}

//------------------------------------------------------------------------------
// witness_entry_if_not_existing()
// creates an directed edge in the History Graph between the node which holds 
// the description of e and the node vh if this links does not already exist.
// parameters:
// edge e       the edge which is the witness for the new node vh
// h_node vh    the node to which the edge should be directed
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  History_Graph<P_VNODE,P_SITE>::
witness_entry_if_not_existing(edge e, h_node vh)
{
    TRACE("  H.witness_entry_if_not_existing: ");AVDDD(pAVD->V,e);TRACEN(" END");

  bool existing = false;
  h_node vh_tmp;
  h_node vh_from = pAVD->V.hist_node(e);
  forall_adj_nodes(vh_tmp,vh_from) {
    if (vh_tmp == vh) existing = true;
  }
  if (!existing) 
    new_edge(vh_from,vh);
}


//------------------------------------------------------------------------------
// new_or_existing_description()
// returns the node in the History Graph which is the representation of the 
// edge e. It creates a new node in the History Graph with the copied 
// information of the given node if there's not already a son of the current 
// node vh with some changed information. This could be the case for edges which 
// are affected at one end and shortened at the other end.
// In Addition it creates an directed edge from hist_node(e)->vh to the new node.
// parameters:
// edge e              the edge whose h_node is searched
// return value:
// h_node              the node which is needed (maybe created, maybe existing)
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
h_node  History_Graph<P_VNODE,P_SITE>::
new_or_existing_description(edge e)
{
  TRACE("  H.new_or_existing_description: ");AVDDD(pAVD->V,e);TRACEN("");

  P_SITE p_s1,p_s2;
  h_node vh = pAVD->V.hist_node(e), vh_lauf;
  P_SITE p = pAVD->V.left_site(e);
  P_SITE q = pAVD->V.right_site(e);

  forall_adj_nodes(vh_lauf,vh) {
    p_s1 = left_site(vh_lauf);
    p_s2 = right_site(vh_lauf);
    if ((p == p_s1)&&(q == p_s2) || (p == p_s2)&&(q == p_s1))
      return vh_lauf;
  }

  // now there was no son of vh with the needed site information thus we have
  // to create one:

  h_node vh_new = new_description(vh); 

  // now a link to the new created node from the edge which it refers to 
  // is only necessary if the edge is still in the Voronoi Diagram 
  // afterwards <=> this is dependent on the fact if it is intersected

  if (!pAVD->V.get_e_t_list_item(e)) {
    // here e is not intersected and thus stays in V
    TRACEN("   e is not intersected => enters Aff_edges");
    pAVD->V.Aff_edges.append(aff_edges_list_info(e,vh_new));
  }

  TRACE("   new node: "); AVDD(vh_new); TRACEN(" END");
  return vh_new;
}

//------------------------------------------------------------------------------
// insert_descriptions_at_v_new()
// copes with the descriptions of the two new edges at a new node in V. 
// The three sites p, q, t have to be inserted into the h_node_info containers 
// of the just examined new edge along the boundary of T ending in v_new. 
// The information for the next new edge along the boundary of T is returned 
// for further adjustments.
// parameters:
// h_node vh    the node which contains the description of v_new as end of the 
//              new edge
// P_SITE p, q  the sites on each side of e
// P_SITE t     the new inserted site
// return value:
// P_SITE       the source node determinig site for the next new edge
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
P_SITE  History_Graph<P_VNODE,P_SITE>::
insert_descriptions_at_v_new(h_node vh, P_SITE p, P_SITE q, P_SITE t)
{
    TRACEN("  H.insert_descriptions_at_v_new");

  // for the new edge the node determining site is q:

  left_site(vh)   = p;
  right_site(vh)  = t;
  target_site(vh) = q;

  return p;
  // this is the determining node for the source node of the next new edge
}

//------------------------------------------------------------------------------
// insert_descriptions_at_v_mod()
// copes with the descriptions of the four concerned edges at a modified node 
// in V. The first visited edge e1 which is outside T when traversing the 
// edges starting in v_mod in a clockwise order is affected. Then after all 
// edges outside T the first edge e2 before entering T again is also affected. 
// If there's only one edge outside T e1 == e2.
// Moreover it completes the description of v_mod as endpoint of the just 
// examined new edge and it returns the site which defines startpoint of the 
// next examined edge. Remember that we are walking around T in clockwise
// order.
// parameters:
// edge e1      the first edge at v_mod outside T = affected edge
// edge e2      the last edge before entering T again = affected edge
// h_node vh    the node which contains the description of v_mod as end of the 
//              new edge
// P_SITE p1    site before e1 (clockwise)
// P_SITE p2    site after e2 (clockwise)
// P_SITE t     the new site to be inserted
// return value:
// P_SITE       the source node determinig site for the next new edge
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
P_SITE  History_Graph<P_VNODE,P_SITE>::
insert_descriptions_at_v_mod (edge e1, edge e2, h_node vh, 
                              P_SITE p1, P_SITE p2, P_SITE t)
{
    TRACEN("  H.insert_descriptions_at_v_mod");
    ASSERT( (source(e1)==source(e2)), insert_descriptions_at_v_mod);

  if (e1 == e2) { // there's only one edge outside T
      TRACEN("   e1 = e2");
  
    h_node vh_e1 = new_or_existing_description(e1);
    // to correct the History Graph for the affected edge e1:

      ASSERT( ((left_site(vh_e1)==p1)||(right_site(vh_e1)==p1)),
              insert_descriptions_at_v_mod);

    edge_end ee;
    if (left_site(vh_e1) == p1) 
    // description of vh_e1 has to be corrected at the right end of the edge
      source_site(vh_e1) = t;
    else
      target_site(vh_e1) = t;

      TRACE("   affected: "); AVDD(vh_e1); TRACEN("");
  }
  else // e1 != e2
  {
      TRACEN("   e1 != e2");
    
    h_node vh_e1 = new_or_existing_description(e1); 
    // to correct the History Graph for the affected edge e1 

      ASSERT( ((left_site(vh_e1)==p1)||(right_site(vh_e1)==p1)), 
              insert_descriptions_at_v_mod);
      TRACE("   affected1: ");AVDD(vh_e1);TRACEN("");

    h_node vh_e2 = new_or_existing_description(e2); 
    // to correct the History Graph for the affected edge e2 
 
      ASSERT( ((left_site(vh_e2)==p2)||(right_site(vh_e2)==p2)),
              insert_descriptions_at_v_mod);

    if (right_site(vh_e2) == p2) 
    // description of vh_e2 has to be corrected at the right end of the edge
      source_site(vh_e2) = t;
    else
      target_site(vh_e2) = t;

      TRACE("   affected2: ");AVDD(vh_e2);TRACEN("");
  }

  // now take care of the just created new edge

  P_SITE q1 = pAVD->V.right_site(e1);
  P_SITE q2 = pAVD->V.left_site(e2);

  left_site(vh) = p1;
  right_site(vh) = t;
  target_site(vh) = q1;

    TRACE("   new edge: ");AVDD(vh);TRACEN("");
    TRACEN("  END of insert");
  
  // now take care of description for next new edge
  return p1;
}

#endif // HISTORY_GRAPH_C

//----------------------------------- EOF --------------------------------------

