
#ifndef ES_ADAPTATION_H
#define ES_ADAPTATION_H

#include <math.h>
#include <LEDA/stream.h>
#include <LEDA/real.h>
#include <LEDA/plane.h>
#include <LEDA/rat_polygon.h>
#include <LEDA/dictionary.h>
#include <LEDA/window.h>
#include <LEDA/ps_file.h>
#include <LEP/avd/avd.h>
#include <LEP/avd/p_handle.h>
#include <LEP/avd/es_plane_ext.h>

enum node_status {UNAFFECTED=0, AFFECTED=1, CLIPPED=2};


inline ostream&  operator<<(ostream& out, node_status ns)
{
  switch (ns) {
  case UNAFFECTED: cout << "UNAFFECTED"; break;
  case AFFECTED: cout << "AFFECTED"; break;
  case CLIPPED: cout << "CLIPPED"; break;
  }
  return out;
}

inline ostream&  operator<<(ostream& out, topology t)
{
  switch (t) {
  case NO_INTERSECTION: cout << "NO_INTERSECTION"; break;
  case TOTAL:           cout << "TOTAL"; break;
  case SOURCE_SEGMENT:  cout << "SOURCE_SEGMENT"; break;
  case TARGET_SEGMENT:  cout << "TARGET_SEGMENT"; break;
  case INTERN:          cout << "INTERN"; break;
  case TWO_SEGMENTS:    cout << "TWO_SEGMENTS"; break;
  }
  return out;
}

#ifdef SF_DEBUG
#define CASESTAT(n,s) casestat(n,s)
#define ZEROSTAT(n,s,sign) zerostat(n,s,sign)
#else
#define CASESTAT(n,s)
#define ZEROSTAT(n,s,sign)
#endif


#ifdef SF_DEBUG
static int count_PPP_S_all;
static int count_PPP_P_all;

static int count_SPP_S_all;
static int count_SPP_P_all;

static int count_SSP_S_all;
static int count_SSP_P_all;

static int count_SSS_S_all;
static int count_SSS_P_all;

static int count_PPP_S_zero;
static int count_PPP_P_zero;

static int count_SPP_S_zero;
static int count_SPP_P_zero;

static int count_SSP_S_zero;
static int count_SSP_P_zero;

static int count_SSS_S_zero;
static int count_SSS_P_zero;

inline void print_filter_stat()
{
  cout << "filter statistics: node_type x inserted_site_type\n"
  cout << "PPP (P0/Pall S0/Sall) ";
  cout << count_PPP_P_zero << "/" << count_PPP_P_all << "  ";
  cout << count_PPP_S_zero << "/" << count_PPP_S_all << endl;
  cout << "SPP (P0/Pall S0/Sall) ";
  cout << count_SPP_P_zero << "/" << count_SPP_P_all << "  ";
  cout << count_SPP_S_zero << "/" << count_SPP_S_all << endl;
  cout << "SSP (P0/Pall S0/Sall) ";
  cout << count_SSP_P_zero << "/" << count_SSP_P_all << "  ";
  cout << count_SSP_S_zero << "/" << count_SSP_S_all << endl;
  cout << "SSS (P0/Pall S0/Sall) ";
  cout << count_SSS_P_zero << "/" << count_SSS_P_all << "  ";
  cout << count_SSS_S_zero << "/" << count_SSS_S_all << endl;
  cout << endl;
}

inline void casestat(p_vnode v, p_site t)
{
  node_type tx=(*v).nt();
  if ((*t).is_seg()) {
    switch(tx) {
      case(PPP): 
        count_PPP_S_all++;
        break;
      case(SPP_disteq): 
      case(SPP_distdiff): 
        count_SPP_S_all++;
        break;
      case(SSP_coltouch): 
      case(SSP_notcol): 
      case(SSP_alltouch): 
        count_SSP_S_all++;
        break;
      case(SSS_interior): 
      case(SSS_coltouch): 
      case(SSS_alltouch): 
        count_SSS_S_all++;
        break;
    }
  } else { // (*t).is_pnt())
    switch(tx) {
      case(PPP): 
        count_PPP_P_all++;
        break;
      case(SPP_disteq): 
      case(SPP_distdiff): 
        count_SPP_P_all++;
        break;
      case(SSP_coltouch): 
      case(SSP_notcol): 
      case(SSP_alltouch): 
        count_SSP_P_all++;
        break;
      case(SSS_interior): 
      case(SSS_coltouch): 
      case(SSS_alltouch): 
        count_SSS_P_all++;
        break;
    }
  }
}

inline void zerostat(p_vnode v, p_site t, int sign)
{
  node_type tx=(*v).nt();
  if (sign == 0 && (*t).is_seg()) {
    switch(tx) {
      case(PPP): 
        count_PPP_S_zero++;
        break;
      case(SPP_disteq): 
      case(SPP_distdiff): 
        count_SPP_S_zero++;
        break;
      case(SSP_coltouch): 
      case(SSP_notcol): 
      case(SSP_alltouch): 
        count_SSP_S_zero++;
        break;
      case(SSS_interior): 
      case(SSS_coltouch): 
      case(SSS_alltouch): 
        count_SSS_S_zero++;
        break;
    }
  }
  if (sign == 0 && (*t).is_pnt()) {
    switch(tx) {
      case(PPP): 
        count_PPP_P_zero++;
        break;
      case(SPP_disteq): 
      case(SPP_distdiff): 
        count_SPP_P_zero++;
        break;
      case(SSP_coltouch): 
      case(SSP_notcol): 
      case(SSP_alltouch): 
        count_SSP_P_zero++;
        break;
      case(SSS_interior): 
      case(SSS_coltouch): 
      case(SSS_alltouch): 
        count_SSS_P_zero++;
        break;
    }
  }
}
#endif



#define sq(t) ((t)*(t))

inline point    rpnt2pnt(const rat_point& p) { return p.to_point(); }
inline segment  rseg2seg(const rat_segment& s) { return s.to_segment(); }
inline void     seg2line_paras(const rat_segment&, real&, real&, real&);
inline void     seg2line_paras(const rat_segment&, integer&, integer&, integer&);

inline int      sorted_switch_label_b2(bool,bool);
  // with the following return constants
  const int tt_label=3;
  const int tf_label=2;
  const int ft_label=1;
  const int ff_label=0;

inline int      sorted_switch_label_b3(bool,bool,bool);
  // with the following return constants
  const int ttt_label=7;
  const int ttf_label=6;
  const int tft_label=5;
  const int tff_label=4;
  const int ftt_label=3;
  const int ftf_label=2;
  const int fft_label=1;
  const int fff_label=0;

inline int      sorted_switch_label_i23(int,int);
  const int UU = 0;
  const int UA = 1;
  const int UC = 2;
  const int AU = 3;
  const int AA = 4;
  const int AC = 5;
  const int CU = 6;
  const int CA = 7;
  const int CC = 8;




enum site_type {PNT, SEG};
#define INFNUM -4711

/*{\Moptions
outfile=es_sites.man
}*/

/*{\Manpage {es_site} {} {Site objects storing Points and Segments} 
            {s}}*/

class es_site {

/*{\Mdefinition An instance of data type |\Mname| stores the site
        information used in the Voronoi Diagram construction. By its
        nature it either stores a |rat_point| or a |rat_segment|.}*/

protected:
  site_type   t;  // flag if site is point or segment
  rat_segment s;  // storing place for point or segment
  int n;          // debbuggin help: objects have numbers
  
  static int count;

  friend bool  operator==(const es_site& s1, const es_site& s2);
  friend bool  operator!=(const es_site& s1, const es_site& s2);
public:
  /*{\Mcreation 3}*/
  es_site()  { n=INFNUM; } 

  es_site(const rat_point& p, bool numbering = true); 
  /*{\Mcreate introduces a site storing a point |p| in the plane.
    if |numbering == true| then the site object gets an incremental
     number.}*/

  es_site(const rat_segment& s); 
  /*{\Mcreate introduces a site storing a line segment |s| in the
    plane.}*/

  es_site(const es_site&); 
  es_site(integer, integer); 
  es_site(integer, integer, integer, integer); 
 ~es_site()  {} 

  /*{\Moperations 3 2}*/

  es_site operator= (const es_site&);
  
  site_type    type() const    {  return t;  }

  bool         is_pnt() const  {  return (t==PNT);  }
  /*{\Mop     return true if |\Mvar| stores a point.}*/
 
  bool         is_seg() const  {  return (t==SEG);  }
  /*{\Mop     return true if |\Mvar| stores a segment.}*/

  rat_segment  rseg() const    { return s;  }
  /*{\Mop     returns the stored segment in |\Mvar|.
        \precond |\Mvar.is_seg()|.}*/

  rat_point    rpnt() const    { return s.start();  }
  /*{\Mop     returns the stored point in |\Mvar|.
        \precond |\Mvar.is_pnt()|.}*/

  int          nr() const      { return n;  }

  point        pnt() const  { return s.start().to_point();  }
  segment      seg() const  { return s.to_segment();  }

  void         draw(window&, color);

  friend ostream& operator<<(ostream&, const es_site&);
  friend void debug(const es_site& s) { cerr << s; }  
  friend void debug_short(const es_site& s) 
  { if (s.nr()==INFNUM) cerr<<"[I]";
    else cerr <<"["<<s.nr()<<"]"; }  

  LEDA_MEMORY(es_site); 

/*{\Mimplementation The data type stores the geometric information
together with a type id which allows to distinguish point and segment
existence. }*/

};



#define p_site p_handle<es_site>
// we eliminate the symbol p_site in the global scope

enum node_type {SSS_alltouch = 0,
                SSS_coltouch, 
                SSS_interior, 
                SSP_alltouch,    // 3
                SSP_notcoll,
                SSP_coltouch, 
                SPP_distdiff,    // 6
                SPP_disteq, 
                PPP};            // 8


/*{\Moptions
outfile=es_vnodes.man
}*/

/*{\Manpage {es_vnode} {} {Node Coordinates Container} {v}}*/


class es_vnode {

/*{\Mdefinition An instance of data type |\Mname| stores the
coordinate information used for the embedding of the Voronoi Diagram
into the plane. It stores point coordintes by the |real| data type.}*/

protected:
  real xCoord;
  real yCoord;
  real wCoord;

  integer xCoordI;
  integer yCoordI;
  integer wCoordI;
  
  double xCoordD;
  double yCoordD;
  double wCoordD;

  double xCoordDS;
  double yCoordDS;
  double wCoordDS;
 
  p_site csites[3];
  node_type t;
  bool rat;
  
  bool RealOk;
  bool DblOk;
  bool IntOk;

  void CompReal();
  void CompDbl();
  void CompInt();

  int SSS_coltouch_Exact();
  int SSS_coltouch_Integer();
  int SSS_coltouch_Double();

  int SSS_interior_Exact();
  int SSS_interior_Integer();
  int SSS_interior_Double();

  int SSP_notcoll_Exact();
  int SSP_notcoll_Integer();
  int SSP_notcoll_Double();

  int SSP_coltouch_Exact();
  int SSP_coltouch_Integer();
  int SSP_coltouch_Double();

  int SPP_distdiff_Exact();
  int SPP_distdiff_Integer();
  int SPP_distdiff_Double();

  int SPP_disteq_Exact();
  int SPP_disteq_Integer();
  int SPP_disteq_Double();

  int PPP_Exact();
  int PPP_Integer();
  int PPP_Double();
  
  void PreCompSSS();
  void PreCompSSP();
  void PreCompSPP();

public:
  /*{\Mcreation 3}*/
  es_vnode () {}

  es_vnode(p_site p, p_site q, p_site r);

  es_vnode(real x, real y, real w, bool i=false); 
  /*{\Mcreate introduces a |\Mname| storing the coordinates of the
     Voronoi node |pqr| in homogeneous form $(x/w,y/w)$.}*/

  es_vnode(const es_vnode&);
  ~es_vnode() {}

  /*{\Moperations 3 2}*/
  
  es_vnode operator=(const es_vnode&);
  bool operator==(const es_vnode&);
  bool operator!=(const es_vnode& vo) {return !operator==(vo);}

  real x() const 
  /*{\Mop returns the numerator of the $x$-coordinate of |\Mvar|.}*/
  { if (!RealOk) const_cast<es_vnode*>(this)->CompReal(); return xCoord; }

  real y() const
  /*{\Mop     returns the numerator of the $y$-coordinate of |\Mvar|.}*/
  { if (!RealOk) const_cast<es_vnode*>(this)->CompReal(); return yCoord; }

  real w() const 
  /*{\Mop     returns the homogenizing common denominator of the 
        coordinates of |\Mvar|.}*/
  { if (!RealOk) const_cast<es_vnode*>(this)->CompReal(); return wCoord; }

  integer xI() const 
  { if (!IntOk) const_cast<es_vnode*>(this)->CompInt(); return xCoordI; }
  integer yI() const 
  { if (!IntOk) const_cast<es_vnode*>(this)->CompInt(); return yCoordI; }
  integer wI() const 
  { if (!IntOk) const_cast<es_vnode*>(this)->CompInt(); return wCoordI; }

  double xD() const 
  { if (!DblOk) const_cast<es_vnode*>(this)->CompDbl(); return xCoordD; }
  double yD() const 
  { if (!DblOk) const_cast<es_vnode*>(this)->CompDbl(); return yCoordD; }
  double wD() const 
  { if (!DblOk) const_cast<es_vnode*>(this)->CompDbl(); return wCoordD; }

  double xDS() const 
  { if (!DblOk) const_cast<es_vnode*>(this)->CompDbl(); return xCoordDS; }
  double yDS() const 
  { if (!DblOk) const_cast<es_vnode*>(this)->CompDbl(); return yCoordDS; }
  double wDS() const 
  { if (!DblOk) const_cast<es_vnode*>(this)->CompDbl(); return wCoordDS; }

  bool IsRat() const 
  { if (!IntOk) const_cast<es_vnode*>(this)->CompInt(); return rat; }

  es_site& csite(int i) const 
  { if (i<0 || i>2) error_handler(1,"es_vnode::csite: index out of bounds");
    return *(csites[i]); }
  node_type nt() const  { return t; }

  double xd() const { return (xD()/wD());  }
  /*{\Mop returns the approximated $x$-coordinate of |\Mvar|.}*/

  double yd() const { return (yD()/wD());  }
  /*{\Mop returns the approximated $y$-coordinate of |\Mvar|.}*/

  point  pnt() const { return point(xd(),yd());  }
  /*{\Mop returns an approximated double point of |\Mvar|.}*/

  friend ostream&  operator<<(ostream&, const es_vnode&);
  friend void debug(const es_vnode& s) { cerr << s; }  
  friend void debug_short(const es_vnode& s) { cerr << s; }  
  
  LEDA_MEMORY(es_vnode);

/*{\Mimplementation The data type stores the coordinate information
of a Voronoi node using a homogeneous representation.}*/

};

inline ostream&  operator<<(ostream& out, node_type nt)
{
  switch (nt) {
  case SSS_alltouch: cout << "SSS_alltouch"; break;
  case SSS_coltouch: cout << "SSS_coltouch"; break;
  case SSS_interior: cout << "SSS_interior"; break;
  case SSP_alltouch: cout << "SSP_alltouch"; break;
  case SSP_notcoll:  cout << "SSP_notcoll";  break;
  case SPP_distdiff: cout << "SPP_distdiff"; break;
  case SPP_disteq:   cout << "SPP_disteq";   break;
  case PPP:          cout << "PPP";          break;
  }
  return out;
}

#undef p_site


/*{\Moptions
outfile=esvd.man
}*/

/*{\Manpage {ES_Voronoi_Diagram} {} 
            {Euclidean Voronoi Diagrams of Segments}{VD}}*/

 
class ES_Voronoi_Diagram : public 
   Abstract_Voronoi_Diagram< p_handle<es_vnode>,
                             p_handle<es_site> >
{
/*{\Mdefinition An instance of data type |\Mname| stores combinatorial
and geometric embedding of the Euclidean Voronoi Diagrams of a set of
line segments and points. The diagram can be accessed in form of a
bidirected graph realizing the Voronoi vertices and edges such that
each edge is contained in exactly one face associated with a
site. Sites and Voronoi vertex coordinates are represented by types
described above. 

The data type is defined in \texttt{LEP/avd/es\_adaptation.h}.
}*/

public:

/*{\Mtypes 3}*/

  typedef es_vnode vnode;
  /*{\Mtypemember the coordinate container type ($\equiv$ |es_node|).}*/

  typedef es_site site;
  /*{\Mtypemember the site type ($\equiv$ |es_site|).}*/

  /*{\Moptions nextwarning=no }*/
  /*
  typedef p_handle<es_vnode> p_vnode;
  */
  /*{\Mtypemember the handle to a coordinate container.}*/

  /*{\Moptions nextwarning=no }*/
  /*
  typedef p_handle<es_site> p_site;
  */
  /*{\Mtypemember the handle to a site.}*/

protected:

  dictionary<rat_point,int> KnownPoints;

  enum bisec_type {PARA, SEGM, RAY, LIN, GAMMA};

  //-----------------------------------------------------------------------
  // THE VIRTUAL INTERFACE:
  // all the following procedures have to be programmed as an interface for
  // the Abstract Voronoi Diagrams
  //-----------------------------------------------------------------------

  bool init_is_possible(p_site, p_site, bool&);
  topology edge_intersection_topology(p_site, p_site, p_site, p_site, 
                                      p_site, p_vnode, p_vnode, bool&);
  bool edge_is_intersected(p_site, p_site, p_site, p_site, p_site, 
                           p_vnode, p_vnode, bool&);

  p_vnode get_node_coords(p_site p, p_site q, p_site r)
  { if (p == infinity() || q == infinity() || r == infinity())
    return gamma_node();
  else
    return p_vnode(vnode(p,q,r));
  }
  // inline with lazy scheme, the coordinates are calculated on request;

  void draw_bisector(p_site, p_site, p_vnode, p_vnode, window&, color);
  void draw_site(p_site, window&, color);


  //-----------------------------------------------------------------------
  // HELPERS for this:
  //-----------------------------------------------------------------------

  // for basic operations:
  bool inner_point_on_edge_is_clipped(p_site, p_site, p_site, p_site, 
                                      p_site, p_vnode, p_vnode);
  bool inner_point_on_edge_remains(p_site, p_site, p_site, p_site, 
                                   p_site, p_vnode, p_vnode);
  bool total_cut_when_both_affected(p_site, p_site, p_site, p_site, 
                                    p_site, p_vnode, p_vnode);


  node_status  calculate_node_status(p_site, p_site, p_site, p_site, 
                                     p_vnode);
  node_status  infinite_node_status(p_site, p_site, p_site);

  int   side_of_circle(p_vnode, const rat_point&);
  bool  is_in_end_perp_stripe(const rat_segment&, p_vnode);
  bool  line_hull_and_order(p_site, p_site, p_site);
  bool  illegal_intersection(p_site, p_site);
  bool  perp_separates(const rat_segment&, const rat_point&, 
                       p_vnode, p_vnode);
  bool  less_than_right_angle(const rat_segment&, const rat_segment&);
  int   orientation(const rat_segment&, vnode&);

  // for get_node_coords:
  vnode foot_of_perpendicular(vnode&, const rat_segment&);
  vnode other_end(p_site, vnode&);

  bool  right_turn(vnode&, vnode&, vnode&);

  // for draw_bisector:
  void  adapt_seg(segment&, bisec_type&, p_vnode, p_vnode);

  // nobody understands me ;-(
  int orientation(const rat_segment& s, const rat_point& p)
  { return ::orientation(s,p); } 
  int orientation(const rat_point& p, const rat_point& q, 
                    const rat_point& r)
  { return ::orientation(p,q,r); } 
  bool right_turn(const rat_point& p, const rat_point& q, 
                  const rat_point& r)
  { return ::right_turn(p,q,r); } 
  rat_point other_end(const rat_segment& s, const rat_point& p) 
  { return ::other_end(s,p); }   



  public:
  /*{\Mcreation 4.5}*/

  ES_Voronoi_Diagram() : Abstract_Voronoi_Diagram<p_vnode,p_site>() {}
  /*{\Mcreate introduces an empty Voronoi Diagram |\Mvar| of type |\Mname| in 
              the plane.}*/

  ~ES_Voronoi_Diagram() {}



  /*{\Moperations 1.5 2}*/

  int   insert(const rat_point& p);
  /*{\Mop     incrementally extends |\Mvar| by inserting |p| into the diagram.
              The return value states if the insertion of the site changed the 
              diagram (2), if it was inserted but only obtained an empty region 
              (1) or if it was an illegal site (0).}*/

  int   insert(const rat_segment& s);
  /*{\Mop incrementally extends |\Mvar| by inserting |s| into the
              diagram.  A segment site is split into its endpoints and
              its relative interior.  where the endpoints are inserted
              first. The return value states if the insertion of the
              site changed the diagram (2), if it was inserted but
              only obtained an empty region (1) or if it was an
              illegal site (0).}*/

  bool  insert(const rat_polygon& P);
  /*{\Mop     incrementally extends |\Mvar| by inserting |P| into the diagram.
              First all vertices are inserted then the corresponding edges are
              added.}*/

  int  animated_insert(const rat_point& p, window& W, int pause);
  int  animated_insert(const rat_segment& s, window& W, int pause);
  bool animated_insert(const rat_polygon& p, window& W, int pause);

  /*{\Moptions nextwarning=no}*/
  /*
    void   get_diagram(GRAPH<p_vnode,p_site>& G);
  */
  /*{\Mop provides the Voronoi diagram in form of a bidirected planar
        map.  The node parameters are handles to |vnode| coordinate
        containers which contain the geometric coordinates of the
        node.  The edge parameters are handles to |site| containers
        which contain the geometric object information of the site
        left of the directed edge.  The planar map encoding the
        diagram is enclosed into a bounding $\Gamma$-cycle of edges
        bounding the plane. For nodes on $\Gamma$ the coordinate
        pointer points to |\Mvar.gamma_node()|. For edges which are part of
        $\Gamma$ clockwise, the site pointer points to the special
        site |\Mvar.infinity()|. For more information on this please see
        the theory of Abstract Voronoi Diagrams as described in
        \cite{meise:phd}. For the types |site| and |vnode| refer to the
        corresponding manpages.}*/


  /*{\Moptions nextwarning=no}*/
  /*
    void   draw_diagram(window& W)
  */
  /*{\Mop     draws the diagram in the window W.}*/

  void   clear()
  /*{\Mop cleans the data structure and reinitializes it to the empty
              diagram.}*/
  { KnownPoints.clear();
    Abstract_Voronoi_Diagram< p_handle<es_vnode>,p_handle<es_site> >::
    clear(); }

};

/*{\Mimplementation The data type is inherited from the abstract data
type which can be adapted to concrete Voronoi Diagrams by providing
the necessary implementation interface procedures. The update follows
a randomized incremental scheme which uses a history graph to spot the
update region in logarithmic time. The update itself works in time
proportional to the size of the update region which is on average
constant work. The space requirement is linear in the size of the
input. For more information on the theory of Abstract Voronoi Diagrams
see \cite{meise:phd}. For the adaptation and numerical optimization used
in this package see \cite{burni:phd} and \cite{funke:dpl}.}*/

/*{\Mexample For an example please refer to [[demo/es_test.c]].
To use this data type please link with [[demo/exp_es_adaptation.c]] and
[[demo/exp_es_plane_ext.c]].}*/



#endif // ES_ADAPTATION_H


