//-----------------------------------------------------------------------
// VORONOI DIAGRAMS WITH POINTS AND EUCLIDEAN DISTANCE BASED ON
// RATIONAL POINTS OF LEDA
//-----------------------------------------------------------------------
 
#ifndef EP_ADAPTATION_H
#define EP_ADAPTATION_H

const int BETWEEN = 0;
const int ID_12   = 1;
const int ID_13   = 2;
const int ELSE    = 3;

#include <LEP/avd/avd.h>

#undef _DEBUG
#ifdef _DEBUGALL
#define _DEBUG
#define EPD(t) debug(t)
#define EPDS(t) debug_short(t)
#else
#define EPD(t)
#define EPDS(t)
#endif
#include <LEP/avd/debug.h>
#include <LEP/avd/p_handle.h>
#include <LEDA/window.h>
#include <LEDA/rat_point.h>

inline  void debug(const rat_point& s) 
{ cerr << "#" << ID_Number(s) << " " << s.to_point(); }

inline void debug_short(const rat_point& s) 
{ cerr << "#" << ID_Number(s);  }

/*{\Manpage {EP_Voronoi_Diagram} {} {Euclidean Voronoi Diagrams of Points} 
            {VD}}*/

class EP_Voronoi_Diagram : public 
  Abstract_Voronoi_Diagram< p_handle<rat_point>, p_handle<rat_point> >
{
/*{\Mdefinition An instance of data type |\Mname| stores combinatorial
and geometric embedding of the Euclidean Voronoi Diagrams of a set of
points. The diagram can be accessed in form of a bidirected graph
realizing the Voronoi vertices and edges such that each edge is
contained in exactly one face associated with a site. Sites and
Voronoi vertex coordinates are represented by LEDA |rat_points|.
                  
The data type is defined in \texttt{LEP/avd/ep\_adaptation.h}.}*/

public:

/*{\Mtypes 3}*/

  typedef rat_point vnode;
  /*{\Mtypemember the coordinate container type.}*/

  typedef rat_point site;
  /*{\Mtypemember the site type.}*/

  /*{\Moptions nextwarning=no }*/
  /*
  typedef p_handle<rat_point> p_vnode;
  */
  /*{\Mtypemember the handle to a coordinate container.}*/

  /*{\Moptions nextwarning=no }*/
  /*
  typedef p_handle<rat_point> p_site;
  */
  /*{\Mtypemember the handle to a site.}*/

protected:
  //-----------------------------------------------------------------------
  // THE VIRTUAL INTERFACE:
  // all the following procedures have to be programmed as an interface for
  // the Abstract Voronoi Diagrams
  //-----------------------------------------------------------------------

  bool init_is_possible(p_site, p_site, bool&);
  topology edge_intersection_topology(p_site, p_site, p_site, p_site, 
                                      p_site, p_vnode, p_vnode, bool&);
  bool edge_is_intersected(p_site, p_site, p_site, p_site, p_site, 
                           p_vnode, p_vnode, bool&);
  p_vnode get_node_coords(p_site, p_site, p_site);
  void draw_bisector(p_site, p_site, p_vnode, p_vnode, window&, color);
  void draw_site(p_site, window&, color);

  //-----------------------------------------------------------------------
  // MEISER INTERFACE:
  // all the following procedures have been proposed by S.MEISER as an
  // modularization of the above functions. See his thesis for more
  // information
  //-----------------------------------------------------------------------

  bool clipped_at_node(p_site, p_site, p_site, p_site, p_site, 
                       p_vnode, p_vnode, bool&);

  bool new_node_on_edge(p_site, p_site, p_site, p_site, p_site, 
                        p_vnode, p_vnode);

  //-----------------------------------------------------------------------
  // modulized further into:
  //-----------------------------------------------------------------------

  bool pqr_in_V(p_site p, p_site q, p_site r, p_site t, p_vnode m,
                bool& error);
  bool pqr_in_V(p_site p, p_site q, p_site r, bool& error);
  int  pqt_between_pqr_and_psq(p_site p, p_site q, p_site r, p_site s, 
                               p_site t, p_vnode mr, p_vnode ms);
  int  iqt_between_iqr_and_isq(p_site q, p_site r, p_site s, p_site t);
  int  itq_between_iqr_and_isq(p_site q, p_site r, p_site s, p_site t);

public:
  /*{\Mcreation 4.5}*/

  EP_Voronoi_Diagram() : Abstract_Voronoi_Diagram<p_vnode,p_site>() {}
  /*{\Mcreate introduces an empty Voronoi Diagram |\Mvar| of type |\Mname| in 
              the plane.}*/
  
  int insert(const rat_point& p)
  /*{\Mop     incrementally extends |\Mvar| by inserting |p| into the diagram.
              The return value states if the insertion of the site changed the 
              diagram (2), if it was inserted but only obtained an empty region 
              (1) or if it was an illegal site (0).}*/
  { return Abstract_Voronoi_Diagram<p_vnode,p_site>::
    insert(p_site(p)); }

  int animated_insert(const rat_point& p, window& W, int pause)
  /*{\Mop as above but animates the insertion in window |W| with a
    step delay of |pause| seconds.}*/
  { return Abstract_Voronoi_Diagram<p_vnode,p_site>::
    animated_insert(p_site(p),W,pause); }

  /*{\Moptions nextwarning=no }*/
  /*
  void  draw_diagram(window& W);
  */
  /*{\Mop draws the diagram in window |W|.}*/

  /*{\Moptions nextwarning=no }*/
  /*
  void  get_diagram_copy(GRAPH<p_vnode,p_site>& G);
  */
  /*{\Mop returns the basic combinatorial map which represents the Voronoi
    diagram.}*/

  /*{\Moptions nextwarning=no }*/
  /*
  GRAPH<p_vnode,p_site>& get_diagram() const 
  */
  /*{\Mop returns the basic combinatorial map which represents the Voronoi
    diagram.}*/
 
};

/*{\Mimplementation The data type is inherited from the abstract data
    type which can be adapted to concrete Voronoi Diagrams by
    providing the necessary implementation interface procedures. The
    update follows a randomized incremental scheme which uses a
    history graph to spot the update region in logarithmic time. The
    update itself works in time proportional to the size of the update
    region which is on average constant work. The space requirement is
    linear in the size of the input. }*/


/*{\Mexample If you want to create the Voronoi diagram of a list of |rat_points|
just program:
\begin{Mverb}
#include <LEP/avd/ep_adaptation.h>
  int main() 
  {
    EP_Voronoi_Diagram V;
    rat_point p;
    forall(p,Pointlist)
      V.insert(p);

    window W;
    W.display();
    V.draw_diagram(W);
  }
\end{Mverb}
Compile and link this program with [[demo/ep_adaptation.c]].}*/

//-----------------------------------------------------------------------
// procedures:
//-----------------------------------------------------------------------

inline int circle_pos(const rat_point& m, const rat_point& r , 
                      const rat_point& x)
{ return sign(x.sqr_dist(m) - r.sqr_dist(m)); }

inline bool  clockwise_oriented(p_handle<rat_point> p, 
                                p_handle<rat_point> q, 
                                p_handle<rat_point> r)
{ return right_turn(*p,*q,*r); }

//------------------------------------------------------------------------------
// collinear_ordered()
// checks whether the three points p, q and r have this order on a line
// through all three.
// preconditions:
// * orientation(p,q,r) == 0 <=> they lie collinear
// parameters:
// point p, q, r        the points to be examined
// return value:
// true         if the order is <p,q,r>
// false        else
//------------------------------------------------------------------------------

inline bool collinear_ordered(rat_point p, rat_point q, rat_point r)
{
    TRACE("collinear_ordered");TRACE(p);TRACE(q);TRACEN(r);

  if (compare(p,r)>0) 
    leda_swap(p,r);  // now p < r
  return (compare(p,q)<0 && compare(q,r)<0);
}

//------------------------------------------------------------------------------
// collinear_or_clockwise_ordered()
// checks if the traversal of the circle through the three sites p, q, r can be 
// done clockwise or if the sites are collinear and their order on the straight 
// line through the sites is p-q-r.
// if p=q or p=r or q=r then false is returned!
// parameters:
// p_site p, q, r
// return values:
// true         if the traversal on the circle is p-q-r clockwise or
//              if the traversal on the line is p-q-r (p!=q!=r)
// false        else
//------------------------------------------------------------------------------

inline bool collinear_or_clockwise_ordered(p_handle<rat_point> p, 
                                           p_handle<rat_point> q, 
                                           p_handle<rat_point> r)
{
    TRACEN("  collinear_or_clockwise_ordered: ");

  int x_prod = orientation(*p,*q,*r);

  if (x_prod == 0)                // p,q,r collinear
  {
      TRACEN("linear");
    // vector test if on straight line through p and q: r is behind q
    // including test on zero denominator
    return collinear_ordered(*p,*q,*r);
  }

    TRACE("clockwise: ");TRACEN((x_prod < 0));
  return (x_prod < 0);
}





#endif // EP_ADAPTATION_H

