#ifndef AVD_GENERAL_C
#define AVD_GENERAL_C

#include "avd.h"

//------------------------------------------------------------------------------
// Abstract_Voronoi_Diagram(), ~Abstract_Voronoi_Diagram()
// constructor and destructor of the abstract data type. 
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
Abstract_Voronoi_Diagram() : 
  V(this), H(this), _infin(), _gamma()
{ initialized = object_error_condition = false; }

template <class P_VNODE, class P_SITE>
Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
~Abstract_Voronoi_Diagram()
{ clear(); }

//------------------------------------------------------------------------------
// clear()
// cleans up all new - data
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
clear()
{ 
  V.clear();
  H.clear();
  R.clear();
  initialized = object_error_condition = false;
}


//------------------------------------------------------------------------------
// initialize_diagram_and_history()
// initializes the Voronoi Diagram and the History Graph with the site t1 and 
// the last site in the site_list if they're different.
// parameters:
// P_SITE t1            the first of the two sites
// return value:
// true                 if V and H is initialized
// false                if not
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
bool  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
initialize_diagram_and_history(P_SITE t1)
{ 
    TRACE(" AV.initialize_diagram_and_history - ");

  P_SITE t2 = R.tail();

  if (init_is_possible(t1,t2,object_error_condition))
  {
    // the node exists 
      TRACEN("Initialization ok as pqr exist! ");

    node vh1 = H.new_node(h_node_info<P_SITE>(infinity(),t1,t2,infinity()));
    node vh2 = H.new_node(h_node_info<P_SITE>(t2,infinity(),t1,t2));
    node vh3 = H.new_node(h_node_info<P_SITE>(t1,t2,infinity(),t1));
    // the three nodes in H besides source

    H.new_edge(H.h_source(),vh1);
    H.new_edge(H.h_source(),vh2);
    H.new_edge(H.h_source(),vh3);
    // the three initial edges

    node v1 = V.new_node(gamma_node());
    node v2 = V.new_node(gamma_node());
    // the two nodes in V

    edge e1 = V.new_bi_edge(v1,v2,t1,t2);
    edge e2 = V.new_bi_edge(v1,v2,infinity(),t1,e1,V.reversal(e1));
    edge e3 = V.new_bi_edge(v1,v2,t2,infinity(),e1,V.reversal(e2));

    // the three edges in V representated by the three nodes in H

    V.secure_links_and_master_correspondence(e1,vh1);
    V.secure_links_and_master_correspondence(e2,vh2);
    V.secure_links_and_master_correspondence(e3,vh3);
    // the links from the leafs in H to the related edges in V and
    // vice versa. We additionally keep the invariant that the 
    // description in vh corresponds to that of master_edge(vh)

    V.E_t_item.init(V,nil);
    V.Mod_item.init(V,nil);
    debug_diagram();
    check_invariants();
    return true;
  }
  return false;
}

//------------------------------------------------------------------------------
// insert()
// inserts a new P_SITE into the Voronoi Diagram. 
// precondition:
// * none
// parameter:
// P_SITE t     the site to be inserted
// return value:
// 2            if the inserted site has a face
// 1            if the inserted site has no face but is part of the diagram
// 0            if the inserted site has illegal position
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
int  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
insert(P_SITE t)
{
    TRACE("\nAV.insert of ");AVDD(t); TRACEN("");

  if (R.size()==0) { // no object yet
    R.push(t);
    return 2;
  }
  
  if (!initialized) {
    R.push(t);

    initialized = initialize_diagram_and_history(t);
    if (object_error_condition) {
        TRACEN("AV.initialize_diagram_and_history: object_error_condition");
      R.pop();
      // clean up the object which created illegal condition
      object_error_condition = false;
      return 0;
    }
    if (initialized)
      return 2;
    return 1;
  }

  if (initialized) {
    // the Voronoi Diagram and the History Graph are initialized
    edge e_intersected;
    R.push(t);
    if (H.dfs_for_intersected_leaf(t,e_intersected,object_error_condition)) {
        TRACE("\nAV.insert -> Leaf found: ");V.debug(e_intersected);TRACEN("");

      V.plane_search_for_intersected_edges(e_intersected,t,
      object_error_condition);
      if (!object_error_condition) {
        V.iterate_around_T_and_build_T_edges(t);
        V.insert_new_edges_and_clean_up();

          debug_diagram();
          check_invariants();
          TRACEN("AV.insert: END");
        return 2;
      }
    }

    if (object_error_condition) {
      R.pop();
        TRACEN("AV.insert: E_t empty and object_error_condition");
      // clean up the object which created illegal condition
      object_error_condition = false;
      return 0;
    }
  }
    TRACEN("AV.insert: E_t empty");
  return 1;
}


//------------------------------------------------------------------------------
// animated_insert()
// inserts a new P_SITE into the Voronoi Diagram and draws the occuring update of the 
// Voronoi Diagram in window W.
// precondition:
// * none
// parameter:
// P_SITE t     the site to be inserted
// window& W    the window in which is drawn
// int pause    the number of waiting cycles after insertion, if pause < 0, the
//              the window waits for a mouse click.
// return value:
// 2            if the inserted site has a face
// 1            if the inserted site has no face but is part of the diagram
// 0            if the inserted site has illegal position
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
int  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
animated_insert(P_SITE t, window& W, int pause)
{
    TRACE("\nAV.animated_insert of ");AVDD(t);TRACEN("");

  if (R.size()==0) { // no object yet
    R.push(t);
    return 2;
  }
  
  if (!initialized) {
    R.push(t);

    initialized = initialize_diagram_and_history(t);
    if (object_error_condition) {
        TRACEN("AV.initialize_diagram_and_history: object_error_condition");
      R.pop();
      // clean up the object which created illegal condition
      object_error_condition = false;
      return 0;
    }
    if (initialized) {
      draw_diagram(W);
      if (pause<0) W.read_mouse();
      else         leda_wait(pause);
      return 2;
    }
    return 1;
  }

  if (initialized) {
  // the Voronoi Diagram and the History Graph are initialized
    edge e_intersected;
    R.push(t);
    if (H.dfs_for_intersected_leaf(t,e_intersected,object_error_condition)) {
        TRACE("\nAV.insert -> Leaf found: ");V.debug(e_intersected);TRACEN("");

      V.plane_search_for_intersected_edges(e_intersected,t,
      object_error_condition);
      if (!object_error_condition) {
        V.iterate_around_T_and_build_T_edges(t);
        V.draw_update_region(t,W, red, green);

        if (pause<0) W.read_mouse();
        else         leda_wait(pause);
        V.insert_new_edges_and_clean_up();
        draw_diagram(W);
          debug_diagram();
          check_invariants();
        return 2;
      }
    }

    if (object_error_condition) {
      R.pop();
        TRACEN("AV.insert: E_t empty and object_error_condition");
      // clean up the object which created illegal condition
      object_error_condition = false;
      return 0;
    }
  }
    TRACEN("AV.insert: E_t empty");
  return 1;
}



//------------------------------------------------------------------------------
// draw_diagram()
// draws the edges of the Voronoi Diagram of V into the window W.
// parameters:
// window& W    the window drawn in
// return value:
// none
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
draw_diagram(window& W)
{
  W.start_buffering();
  W.clear();
  P_SITE p;
  forall(p,R) draw_site(p,W,blue2); // draw sites dark blue
  V.draw_diagram(W,blue);           // draw edges blue
  W.flush_buffer();
  W.stop_buffering();
}

//------------------------------------------------------------------------------
// debug_diagram()
// prints a representation of the Voronoi diagram
// parameters:
// return value:
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
void  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
debug_diagram()
{ 
#ifdef _DEBUGAVD
  TRACEN("AV.debug_diagram: ");
  H.debug_self();
  //V.print();
  V.debug_self();
  TRACEN("AV.debug_diagram END\n");
#endif
}

//------------------------------------------------------------------------------
// check_invariants()
// we check the consistence of the data structures after an insert:
// * planar map property of Voronoi diagram map
// * link bidirection of edges in V and leafs in H
// * acyclicity of history graph
// parameters:
// return value:
//------------------------------------------------------------------------------

#include <LEDA/edge_array.h>

template <class P_VNODE, class P_SITE>
void  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>::
check_invariants()
{
#ifdef _CHECKAVD
  if (!Is_Planar_Map(V))
    error_handler(1,"check_invariants: V no planar map.");

  edge e;
  forall_edges(e,V) {
    if (!V.hist_node(e))
      error_handler(1,"check_invariants: no h_node link in edge.");
  }

  // checking face loops:
  edge_array<bool> Vis(V,false);
  forall_edges(e,V) {
    if (!Vis[e]) {
      edge el = e;
      do {
        Vis[el]=true;
        el = V.cyclic_adj_succ(V.reversal(el));
      } while (el != e);
    }
  }

  if (!Is_Acyclic(H))
    error_handler(1,"check_invariants: H is cyclic.");

  h_node h;
  forall_nodes(h,H) {
    if (H.outdeg(h) > 5)
      error_handler(1,"check_invariants: H outdeg > 5.");
    if (H.outdeg(h) == 0) {
      if (!H.master_edge(h))
        error_handler(1,"check_invariants: master edge in H not set.");
      if (H.left_site(h) !=  V.left_site(H.master_edge(h)))
        error_handler(1,"check_invariants: master edge in H not correct.");
    }
  }
#endif
}

#endif // AVD_GENERAL_C

//----------------------------------- EOF --------------------------------------


