
#ifndef ABSTR_VORONOI_H
#define ABSTR_VORONOI_H

//------------------------------------------------------------------------------
// Abstract Voronoi Diagrammes
// theory:          S. Meiser "Zur Konstruktion abstrakter Voronoidiagramme"
// implementation:  M. Seel "Eine Implementierung abstrakter Voronoidiagramme"
// revision:        M. Seel 1997
//------------------------------------------------------------------------------

enum edge_end { SOURCE, TARGET };
enum topology { NO_INTERSECTION=-1,
                  TOTAL=0, 
                  SOURCE_SEGMENT=1, 
                  TARGET_SEGMENT=2, 
                  INTERN=3, 
                  TWO_SEGMENTS=4 };

//#define _CHECKAVD       // for asserts and co
//#define _DEBUGAVD       // for debugging the main actions
//#define _DEBUGALL      // for debugging the tiny little things
// for debugging make defines visual or use compiler flag -D

#ifdef _DEBUGAVD
#define _DEBUG
#define AVDD(t) debug(t)
#define AVDDD(T,t) (T).debug(t)
#define AVDDS(t) debug_short(t)
#else
#define AVDD(t)
#define AVDDD(T,t)
#define AVDDS(t)
#endif
// for debugging we give any object a debug() and debug_short() global

#ifdef _CHECKAVD
#define _ASSERT
#endif

#include <LEP/avd/debug.h>
#include <LEP/avd/lepconfig.h>
#include <LEDA/basic.h>
#include <LEDA/stack.h>
#include <LEDA/list.h>
#include <LEDA/graph.h>
#include <LEDA/ugraph.h>
#include <LEDA/window.h>

//------------------------------------------------------------------------------
// basic data structures:
//------------------------------------------------------------------------------

// some mnemonic names:

typedef node h_node;
typedef list_item mod_list_item;
typedef list_item e_t_list_item;
typedef list_item t_edges_list_item;

//------------------------------------------------------------------------------
// structure definition with slots:
// implementation of procedures see abstr_voro.c
//------------------------------------------------------------------------------


template <class P_SITE>
class h_node_info {
public:
   P_SITE      p_s[4];       // the edge is defined by 4 sites
                             // p_s[0] is defining site at source
                             // p_s[1] is defining site left of edge
                             // p_s[2] is defining site right of edge
                             // p_s[3] is defining site at target
   edge        e;            // the link to the master edge
   bool        visited;      // a flag for dfs

   h_node_info() { e = nil; visited = false; }
   h_node_info(P_SITE r, P_SITE p, P_SITE q, P_SITE s)
   { p_s[0] = r; p_s[1] = p; p_s[2] = q; p_s[3] = s; e = nil; visited = false; }

   h_node_info(const h_node_info<P_SITE>& org)
   { p_s[0] = org.p_s[0]; p_s[1] = org.p_s[1]; 
     p_s[2] = org.p_s[2]; p_s[3] = org.p_s[3];
     e = org.e; visited = false; }
   void switch_ends(edge e_rev)
   { leda_swap(p_s[1],p_s[2]); leda_swap(p_s[0],p_s[3]); e=e_rev; }

   friend ostream& operator<<(ostream& O, const h_node_info<P_SITE>& h)
   { O << "[" << (h.p_s[0]) << "," << (h.p_s[1]) << ",";
     O << (h.p_s[2]) << "," << (h.p_s[3]) << "]";
     return O; }
   friend istream& operator>>(istream& I, const h_node_info<P_SITE>& h)
   { return I; }

   LEDA_MEMORY(h_node_info);
};




struct e_t_list_info {
   edge        e;            // link to the master edge in V
   topology    topol;        // topology of intersected edge
   node        vn[2];        // the possible two nodes on the edge
                             // vn[0] refers to the node incident to
                             // source() if existing
                             // vn[1] refers to the node incident to 
                             // target() if existing
                             // order: source() < vn[0] < vn[1] < target() 
                             // on edge
   e_t_list_info() 
   { e = nil; vn[0] = vn[1] = nil; topol = (topology)-314159; }
   e_t_list_info(edge ei, topology top, node v1, node v2)
   { e = ei; topol = top; vn[0] = v1; vn[1] = v2; }

   friend ostream& operator<<(ostream& O, const e_t_list_info& et)
   { O << et.topol; return O; }
   friend istream& operator>>(istream& I, const e_t_list_info& et)
   { return I; }
   LEDA_MEMORY(e_t_list_info);
};


template <class P_SITE>
struct t_edges_list_info {
   node        v[2];         // start & end nodes of the new edge
   P_SITE      p_s[2];       // the two sites of the new edge
   edge        e[2];         // edges e[i] inzident to nodes v[i], are the 
                             // edges after which the new edge has to be 
                             // inserted at v[i]
   h_node      vh;           // the corresponding node of H

   t_edges_list_info()
   { v[0] = v[1] = nil; e[0] = e[1] = nil; vh = nil; }
   t_edges_list_info(node v1, node v2, P_SITE p1, P_SITE p2, 
        edge e1, edge e2, h_node vhi)
   { v[0] = v1; v[1] = v2; p_s[0] = p1; p_s[1] = p2; 
     e[0] = e1; e[1] = e2; vh = vhi; }

   friend ostream& operator<<(ostream& O, const t_edges_list_info<P_SITE>& te)
   { O << "[" << te.p_s[0] << "," << te.p_s[1] << "]"; return O; }
   friend istream& operator>>(istream& I, const t_edges_list_info<P_SITE>& te)
   { return I; }
 
   LEDA_MEMORY(t_edges_list_info);
};


struct aff_edges_list_info {
  edge     e;
  h_node   vh;
  aff_edges_list_info()
  { e = nil; vh = nil; }
  aff_edges_list_info(edge ei, h_node vi)
  { e = ei; vh = vi; }

  friend ostream& operator<<(ostream& O, const aff_edges_list_info& te)
  { return O; }
  friend istream& operator>>(istream& I, const aff_edges_list_info& te)
  { return I; }

  LEDA_MEMORY(aff_edges_list_info);
};


//------------------------------------------------------------------------------
// the basic data structures: Voronoi Diagram VD; History Graph H
// and the top organization |Abstract Voronoi Diagram| which has it all
// implementation of Voronoi_Diagram see voronio_diagram.c
// implementation of History_Graph see history_graph.c
// implementation of Abstract_Voronoi_Diagram see avd_general.c
//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE> class History_Graph;
template <class P_VNODE, class P_SITE> class Abstract_Voronoi_Diagram;

template <class P_VNODE, class P_SITE>
class Voronoi_Diagram : public GRAPH<P_VNODE,P_SITE>
{
  friend class Abstract_Voronoi_Diagram<P_VNODE,P_SITE>;
  friend class History_Graph<P_VNODE,P_SITE>;

  Abstract_Voronoi_Diagram<P_VNODE,P_SITE>* pAVD; // a reference to mummy
  list<e_t_list_item>       V_new;      // all new nodes 
  list<edge>                V_mod;      // all modified nodes
  list<edge>                V_del;      // all deleted nodes
  list<e_t_list_info>       E_t;        // all intersected edges  
  list< t_edges_list_info<P_SITE> >
                            T_edges;    // all new created edges
  list<aff_edges_list_info> Aff_edges;  // all affected edges
  node_map<mod_list_item>   Mod_item;   // to get from V to V_mod/V_del-items
  edge_map<h_node>          H_node;     // to get from E to H-nodes
  edge_map<e_t_list_item>   E_t_item;   // to get from E to E_t-items


  Voronoi_Diagram(Abstract_Voronoi_Diagram<P_VNODE,P_SITE>* p=nil) : 
    Mod_item(*this), H_node(*this), E_t_item(*this) 
  { pAVD = p; }
  ~Voronoi_Diagram() {}

  // first simple inlines:
  h_node&       hist_node(edge e) { return H_node[e]; }
  P_SITE        left_site(edge e) { return inf(e); }
  P_SITE        right_site(edge e) { return inf(reversal(e)); }
  P_SITE        source_site(edge e) { return inf(reversal(cyclic_adj_succ(e))); }
  P_SITE        target_site(edge e) { return inf(reversal(cyclic_adj_succ(
                                        reversal(e)))); }

  e_t_list_info edge_basic_intersection_operation(edge, P_SITE, bool&);

   // simplifiers:
   void       clear();
   edge       first_not_clipped_edge(edge);
   bool       clipped_at_source(edge);
   void       fill_in_top_info(e_t_list_item, P_SITE);
   void       put_shortened_edge_into_T_edges(e_t_list_item, P_SITE);
   edge       new_bi_edge(node, node, P_SITE, P_SITE, edge=nil, edge=nil);
   void       secure_links_and_master_correspondence(edge, h_node);

   void       plane_search_for_intersected_edges(edge, P_SITE, bool&);
   void       initialize_iteration_around_T(node&, edge&, P_SITE&, edge&);
   bool       next_node_on_bd_T(node&, edge&, P_SITE, h_node);
   void       iterate_around_T_and_build_T_edges(P_SITE);
   void       insert_new_edges_and_clean_up();

   // drawing:
   void       draw_edge(edge, window&, color);
   void       draw_update_region(P_SITE, window&, color, color);
   void       draw_diagram(window&, color);

   
   //------------------------------------------------------------------------------
   // put_into_T_edges()
   // inserts a edge describing container in T_edges.
   // parameters:
   // node v1,v2   the nodes which will be connected by the edge
   // P_SITE p1,p2 the two sites whose voronoi region will be separated
   // edge e1,e2   the orientation edges adjacent to the nodes v1, v2
   // h_node vh    the node in H in which the description of the edge is stored
   // return value:
   // none
   //------------------------------------------------------------------------------

   void  put_into_T_edges(node v1, node v2, P_SITE p1, P_SITE p2, 
                          edge e1, edge e2, h_node vh)
   { TRACE("  V.put_into_T_edges: ");AVDD(v1);AVDD(v2);AVDDS(p1);AVDDS(p2);
     AVDD(e1);AVDD(e2);TRACEN("");
     ASSERT( (!e1||v1==source(e1) && !e2||v2==source(e2)), put_into_T_edges);
     T_edges.append(t_edges_list_info<P_SITE>(v1,v2,p1,p2,e1,e2,vh)); 
   }

   //------------------------------------------------------------------------------
   // Mod list updates
   //------------------------------------------------------------------------------
   // put_into_V_mod()
   // organizes the list V_mod. A node v which is not yet stored as a modified node
   // is inserted into V_mod, as it's the first time an edge adjacent to v is 
   // examined as intersected.
   // parameters:
   // edge e       the edge which is examined, whose source is modified 
   //              and which lies inside T
   //------------------------------------------------------------------------------
   // move_into_V_del()
   // when all edges of the node v are deleted it has to be shifted from V_mod
   // to V_del.
   // parameters:
   // node v            the node which is to be put into one of the lists
   // mod_list_item mit the item where the node is stored in V_mod
   //------------------------------------------------------------------------------
   // get_mod_list_item()
   // returns the item where a node is referenced as modified
   // parameters:
   // node v       the node which is examined 
   // return value:
   // the item if v in V_mod or V_del
   // nil if v not in any list
   //------------------------------------------------------------------------------
   // clear mod_lists()
   // cleans up both intermediate lists V_mod and V_del
   //------------------------------------------------------------------------------

   void  put_into_V_mod(edge e)
   {
       TRACE("    V.put_into_V_mod: ");AVDD(source(e));AVDD(e);TRACEN("");
     Mod_item[source(e)] = V_mod.append(e);
     // append a container to list V_mod marking that the node source(e)
     // is already in V_mod
   }

   void  move_into_V_del(node v)
   { TRACE("    V.move_into_V_del: ");AVDD(v);
     mod_list_item mit = Mod_item[v];  // get the list item via hash table
     edge e = V_mod.del_item(mit);     // delete item in V_mod
     AVDD(e);TRACEN("");
     Mod_item[v] = V_del.append(e);    // insert into V_del; 
     // this shifts the container with the edge information from V_mod to V_del
   }

   mod_list_item get_mod_list_item(node v)
   { return Mod_item[v]; }

   void clear_mod_lists()
   { Mod_item.init();
     V_mod.clear();
     V_del.clear();
   }

   //------------------------------------------------------------------------------
   // put_into_V_new()
   // inserts a link to the new nodes stored in an e_t_list_info container into 
   // the new_list in V.
   // parameters:
   // e_t_list_item e_it   an item in E_t
   //------------------------------------------------------------------------------

   void  put_into_V_new(e_t_list_item e_it)
   { TRACE("    V.put_into_V_new: ");AVDD(E_t.inf(e_it));TRACEN("");
     V_new.append(e_it); 
   }


   //------------------------------------------------------------------------------
   // E_t organizers:
   //------------------------------------------------------------------------------
   // put_into_E_t(): 
   // We store the info container in list E_t and make the list items 
   // referencable from e and reversal(e) via the hash array E_t_item.
   // parameters:
   // edge e                          the edge the container refers to
   // const e_t_list_info_item& p_et_info  the container to be stored
   //------------------------------------------------------------------------------

   void test_E_t_map()
   { TRACE("      test_E_t_map: ");
     edge e; forall_edges(e,(*this)) {AVDDS(e); TRACE(E_t_item[e]); }
     TRACEN("");
   }

   void  put_into_E_t(edge e, const e_t_list_info& et_info)
   { e_t_list_item et = E_t.append(et_info); 
     E_t_item[e] = et; 
     E_t_item[reversal(e)] = et;
     TRACE("    V.put_into_E_t: ");AVDD(e);AVDD(et_info);TRACEN("");
   }

   e_t_list_item get_e_t_list_item(edge e)
   { return E_t_item[e]; }

   e_t_list_info& get_topology_box(edge e) 
   { ASSERT(get_e_t_list_item(e), get_topology_box);
     e_t_list_item eit = E_t_item[e]; 
     return E_t[eit]; }

   topology get_topology(edge e, node& v1, node& v2) 
   {   TRACE("        get_topology: ");AVDD(e);
       ASSERT(get_e_t_list_item(e), get_topology);
     e_t_list_info& e_inf = E_t[E_t_item[e]]; 
       AVDD(e_inf); TRACEN("");
     topology res = e_inf.topol; 
     v1 = e_inf.vn[0], v2 = e_inf.vn[1];
     if (e_inf.e != e) { // not master edge!
         TRACEN("        swapping topology");
       if (res == SOURCE_SEGMENT) res=TARGET_SEGMENT;
       else if (res == TARGET_SEGMENT) res=SOURCE_SEGMENT;
       leda_swap(v1,v2);
     }
       TRACE("        ");AVDD(v1);AVDD(v2);TRACEN("");
     return res;
   }

   void clear_E_t()
   { E_t_item.init();
     E_t.clear();
   }


   
   //------------------------------------------------------------------------------
   // debugging containers:
   //------------------------------------------------------------------------------

   void debug(const P_SITE& p) 
   {
   #ifdef _DEBUGAVD
     ::debug(p); 
   #endif
   }

   void debug(node v)
   {
   #ifdef _DEBUGAVD
     if (v==nil) { cerr << "nil-node "; return; }
     cerr << "node[";
     if ((*this)[v] == pAVD->gamma_node())
       cerr << "gamma node";
     else
       ::debug((*this)[v]);
     cerr << "] ";
   #endif
   }

   void debug(edge e)
   {
   #ifdef _DEBUGAVD
     if (e==nil) { cerr << "nil-edge "; return; }
     cerr << "edge[";
     debug_short(left_site(e));
     cerr << ",";
     debug_short(right_site(e));
     cerr << "|" << int(hist_node(e));
     cerr << "] ";
   #endif
   }

   void debug_short(edge e)
   {
   #ifdef _DEBUGAVD
     if (e==nil) { cerr << "nil-edge "; return; }
     cerr << "[";
     debug_short(left_site(e));
     cerr << ",";
     debug_short(right_site(e));
     cerr << "]";
   #endif
   }

   void debug_short(const P_SITE& p)
   {
   #ifdef _DEBUGAVD
     ::debug_short(p);
   #endif
   }

   void debug(e_t_list_info et_inf)
   {
   #ifdef _DEBUGAVD
     //cerr << "  ";
     debug(et_inf.e);
     switch (et_inf.topol) {
     case TOTAL:          cerr << " TOTAL "; break;
     case SOURCE_SEGMENT: cerr << " SOURCE_SEGMENT "; break;
     case TARGET_SEGMENT: cerr << " TARGET_SEGMENT "; break;
     case INTERN:         cerr << " INTERN "; break;
     case TWO_SEGMENTS:   cerr << " TWO_SEGMENTS "; break;
     }
   #endif
   }

   //------------------------------------------------------------------------------
   // debug procedure for V:
   //------------------------------------------------------------------------------

   void debug_nodes()
   {
   #ifdef _DEBUGAVD
     node n;
     cerr << " nodes in V:\n";
     forall_nodes(n,*this)
     { cerr << "   "; debug(n); cerr << "\n"; }
   #endif
   }

   void debug_node_lists()
   {
   #ifdef _DEBUGAVD
     cerr << " lists of V: ";
     cerr << "\n  V_mod: ";
     edge e_mod;
     forall(e_mod,V_mod)
     { debug(source(e_mod)); }
     cerr << "\n  V_del: ";
     forall(e_mod,V_del)
     { debug(source(e_mod)); }
     cerr << "\n  V_new:\n";
     e_t_list_item e_it;
     forall(e_it,V_new)
     { debug(E_t[e_it]); }
     cerr << "\n";
   #endif
   }

   void debug_self()
   {
   #ifdef _DEBUGAVD
     cerr << "  V:debug_self()\n";
     node v;
     edge e;
     forall_nodes(v,*this) {
       cerr << "   "; debug(v); cerr<< "\n     ";
       forall_adj_edges(e,v) {
         debug(e);
       }
       cerr << endl;
     }
     // cerr << " printout:\n" << *this << endl;
   #endif
   }
       

   //------------------------------------------------------------------------------
   // end of debug 
   //------------------------------------------------------------------------------



};

//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE>
class History_Graph: public GRAPH< h_node_info<P_SITE> , void* >
{
   friend class Abstract_Voronoi_Diagram<P_VNODE,P_SITE>;
   friend class Voronoi_Diagram<P_VNODE,P_SITE>;

   Abstract_Voronoi_Diagram<P_VNODE,P_SITE>* pAVD; // a reference to mummy
   h_node  vh_source;                   // the source of the History Graph:

   History_Graph(Abstract_Voronoi_Diagram<P_VNODE,P_SITE>* p=nil) 
   { vh_source = new_node(h_node_info<P_SITE>()); pAVD=p; }

   void clear() // reinits the History_Graph 
   { GRAPH< h_node_info<P_SITE> , void* >::clear();
     vh_source = new_node(h_node_info<P_SITE>()); }
   ~History_Graph() { clear(); }
  
   // inlines:
   h_node       h_source() { return vh_source; }
   P_SITE&      left_site(h_node h)   { return (*this)[h].p_s[1]; }
   P_SITE&      right_site(h_node h)  { return (*this)[h].p_s[2]; }
   P_SITE&      source_site(h_node h) { return (*this)[h].p_s[0]; }
   P_SITE&      target_site(h_node h) { return (*this)[h].p_s[3]; }
   edge&        master_edge(h_node h) { return (*this)[h].e; }

   bool         visited(h_node h)     { return (*this)[h].visited; }      
   void         mark_visited(h_node h)   { (*this)[h].visited = true; }
   void         mark_unvisited(h_node h) { (*this)[h].visited = false; }

   bool         description_is_intersected(h_node, P_SITE, bool&);
   void         re_init(h_node vh);
   bool         go_for_intersected_leaf(h_node, P_SITE, edge&, bool&);

   P_SITE       insert_descriptions_at_v_new(h_node, P_SITE, P_SITE, P_SITE);
   P_SITE       insert_descriptions_at_v_mod(edge, edge, h_node, 
                P_SITE, P_SITE, P_SITE);

   h_node       new_or_existing_description(edge); 
   h_node       new_witness_entry(edge, P_SITE);
   void         witness_entry(edge, h_node);
   void         witness_entry_if_not_existing(edge, h_node);
   bool         dfs_for_intersected_leaf(P_SITE, edge&, bool&);

   //------------------------------------------------------------------------------
   // new_description()
   // creates a new node in the History Graph with the copied information of the 
   // given node. In Addition it creates an directed edge from the parameter node 
   // to the new node.
   // parameters:
   // h_node vh    the node which is copied
   // return value:
   // h_node       the node which is created
   //------------------------------------------------------------------------------

   h_node new_description(h_node vh)
   { TRACE("  H.new_description from: ");AVDD(vh);TRACEN("");
     h_node vh_new = new_node(inf(vh)); // create new node with copied h_node_info
     new_edge(vh,vh_new);               // create link
     return vh_new;
   }

   
   //------------------------------------------------------------------------------
   // debugging h_node containers:
   //------------------------------------------------------------------------------

   void debug(const P_SITE& p) 
   { ::debug(p); }

   void debug(const h_node_info<P_SITE>& p)
   { cerr << "[";
     debug_short(p.p_s[0]); cerr << ",";
     debug_short(p.p_s[1]); cerr << ",";
     debug_short(p.p_s[2]); cerr << ",";
     debug_short(p.p_s[3]); cerr << "]";
   }

   void debug(h_node vh)
   {
   #ifdef _DEBUGAVD
     cerr << "hnode"; 
     debug((*this)[vh]);
     cerr << " ";
   #endif
   }

   //------------------------------------------------------------------------------
   // debugging the History Graph:
   //------------------------------------------------------------------------------

   void debug_self()
   {
   #ifdef _DEBUGAVD
     TRACEN(" H.debug_self: ");
     cerr << "   hnode[source]\n";
     h_node w;
     forall_adj_nodes(w,vh_source) {
       debug_rec(3,w);
     }
     forall_nodes(w,*this) {
       mark_unvisited(w);
     }
   #endif
   }

   void debug_rec(int i, h_node v)
   {
   #ifdef _DEBUGAVD
     cerr << "   ";
     for (int j=0;j<i;j++)
     { cerr << " "; }
     debug(v);
     if (visited(v)) {
       cerr << "-->\n"; return;
     }
     mark_visited(v);
     cerr << "\n";
     h_node w;
     forall_adj_nodes(w,v)
       debug_rec(i+3,w);
   #endif
   }



};

//------------------------------------------------------------------------------

template <class P_VNODE, class P_SITE> class Abstract_Voronoi_Diagram;
template <class P_VNODE, class P_SITE>
void xfig_diagram(Abstract_Voronoi_Diagram<P_VNODE,P_SITE>&, window&);

/*{\Manpage {Abstract_Voronoi_Diagram} {P_VNODE, P_SITE} 
            {Abstract Voronoi Diagrams in the plane} AVD}*/

/*{\Mtext
\newcommand{\reals}{\,{\rm\hbox{I\kern-.15em R}}}
\newcommand{\nats}{\,{\rm\hbox{I\kern-.15em N}}}
\newcommand{\bd}{{\it bd}}\newcommand{\VR}{{\it V\!R}}
\newcommand{\oVR}[1]{\overline{{\it V\!R}(#1)}}
}*/

/*{\Mdefinition Let $n\in\nats$ and $S =\{1,\ldots,n-1\}$ be a set of
so called \emph{sites}.  A \emph{dominance system} ${\cal
DS}(S):=\{D(p,q) : p,q \in S, p \neq q\}$ over $S$ is defined as a
family of path connected open subsets of $\reals^2$ with the
properties that\\ (1) $\bd~D(p,q) = \bd~D(q,p)$ and\\ (2) $D(p,q)$,
$D(q,p)$ and $\bd~D(p,q)$ are a disjoint partition of $\reals^2$,
$\forall p \neq q \in S$ and\\ (3) if $\emptyset \neq D(p,q) \neq
\reals^2 $ then $\bd~D(p,q)$ is homeomorphic to $(0,1)$

$D(p,q)$ is called {\em dominance region} of $p$ over $q$ and $J(p,q)
:= \bd~D(p,q)$ is called {\em Bisektor} of $p$ and $q$. $\VR(p,S) :=
\bigcap_{q \in S, q \neq p} D(p,q)$ is the {\em Voronoi region} of $p$
with respect to $S$. And $V(S) := \bigcup_{p,q \in S, p \neq q}
\oVR{p,S} \cap \oVR{q,S}$ is called the {\em Voronoi diagram} of $S$.

A dominance system with the following properties is called {\em valid}:\\
(1) Any two bisectors intersect only in a finite number of connected components.\\
(2) For any non-empty subset $S^\prime \subset S$ we demand that
$\forall\ p \in S^\prime : V\!R(p,S^\prime) \neq \emptyset \quad
\Longrightarrow \quad V\!R(p,S^\prime)$ is path-connected and
$\reals^2 = V(S^\prime) \cup \bigcup_{p \in S^\prime}V\!R(p,S^\prime)$

The data type |\Mname| can model any Voronoi diagram implied by a
valid dominance system. We store a combinatorial embedding of of
$V(S)$ in form of a bidirected graph $G(S)$ realizing the Voronoi
vertices and edges such that each edge is contained in exactly one
face associated with a site. All edges store an object of type
|P_SITE| and all nodes store an object of type |P_VNODE| where both
types are handles to objects representing sites and node
coordinates. For the requirements of |P_SITE| and |P_VNODE| refer to
the description in \ref{A handle wrapper for objects} below.}*/

template <class P_VNODE, class P_SITE>
class Abstract_Voronoi_Diagram
{
  friend class Voronoi_Diagram<P_VNODE,P_SITE>;
  friend class History_Graph<P_VNODE,P_SITE>;

public:

  /*{\Mtypes 3}*/

  typedef P_VNODE p_vnode;
  /*{\Mtypemember the node coordinate handle type}*/

  typedef P_SITE p_site;
  /*{\Mtypemember the site coordinate handle type}*/

protected:
  P_SITE  _infin;
  P_VNODE _gamma;
    // the abstract datatypes are the following ones:
  Voronoi_Diagram<P_VNODE,P_SITE>    V;
    // the graph structure of the Voronoi Diagram
  History_Graph<P_VNODE,P_SITE>      H;        
    // the history structure
  list<P_SITE>       R;        
    // all sites are stored here

  bool               initialized;
  bool               object_error_condition;

  bool               initialize_diagram_and_history(P_SITE);

  friend void        xfig_diagram EGCSNTA (Abstract_Voronoi_Diagram&, window&);
  void               check_invariants();


public:
/*{\Mcreation AVD 6}*/

  Abstract_Voronoi_Diagram();
  /*{\Mcreate creates an empty object |\Mvar| of type |\Mname|.}*/

  ~Abstract_Voronoi_Diagram();

/*{\Moperations 3 3}*/          

  void  clear();
  /*{\Mop reinitializes the data structure to the empty diagram.}*/

  P_SITE  infinity() const   { return _infin; }
  P_VNODE gamma_node() const { return _gamma; }

  int   insert(p_site t);
  /*{\Mop inserts a new site object |t| into the Voronoi diagram. The return
    value $\in \{0,1,2\}$ states if the the site was illegal $(0)$, got only
    an empty Voronoi region $(1)$ or was inserted into the diagram $(2)$.}*/

  int   animated_insert(p_site t, window& W, int secs);
  /*{\Mop inserts a new site object |t| into the Voronoi diagram as above but
    animates the update operation in the window |W| with a delay of |secs|
    seconds. }*/

  void  draw_diagram(window& W);
  /*{\Mop draws the diagram in window |W|.}*/

  void  get_diagram_copy(GRAPH<P_VNODE,P_SITE>& G)
  /*{\Mop returns the basic combinatorial map which represent the Voronoi
    diagram.}*/
  { G = GRAPH<P_VNODE,P_SITE>(V); }

  GRAPH<P_VNODE,P_SITE>& get_diagram() const 
  /*{\Mop returns the basic combinatorial map which represent the Voronoi
    diagram.}*/
  { return (GRAPH<P_VNODE,P_SITE>&) V; }

  void  debug_diagram();

protected:
  
  //------------------------------------------------------------------------------
  // the two subprocedures of the basic operation which are called within 
  // History_Graph and Voronoi_Diagram:
  // >> edge_is_intersected()
  // >> edge_intersection_topology()
  // where the first check serves as the search condition within the history
  // graph, the latter implements our basic operation. 
  // 
  // >> init_is_possible()
  // was added to cope with initialization state of the diagram
  //
  // then there are the three other operations 
  // >> get_node_coords()
  // >> draw_bisector()
  // >> draw_site()
  // to get concerete nodes coordinates and to visualize the diagram
  // 
  // All theses operations have to be specialized for a concrete Voronoi diagram
  // calculation.
  //------------------------------------------------------------------------------

  /*{\Mtext \headerline{The Geometric Interface Operations} The
  following geometric operations have to be implemented by the user of
  the AVD framework to adapt |\Mname| to a concrete Voronoi diagram.}*/

  virtual bool init_is_possible(p_site p, p_site q, bool& b)=0;
  /*{\Mop returns true if $D(p,q)\neq\emptyset$ and
  $D(q,p)\neq\emptyset$ such that the |\Mvar| can be initialized. |b| is
  set to |false| if |p| is \emph{illegal} with respect to |q| or vice
  versa.}*/

  virtual topology edge_intersection_topology(p_site p, p_site q, 
                                              p_site r, p_site s, 
                                              p_site t, 
                                              p_vnode vpqr, p_vnode vpsq, 
                                              bool& b)=0;
  /*{\Mop returns the intersection topology of the Voronoi edge $pqrs$
  with respect to the new site |t|. |b| is set to |false| if |t| is
  \emph{illegal} with respect to any of the site $p,q,r,s$.}*/

  virtual bool edge_is_intersected(p_site p, p_site q, p_site r , p_site s, 
                                   p_site t, 
                                   p_vnode vpqr, p_vnode vpsq, 
                                   bool& b)=0;
  /*{\Mop returns true if the Voronoi edge $pqrs$ is intersected with respect
  to the new site |t|. |b| is set to |false| if |t| is \emph{illegal}
  with respect to any of the site $p,q,r,s$.}*/


  virtual p_vnode get_node_coords(p_site p , p_site q, p_site r)=0;
  /*{\Mop returns the coordinates of the Voronoi node $pqr$ in form
  of a |p_vnode| container.}*/

  virtual void draw_bisector(p_site p, p_site q, 
                             p_vnode vpqr, p_vnode vpsq, 
                             window& W, color c)=0;
  /*{\Mop draws the Voronoi edge separating the Voronoi regions of $p$
  and $q$ adjacent to the Voronoi nodes with coordinate container $vpqr$
  and $vpsq$ in window |W| in color |c|.}*/


  virtual void draw_site(p_site p, window& W, color c)=0;
  /*{\Mop draws the site $p$ in window |W| in color |c|.}*/


};


/*{\Mimplementation The calculation of the abstract Voronoi diagram
follows a randomized incremental construction scheme which uses a
LEDAs bidirected graph data type to represent the Voronoi diagram and
a history graph as the update search structure. A random insertion
sequence of $n$ objects takes time $O(n \log n)$ and space $O(n)$.}*/

/*{\Mexample To use the framework a user has to provide five basic
geometric primitives which are used by the combinatorial layer to
construct the Voronoi diagram structure. Two examples have been
implemented within the framework. The standard Voronoi diagram of
points under Euclidean Metric and an extended version of Euclidean
Voronoi diagrams of line segments and points. Both come with the
package and can be found in the demo directory.}*/


//------------------------------------------------------------------------------
// inlining of code:
//------------------------------------------------------------------------------

#include <avd_general.C>
#include <history_graph.C>
#include <voronoi_diagram.C>

#endif // avd.h

