
#include <LEP/avd/es_adaptation.h>
#include <LEP/avd/param_handler.h>
#include <LEDA/file.h>
#include <LEDA/map.h>

inline sqr(int i) { return i*i; }

static window* pW;
static ES_Voronoi_Diagram* pAVD;
static file_ostream* pTo;
static string examples_dir = "input";
static map<int,string> ex_file;

const string str_site("site");
const string str_poly("poly");
static int grid = 1;
static int scale = 5;
static int xmin;
static int xmax;
static int ymin;
static bool DRAW=true;
static bool ANIMATE=true;
static int SECS = -2;
enum { EXAMPLE_BUTTON = 200, CLEAN_BUTTON, ZOOM_BUTTON, HELP_BUTTON, 
       END_BUTTON, ANIM_BUTTON };

static rat_point rpnt(point p)
{ return rat_point(integer(p.xcoord()),integer(p.ycoord())); }

static void redraw_diagram() 
{ 
  //cout << "redraw_diagram(" << xmin << "," << xmax << "," << ymin;
  //cout << "," << grid << ")\n";
  pW->init(xmin,xmax,ymin,grid);
  pAVD->draw_diagram(*pW); 
}

static void rescale_window(int x0, int y0, int s)
{
  scale = s;
  xmin = x0-s*10;
  xmax = x0+s*10;
  ymin = y0-s*10;
  grid = leda_max(1,(xmax-xmin)/100);
  //cout << "rescale_window(" << x0 << "," << y0 << "," << scale << ",";
  //cout << grid << ")\n";
}



inline void 
create_bounding_box(point& pmin, point& pmax, const point& test)
{
  pmin = point(leda_min(pmin.xcoord(), test.xcoord()),
               leda_min(pmin.ycoord(), test.ycoord()));
  pmax = point(leda_max(pmax.xcoord(), test.xcoord()),
               leda_max(pmax.ycoord(), test.ycoord()));
}

void fill_diagram(string ifile, bool ANIMATE, int SECS)
{ 
  file_istream from(ifile);
  file_ostream& to = *pTo;
  if (!from)
    error_handler(1,"there is no input file "+ifile+"!");
  
  point p,q,pmin,pmax;
  string istr1,istr2;
  bool first = true;
  while ((from >> istr1) && (from >> istr2)) {
      /* this loop reads the lines with label istr1 and
         pointstring istr2 */
   if (first)
      first = false;
    else
      to << "\n";

   cout << istr1 << " " << istr2 << "\n";
   string_istream is(istr2);
   if (istr1 == str_site) {
     /* this handles site input lines where two consecutive
        points define a site -- two equal points a point site
        and two different points a segment site */
      to << istr1 << " ";
      while ((is >> p) && (is >> q)) {
        // we assume that the input is on the integer grid
        create_bounding_box(pmin,pmax,p);
        create_bounding_box(pmin,pmax,q);
        bool legal;
        if (ANIMATE) {
          if (p == q)
            legal = pAVD->animated_insert(rpnt(p),*pW,SECS);
          else
            legal = pAVD->animated_insert(rat_segment(rpnt(p),rpnt(q)),
                                            *pW,SECS);
        } else {
          if (p == q)
            legal = pAVD->insert(rpnt(p));
          else
          legal = pAVD->insert(rat_segment(rpnt(p),rpnt(q)));
        }
        to << p << q;
        to.flush();
        if (!legal) 
          cout << "illegal object: " << p << q << "\n";
      }
    }
    else if (istr1 == str_poly)
    {
      /* this handles polygon input lines where all points
         define the vertices of a closed polygon */

      to << istr1 << " ";
      list<rat_point> p_list;
      while (is >> p) {
        create_bounding_box(pmin,pmax,p);
        p_list.append(rpnt(p));
        to << p;
        to.flush();
      }

      rat_polygon poly(p_list);
      if (ANIMATE) {
        if (!pAVD->animated_insert(poly,*pW,SECS)) 
          cout << "illegal polygon: " << poly << "\n";
      } else {
        if (!pAVD->insert(poly)) 
          cout << "illegal polygon: " << poly << "\n";
      }
    }
    else
      error_handler(1,"wrong label - your input file is corrupted!");
    if (!is.eof())
      error_handler(1,"point list - your input file is corrupted!");
  }
  if (!from.eof())
    error_handler(1,"line structure - your input file is corrupted!");

  double dx = pmax.xcoord()-pmin.xcoord();
  double dy = pmax.ycoord()-pmin.ycoord();
  int x0 = int(pmin.xcoord()+dx/2), y0 = int(pmin.ycoord()+dy/2);
  int dmax = int(leda_max(dx,dy))/2;
  rescale_window(x0,y0,dmax/10+2);    
}

void do_examples(int i)
{
  fill_diagram(examples_dir+"/"+ex_file[i], ANIMATE, SECS);
  redraw_diagram();  
}



inline window& operator>>(window& W, rat_polygon& P)
{
  polygon Pd;
  W >> Pd;
  point p;
  list<rat_point> Lpr;
  forall(p, Pd.vertices())
    Lpr.append(rat_point(int(p.xcoord()),int(p.ycoord())));
  P = rat_polygon(Lpr);
  return W;
}

main(int argc, char* argv[])
{
  int count = 0;

  string explain = " use negative anim_seconds to disable animation \n";
  explain +=       " use negative scale to disable online modus\n.";
  param_handler params(argc,argv,
  "[inputfile] [anim_seconds] [scale] [xmin xmax ymin wx wy]",0,8);
  // we go for: anim_seconds 
  // < -1 : don't animate
  // = -1 : single_step
  // >= 0 : pause in seconds

  file_ostream to("log");
  pTo = &to;
  string ifile;

  int wx = 800, wy = 850;

  ES_Voronoi_Diagram AVD;
  pAVD = &AVD;

  params.set(ifile,1);
  params.set(SECS,2);
  params.set(scale,3);
  params.set(xmin,4);
  params.set(xmax,5);
  params.set(ymin,6);
  params.set(wx,7);
  params.set(wy,8);

  if (scale<0) DRAW=false; // we don't draw for negative scale
  if (SECS < -1) ANIMATE=false;

  
    window W(wx, wy, "Euclidean Voronoi Diagrams of Line Segments (AVD Adaptation)");

    list<string> examples = get_files(examples_dir,"*");
    menu exmenu;
    string s;
    int i=0;
    forall(s,examples) {
      ex_file[i] = s;
      exmenu.button(s,i++,do_examples);
    }

    W.button("Examples", EXAMPLE_BUTTON, exmenu);
    W.button("Animation",ANIM_BUTTON);
    W.button("Clean",CLEAN_BUTTON);
    W.button("Zoom View",ZOOM_BUTTON);
    W.button("Help",HELP_BUTTON);
    W.button("End",END_BUTTON);

    panel  P("Zoom window"),H("Help on avd"),A("Animation Setup");
    pW = &W;
    W.set_redraw(redraw_diagram);
    W.set_show_coordinates(true);
    W.set_node_width(4);
    rescale_window(0,0,scale);
    W.init(xmin,xmax,ymin,grid);
    W.flush_buffer();
    W.set_grid_mode(grid);

    P.text_item("Viewer centered around the mouse click!");
    P.int_item("Scale",scale,1,30);
    enum { PINB =100, POUTB, PDEFB, PRECB, PCANCB };
    P.button("Zoom In",PINB);
    P.button("Zoom Out",POUTB);
    P.button("Default",PDEFB);
    P.button("Recenter",PRECB);
    P.button("Cancel",PCANCB);

    H.text_item("You can input sites by two consecutive mouse clicks into the window.");
    H.text_item("Two identical points trigger one point site two different points");
    H.text_item("trigger a segment site. A simple polygon can be inserted more");
    H.text_item("efficiently via the right mouse button. The window can be zoomed");
    H.text_item("via the <Zoom View> Button.");
    H.button("Done");

    A.text_item("You can animate the construction either in single step ");
    A.text_item("(delay=-1 means triggering updates by mouse clicks) or ");
    A.text_item("in an animation with update delay.");
    A.bool_item("Animation (on/off)",ANIMATE);
    A.int_item("Animation delay",SECS,-1,10);
    A.button("Done");
    W.display();



  double x0=0,y0=0,x1,y1;
  float starttime = used_time();

  if (params.num() >= 1) 
    fill_diagram(ifile, ANIMATE, SECS);

  float endtime = used_time(starttime);
  cout << "used time for construction: " << endtime << endl;

  if (DRAW==true) 
      {
        redraw_diagram();

        // for the setting of the window redrawing function there's redraw_diagram
        // calling the redraw function AV.draw_diagram 

        rat_polygon Poly;
        rat_point v,vo;
        rat_segment st;
        point p,q;
        bool loop = true, first=true;
        int result = -1;
        int but = W.read_mouse(p);
        while (loop) {
          switch (but) {

          case MOUSE_BUTTON(1):
            if (first) {
              to << str_site << " ";
              first = false;
            }
            W.read_mouse_seg(p,q);
            to << p << q;
            to.flush();
            st = rat_segment(rpnt(p),rpnt(q));
            if (ANIMATE) {
              if (!AVD.animated_insert(st,W,SECS)) {
                W.acknowledge("Site is illegal!\n");
                cout << "illegal site: " << st << "\n";
              }
              
            } else {
              if (!AVD.insert(st)) {
                W.acknowledge("Site is illegal!\n");
                cout << "illegal site: " << st << "\n";
              }
              redraw_diagram();
            }
            break;
         case MOUSE_BUTTON(3):
            if (!first)
              to << "\n";
            to << str_poly << " ";
            W >> Poly;
            forall(v,Poly.vertices())
              to << v.to_point();
            to << "\n";
            first = true;
            to.flush();
            if (ANIMATE) {
              if (!AVD.animated_insert(Poly,W,SECS))
              {
                W.acknowledge("Polygon is illegal!\n");
                cout << "illegal polygon: " << Poly << "\n";
              }  
            } else {
              if (!AVD.insert(Poly))
              {
                W.acknowledge("Polygon is illegal!\n");
                cout << "illegal polygon: " << Poly << "\n";
              }
              redraw_diagram();
            }
            break;
          case MOUSE_BUTTON(2):
          case ZOOM_BUTTON:
            result = P.open();
            if (result == PINB)
              scale = leda_max(1,scale-1);
            else if (result == POUTB)
              scale += 1;
            else if (result == PDEFB)
              scale = 5;
            if (result != PCANCB) {
              W.read_mouse(x0,y0);
              rescale_window(int(x0),int(y0),scale);
            }
            redraw_diagram();
            break;
          case HELP_BUTTON:
            H.open();
            break;
          case ANIM_BUTTON:
            SECS=-1;
            A.open();
            break;
          case END_BUTTON:
            loop=false;
            SECS=0;
            continue;
            break;
          case CLEAN_BUTTON:
            AVD.clear();
            W.clear();
            break;
          }
          but=W.read_mouse(p);
        }
        to << "\n";
        to.flush();
        W.clear();
      }


  if (ANIMATE) leda_wait(10*SECS);
  AVD.clear();
  to << endl;
  to.close();
}


