
#include <LEP/avd/expcomp.h>
#include <LEP/avd/debug.h>
#include <LEP/avd/es_adaptation.h>

// we first specialize the following symbols
#define site    es_site
#define vnode   es_vnode
#define p_site  p_handle<es_site>
#define p_vnode p_handle<es_vnode>


//#define _DEBUG
//#define _DEBUG_LOW
//#define _ASSERTS
//#define _DEBUG_DRAW

#ifdef _DEBUG_LOW
#define LTRACE(t) cerr << "     " << (t)
#else
#define LTRACE(t) 
#endif

#ifdef _DEBUG_LOW
#define LTRACEN(t) cerr << "     " << (t) << "\n"
#else
#define LTRACEN(t) 
#endif

int site::count = 0; // -1


inline void seg2line_paras(const rat_segment& s, 
            real& a, real& b, real& c)
{
    ASSERT((s.W1()==1),seg2line_paras);
    ASSERT((s.W2()==1),seg2line_paras);

  a = s.Y1() - s.Y2();
  b = s.X2() - s.X1();
  c = s.X1()*s.Y2() - s.X2()*s.Y1();
}

inline void seg2line_paras(const rat_segment& s, 
            integer& a, integer& b, integer& c)
{
    ASSERT((s.W1()==1),seg2line_paras);
    ASSERT((s.W2()==1),seg2line_paras);

  a = s.Y1() - s.Y2();
  b = s.X2() - s.X1();
  c = s.X1()*s.Y2() - s.X2()*s.Y1();
}


inline int sorted_switch_label_b2(bool b1, bool b2)
{
  int res = 0;
  if (b2) res += 1;
  if (b1) res += 2;
  return res;
}

inline int sorted_switch_label_b3(bool b1, bool b2, bool b3)
{
  int res = 0;
  if (b3) res += 1;
  if (b2) res += 2;
  if (b1) res += 4;
  return res;
}

inline int sorted_switch_label_i23(int i1, int i2)
{ return i1*3+i2; }



site::site(const rat_point& P_init, bool inc)
{
  t = PNT;
  s = rat_segment(P_init,P_init);
  if (inc)
    n = ++count;
  else
    n = -count;
}

site::site(const rat_segment& s_init)
{ 
  if (s_init.start() == s_init.end())
    t = PNT;
  else
    t = SEG;
  s = s_init;
  n = ++count;
}

site::site(integer x, integer y)
{
  t = PNT;
  s = rat_segment(x,y,x,y);
  n = ++count;
}

site::site(integer x0, integer y0, integer x1, integer y1)
{ 
  if (x0 == x1 && y0 == y1)
    t = PNT;
  else
    t = SEG;
  s = rat_segment(x0,y0,x1,y1);
  n = ++count;
}

site::site(const site& s_init)
{
  t = s_init.t;
  s = s_init.s;
  n = s_init.n;
}

site  site::operator= (const site& so)
{
  t = so.t;
  s = so.s;
  n = so.n;  
  return *this;
}

void  site::draw(window& W, color c)
{
#ifdef _DEBUG_DRAW
  W.set_node_width(10);
  if (t == PNT)
    W.draw_int_node(rpnt2pnt(s.start()),n);
  else // t == SEG
  {
    W.draw_segment(rseg2seg(s),c);
    point S((s.start().xcoordD()+s.end().xcoordD())/2.0,
             (s.start().ycoordD()+s.end().ycoordD())/2.0);
    W.draw_int_node(S,n);
  }
#else
  if (t == PNT) {
    W.set_node_width(4);
    W.draw_filled_node(pnt(),c);
  } else {// t == SEG
    W.set_line_width(3);
    W.draw_segment(seg(),c);
    W.set_line_width(1);
  }
#endif
}

bool  operator==(const site& s1, const site& s2)
{
  if (s1.type() != s2.type()) return false;
  return (s1.s == s2.s);
}

bool  operator!=(const site& s1, const site& s2)
{
  return !(s1 == s2);
}

ostream&  operator<<(ostream& out, const site& p)
{
  if (p.nr() == INFNUM)
    out << " infinity"; 
  else if (p.is_pnt())
    out << " pnt #" << p.nr() << " " << p.pnt();
  else // p.is_seg()
    out << " seg #" << p.nr() << " " << p.seg();
  return out;
}


vnode::vnode(p_site p, p_site q, p_site r)
{
  // this is only called in the context that |pqr| exists
  // now |p,q,r != \infty| which means either pnt or seg:
  RealOk=DblOk=IntOk=rat=false;

  switch(sorted_switch_label_b3((*p).is_seg(),(*q).is_seg(),(*r).is_seg())) {
    case ttt_label:
        csites[0]=p; csites[1]=q; csites[2]=r;
        PreCompSSS();
        break;
    case ttf_label:
        csites[0]=p; csites[1]=q; csites[2]=r;
        PreCompSSP();
        break;
    case tft_label:
        csites[0]=r; csites[1]=p; csites[2]=q;
        PreCompSSP();
        break;
    case tff_label:
        csites[0]=p; csites[1]=q; csites[2]=r;
        PreCompSPP();
        break;
    case ftt_label:
        csites[0]=q; csites[1]=r; csites[2]=p;
        PreCompSSP();
        break;
    case ftf_label:
        csites[0]=q; csites[1]=r; csites[2]=p;
        PreCompSPP();
        break;
    case fft_label:
        csites[0]=r; csites[1]=p; csites[2]=q;
        PreCompSPP();
        break;
    case fff_label:
        csites[0]=p; csites[1]=q; csites[2]=r;
        t=PPP;
        break;
  }
  LTRACE("es_vnode construction");
  LTRACE(nt());LTRACE(csite(0));LTRACE(csite(1));LTRACEN(csite(2));
}

vnode::vnode(real xr, real yr, real wr, bool i) 
{
  xCoord=xr;
  yCoord=yr;
  wCoord=wr;  
  RealOk=true;
  error_handler(0,"am I ever called?");
}

vnode::vnode(const vnode& vn)
{
  xCoord = vn.xCoord;
  yCoord = vn.yCoord;
  wCoord = vn.wCoord;

  xCoordI = vn.xCoordI;
  yCoordI = vn.yCoordI;
  wCoordI = vn.wCoordI;

  xCoordD = vn.xCoordD;
  yCoordD = vn.yCoordD;
  wCoordD = vn.wCoordD;

  xCoordDS = vn.xCoordDS;
  yCoordDS = vn.yCoordDS;
  wCoordDS = vn.wCoordDS;

  csites[0] = vn.csites[0];
  csites[1] = vn.csites[1];
  csites[2] = vn.csites[2];
  t = vn.t;
  rat = vn.rat;

  RealOk=vn.RealOk;
  DblOk=vn.DblOk;
  IntOk=vn.IntOk;
}

vnode vnode::operator=(const vnode& vn)
{
  xCoord = vn.xCoord;
  yCoord = vn.yCoord;
  wCoord = vn.wCoord;

  xCoordI = vn.xCoordI;
  yCoordI = vn.yCoordI;
  wCoordI = vn.wCoordI;

  xCoordD = vn.xCoordD;
  yCoordD = vn.yCoordD;
  wCoordD = vn.wCoordD;

  xCoordDS = vn.xCoordDS;
  yCoordDS = vn.yCoordDS;
  wCoordDS = vn.wCoordDS;

  csites[0] = vn.csites[0];
  csites[1] = vn.csites[1];
  csites[2] = vn.csites[2];
  t = vn.t;
  rat = vn.rat;

  RealOk=vn.RealOk;
  DblOk=vn.DblOk;
  IntOk=vn.IntOk;

  return *this;
}

bool vnode::operator==(const vnode& vn)
{
  return ((x()*vn.w() == vn.x()*w()) && (y()*vn.w() == vn.y()*w()));
}

ostream&  operator<<(ostream& out, const vnode& vn)
{
  out << "<";
  out << vn.csite(0) << ",";
  out << vn.csite(1) << ",";
  out << vn.csite(2);
  out << "> = ";
  out << "(" << vn.xd() << "," << vn.yd() << ")";
  return out;
}




void vnode::PreCompSSS()
{
  rat_segment l1=csite(0).rseg();
  rat_segment l2=csite(1).rseg();
  rat_segment l3=csite(2).rseg();
    LTRACE("sss_coords: ");LTRACE(l1);LTRACE(l2);LTRACEN(l3);

  rat_point tpnt12, tpnt13, tpnt23;
  bool touch12 = endpnts_touch(l1,l2,tpnt12);
  bool touch13 = endpnts_touch(l1,l3,tpnt13);
  bool touch23 = endpnts_touch(l2,l3,tpnt23);
  // provides information about common endpoints

  
  /* All segments might touch in one point.  This piece of code checks
     the special case 3 of the above list. As we calculated touch
     conditions and touch points for the three segments this is
     easy. Note that here the node is rational.  */
  if (touch12 && touch13 && (tpnt12 == tpnt13))
  {
      LTRACEN("  all segments touch in one point");
    xCoordI=tpnt12.X();
    yCoordI=tpnt12.Y();
    wCoordI=tpnt12.W();
    if (sign(wCoordI)<0)
    {
      xCoordI=-xCoordI;
      yCoordI=-yCoordI;
      wCoordI=-wCoordI;
    }
    xCoord=xCoordI; 
    yCoord=yCoordI; 
    wCoord=wCoordI;
    xCoordD=xCoordI.to_double(); 
    yCoordD=yCoordI.to_double(); 
    wCoordD=wCoordI.to_double();
    xCoordDS=(abs(xCoordI)).to_double();
    yCoordDS=(abs(yCoordI)).to_double();
    wCoordDS=(abs(wCoordI)).to_double();
    t=SSS_alltouch;
    DblOk=IntOk=RealOk=true;
    rat=true;
    return;
  }


  int l1_rel_l2, l1_rel_l3, l2_rel_l1, l2_rel_l3, l3_rel_l1, l3_rel_l2;
  orient_segs(l1,l2,l1_rel_l2,l2_rel_l1);
  orient_segs(l1,l3,l1_rel_l3,l3_rel_l1);
  orient_segs(l2,l3,l2_rel_l3,l3_rel_l2);
    
  if ((l1_rel_l2 == ON) || (l2_rel_l3 == ON) || (l1_rel_l3 == ON) )
    t=SSS_coltouch; // sss_coords - two segments are collinearly touching
  else
    t=SSS_interior; // sss_coords - node is implied by interior touching points
}



void vnode::PreCompSSP()
{
  rat_segment l1=csite(0).rseg();
  rat_segment l2=csite(1).rseg();
  rat_point   P =csite(2).rpnt();
    LTRACE("ssp_coords: ");LTRACE(l1);LTRACE(l2);LTRACEN(P);

  bool p_is_l1_endpnt = (P == l1.start() || P == l1.end());
  bool p_is_l2_endpnt = (P == l2.start() || P == l2.end());
  
  if (p_is_l1_endpnt && p_is_l2_endpnt) {
    // ssp_coords P is common endpoint
    integer xp = P.X(),
            yp = P.Y(),
            zp = P.W(); // initializes the homogenous coords of P

    if (sign(zp)<0)
    {
      xp=-xp;
      yp=-yp;
      zp=-zp;
    }
    xCoordI=xp; 
    yCoordI=yp; 
    wCoordI=zp;
    xCoord=xCoordI; 
    yCoord=yCoordI; 
    wCoord=wCoordI;
    xCoordD=xCoordI.to_double(); 
    yCoordD=yCoordI.to_double(); 
    wCoordD=wCoordI.to_double();
    xCoordDS=(abs(xCoordI)).to_double();
    yCoordDS=(abs(yCoordI)).to_double();
    wCoordDS=(abs(wCoordI)).to_double();
    t=SSP_alltouch;
    DblOk=IntOk=RealOk=true;
    rat=true;
  }

  else
  {
    integer a1,b1,c1,a2,b2,c2;
    int l1_p_or = orientation(l1,P);
    int l2_p_or = orientation(l2,P);

    if (l1_p_or == RIGHT)
      invert_seg(l1);
    if (l2_p_or == RIGHT)
      invert_seg(l2);

    integer dx1 = l1.dx();
    integer dy1 = l1.dy();
    integer dx2 = l2.dx();
    integer dy2 = l2.dy();
    

    if (l1_p_or == ON)
    {
      if (dx1*dy2 == dx2*dy1 && 
          sign(dx1) == sign(dx2) && sign (dy1) == sign(dy2) ||
          P == l1.start())
        // l1 and l2 are parallel and equally directed or
        // l1 and l2 are not parallel and the node is on the right
        invert_seg(l1);
    }
    if (l2_p_or == ON)
    {
      if (dx1*dy2 == dx2*dy1 && 
          sign(dx1) == sign(dx2) && sign(dy1) == sign(dy2) ||
          P == l2.end())
        invert_seg(l2);
    }
  
    seg2line_paras(l1,a1,b1,c1); // initializes the homogenous coords of l1
    seg2line_paras(l2,a2,b2,c2); // initializes the homogenous coords of l2

    if (((a1*b2-a2*b1)!=0) || ((a1*c2-a2*c1)!=0) || ((b1*c2-b2*c1)!=0))
      t=SSP_notcoll;  // ssp_coords with different segment support lines
    else 
      t=SSP_coltouch; // ssp_coords with one segment support line: l1 == l2
  }
}


void vnode::PreCompSPP()
{
  rat_segment l=csite(0).rseg();
  rat_point   P=csite(1).rpnt();
  rat_point   Q=csite(2).rpnt();

    LTRACE("spp_coords: ");LTRACE(l);LTRACE(P);LTRACEN(Q);
    ASSERT((orientation(l,P)!=-orientation(l,Q)),spp_coords);

  if (right_turn(l.start(),l.end(),P))
    invert_seg(l);
    // care for the right orientation of l

  integer a,b,c;
  seg2line_paras(l,a,b,c);     // initializes the homogenous coords of l
  // first we again translate the configuration according to the
  // translation of P into the origin (0,0,1)
  // the translated coordinates are marked by doubled dir characters

  int sign_res;
  BEGIN_PREDICATE
    {
      DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W() Q.X() Q.Y() Q.W();
      DECLARE_ATTRIBUTES integer_type FOR a b;
      integer xp=P.X(); integer yp=P.Y(); integer wp=P.W();
      integer xq=Q.X(); integer yq=Q.Y(); integer wq=Q.W();
      
      integer xxq = xq*wp - xp*wq;
      integer yyq = yq*wp - yp*wq;
      integer aa = a*wp;
      integer bb = b*wp;
      integer res=aa*xxq+bb*yyq;
      sign_res=sign(res);
    }
  END_PREDICATE

  // According to C.Burnikel we have to examine two cases here
  // ((aa*xxq + bb*yyq)!=0) ?
  if (sign_res !=0 )
    t=SPP_distdiff; // spp_coords with different distances of P and Q to l
  else
    t=SPP_disteq;   // spp_coords with equal distance of P and Q to l
}



void vnode::CompDbl()
{
    LTRACE("CompDbl "); 
    if (DblOk) cout<<"DblOk is true"<<endl;

  switch(t) {
    case(SSS_coltouch): SSS_coltouch_Double(); break;
    case(SSS_interior): SSS_interior_Double(); break;
    case(SSP_notcoll):  SSP_notcoll_Double();  break;
    case(SSP_coltouch): SSP_coltouch_Double(); break;
    case(SPP_distdiff): SPP_distdiff_Double(); break;
    case(SPP_disteq):   SPP_disteq_Double();   break;
    case(PPP):          PPP_Double();          break;
    default: error_handler(1,"CompDbl: reached illegal default switch."); break;
  }

  DblOk=true;

  if (float_abs(wCoordD) <= wCoordDS*22*eps1) {
    /* if sign of wCoordD is unsure, invalidate floating-point approximations) */
    xCoordDS=pInf_double;
    yCoordDS=pInf_double;
    wCoordDS=pInf_double;
  }
  else if (sign(wCoordD)<0) {
    xCoordD=-xCoordD;
    yCoordD=-yCoordD;
    wCoordD=-wCoordD;
  }
}


void vnode::CompInt()
{
    LTRACE("CompInt"); LTRACEN(*this);
    ASSERT(!rat, "Error SPP not rational");

  switch(t) {
    case(SSS_coltouch):  rat=SSS_coltouch_Integer(); break;
    case(SSS_interior):  rat=SSS_interior_Integer(); break;
    case(SSP_notcoll):   rat=SSP_notcoll_Integer();  break;
    case(SSP_coltouch):  rat=SSP_coltouch_Integer(); break;
    case(SPP_distdiff):  rat=SPP_distdiff_Integer(); break;
    case(SPP_disteq):    rat=SPP_disteq_Integer();   break;
    case(PPP):           rat=PPP_Integer();          break;
    default: error_handler(1,"CompInteger: reached illegal default switch."); break;
  }

  if (sign(wCoordI)<0) {
    xCoordI=-xCoordI;
    yCoordI=-yCoordI;
    wCoordI=-wCoordI;
  }

  IntOk=true;
  if (rat==true) {
    xCoord=xCoordI; 
    yCoord=yCoordI; 
    wCoord=wCoordI;
    xCoordD=xCoordI.to_double(); 
    yCoordD=yCoordI.to_double(); 
    wCoordD=wCoordI.to_double();
    xCoordDS=(abs(xCoordI)).to_double(); 
    yCoordDS=(abs(yCoordI)).to_double(); 
    wCoordDS=(abs(wCoordI)).to_double();
    DblOk=RealOk=true;
  }
}


void vnode::CompReal()
{
    LTRACE("CompReal ");LTRACEN(*this);

  switch(t) {
    case(SSS_coltouch):  SSS_coltouch_Exact(); break;
    case(SSS_interior):  SSS_interior_Exact(); break;
    case(SSP_notcoll):   SSP_notcoll_Exact();  break;
    case(SSP_coltouch):  SSP_coltouch_Exact(); break;
    case(SPP_distdiff):  SPP_distdiff_Exact(); break;
    case(SPP_disteq):    SPP_disteq_Exact();   break;
    case(PPP):           PPP_Exact();          break;
    default: error_handler(1,"CompReal: reached illegal default switch."); break;
  }

  if (sign(wCoord)<0) {
    xCoord=-xCoord;
    yCoord=-yCoord;
    wCoord=-wCoord;
  }
  xCoordD=xCoord.to_double(); 
  yCoordD=yCoord.to_double(); 
  wCoordD=wCoord.to_double();
  xCoordDS=ceil(fabs(xCoord.to_double())*
                ((xCoord.get_double_error()+eps)/eps));
  yCoordDS=ceil(fabs(yCoord.to_double())*
                ((yCoord.get_double_error()+eps)/eps));
  wCoordDS=ceil(fabs(wCoord.to_double())*
                ((wCoord.get_double_error()+eps)/eps));
  RealOk=DblOk=true;
}



DEFINE_FUNCTION vnode::PPP_() AS poly_rid_type;

  rat_point P=csite(0).rpnt();
  rat_point Q=csite(1).rpnt();
  rat_point R=csite(2).rpnt();

BEGIN_COMPUTATION
{
DECLARE_ATTRIBUTES poly_id_type FOR 
  P.X() P.Y() P.W() Q.X() Q.Y() Q.W() R.X() R.Y() R.W();
DECLARE_INTFLAG 1;
  real xpi = P.X(); 
  real ypi = P.Y(); 
  real wpi = P.W(); 
  real xqi = Q.X(); 
  real yqi = Q.Y(); 
  real wqi = Q.W(); 
  real xri = R.X(); 
  real yri = R.Y(); 
  real wri = R.W();
          
  xCoord = wqi * wri * (sq(xpi) + sq(ypi)) * (yqi*wri - yri*wqi) +
           wpi * wri * (sq(xqi) + sq(yqi)) * (yri*wpi - ypi*wri) +
           wpi * wqi * (sq(xri) + sq(yri)) * (ypi*wqi - yqi*wpi);
  yCoord = - (wqi * wri * (sq(xpi) + sq(ypi)) * (xqi*wri - xri*wqi) +
              wpi * wri * (sq(xqi) + sq(yqi)) * (xri*wpi - xpi*wri) +
              wpi * wqi * (sq(xri) + sq(yri)) * (xpi*wqi - xqi*wpi));
  wCoord = 2 * wpi * wqi * wri * 
           (wpi*(xqi*yri - xri*yqi) + 
            wqi*(xri*ypi - xpi*yri) + 
            wri*(xpi*yqi - xqi*ypi));

}
END_COMPUTATION
END_FUNCTION


DEFINE_FUNCTION vnode::SPP_distdiff_() AS poly_rid_type;

  rat_segment l=csite(0).rseg();
  rat_point   P=csite(1).rpnt();
  rat_point   Q=csite(2).rpnt();

  if (right_turn(l.start(),l.end(),P))
    invert_seg(l);  
    /* care for the right orientation of l */

  integer a,b,c;
  seg2line_paras(l,a,b,c);     // initializes the homogenous coords of l

BEGIN_COMPUTATION
{
DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W() Q.X() Q.Y() Q.W();
DECLARE_ATTRIBUTES integer_type FOR a b c;
DECLARE_INTFLAG 1;
  real xp=P.X();
  real yp=P.Y();
  real zp=P.W();
  real xq=Q.X();
  real yq=Q.Y();
  real zq=Q.W();
  
  /* first we again translate the configuration according to the */ 
  /* translation of P into the origin (0,0,1) */
  /* the translated coordinates are marked by doubled dir characters */

  real xxq = xq*zp - xp*zq;
  real yyq = yq*zp - yp*zq;

  real R = (sq(xxq)+sq(yyq));
  real S1=(a*xp + b*yp + c*zp);
  real S2=(a*xq + b*yq + c*zq);
  real S = zp * zq * R * (sq(a)+sq(b)) * S1*S2; 
  real S_W = sqrt(S);

  real zzq = zp*zq;
  real aa = a*zp;
  real bb = b*zp;
  real cc = a*xp + b*yp + c*zp;

  real A = (a*xxq + b*yyq);
  real X = sq(A);
  real T = R * A;
  real I = b * T  +  2 * cc * xxq * zq * (b*xxq-a*yyq);
  real J = a * T  +  2 * cc * yyq * zq * (a*yyq-b*xxq);

    /* the result coordinates of the $lPQ$ node of the translated configuration: */
    /* the factor 1 in the two lines below is kappa */
  real xv_trans = J  +  2 * 1 * yyq * S_W;
  real yv_trans = I  -  2 * 1 * xxq * S_W;
  real zv_trans = 2 * zzq * X;


  xCoord = xv_trans*zp + xp*zv_trans;
  yCoord = yv_trans*zp + yp*zv_trans;
  wCoord = zv_trans*zp;
}
END_COMPUTATION
END_FUNCTION



DEFINE_FUNCTION vnode::SPP_disteq_() AS poly_rid_type;

  rat_segment l=csite(0).rseg();
  rat_point   P=csite(1).rpnt();
  rat_point   Q=csite(2).rpnt();

  if (right_turn(l.start(),l.end(),P))
    invert_seg(l);  
    /* care for the right orientation of l */

  integer a,b,c;
  seg2line_paras(l,a,b,c);     // initializes the homogenous coords of l

BEGIN_COMPUTATION
{
DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W() Q.X() Q.Y() Q.W();
DECLARE_ATTRIBUTES integer_type FOR a b c;
DECLARE_INTFLAG 1;

  real xp=P.X(); 
  real yp=P.Y(); 
  real zp=P.W();
  real xq=Q.X(); 
  real yq=Q.Y();
  real zq=Q.W();

  /* first we again translate the configuration according to the */ 
  /* translation of P into the origin (0,0,1) */
  /* the translated coordinates are marked by doubled dir characters */

  real xxq = xq*zp - xp*zq;
  real yyq = yq*zp - yp*zq;
  real zzq = zp*zq;
  real aa = a*zp;
  real bb = b*zp;
  real cc = a*xp + b*yp + c*zp;

  real B = (sq(a)+sq(b));
  real C = (sq(xxq)+sq(yyq));

    /* the result coordinates of the $lPQ$ node (translated): */
  real xv_trans = a * B * C  +  4 * cc * B * xxq * zq  -  
               4 * a * sq(cc) * sq(zq);
  real yv_trans = b * B * C  +  4 * cc * B * yyq * zq  -  
               4 * b * sq(cc) * sq(zq);
  real zv_trans = 8 * B * cc * sq(zq) * zp;

  xCoord = xv_trans*zp + xp*zv_trans;
  yCoord = yv_trans*zp + yp*zv_trans;
  wCoord = zv_trans*zp;
}
END_COMPUTATION
END_FUNCTION



DEFINE_FUNCTION vnode::SSP_notcoll_() AS poly_rid_type;

  rat_segment l1=csite(0).rseg();
  rat_segment l2=csite(1).rseg();
  rat_point   P =csite(2).rpnt();

    int l1_p_or = orientation(l1,P);
    int l2_p_or = orientation(l2,P);


    if (l1_p_or == RIGHT)
      invert_seg(l1);
    if (l2_p_or == RIGHT)
      invert_seg(l2);

    integer dx1 = l1.dx();
    integer dy1 = l1.dy();
    integer dx2 = l2.dx();
    integer dy2 = l2.dy();
    
    if (l1_p_or == ON)
    {
      if (dx1*dy2 == dx2*dy1 && 
          sign(dx1) == sign(dx2) && sign (dy1) == sign(dy2) ||
          P == l1.start())
        // l1 and l2 are parallel and equally directed or
        // l1 and l2 are not parallel and the node is on the right
        invert_seg(l1);
    }
    if (l2_p_or == ON)
    {
      if (dx1*dy2 == dx2*dy1 && 
          sign(dx1) == sign(dx2) && sign(dy1) == sign(dy2) ||
          P == l2.end())
        invert_seg(l2);
    }
    integer a1,b1,c1,a2,b2,c2;
  
    seg2line_paras(l1,a1,b1,c1); // initializes the homogenous coords of l1
    seg2line_paras(l2,a2,b2,c2); // initializes the homogenous coords of l2

    int sign_s, sign_r;
    if (a1 == 0 && a2 == 0) 
      sign_s = sign(b1); 
      // the trivial case with x-axis parallel lines
      // note that |sign(b2) != sign(b1)|
    else
      sign_s = sign(b1*abs(a2) - b2*abs(a1));
      // the previous case

     if (b1 == 0 && b2 == 0) 
      sign_r = sign(a1); 
      // the trivial case with y-axis parallel lines
      // note that |sign(a2) != sign(a1)|
    else
      sign_r = sign(a1*abs(b2) - a2*abs(b1));
      // the previous case

    int kappa = -1; // compare Lemma 12

BEGIN_COMPUTATION
{
DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W();
DECLARE_ATTRIBUTES integer_type FOR a1 b1 c1 a2 b2 c2;
DECLARE_ATTRIBUTES simple_type int_val FOR sign_s sign_r kappa;
DECLARE_INTFLAG 1;

    real xp = P.X();
    real yp = P.Y();
    real zp = P.W(); /* initializes the homogenous coords of P */

    real cc1 = a1*xp + b1*yp + c1*zp;
    real cc2 = a2*xp + b2*yp + c2*zp;

    real D12_2 = (sq(a1)+sq(b1))*(sq(a2)+sq(b2));
    real D12 = sqrt(D12_2);
 
    real D = 2*cc1*cc2;
    real F = D * (b1*b2 - a1*a2 + D12 );
    real G = D * (a1*a2 - b1*b2 + D12 ); 
    real F_W = sqrt(F);
    real G_W = sqrt(G);
 
    real X = - a1*a2 - b1*b2 + D12;
    real I = b1*cc2 + b2*cc1;
    real J = a1*cc2 + a2*cc1;

    real xv_trans = J + kappa * sign_s * G_W;
    real yv_trans = I - kappa * sign_r * F_W;
    real zv_trans = zp * X;

    /* now this is the node of the translated configuration */
    /* we have to translate it according to the coordinates of P */
    xCoord = xv_trans*zp + xp*zv_trans;
    yCoord = yv_trans*zp + yp*zv_trans;
    wCoord = zv_trans*zp;
}
END_COMPUTATION
END_FUNCTION



DEFINE_FUNCTION vnode::SSP_coltouch_() AS poly_rid_type;

  rat_segment l1=csite(0).rseg();
  rat_segment l2=csite(1).rseg();
  rat_point    P=csite(2).rpnt();

  integer xv,yv,zv;
  bool p_is_l1_endpnt = (P == l1.start() || P == l1.end());
  bool p_is_l2_endpnt = (P == l2.start() || P == l2.end());
  
    int l1_p_or = orientation(l1,P);
    int l2_p_or = orientation(l2,P);


    if (l1_p_or == RIGHT)
      invert_seg(l1);
    if (l2_p_or == RIGHT)
      invert_seg(l2);

    integer dx1 = l1.dx();
    integer dy1 = l1.dy();
    integer dx2 = l2.dx();
    integer dy2 = l2.dy();
    
    if (l1_p_or == ON)
    {

      if (dx1*dy2 == dx2*dy1 && 
          sign(dx1) == sign(dx2) && sign (dy1) == sign(dy2) ||
          P == l1.start())
        // l1 and l2 are parallel and equally directed or
        // l1 and l2 are not parallel and the node is on the right
        invert_seg(l1);
    }
    if (l2_p_or == ON)
    {

      if (dx1*dy2 == dx2*dy1 && 
          sign(dx1) == sign(dx2) && sign(dy1) == sign(dy2) ||
          P == l2.end())
        invert_seg(l2);
    }
  
    integer a1,b1,c1;
    seg2line_paras(l1,a1,b1,c1); // initializes the homogenous coords of l1
      
    integer xt,yt,zt;
    if (l1.end()==l2.start())
    {
      xt = l1.X2();
      yt = l1.Y2();
      zt = l1.W2(); 
    }
    else if (l1.start()==l2.end())
    {
      xt = l1.X1();
      yt = l1.Y1();
      zt = l1.W1();
    }
    // now T = (xt,yt,zt) is the common endpoint on the line
   

BEGIN_COMPUTATION
{ 
DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W();
DECLARE_ATTRIBUTES integer_type FOR a1 b1 xt yt zt; 
DECLARE_INTFLAG 1;
    real xp = P.X();
    real yp = P.Y();
    real zp = P.W(); /* initializes the homogenous coords of P */

    /* we determine the middle perpendicular l3 of line segment TP */
    real a3 = 2*zp*zt * (xt*zp-xp*zt);
    real b3 = 2*zp*zt * (yt*zp-yp*zt);
    real c3 = (sq(xp)+sq(yp)) * sq(zt) - 
              (sq(xt)+sq(yt)) * sq(zp);
   
    /* now we determine the perpendicular line l4 of l1=l2 in T */
    real a4 = b1*zt;
    real b4 = -a1*zt;
    real c4 = a1*yt - b1*xt;

    /* and now the intersection point of the two lines: */
    xCoord = b3*c4 - b4*c3;
    yCoord = a4*c3 - a3*c4;
    wCoord = a3*b4 - a4*b3;
}
END_COMPUTATION
END_FUNCTION



DEFINE_FUNCTION vnode::SSS_coltouch_() AS poly_rid_type;

  rat_segment l1=csite(0).rseg();
  rat_segment l2=csite(1).rseg();
  rat_segment l3=csite(2).rseg();

  rat_point tpnt12, tpnt13, tpnt23;
  bool touch12 = endpnts_touch(l1,l2,tpnt12);
  bool touch13 = endpnts_touch(l1,l3,tpnt13);
  bool touch23 = endpnts_touch(l2,l3,tpnt23);

  int l1_rel_l2, l1_rel_l3, l2_rel_l1, l2_rel_l3, l3_rel_l1, l3_rel_l2;
  orient_segs(l1,l2,l1_rel_l2,l2_rel_l1);
  orient_segs(l1,l3,l1_rel_l3,l3_rel_l1);
  orient_segs(l2,l3,l2_rel_l3,l3_rel_l2);
  // provides information about orientation of supporting lines
    
  rat_segment l_1, l_2;
 
  if (l1_rel_l2 == ON)
  {
    if (l2.start() == tpnt12)
      invert_seg(l2);
    if (orientation(l3,tpnt12) == RIGHT)
      invert_seg(l3);
    l_1=l2; l_2=l3;
  }
  if (l1_rel_l3 == ON)
  {
    if (l1.start() == tpnt13)
      invert_seg(l1);
    if (orientation(l2,tpnt13) == RIGHT)
      invert_seg(l2);
    l_1=l1; l_2=l2;
  }
  if (l2_rel_l3 == ON)
  {
    if (l3.start() == tpnt23)
      invert_seg(l3);
    if (orientation(l1,tpnt23) == RIGHT)
      invert_seg(l1);
    l_1=l3; l_2=l1;
  }


  integer a1,b1,c1,a2,b2,c2;
  seg2line_paras(l_1,a1,b1,c1);
  seg2line_paras(l_2,a2,b2,c2);  

  rat_point P=l_1.end();

BEGIN_COMPUTATION
{
DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W();
DECLARE_ATTRIBUTES integer_type FOR a1 b1 c1 a2 b2 c2;
DECLARE_INTFLAG 1;

  real D1_2 = sq(a1)+sq(b1);
  real D2_2 = sq(a2)+sq(b2);
  real D12_2 = D1_2*D2_2;
  real D12 = sqrt(D12_2);
 
  real a3 = a1*D12-a2*D1_2;   /* simply multiplied by D1 */
  real b3 = b1*D12-b2*D1_2;
  real c3 = c1*D12-c2*D1_2;
  /* this is the bisector line |l3| between |l_1| and |l_2| */
  /* it's either a half angle or a middle parallel line */

  real a4 =   b1*P.W();
  real b4 = - a1*P.W();
  real c4 = a1*P.Y() - b1*P.X();
  /* this is the perpendicular line |l4| of |l_1| in |T| */
  
  xCoord = b3*c4 - b4*c3;
  yCoord = a4*c3 - a3*c4;
  wCoord = a3*b4 - a4*b3;
  /* this is the intersection point of the two lines */
}

END_COMPUTATION
END_FUNCTION




DEFINE_FUNCTION vnode::SSS_interior_() AS poly_rid_type;

  rat_segment l1=csite(0).rseg();
  rat_segment l2=csite(1).rseg();
  rat_segment l3=csite(2).rseg();

  int l1_rel_l2, l1_rel_l3, l2_rel_l1, l2_rel_l3, l3_rel_l1, l3_rel_l2;
  orient_segs(l1,l2,l1_rel_l2,l2_rel_l1);
  orient_segs(l1,l3,l1_rel_l3,l3_rel_l1);
  orient_segs(l2,l3,l2_rel_l3,l3_rel_l2);

  bool l1_is_oriented = (l2_rel_l1 == LEFT || l3_rel_l1 == LEFT);
  bool l2_is_oriented = (l1_rel_l2 == LEFT || l3_rel_l2 == LEFT);
  bool l3_is_oriented = (l1_rel_l3 == LEFT || l2_rel_l3 == LEFT);
  //  this states if a segment is already correctly oriented 

  if (!l1_is_oriented && (l2_rel_l1 == RIGHT || l3_rel_l1 == RIGHT))
  {  
    invert_seg(l1);  
    l1_is_oriented = true;  
  }
  if (!l2_is_oriented && (l1_rel_l2 == RIGHT || l3_rel_l2 == RIGHT))
  {  
    invert_seg(l2);  
    l2_is_oriented = true;  
  }
  if (!l3_is_oriented && (l1_rel_l3 == RIGHT || l2_rel_l3 == RIGHT))
  {  
    invert_seg(l3);  
    l3_is_oriented = true;  
  }  
  //  this orients the segments which are not correctly oriented
  //  now only one segment might not be oriented

  if (!l1_is_oriented)
  {
    if (l2_rel_l3 == UNDEC && orientation(l1,l3.start()) != RIGHT ||
       l3_rel_l2 == UNDEC && orientation(l1,l2.start()) != LEFT ||
       l2_rel_l3 != UNDEC && l3_rel_l2 != UNDEC && 
       orientation(l1,l2.start()) != LEFT)
      /* l2 is $s_n$ and l3 is $s_p$ thus the first line refers to case 2, the second 
         refers to case 3 and the last two lines refer to case1 */
      invert_seg(l1);
    l1_is_oriented = true;
  }

  if (!l2_is_oriented)
  {
    if (l3_rel_l1 == UNDEC && orientation(l2,l1.start()) != RIGHT ||
       l1_rel_l3 == UNDEC && orientation(l2,l3.start()) != LEFT ||
       l1_rel_l3 != UNDEC && l3_rel_l1 != UNDEC && 
       orientation(l2,l3.start()) != LEFT)
      /* l3 is $s_n$ and l1 is $s_p$ thus the first line refers to case 2, the second 
         refers to case 3 and the last two lines refer to case1 */
      invert_seg(l2);
    l2_is_oriented = true;
  }

  if (!l3_is_oriented)
  {
    if (l1_rel_l2 == UNDEC && orientation(l3,l2.start()) != RIGHT ||
       l2_rel_l1 == UNDEC && orientation(l3,l1.start()) != LEFT ||
       l1_rel_l2 != UNDEC && l2_rel_l1 != UNDEC && 
       orientation(l3,l1.start()) != LEFT)
      /* l1 is $s_n$ and l2 is $s_p$ thus the first line refers to case 2, the second 
         refers to case 3 and the last two lines refer to case1 */
      invert_seg(l3);
    l3_is_oriented = true;
  }

 
  integer a1,b1,c1,a2,b2,c2,a3,b3,c3;
  seg2line_paras(l1,a1,b1,c1); 
  seg2line_paras(l2,a2,b2,c2); 
  seg2line_paras(l3,a3,b3,c3); 
  // initializes the homogenous coords of |l1,l2,l3|

BEGIN_COMPUTATION
{  
DECLARE_ATTRIBUTES integer_type FOR a1 b1 c1 a2 b2 c2 a3 b3 c3;
DECLARE_INTFLAG 1;

  real D1_2 = sq(a1)+sq(b1);
  real D2_2 = sq(a2)+sq(b2);
  real D3_2 = sq(a3)+sq(b3);

  real D12 = sqrt(D1_2*D2_2);
  real D13 = sqrt(D1_2*D3_2);
 

  xCoord = (c1*b2-c2*b1)*D13 + (c2*b3-c3*b2)*D1_2 + (c3*b1-c1*b3)*D12;
  yCoord = -((c1*a2-c2*a1)*D13 + (c2*a3-c3*a2)*D1_2 + (c3*a1-c1*a3)*D12);
  wCoord = -((a1*b2-a2*b1)*D13 + (a2*b3-a3*b2)*D1_2 + (a3*b1-a1*b3)*D12);
  // this is the intersection point of the two lines
}
END_COMPUTATION
END_FUNCTION



int ES_Voronoi_Diagram::
insert(const rat_point& q)
{
    TRACEN("\n>>----------------------------------<<");
    TRACE("ES.insert of ");TRACEN(q);
   rat_point p(q);
   if (p.W() != 1) {
     p = rat_point(round(p.xcoord()),round(p.ycoord()));
      error_handler(1,"insert: rounding point to grid!");
   }

   if (KnownPoints.lookup(p)==NULL)
   {
        KnownPoints.insert(p, 1);
        return Abstract_Voronoi_Diagram<p_vnode,p_site>::
               insert(p_site(site(p)));
   }
}

int ES_Voronoi_Diagram::
insert(const rat_segment& t)
{
    TRACEN("\n>>----------------------------------<<");
    TRACE("ES.insert of ");
    TRACEN(t);

    rat_segment s(t);
    if (s.is_trivial())
      return insert(s.source());
 
    if (s.W1() != 1 || s.W2() != 1) {
      s = rat_segment(round(s.xcoord1()),round(s.ycoord1()),
                      round(s.xcoord2()),round(s.ycoord2()));
      error_handler(1,"insert: rounding segment to grid!");
    }

    if (KnownPoints.lookup(s.start())==NULL)
    {
      KnownPoints.insert(s.start(), 1);
      p_site epnt1(site(s.start()));

      TRACEN("<<AVD.insert of epnt1");
      if (Abstract_Voronoi_Diagram<p_vnode,p_site>::
          insert(epnt1) == 0)
        return 0;
    }

    if (KnownPoints.lookup(s.end())==NULL)
    {
      KnownPoints.insert(s.end(), 1);
      p_site epnt2(site(s.end()));
      TRACEN("<<AVD.insert of epnt2");
      if (Abstract_Voronoi_Diagram<p_vnode,p_site>::
          insert(epnt2) == 0)
        return 0;
    }

      TRACEN("<<AVD.insert of interior seg");
    return Abstract_Voronoi_Diagram<p_vnode,p_site>::
           insert(p_site(site(s))); 
}


bool ES_Voronoi_Diagram::
insert(const rat_polygon& P)
{
    TRACEN("\n>>----------------------------------<<");
    TRACE("ES.insert of polygon");
    TRACEN(P);

  bool legal = true;
  list<rat_point> LP;
  rat_point p;
  forall (p, P.vertices())
  {
    if (p.W()!=1) {
      p = rat_point(round(p.xcoord()), round(p.ycoord()));
      error_handler(1,"insert: rounding point to grid!");
    }
    LP.append(p);
      
    if (legal) {
      legal = insert(p);
    }
  }

  rat_point p_prev,p_act,p_first;
  bool first = true;

  forall (p, LP) {
    if (legal)
    {
      if (first) {
        p_first = p_act = p;
        first = false;
      }
      else {
        p_prev = p_act;
        p_act  = p;
        legal = Abstract_Voronoi_Diagram<p_vnode,p_site>::insert(
                p_site(site(rat_segment(p_prev,p_act))));
      }
    }
  }

  if (legal)
    legal = Abstract_Voronoi_Diagram<p_vnode,p_site>::
            insert(p_site(site(rat_segment(p_act,p_first))));
  return legal;
}

//--------------------------------------------------------------------------

int ES_Voronoi_Diagram::
animated_insert(const rat_point& p, window& W, int pause)
{
    TRACEN("\n>>----------------------------------<<");
    TRACE("ES.animated_insert of "); TRACEN(p);
    ASSERT(p.W()==1,animated_insert_pnt);

  if (KnownPoints.lookup(p)==NULL) {
    dic_item dit = KnownPoints.insert(p,1);
    int retval = Abstract_Voronoi_Diagram<p_vnode,p_site>::
      animated_insert(p_site(site(p)),W,pause);
    if (retval == 0) 
    // to catch case where illegal point is repeatedly inserted
      KnownPoints.del_item(dit);
    return retval;
  }
  else 
    return 1;
}


int ES_Voronoi_Diagram::
animated_insert(const rat_segment& s, window& W, int pause)
{
    TRACEN("\n>>----------------------------------<<");
    TRACE("ES.animated_insert of "); TRACEN(s);
    ASSERT(s.W1()==1 && s.W2()==1,animated_insert_seg);

  if (s.is_trivial())
    return animated_insert(s.source(),W,pause);
  else {
    if (animated_insert(s.source(),W,pause) == 0)
      return 0;
    if (animated_insert(s.target(),W,pause) == 0)
      return 0;

        TRACEN("<<AVD.insert of interior seg");
    return Abstract_Voronoi_Diagram<p_vnode,p_site>::
      animated_insert(p_site(site(s)),W,pause); 
  }
}


bool ES_Voronoi_Diagram::
animated_insert(const rat_polygon& P, window& W, int pause)
{
    TRACEN("\n>>----------------------------------<<");
    TRACE("ES.animated_insert of polygon"); TRACEN(P);

  rat_point v;
  bool legal = true;
  forall (v, P.vertices())
    if (legal) legal = animated_insert(v,W,pause);

  rat_segment e;
  forall (e, P.edges())
    if (legal) legal = Abstract_Voronoi_Diagram<p_vnode,p_site>::
      animated_insert(p_site(site(e)),W,pause);

  return legal;
}



bool ES_Voronoi_Diagram::
init_is_possible(p_site t1, p_site t2, bool& error)
{
  if(error = illegal_intersection(t1,t2)) 
    return false;
  if ((*t1).is_pnt() && (*t2).is_pnt() && (*t1).rpnt()==(*t2).rpnt())
    return false;
  return true;
}



topology  ES_Voronoi_Diagram::
edge_intersection_topology(p_site p, p_site q, p_site r, p_site s, p_site t, 
                           p_vnode mr, p_vnode ms, bool& error)  
{ 
    TRACE(">>edge_intersection_topology: ");
    TRACE(*p);TRACE(*q);TRACE(*r);TRACE(*s);TRACEN(*t);

  {
    error = (illegal_intersection(p,t) || illegal_intersection(q,t) || 
             illegal_intersection(r,t) || illegal_intersection(s,t));
  }


  if (error)
    return NO_INTERSECTION;

  node_status pqr_status = calculate_node_status(p,q,r,t,mr);
  node_status psq_status = calculate_node_status(q,p,s,t,ms);
  topology result;

    TRACE(" node stati: ");TRACE(pqr_status);TRACEN(psq_status);

  /* in the following case switching the labels determine
     U (unaffected), A (affected), C (clipped) */
  switch(sorted_switch_label_i23(pqr_status,psq_status)){
  case UC: 
    result = TARGET_SEGMENT;
    break;
  case CU: 
    result = SOURCE_SEGMENT;
    break;
  case UA:  
    if (inner_point_on_edge_is_clipped(p,q,r,s,t,mr,ms))
      result = TARGET_SEGMENT;
    else
      result = NO_INTERSECTION;
    break;
  case AU: 
    if (inner_point_on_edge_is_clipped(p,q,r,s,t,mr,ms))
      result = SOURCE_SEGMENT;
    else
      result = NO_INTERSECTION;
    break;
  case CA:  
    if (inner_point_on_edge_remains(p,q,r,s,t,mr,ms))
      result = SOURCE_SEGMENT;
    else
      result = TOTAL;
    break;
  case AC: 
    if (inner_point_on_edge_remains(p,q,r,s,t,mr,ms))
      result = TARGET_SEGMENT;
    else
      result = TOTAL;
    break;
  case UU: 
    if (inner_point_on_edge_is_clipped(p,q,r,s,t,mr,ms))
      result = INTERN;
    else
      result = NO_INTERSECTION;
    break;
  case AA: 
    if (total_cut_when_both_affected(p,q,r,s,t,mr,ms))
      result = TOTAL;
    else
      result = NO_INTERSECTION;
    break;
  case CC: 
    if (inner_point_on_edge_remains(p,q,r,s,t,mr,ms))
      result = TWO_SEGMENTS;
    else
      result = TOTAL;
    break;
  }

    TRACE("return value = ");TRACEN(result);
  return result;
}



bool ES_Voronoi_Diagram::
inner_point_on_edge_is_clipped(p_site p, p_site q, p_site r, p_site s, 
                               p_site t, p_vnode mr, p_vnode ms)
{
    TRACE("  inner_point_on_edge_is_clipped ");  
    TRACE(*p);TRACE(*q);TRACE(*r);TRACE(*s);TRACEN(*t);

  {
    if (p == infinity())
      return (r == s && line_hull_and_order(r,q,t));
    if (q == infinity())
      return (r == s && line_hull_and_order(r,p,t));
  }


  
  /* now all sites are finite */
  
  if ((*t).is_seg())
    return false;    // see Lemma 10

  rat_point P;
  rat_segment sp,sq;
  p_vnode vl,vr;
  rat_point T = (*t).rpnt();
  switch(sorted_switch_label_b2((*p).is_pnt(),(*q).is_pnt())) {
  case tt_label: 
    return false; 
    break;
  case tf_label:
    P = (*p).rpnt(); 
    sq = (*q).rseg(); 
    vl = mr; 
    vr = ms;
    {
      int p_rel_sq = orientation(sq,P);
      if (p_rel_sq == ON)
        return false;
      else // $P \not\in l_q$
        if ((orientation(sq,T) == p_rel_sq) && 
            (cmp_pnt_line_distance(T,P,sq) < 0) && 
            perp_separates(sq,T,vl,vr)) 
          return true;
        else
          return false;
    }


    break;
  case ft_label:
    P = (*q).rpnt(); 
    sq = (*p).rseg(); 
    vl = ms; 
    vr = mr;
    {
      int p_rel_sq = orientation(sq,P);
      if (p_rel_sq == ON)
        return false;
      else // $P \not\in l_q$
        if ((orientation(sq,T) == p_rel_sq) && 
            (cmp_pnt_line_distance(T,P,sq) < 0) && 
            perp_separates(sq,T,vl,vr)) 
          return true;
        else
          return false;
    }


    break;
  case ff_label: 
    sp = (*p).rseg(); 
    sq = (*q).rseg(); 
    {
        LTRACEN("clipped inner point and ss");
      if (collinear(sp,sq))
      {
          LTRACEN("p and q collinear");
        return false;
      }
      else // $l_{sp} \neq l_{sq}$
      {
          LTRACEN("p and q not collinear");

        int m_rel_sp = orientation(sp,*mr);
        int m_rel_sq = orientation(sq,*mr);
        if (m_rel_sp == 0)
          m_rel_sp =  orientation(sp,*ms);
        if (m_rel_sq == 0)
          m_rel_sq =  orientation(sq,*ms);

        /* Lemma 9 demands that $T$ lies on the same side
           of the lines through |p| and |q| as the nodes
           |mr| and |ms|. However we have to cope with the
           fact, that |mr| (or |ms|) might be on both lines.
           In this case we check the corresponding other node
           as not both nodes can be on both lines */
        rat_segment s_close;
        p_vnode vl,vr;
        if (cmp_seg_pnt_distance(sp,sq,T) < 0)
        {
          s_close = sp;
          vl = ms;
          vr = mr;
        }
        else
        {
          s_close = sq;
          vl = mr;
          vr = ms;
        }

        if (orientation(sp,T) == m_rel_sp && 
            orientation(sq,T) == m_rel_sq && 
            perp_separates(s_close,T,vl,vr)) 
          return true;
        else
          return false;
      }
    }


    break;
  }  
}


bool  ES_Voronoi_Diagram::
total_cut_when_both_affected(p_site p, p_site q, p_site r, p_site s, 
                             p_site t, p_vnode mr, p_vnode ms)
{
    TRACE("  total_cut_when_both_affected ");  
    TRACE(*p);TRACE(*q);TRACE(*r);TRACE(*s);TRACEN(*t);

  rat_point P;
  rat_segment sq;
  switch(sorted_switch_label_b2((*p).is_pnt(),(*q).is_pnt())) {
  case tt_label: 
    return false; // |t| as pnt or seg 
    break;
  case tf_label:
    P = (*p).rpnt(); 
    sq = (*q).rseg();
    {
      if (collinear(sq,P)) // |P| is endpoint of |sq|
      {
        if ((*t).is_seg() && collinear((*t).rseg(),P))
          /* $l_t = l_q$ thus  |t| gets |P|'s Voronoi region */
          return true;
        else
          /* |t| seg tangentially touching the clearance circles
             or |t| is pnt (and equal to |P|) */
          return false;
      }
      else // $P \not\in l_{sq}$
      {
        if ((*t).is_pnt() &&
            cmp_pnt_line_distance((*t).rpnt(),P,sq) < 0)
          return true;
        else 
          /* |t| is pnt and further away from $l_q$ than |P| or equal to |P|
             or |t| is seg  tangentially touching the clearance circles */
          return false;      
      }
    }

    break;
  case ft_label:
    P = (*q).rpnt(); 
    sq = (*p).rseg();
    {
      if (collinear(sq,P)) // |P| is endpoint of |sq|
      {
        if ((*t).is_seg() && collinear((*t).rseg(),P))
          /* $l_t = l_q$ thus  |t| gets |P|'s Voronoi region */
          return true;
        else
          /* |t| seg tangentially touching the clearance circles
             or |t| is pnt (and equal to |P|) */
          return false;
      }
      else // $P \not\in l_{sq}$
      {
        if ((*t).is_pnt() &&
            cmp_pnt_line_distance((*t).rpnt(),P,sq) < 0)
          return true;
        else 
          /* |t| is pnt and further away from $l_q$ than |P| or equal to |P|
             or |t| is seg  tangentially touching the clearance circles */
          return false;      
      }
    }

    break;
  case ff_label: 
    {
      if (collinear((*p).rseg(),(*q).rseg()))
        return false;
      else
        return true;
    }


    break;
  }
}



bool  ES_Voronoi_Diagram::
inner_point_on_edge_remains(p_site p, p_site q, p_site r, p_site s, 
                            p_site t, p_vnode mr, p_vnode ms)
{
    TRACE("  inner_point_on_edge_remains ");  
    TRACE(*p);TRACE(*q);TRACE(*r);TRACE(*s);TRACEN(*t);

  {
    if (p == infinity())
      return (r == s && line_hull_and_order(r,t,q));
    if (q == infinity())
      return (r == s && line_hull_and_order(r,t,p));
  }

  
  rat_point P,Q;
  rat_segment sp,sq;
  p_vnode vr,vs;
  switch(sorted_switch_label_b2((*p).is_pnt(),(*q).is_pnt())) {
  case tt_label:
    P = (*p).rpnt(); 
    Q = (*q).rpnt(); 
    vr = mr; 
    vs = ms;
    {
        LTRACEN("remaining inner point and pp");
      if ((*t).is_pnt())
      {
          LTRACEN("t is pnt");
        return false;
      }
      else // |t| is segment
      {
        rat_segment st = (*t).rseg(); 
        if (collinear(st,P) || collinear(st,Q))
        {
            LTRACEN("p or q on l_t");
          return false;
        }

        int pq_rel_lt = cmp_pnt_line_distance(P,Q,st);
        if (pq_rel_lt == 0)
        {
            LTRACEN("p and q have equal distance from l_t");
          return false;
        }

        if (pq_rel_lt > 0) // |Q| closer to $l_{st}$ than |P|
        {
          leda_swap(P,Q);
          leda_swap(vs,vr);
        }

        /* now P is closer to $l_t$ than Q, note that vs or vr could
           be gamma_vnode when the node lies on $\Gamma$. */
        rat_segment tp_perp(perpendicular_base_point(st,P),P);

          LTRACEN("examination results: ");
          LTRACE(P);LTRACE(Q);LTRACEN(tp_perp); 
          if (vr!=gamma_node()) 
          {LTRACE(*vr);LTRACEN(orientation(tp_perp,*vr));}
          if (vs!=gamma_node()) 
          {LTRACE(*vs);LTRACEN(orientation(tp_perp,*vs));}

        if ((vs==gamma_node() || orientation(tp_perp,*vs) == LEFT) && 
            (vr==gamma_node() || orientation(tp_perp,*vr) == RIGHT))
          return true;
        else
          return false;
      }   
    }

    break;
  case tf_label:
    P = (*p).rpnt(); 
    sq = (*q).rseg(); 
    vr = mr; 
    vs = ms;
    {
        LTRACE("remaining inner point and ps");
      if (is_endpnt(P,sq)) { // |P| endpoint of segment |sq| 
          LTRACEN("P endpoint of q");
        if (t->is_seg() && is_endpnt(P,t->rseg()))
          return false;
        else
          return true; 
      }
      /* now |p| is not in $l_q$ */

      if ((*t).is_pnt())  {
          LTRACEN("t is point");
        if (cmp_pnt_line_distance(P,(*t).rpnt(),sq) < 0 &&
            perp_separates(sq,P,vr,vs))
          return true;
        else // |T| closer to |sq| than |P|
          return false;
      }

      /* now |t| is segment  */
      rat_segment st = (*t).rseg();

      // we have to examine if |P| is endpoint of |t| or not 

      int p_rel_t = orientation(st,P);
      if ((p_rel_t != ON) && perp_separates(st,P,vs,vr))
      {
        /* this separation case cannot happen with $P \in l_t$ 
           and both nodes clipped by |t| */
           LTRACEN("P not on l_t and sep argument");
         return true;
      }
     
      rat_segment tp_perp;
      if (p_rel_t == ON) 
      {
          LTRACEN("P on l_t");
        rat_point To = other_end(st,P);
        tp_perp = rat_segment(To.rotate90(P),P);       
        // by this construction the nodes should be left  
      } 
      else 
      { 
          LTRACEN("P not on l_t");
        tp_perp = rat_segment(perpendicular_base_point(st,P),P);
        if (orientation(tp_perp,*vr) == RIGHT)
          invert_seg(tp_perp);          
        if (orientation(tp_perp,*vs) == RIGHT)
          return false;
          // now we can guarantee that the nodes are both left of $l_t$
      }
      if (orientation(sq,*vr) == RIGHT)
        invert_seg(sq);             
        // also the nodes are left of $l_q$ now

      if (perp_separates(sq,P,vr,vs) && 
          less_than_right_angle(sq,tp_perp))
        return true;
      return false;        
    }


    break;
  case ft_label:
    P = (*q).rpnt(); 
    sq = (*p).rseg(); 
    vr = ms; 
    vs = mr; 
    {
        LTRACE("remaining inner point and ps");
      if (is_endpnt(P,sq)) { // |P| endpoint of segment |sq| 
          LTRACEN("P endpoint of q");
        if (t->is_seg() && is_endpnt(P,t->rseg()))
          return false;
        else
          return true; 
      }
      /* now |p| is not in $l_q$ */

      if ((*t).is_pnt())  {
          LTRACEN("t is point");
        if (cmp_pnt_line_distance(P,(*t).rpnt(),sq) < 0 &&
            perp_separates(sq,P,vr,vs))
          return true;
        else // |T| closer to |sq| than |P|
          return false;
      }

      /* now |t| is segment  */
      rat_segment st = (*t).rseg();

      // we have to examine if |P| is endpoint of |t| or not 

      int p_rel_t = orientation(st,P);
      if ((p_rel_t != ON) && perp_separates(st,P,vs,vr))
      {
        /* this separation case cannot happen with $P \in l_t$ 
           and both nodes clipped by |t| */
           LTRACEN("P not on l_t and sep argument");
         return true;
      }
     
      rat_segment tp_perp;
      if (p_rel_t == ON) 
      {
          LTRACEN("P on l_t");
        rat_point To = other_end(st,P);
        tp_perp = rat_segment(To.rotate90(P),P);       
        // by this construction the nodes should be left  
      } 
      else 
      { 
          LTRACEN("P not on l_t");
        tp_perp = rat_segment(perpendicular_base_point(st,P),P);
        if (orientation(tp_perp,*vr) == RIGHT)
          invert_seg(tp_perp);          
        if (orientation(tp_perp,*vs) == RIGHT)
          return false;
          // now we can guarantee that the nodes are both left of $l_t$
      }
      if (orientation(sq,*vr) == RIGHT)
        invert_seg(sq);             
        // also the nodes are left of $l_q$ now

      if (perp_separates(sq,P,vr,vs) && 
          less_than_right_angle(sq,tp_perp))
        return true;
      return false;        
    }


    break;
  case ff_label:
    sp = (*p).rseg(); 
    sq = (*q).rseg();
    {
      if (collinear(sp,sq))
        return true;
      else
        return false;
    }


    break;
  }
}



bool  ES_Voronoi_Diagram::
edge_is_intersected(p_site p, p_site q, p_site r, p_site s, p_site t, 
                    p_vnode mr, p_vnode ms, bool& error)
{
    TRACEN(">>edge_is_intersected: ");
    TRACE(*p);TRACE(*q);TRACE(*r);TRACE(*s);TRACEN(*t);

  {
    error = (illegal_intersection(p,t) || illegal_intersection(q,t) || 
             illegal_intersection(r,t) || illegal_intersection(s,t));
  }



    TRACE(" error status: ");TRACEN(error);

  if (error)
    return false;

  if ((*t).is_seg())
  /* in case that |t| is a segment we have to look first for a
     conflict with one of its endpoints, as we might hit edges
     within the history graph where the endpoints are not yet 
     inserted; if there is none we check with the interior */
  {
      TRACEN(" endpoint check first");
      site epnt1((*t).rseg().start(),false);
      site epnt2((*t).rseg().end(),false);
      if (edge_is_intersected(p,q,r,s,p_site(epnt1),mr,ms,error) || 
          edge_is_intersected(p,q,r,s,p_site(epnt2),mr,ms,error)) {
        /* there can be no error condition here, as endpoints
           are inserted before the interior and the error would
           have stopped the insertion there */
          TRACEN(" endpoint check already shows conflict");
        return true;
      }
  }

  node_status pqr_status = calculate_node_status(p,q,r,t,mr);
  node_status psq_status = calculate_node_status(q,p,s,t,ms);
  bool result;

    TRACE(" node stati: ");TRACE(pqr_status);TRACEN(psq_status);

  /* in the following case switching the labels determine
     U (unaffected), A (affected), C (clipped) */
  switch(sorted_switch_label_i23(pqr_status,psq_status)){
  case UC:
  case CU:
  case AC:
  case CA: 
  case CC: 
    result = true; 
    break;
  case UA: 
  case AU: 
  case UU: 
    result = inner_point_on_edge_is_clipped(p,q,r,s,t,mr,ms);
    break;
  case AA: 
    result = total_cut_when_both_affected(p,q,r,s,t,mr,ms);
    break;
  }  

    TRACE("return value = ");TRACEN(result);
  return result;
}


node_status  ES_Voronoi_Diagram::
calculate_node_status(p_site p, p_site q, p_site r, p_site t, p_vnode mr)
{
  {
    if (p == infinity())
      return infinite_node_status(q,r,t);
    if (q == infinity())
      return infinite_node_status(r,p,t);
    if (r == infinity())
      return infinite_node_status(p,q,t);
  }


  if ((*t).is_pnt())
    {
      int sgn_res = side_of_circle(mr,(*t).rpnt());
      switch (sgn_res) {
        case IN:  return CLIPPED;    break;
        case ON:  return AFFECTED;   break;
        case OUT: return UNAFFECTED; break;
      }
    }


  else
    {
        LTRACE("finite node status with |t| segment ");
        LTRACE(*mr);
        LTRACEN(*t);

      int sign_res;
      integer at,bt,ct;
      seg2line_paras((*t).rseg(),at,bt,ct);

      if ((*mr).nt() > 2) { // nt != SSS_xxx
        rat_point P = mr->csite(2).rpnt();

        CASESTAT(mr,t);

        {
          vnode& vr = *mr;
          BEGIN_PREDICATE
            {
              DECLARE_ATTRIBUTES poly_rid_type good_sup FOR vr.x() vr.y() vr.w();
              DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y();
              DECLARE_ATTRIBUTES integer_type FOR at bt ct;
              DECLARE_INDEX 42 FOR vr.x() vr.y() vr.w();
              DECLARE_INTFLAG vr.IsRat();

              real res = sq(at*vr.x() + bt*vr.y() + ct*vr.w()) -
                (sq(at)+sq(bt))*(sq(vr.x() - P.X()*vr.w()) + sq(vr.y() - P.Y()*vr.w()));
              sign_res=sign(res);
            }
          END_PREDICATE
        }

          ZEROSTAT(mr,t,sign_res);
          LTRACE("point compare (l_t-circle-pos) = ");LTRACEN(sign(res));
      }
      else { // we only have segments
        integer a1,b1,c1;
        seg2line_paras((*mr).csite(0).rseg(),a1,b1,c1);

          CASESTAT(mr,t);

        {
          vnode& vr = *mr;
          BEGIN_PREDICATE
            {
              DECLARE_ATTRIBUTES poly_rid_type good_sup FOR vr.x() vr.y() vr.w();
              DECLARE_ATTRIBUTES integer_type FOR a1 b1 c1;
              DECLARE_ATTRIBUTES integer_type FOR at bt ct;
              DECLARE_INDEX 42 FOR vr.x() vr.y() vr.w();
              DECLARE_INTFLAG vr.IsRat();
              real res = sq(at*vr.x() + bt*vr.y() + ct*vr.w()) * (sq(a1)+sq(b1)) -
                sq(a1*vr.x() + b1*vr.y() + c1*vr.w()) * (sq(at)+sq(bt));
              sign_res=sign(res);
            }
          END_PREDICATE
        }

          ZEROSTAT(mr,t,sign_res);
          LTRACE("all seg compare (l_t-circle-pos) = ");LTRACEN(sign(res));
      }
      // now we know if the line supporting |t| intersects our circle
      // accuracy adaptation needed ?

      switch (sign_res) {
        case IN: // the line $l_t$ intersects the circle
          if (side_of_circle(mr,(*t).rseg().start()) < 0 ||
              side_of_circle(mr,(*t).rseg().end()) < 0 ||
              is_in_end_perp_stripe((*t).rseg(),mr))
            return CLIPPED;
          else
            return UNAFFECTED;
          break;
        case ON:  // the line $l_t$ touches the circle
          if (is_in_end_perp_stripe((*t).rseg(),mr))
            return AFFECTED;
          else
            return UNAFFECTED;
          break;
        case OUT: // the line $l_t$ passes the circle 
          return UNAFFECTED;
          break;
      }
    }

  error_handler(1,"calculate_node_status: should not end here");
}


node_status  ES_Voronoi_Diagram::
infinite_node_status(p_site p, p_site q, p_site t)
{
    TRACE("  infinite_node_status ");
    TRACE(*p);TRACE(*q);TRACE(*t);

  node_status result;

  switch(sorted_switch_label_b2((*p).is_pnt(),(*q).is_pnt())) {
  case tt_label: 
      ASSERT(((*p).rpnt()!=(*q).rpnt()),infinite_node_status);

    if ((*t).is_pnt())
    {
      /* the infinity node disappears if
         - |t| is right of $\overline{pq}$
         - |t| is in the interior of $\overline{pq}$ */

      int pqt_or = orientation((*p).rpnt(),(*q).rpnt(),(*t).rpnt());

      if (pqt_or == RIGHT || 
          (pqt_or == ON && 
           collinear_ordered((*p).rpnt(),(*t).rpnt(),(*q).rpnt())))
        result = CLIPPED;
      else
        result = UNAFFECTED;
    }
    else // |(*t).is_seg()|
    {
      /* the infinity node disappears if
         - one of the endpoints is right of $\overline{pq}$
         - one of the endpoints is in the interior of $\overline{pq}$
         - the segment is the interior of $\overline{pq}$ */

      rat_point Ts = (*t).rseg().start();
      rat_point Te = (*t).rseg().end();
      int pqtstart_or = orientation((*p).rpnt(),(*q).rpnt(),Ts);
      int pqtend_or = orientation((*p).rpnt(),(*q).rpnt(),Te);
      if (pqtstart_or == RIGHT || pqtend_or == RIGHT || 
          (pqtstart_or == ON && collinear_ordered((*p).rpnt(),Ts,(*q).rpnt())) || 
          (pqtend_or == ON && collinear_ordered((*p).rpnt(),Te,(*q).rpnt())) || 
          (is_endpnt((*p).rpnt(),(*t).rseg()) && 
           is_endpnt((*q).rpnt(),(*t).rseg())))
        result = CLIPPED;
      else
        result = UNAFFECTED;
    }
    break;

  case tf_label: 
      ASSERT((is_endpnt((*p).rpnt(),(*q).rseg())),infinite_node_status);
    if ((*t).is_pnt())
    {
      rat_point P = (*p).rpnt();
      rat_point Q = other_end((*q).rseg(),P);
      if (right_turn(P,Q,(*t).rpnt()))
        result = CLIPPED;
      else
        result = UNAFFECTED;
    }
    else // |(*t).is_seg()|
    {
      rat_point P = (*p).rpnt();
      rat_point Q = other_end((*q).rseg(),P);
      rat_point Ts = (*t).rseg().start();
      rat_point Te = (*t).rseg().end();
      int pqtstart_or = orientation(P,Q,Ts);
      int pqtend_or = orientation(P,Q,Te);
      if (pqtstart_or == RIGHT || pqtend_or == RIGHT || 
          (pqtstart_or == ON && pqtend_or == ON && 
           is_endpnt(P,(*t).rseg())))
        result = CLIPPED;
      else
        result = UNAFFECTED;       
    }
    break;

  case ft_label: 
      ASSERT((is_endpnt((*q).rpnt(),(*p).rseg())),infinite_node_status);
    if ((*t).is_pnt())
    {
      rat_point Q = (*q).rpnt();
      rat_point P = other_end((*p).rseg(),Q);
      if (right_turn(P,Q,(*t).rpnt()))
        result = CLIPPED;
      else
        result = UNAFFECTED;
    }
    else // |(*t).is_seg()|
    {
      rat_point Q = (*q).rpnt();
      rat_point P = other_end((*p).rseg(),Q);
      rat_point Ts = (*t).rseg().start();
      rat_point Te = (*t).rseg().end();
      int pqtstart_or = orientation(P,Q,Ts);
      int pqtend_or = orientation(P,Q,Te);
      if (pqtstart_or == RIGHT || pqtend_or == RIGHT || 
          (pqtstart_or == ON && pqtend_or == ON && 
           is_endpnt(Q,(*t).rseg())))
        result = CLIPPED;
      else
        result = UNAFFECTED;       
    }
    break;

  case ff_label: 
      ASSERT((collinear((*p).rseg(),(*q).rseg()) && 
              (is_endpnt((*p).rseg().start(),(*q).rseg()) || 
               is_endpnt((*p).rseg().end(),(*q).rseg()))), 
             infinite_node_status);

    if ((*t).is_pnt())
    {
      rat_point P = (*p).rseg().start();
      rat_point Q = other_end((*q).rseg(),P);
      if (right_turn(P,Q,(*t).rpnt()))
        result = CLIPPED;
      else
        result = UNAFFECTED;
    }
    else // |(*t).is_seg()|
    {
      rat_point P = (*p).rseg().start();
      rat_point Q = other_end((*q).rseg(),P);
      rat_point Ts = (*t).rseg().start();
      rat_point Te = (*t).rseg().end();
      int pqtstart_or = orientation(P,Q,Ts);
      int pqtend_or = orientation(P,Q,Te);
      if (pqtstart_or == RIGHT || pqtend_or == RIGHT)
        result = CLIPPED;
      else
        result = UNAFFECTED;       
    }
    break;
  }  

    TRACEN(result);
  return result;
}


int ES_Voronoi_Diagram::
side_of_circle(p_vnode vn, const rat_point& T)
{
    LTRACE("side_of_circle ");
    LTRACE(*vn);
    LTRACEN(T);

  int sign_res = NO_IDEA;
  vnode& v = *vn;
  node_type nt=v.nt();

  if (nt==PPP) {
    if (T==(v.csite(2).rpnt())) sign_res=0;
    if (T==(v.csite(1).rpnt())) sign_res=0;
    if (T==(v.csite(0).rpnt())) sign_res=0;

  }
  if (nt==SPP_distdiff || nt==SPP_disteq) {
    if (T==(v.csite(2).rpnt())) sign_res=0;
    if (T==(v.csite(1).rpnt())) sign_res=0;
  }
  if (nt==SSP_alltouch || nt==SSP_notcoll || nt==SSP_coltouch) {
    if (T==(v.csite(2).rpnt())) sign_res=0;
  }

  if (sign_res!=NO_IDEA) {
    LTRACE("Jump out ");LTRACEN(v.nt());
    return sign_res;
  } else {
    LTRACE("No jump: ");LTRACE((T==(v.csite(2).rpnt())));LTRACEN(v.nt());
  }

  if (v.nt() > 2) { // != SSS_xxx
    rat_point P=v.csite(2).rpnt();
      CASESTAT(vn,t);
    {
    BEGIN_PREDICATE
    {
    DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() T.X() T.Y();
    DECLARE_ATTRIBUTES poly_rid_type good_sup FOR v.x() v.y() v.w();
    DECLARE_INDEX 42 FOR v.x() v.y() v.w();
    DECLARE_INTFLAG v.IsRat();
      real res;
      res = v.w() * (sq(T.X())+sq(T.Y())-sq(P.X())-sq(P.Y())) + 
            2 * (v.x()*(P.X()-T.X()) + v.y()*(P.Y()-T.Y()));
      sign_res=sign(res);
    }
    END_PREDICATE
    }
      ZEROSTAT(vn,t,sign_res);
  } 
  else { // SSS comparison follows (we have no point on the circle)
    integer a1,b1,c1;
    seg2line_paras(v.csite(0).rseg(),a1,b1,c1);
      CASESTAT(vn,t);
    {
    BEGIN_PREDICATE
    {
    DECLARE_ATTRIBUTES integer_type FOR a1 b1 c1;
    DECLARE_ATTRIBUTES poly_id_type FOR T.X() T.Y();
    DECLARE_ATTRIBUTES poly_rid_type good_sup FOR v.x() v.y() v.w();
    DECLARE_INDEX 18 FOR v.x() v.y() v.w();
    DECLARE_INTFLAG v.IsRat();
      real res = (sq(a1)+sq(b1))*(sq(v.x() - T.X()*v.w()) + 
                                  sq(v.y() - T.Y()*v.w())) - 
                 sq(a1*v.x() + b1*v.y() + c1*v.w());
      sign_res=sign(res);
    }
    END_PREDICATE
    }
      ZEROSTAT(vn,t,sign_res);
  }
  return sign_res;
}




bool ES_Voronoi_Diagram::
is_in_end_perp_stripe(const rat_segment& s, p_vnode vn)
{
    ASSERT((s.W1()==1),is_in_end_perp_stripe);
    ASSERT((s.W2()==1),is_in_end_perp_stripe);
    LTRACEN("is_in_end_perp_stripe: ");

  vnode& v = *vn;
  integer a,b,c;
  seg2line_paras(s,a,b,c);

  int sgn_s1, sgn_s2;

  rat_point P=s.start();
  rat_point Q=s.end();

  BEGIN_PREDICATE
    {
      DECLARE_ATTRIBUTES poly_rid_type good_sup FOR v.x() v.y() v.w();
      DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y();
      DECLARE_INDEX 42 FOR v.x() v.y() v.w();
      DECLARE_ATTRIBUTES integer_type FOR a b c;
      DECLARE_INTFLAG v.IsRat();
      real s1=b*v.x() - a*v.y() + (a*P.Y()-b*P.X())*v.w();
      sgn_s1=sign(s1);
    }
  END_PREDICATE
    LTRACE("first sign: ");LTRACEN(sgn_s1);

  BEGIN_PREDICATE
    {
      DECLARE_ATTRIBUTES poly_rid_type good_sup FOR v.x() v.y() v.w();
      DECLARE_ATTRIBUTES poly_id_type FOR Q.X() Q.Y();
      DECLARE_INDEX 42 FOR v.x() v.y() v.w();
      DECLARE_ATTRIBUTES integer_type FOR a b c;
      DECLARE_INTFLAG v.IsRat();
      real s2 = b*v.x() - a*v.y() + (a*Q.Y()-b*Q.X())*v.w();
      sgn_s2=sign(s2);
    }
  END_PREDICATE
    LTRACE("second sign: ");LTRACEN(sgn_s2);

  return (sgn_s1 != sgn_s2);
  /* if |(sgn_s1 == sgn_s2)| the perpendicular through |vn|
     intersects $l_s$ outside of |s| */
}


bool  ES_Voronoi_Diagram::
line_hull_and_order(p_site pp1, p_site pp2, p_site pp3)
{
    TRACE("  line_hull_and_order");TRACE(*pp1);TRACE(*pp2);TRACE(*pp3);

  rat_segment p12dir,s1,s2;
  const site& p1 = *pp1;
  const site& p2 = *pp2;
  const site& p3 = *pp3;
  rat_point P1,P2,P3;
  bool result;

    switch(sorted_switch_label_b2(p1.is_pnt(),p2.is_pnt())) {
    case tt_label: 
      P1 = p1.rpnt();
      P2 = p2.rpnt();
      p12dir = rat_segment(P1,P2);
      /* now |P1| and |P2| span the hull */

      if (p3.is_pnt()) 
      {
        result = (collinear(p12dir,p3.rpnt()) && 
                  collinear_ordered(P1,P2,p3.rpnt()));
      }
      else // |p3.is_seg()|
        result = (collinear(p12dir,p3.rseg()) && 
                  collinear_ordered(P1,P2,other_end(p3.rseg(),P2)));
      break;
    case tf_label: 
      P1 = p1.rpnt();
      s2 = p2.rseg();
      if (!collinear(s2,P1))
        result = false;
      else /* now |P1| is on $l_{s2}$ */
      {
        if (collinear_ordered(P1,s2.end(),s2.start()) || P1 == s2.end())
          invert_seg(s2);
        P2 = s2.end();
        p12dir = rat_segment(P1,P2);
        /* now |P1| and |P2| span the hull */

        if (p3.is_pnt()) 
          result = (collinear(p12dir,p3.rpnt()) && 
                    (collinear_ordered(P1,P2,p3.rpnt()) || 
                     P2 == p3.rpnt()));
        else // |p3.is_seg()|
          result = (collinear(p12dir,p3.rseg()) && 
                    collinear_ordered(P1,P2,other_end(p3.rseg(),P2)));
      }
      break;
    case ft_label: 
      s1 = p1.rseg();
      P2 = p2.rpnt();
      if (!collinear(s1,P2))
        result =  false;
      else /* now |P2| is on $l_{s1}$ */
      {
        if (collinear_ordered(s1.end(),s1.start(),P2) || s1.start() == P2)
          invert_seg(s1);
        P1 = s1.start();
        p12dir = rat_segment(P1,P2);
        /* now |P1| and |P2| span the hull */

        if (p3.is_pnt()) 
          result = (collinear(p12dir,p3.rpnt()) && 
                    collinear_ordered(P1,P2,p3.rpnt()));
        else // |p3.is_seg()|
          result = (collinear(p12dir,p3.rseg()) && 
                    collinear_ordered(P1,P2,other_end(p3.rseg(),P2)));
      }
      break;
    case ff_label: 
      s1 = p1.rseg();
      s2 = p2.rseg();
      if (!collinear(s1,s2))
        result = false;
      else  /* now $l_{s1} = l_{s2}$ */
      {
        if (collinear_ordered(s1.end(),s1.start(),other_end(s2,s1.start())))
          invert_seg(s1);
        if (collinear_ordered(other_end(s1,s2.end()),s2.end(),s2.start()))
          invert_seg(s2);
        P1 = s1.start();
        P2 = s2.end();
        p12dir = rat_segment(P1,P2);
        /* now |P1| and |P2| span the hull */

        if (p3.is_pnt()) 
          result = (collinear(p12dir,p3.rpnt()) && 
                    (collinear_ordered(P1,P2,p3.rpnt()) || 
                     P2 == p3.rpnt()));
        else // |p3.is_seg()|
          result = (collinear(p12dir,p3.rseg()) && 
                    collinear_ordered(P1,P2,other_end(p3.rseg(),P2)));
      }
      break;
    }  

      TRACEN(result);
    return result;
}



bool ES_Voronoi_Diagram::
illegal_intersection(p_site pp1, p_site pp2)
{
  if (pp1 == infinity()) return false;
  if (pp2 == infinity()) return false;

  const site& p1 = *pp1;
  const site& p2 = *pp2;
  rat_point P1,P2,P3,P4;

  if (p1.type() == p2.type())
  {
    if (p1.is_pnt())
      return false;  // two points can't have illegal position
    else
      if (segments_intersect_internally(p1.rseg(),p2.rseg()))
      {
          TRACEN("illegal_intersection: segments intersect internally!");
          return true;
      }
  }

  // now the mixed cases:
  if (p1.is_seg()) // and |p2.is_pnt()|
  {
    P1 = p1.rseg().start();
    P2 = p1.rseg().end();
    P3 = p2.rpnt();
  }
  if (p2.is_seg()) // and |p1.is_pnt()|
  {
    P1 = p2.rseg().start();
    P2 = p2.rseg().end();
    P3 = p1.rpnt();
  }

  int p3_rel_p12 = orientation(P1,P2,P3);
  if ((p3_rel_p12 == ON) && collinear_ordered(P1,P3,P2))
  {
    // |P3| on the segment between |P1| and |P2| is illegal
      TRACEN("illegal_intersection: point in interior of segment!"); 
    return true;
  }
  return false;
 
}



int  ES_Voronoi_Diagram::
orientation(const rat_segment& s, vnode& vn)
{
  rat_point P=s.start();
  rat_point Q=s.end();

  int sign_res;

  BEGIN_PREDICATE
  {
    DECLARE_ATTRIBUTES poly_id_type FOR P.X() P.Y() P.W() Q.X() Q.Y() Q.W();
    DECLARE_ATTRIBUTES poly_rid_type good_sup FOR vn.x() vn.y() vn.w();
    DECLARE_INDEX 42 FOR vn.x() vn.y() vn.w();
    DECLARE_INTFLAG vn.IsRat();
    real res = (P.X()*Q.W()-Q.X()*P.W()) * (P.Y()*vn.w()-vn.y()*P.W()) - 
               (P.Y()*Q.W()-Q.Y()*P.W()) * (P.X()*vn.w()-vn.x()*P.W());
    sign_res=sign(res);
  }
  END_PREDICATE

  return (sign_res);
}



bool  ES_Voronoi_Diagram::
less_than_right_angle(const rat_segment& s1, const rat_segment& s2)
{
  integer dx1 = s1.X2()*s1.W1() - s1.X1()*s1.W2();
  integer dy1 = s1.Y2()*s1.W1() - s1.Y1()*s1.W2();
  integer dx2 = s2.X2()*s2.W1() - s2.X1()*s2.W2();
  integer dy2 = s2.Y2()*s2.W1() - s2.Y1()*s2.W2();
  return ((dx1*dx2 + dy1*dy2) > 0);
}



bool  ES_Voronoi_Diagram::
perp_separates(const rat_segment& l, const rat_point& P, 
               p_vnode ml, p_vnode mr)
{
  rat_point B_l = perpendicular_base_point(l,P);
  rat_segment perp(B_l,P);
  return (orientation(perp,*ml) == LEFT && 
          orientation(perp,*mr) == RIGHT);
}



void ES_Voronoi_Diagram::
adapt_seg(segment& s, bisec_type& t, p_vnode si, p_vnode ei)
{
    TRACE("    adapt_seg: ");TRACEN(s);
    
  switch(sorted_switch_label_b2((si != gamma_node()),(ei != gamma_node()))) {
  case tt_label: // both not on gamma
    s = segment((*si).pnt(),(*ei).pnt());
    t = SEGM;    
    break;
  case tf_label: // only ei on gamma
    move_seg(s,(*si).pnt());
    t = RAY;
    break;
  case ft_label: // only si on gamma
    invert_seg(s);
    move_seg(s,(*ei).pnt());
    t = RAY;
    break;
  case ff_label: // both on gamma
    t = LIN;
    break;  
  }
    TRACE("    adapted: ");TRACEN(s);
}




vnode ES_Voronoi_Diagram::
other_end(p_site p, vnode& v)
{
  if ((*p).is_pnt())
    return v;
  else { // |(*p).is_seg()|
    rat_point PS((*p).rseg().start());
    rat_point PE((*p).rseg().end());
    vnode ps(PS.X(),PS.Y(),PS.W());
    vnode pe(PE.X(),PE.Y(),PE.W());
    if (ps == v)
      return pe;
    else
      return ps;
  }
}




inline bool in_window(window& W, point p)
{
  return (p.xcoord() >= W.xmin() && p.xcoord() <= W.xmax() &&
          p.ycoord() >= W.ymin() && p.ycoord() <= W.ymax());
}

inline bool crossing_window_border(window& W, segment s)
{
  if (in_window(W,s.source()) && !in_window(W,s.target()) ||
      !in_window(W,s.source()) && in_window(W,s.target()))
    return true;
}

void  ES_Voronoi_Diagram::
draw_bisector(p_site pp, p_site pq, p_vnode si, p_vnode ei, window& W, 
              color c)
{
    TRACEN("draw_bisector:");

  bisec_type t;    // the type of the bisector piece;
  segment s;       // start and end point on the bisector
  point   Fo;      // the focus if PARA
  segment ba;      // the baseline if PARA 
  
  // first we analyze the type and store it in t

  if (pp == infinity() || pq == infinity()) {
    t = GAMMA;
    return;
  }

  const site& p = *pp;
  const site& q = *pq;
  // and now the finite cases:
  switch(sorted_switch_label_b2(p.is_pnt(),q.is_pnt())) {
  case tt_label: // both are pnts
    s = mid_perp_seg(p.pnt(),q.pnt());
    adapt_seg(s,t,si,ei);
    break;

  case tf_label: 
    // we know that if (point) p lies on lq then p must be an
    // endpoint of (segment) q as endpoints are inserted before
    // their segment interior
    if (p.rpnt() == q.rseg().start())
    {
      s = end_perp_seg(q.seg().start(),q.seg().end());
      adapt_seg(s,t,si,ei);
    }
    else if (p.rpnt() == q.rseg().end())
    {
      s = end_perp_seg(q.seg().end(),q.seg().start());
      adapt_seg(s,t,si,ei);
    }
    else 
    {
      // we know that in this case si and ei must have finite coordinates
      // and limit a parabola piece
      s = segment((*si).pnt(),(*ei).pnt());
      Fo = p.pnt();
      ba = q.seg();
      t = PARA;
    }
    break;

  case ft_label: 
    // we know that if q lies on lp then q must be an
    // endpoint of p as endpoints are inserted before
    // their segment interior
    if (q.rpnt() == p.rseg().start())
    {
      s = end_perp_seg(p.seg().start(),p.seg().end());
      invert_seg(s);
      adapt_seg(s,t,si,ei);
    }
    else if (q.rpnt() == p.rseg().end())
    {
      s = end_perp_seg(p.seg().end(),p.seg().start());
      invert_seg(s);
      adapt_seg(s,t,si,ei);
    }
    else 
    {
      // we know in this case that si and ei must have finite coordinates
      // and limit a parabola piece 
      s = segment((*ei).pnt(),(*si).pnt());
      Fo = q.pnt();
      ba = p.seg();
      t = PARA;
    }
    break;

  case ff_label: 
    // it might happen that the segments |p| and |q| are collinear
    // and touch in an endpoint, if this is not the case then
    // we know that si and ei must have finite coordinates
    // and limit a line segment
    if (collinear(p.rseg(),q.rseg()))
    {
        ASSERT(is_endpnt(p.rseg().start(),q.rseg())||
               is_endpnt(p.rseg().end(),q.rseg()),
               bisec_elem_collinear_segments);

      if (is_endpnt(p.rseg().start(),q.rseg()))
        s = end_perp_seg(p.seg().start(),p.seg().end());
      else // |is_endpnt(p.rseg().end(),q.rseg())|
        s = end_perp_seg(p.seg().end(),p.seg().start());
      invert_seg(s);
      adapt_seg(s,t,si,ei);
    }
    else
    {      
      s = segment((*si).pnt(),(*ei).pnt());
      t = SEGM;
    }
    break;
  }  

  // then we draw it:
  switch (t) {
  case GAMMA: TRACEN("   GAMMA");
    break;
  case PARA:  TRACE("   PARA ");TRACE(s);TRACE(Fo);TRACEN(ba);
    draw_parabola(Fo,ba,s,W,c);
    break;
  case SEGM:  TRACE("   SEGMENT ");TRACEN(s);
    //if (crossing_window_border(W,s)) {
    //  cerr << "SEGMENT: " << s << endl;
    //  W.draw_segment(s,red);
    //} else
    W.draw_segment(s,c);
    break;
  case RAY:   TRACE("   RAY ");TRACEN(s);
    W.draw_ray(s,c);
    break;
  case LIN:   TRACE("   LINE ");TRACEN(s);
    W.draw_line(s,c);
    break;
  }
}




void  ES_Voronoi_Diagram::
draw_site(p_site p, window& W, color c)
{ (*p).draw(W,c); }




bool ES_Voronoi_Diagram::
right_turn(vnode& p1, vnode& p2, vnode& p3)
{
  int sign_D;

  BEGIN_PREDICATE
    {
      DECLARE_ATTRIBUTES poly_rid_type good_sup FOR 
      p1.x() p1.y() p1.w() p2.x() p2.y() p2.w() p3.x() p3.y() p3.w();
      DECLARE_INDEX 42 FOR 
      p1.x() p1.y() p1.w() p2.x() p2.y() p2.w() p3.x() p3.y() p3.w();
      real D = (p1.x()*p2.w()-p2.x()*p1.w()) * (p1.y()*p3.w()-p3.y()*p1.w()) - 
               (p1.y()*p2.w()-p2.y()*p1.w()) * (p1.x()*p3.w()-p3.x()*p1.w());
      sign_D = sign(D);
    }
  END_PREDICATE
  //if (sign_D==0) cout<<"RightTurn "<<endl;
  return (sign_D < 0);
}


vnode ES_Voronoi_Diagram::
foot_of_perpendicular(vnode& v, const rat_segment& l)
{
    TRACEN("foot_of_perpendicular: ");
  real a,b,c;
  seg2line_paras(l,a,b,c);
  real vx = v.x();
  real vy = v.y();
  real vz = v.w();

  real x = - vy*a*b + vx*b*b - a*c*vz;
  real y = - vx*a*b + vy*a*a - b*c*vz;
  real z = vz*(sq(a)+sq(b));
  // according to C.Burnikel
  vnode vfp(x,y,z);
    TRACEN(vfp);

  return vfp;
}


#undef site
#undef vnode
#undef p_site
#undef p_vnode


/* --------------------------------
          end of file
   -------------------------------- */


