
#include <LEDA/rat_segment.h>
#include <LEP/avd/ep_adaptation.h>

#define p_vnode p_handle<rat_point>
#define p_site p_handle<rat_point>

//------------------------------------------------------------------------------
// edge_intersection_topology()
// calculates if the edge psqr is intersected by the new site t and returns a
// container which holds the calculated result.
// parameters:
// p_site p,q,r,s       the sites which build the edge 
// p_site t             the new site
// p_vnode mr,ms                the node coordinates of the pqr and psq nodes
// bool& error          an error flag in case of object restrictions
// return value:
// one of the topology values
//------------------------------------------------------------------------------

topology  EP_Voronoi_Diagram::
edge_intersection_topology(p_site p, p_site q, p_site r, p_site s, 
                           p_site t, p_vnode mr, p_vnode ms, bool& error)
{
    TRACE("    edge_intersection_topology: ");
    EPD(p);EPD(q);EPD(r);EPD(s);TRACEN("");

  bool source_clipped = clipped_at_node(p,q,r,s,t,mr,ms,error);
    TRACE("    clipped at node return: "); TRACE((int)source_clipped); TRACEN("");

  bool target_clipped = clipped_at_node(q,p,s,r,t,ms,mr,error);
    TRACE("    clipped at node return: "); TRACE((int)target_clipped); TRACEN("");

  if (error) return NO_INTERSECTION;
  // error in case that the new object t has an illegal character compared
  // to the four objects determining the edge.

  if (source_clipped != target_clipped)
  {
       TRACEN("    source_clipped != target_clipped");
    if (source_clipped) 
      return SOURCE_SEGMENT;
    else
      return TARGET_SEGMENT;
  }
  else // both intersected or both not intersected
  {
    bool new_node_exists = new_node_on_edge(p,q,r,s,t,mr,ms);
    bool both_clipped = (source_clipped && target_clipped);

      TRACEN("    both intersected or both not intersected: ");
      TRACE("    s_i / t_i / n_n_e ");
      TRACE((int)source_clipped);
      TRACE(" ");
      TRACE((int)target_clipped);
      TRACE(" ");
      TRACE((int)new_node_exists);
      TRACEN(" ");

    if (new_node_exists && both_clipped) 
      return TWO_SEGMENTS;
    if (new_node_exists && !both_clipped) 
      return INTERN;
    if (!new_node_exists && both_clipped) 
      return TOTAL;
  }
  return NO_INTERSECTION;
}

//------------------------------------------------------------------------------
// edge_is_intersected()
// calculates if the edge psqr is intersected by the new site t.
// parameters:
// p_site p,q,r,s       the node containing the edge information
// p_site t             the new site
// bool& error          an error flag in case of object restrictions
// return value:
// true         if the edge is intersected
// false                else
//------------------------------------------------------------------------------

bool  EP_Voronoi_Diagram::
edge_is_intersected(p_site p, p_site q, p_site r, p_site s, p_site t, 
                    p_vnode mr, p_vnode ms, bool& error)
{
    TRACEN("edge_is_intersected: ");

  bool intersected;
  intersected = clipped_at_node(p,q,r,s,t,mr,ms,error);
  if (!(error || intersected))
    intersected = clipped_at_node(q,p,s,r,t,ms,mr,error);
  if (!(error || intersected))
    intersected = new_node_on_edge(p,q,r,s,t,mr,ms);
  return intersected;
}

//------------------------------------------------------------------------------
// init_is_possible()
// calculates if the initialization of the Voronoi digramm with the two sites
// t1 and t2 is possible
// parameters:
// p_site t1,t2         the two sides which might contribute to the initial
//                      diagram
// bool& error          a flag which shows if the two sides intersect illegally
// return value:
// true                 if V({t1,t2,infinity()}) exists
// false                else
//------------------------------------------------------------------------------

bool  EP_Voronoi_Diagram::
init_is_possible(p_site t1, p_site t2, bool& error)
{ 
  return pqr_in_V(t1,t2,infinity(),error);
}


//------------------------------------------------------------------------------
// clipped_at_node()
// calculates if the edge <p,q,r,s> is clipped at the node <p,q,r>, when the new
// site t is inserted into the Voronoi Diagram.
// parameters:
// p_site p,q,r,s       the sites defining the edge 
// p_site t             the new site
// p_vnode mr,ms        the coordinates of the end points of finite edges
// bool& error  an error flag in case of object restrictions
// return value:
// true         if the edge is clipped at the examined edge end
// false        else
//------------------------------------------------------------------------------

bool EP_Voronoi_Diagram::
clipped_at_node(p_site p, p_site q, p_site r, p_site s, p_site t, 
                p_vnode mr, p_vnode ms, bool& error)
{
    TRACE("    clipped_at_node: ");EPD(p);EPD(q);EPD(r);EPD(s);EPD(t);
    TRACEN("");

  if (!pqr_in_V(p,q,r,t,mr,error))
  {
    if (error) return false;
    //  return without examination in case of illegal object condition

      TRACEN("      pqr does not exist");

    if (p==infinity())
      return (iqt_between_iqr_and_isq(q,r,s,t) != ID_12);
    else if (q==infinity())
      return (itq_between_iqr_and_isq(p,s,r,t) != ID_13); 
    else
      return (pqt_between_pqr_and_psq(p,q,r,s,t,mr,ms) != ID_12);
  }
  else // pqr exists in V({p,q,r,t})
  {
    if (error) return false;
    // return without examination in case of illegal object condition

      TRACEN("      pqr does exist");

    if (p==infinity())
      return (itq_between_iqr_and_isq(q,r,s,t) == ID_12);
    else if (q==infinity())
      return (iqt_between_iqr_and_isq(p,s,r,t) == ID_13);
    else
      return (pqt_between_pqr_and_psq(q,p,s,r,t,ms,mr) == ID_13);
  }
}

//------------------------------------------------------------------------------
// new_node_on_edge()
// calculates if a new node <p,t,q> exists on the edge <p,q,r,s> 
// after the new site t was inserted into the Voronoi Diagram.
// parameters:
// p_site p,q,r,s       the sites defining the edge 
// p_site t             the new site
// p_vnode mr,ms        the coordinates of the end points of finite edges
// return value:
// true         if there is a new point on the edge
// false        else
//------------------------------------------------------------------------------

bool EP_Voronoi_Diagram::
new_node_on_edge(p_site p, p_site q, p_site r, p_site s,
                 p_site t, p_vnode mr, p_vnode ms)
{
    TRACE("    new_node_on_edge ");EPD(p); EPD(q); EPD(r); EPD(s);
    EPD(t);TRACEN("");

  if (p==infinity())
    return (itq_between_iqr_and_isq(q,r,s,t) == BETWEEN);
  else if (q==infinity())
    return (itq_between_iqr_and_isq(p,s,r,t) == BETWEEN); 
  else
    return (pqt_between_pqr_and_psq(p,q,r,s,t,mr,ms) == BETWEEN);
}



//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// the 5 + 2 fantastic procedures::
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// pqr_in_V(,,)
// calculates if the node <p,q,r> exists in the Voronoi Diagramm of the 3 sites
// p, q, r. It just examines the position of the sites in the plane.
// preconditions:
// * p, q, r in R (infinity in R)
// parameters:
// p_site p, q, r       the three sites examined
// bool& error  an error flag in case of object restrictions
// return value:
// true         if <p,q,r> in V({p,q,r}) exists
// false        else
//------------------------------------------------------------------------------

bool EP_Voronoi_Diagram::
pqr_in_V(p_site p, p_site q, p_site r, bool& error)
{
    TRACE("pqr_in_V3: "); EPD(p); EPD(q); EPD(r);TRACEN("");

  if (p == infinity())
    return (*q != *r);
  if (q == infinity())
    return (*p != *r);
  if (r == infinity())
    return (*p != *q);
      
  return clockwise_oriented(p,q,r);
  // thus returns false if p,q,r are collinear or if there's equality of two
  // sites or if their orientation is wrong.
}

//------------------------------------------------------------------------------
// pqr_in_V(,,,)
// calculates if the node <p,q,r> exists in the Voronoi Diagramm of the 4 sites
// p, q, r and the new site t. It gets the 3 old sites from the h_node_info 
// structure together with the information which end of the edge is examined.
// preconditions:
// * p, q, r in R (infinity in R)
// * t in R \ {infinity}
// parameters:
// p_site p,q,r         the sites which build the node
// p_site t             the new object
// rat_point* m         the calculated circle center coords of the <p,q,t>
// bool& error  an error flag in case of object restrictions
// return value:
// true         if <p,q,r> in V({p,q,r,t}) exists
// false        else
//------------------------------------------------------------------------------

bool EP_Voronoi_Diagram::
pqr_in_V(p_site p, p_site q, p_site r, p_site t, p_vnode m, bool& error)
{
    TRACE("pqr_in_V4: "); 
    EPDS(p); EPDS(q); EPDS(r); EPDS(t); TRACEN("");

  // now: p!=q!=r
  if (p==infinity()) 
    return (!collinear_or_clockwise_ordered(r,t,q));
    // <i,q,r> does not exist if r,t,q are collinear or 
    // if they build a triangle rtq
    // if t=q or t=r then <i,q,r> does furtheron exist
  if (q==infinity()) 
    return (!collinear_or_clockwise_ordered(p,t,r));
    // <i,r,p> does not exist if p,t,r are collinear or 
    // if they build a triangle ptr
    // if t=p or t=r then <i,r,p> does furtheron exist
  if (r==infinity()) 
    return (!collinear_or_clockwise_ordered(q,t,p));
    // <i,p,q> does not exist if q,t,p are collinear or 
    // if they build a triangle qtp
    // if t=p or t=q then <i,p,q> does furtheron exist

  // now: p!=i,q!=i,r!=i,t!=i

  return (circle_pos(*m,*p,*t) >= 0);
  // <p,q,r> does not exist in V({p,q,r,t}) if t is closer to m than p
  // if t=p or t=q or t=r then the node <p,q,r> exists and true is returned
} 

//------------------------------------------------------------------------------
// pqt_between_pqr_and_psq()
// calculates if there's a node <p,q,t> between the two nodes which define an 
// edge in VR.
// preconditions:
// * p!=infinity & q!=infinity
// * t!=infinity
// * <p,q,r>, <p,s,q>, <p,q,t> exist i.e. p!=q!=s;p!=q!=r;p!=q!=t
// *                                      p,q,t are not collinear
// parameters:
// p_site p,q,r,s       the sites which build the edge
// p_site t             the new object
// return value:
// BETWEEN      if <p,q,r> < <p,q,t> < <p,s,q>
// ID_12        if <p,q,r> = <p,q,t>
// ID_13        if <p,s,q> = <p,q,t>
// ELSE         else
//------------------------------------------------------------------------------

int EP_Voronoi_Diagram::
pqt_between_pqr_and_psq(p_site p, p_site q, p_site r, p_site s, p_site t,
                             p_vnode mr, p_vnode ms)
{
    TRACE("pqt_between_pqr_and_psq "); 
    EPDS(p); EPDS(q); EPDS(r); EPDS(s); EPDS(t); TRACEN("");
 
  bool dummy;
  int t_rel_mr, t_rel_ms;

  if (!clockwise_oriented(p,q,t)) 
    return ELSE;
    // this is the case if there's no <p,q,t> node in VR
    // <=> (t=p or t=q) or p,q,t are collinear 

  if (r==infinity())
    t_rel_mr = -1; // IN
  else
    t_rel_mr = circle_pos(*mr,*p,*t);

  if (s==infinity())
    t_rel_ms = 1;  // OUT
  else 
    t_rel_ms = circle_pos(*ms,*p,*t);

  if (t_rel_mr == 0) 
    return ID_12;
  if (t_rel_ms == 0) 
    return ID_13;
  if (t_rel_mr < 0 && t_rel_ms > 0) 
    return BETWEEN;
  return ELSE;
}

//------------------------------------------------------------------------------
// iqt_between_iqr_and_isq()
// calculates the relative position of the three nodes v = <infinity,q,t>, u =
// <infinity,q,r> and w = <infinity,s,q> on the oriented bisector between 
// infinity (on the left) and q. 
// preconditions:
// * q!=infinity & r!=infinity & s!=infinity
// * t!=infinity
// parameters:
// p_site q,r,s the sites defining the edge
// p_site t     the new site
// return value:
// BETWEEN      if the order is u < v < w 
// ID_12        if u = v
// ID_13        if v = w
// ELSE         if v < u or w < v
//              or t=q (<i,q,t> does not exist)
//------------------------------------------------------------------------------

int EP_Voronoi_Diagram::
iqt_between_iqr_and_isq(p_site q, p_site r, p_site s, p_site t)
{
    TRACEN("iqt_between_iqr_and_isq ");
    EPDS(q); EPDS(r); EPDS(s); EPDS(t); TRACEN(""); 
    ASSERT( (r==s)||(collinear_or_clockwise_ordered(r,q,s)), 
          "iqt_between_iqr_and_isq():nodes 2,3 don't exist!\n");

  if (*t == *q) 
    return ELSE;
  if (*t == *r) 
    return ID_12;
  if (*t == *s) 
    return ELSE;

  // now: t!=r,s,q

  if (r == s)
    if (!collinear_or_clockwise_ordered(t,r,q)) 
      return BETWEEN;
    else
      return ELSE;
  else // r != s
    if (collinear_or_clockwise_ordered(r,t,q) && 
        collinear_or_clockwise_ordered(t,q,s))
      return BETWEEN;
    else
      return ELSE;
}

//------------------------------------------------------------------------------
// itq_between_iqr_and_isq()
// calculates the relative position of the three nodes v = <infinity,t,q>, u = 
// <infinity,q,r> and w = <infinity,s,q> on the oriented bisector between 
// infinity (on the left) and q. 
// preconditions:
// * q!=infinity & r!=infinity & s!=infinity
// * t!=infinity
// parameters:
// p_site q,r,s the sites defining the edge
// p_site t     the new site
// return value:
// BETWEEN      if the order is u < v < w 
// ID_12        if u = v
// ID_13        if v = w
// ELSE         if t=q (<i,t,q> does not exist)
//              or v < u or w < v
//------------------------------------------------------------------------------

int EP_Voronoi_Diagram::
itq_between_iqr_and_isq(p_site q, p_site r, p_site s, p_site t)
{
    TRACE("itq_between_iqr_and_isq ");
    EPDS(q); EPDS(r); EPDS(s); EPDS(t); TRACEN("");
    ASSERT( (r==s)||(collinear_or_clockwise_ordered(r,q,s)), 
            "itq_between_iqr_and_isq():nodes 2,3 don't exist!\n");

  if (*t == *q) return ELSE;
  if (*t == *s) return ID_13;
  if (*t == *r) return ELSE;

  // now: t!=r,s,q

  if (r == s)
    if (!collinear_or_clockwise_ordered(q,r,t)) 
      return BETWEEN;
    else
      return ELSE;
  else // r != s
    if (collinear_or_clockwise_ordered(r,q,t) && 
        collinear_or_clockwise_ordered(q,t,s))
      return BETWEEN;
    else
      return ELSE;
}

//------------------------------------------------------------------------------
// get_node_coords()
// calculates the coordinates of the node between the sites p, q and r which 
// is the center of the circle through the three sites p, q, r
// preconditions:
// * p!=q!=r
// parameters:
// p_site p, q, r       the three sites which build the node
// return value:
// rat_point(circle center) 
//------------------------------------------------------------------------------

p_vnode EP_Voronoi_Diagram::
get_node_coords(p_site p, p_site q, p_site r)
{
  // the procedure is only called in the
  // context where <p,q,r> exists.
  if (p == infinity() || q == infinity() || r == infinity())
      return gamma_node();

  rat_point pqm = center(*p,*q), prm = center(*p,*r);
  // the points which divide the segments pq and pr in halves

  rat_point pqt = (*p).rotate90(pqm);
  rat_point prt = (*p).rotate90(prm);
  rat_segment pq_bis(pqm,pqt);
  rat_segment pr_bis(prm,prt);
  vnode int_rpnt;
  if (!pq_bis.intersection_of_lines(pr_bis,int_rpnt))
    error_handler(1,"circle_center: bisecs don't intersect!\n");
  
  return p_vnode(int_rpnt);
}

//------------------------------------------------------------------------------
// draw_bisector()
// draws the edge between the sites p and q in the window w.
// parameters:
// p_site p     the site left of the edge looking from the source
// p_site q     the site right of the edge looking from the source
// rat_point* n1        the finite source coordinates of the source node if finite
// rat_point* n2        the finite target coordinates of the target node if finite
// window& W
// return value:
// none
//------------------------------------------------------------------------------

void EP_Voronoi_Diagram::
draw_bisector(p_site p, p_site q, p_vnode n1, p_vnode n2, window& W, color c)
{ 
  TRACE("draw_bisector: ");EPD(p);EPD(q);EPD(n1);EPD(n2);TRACEN("");
  double dx1,dx2,dy1,dy2;

  if ((n1 != gamma_node()) && (n2 != gamma_node()))
  {
    W.draw_segment((*n1).to_point(),(*n2).to_point(),c);
    return;
  }
  if ((n1 == gamma_node()) && (n2 == gamma_node()))
  { 
    
    point mp = center(*p,*q).to_point();
    point th = (*q).to_point().rotate90(mp);
    W.draw_line(mp,th,c);
    return;
  }

  point p1, p2, from;
  point mp = center(*p,*q).to_point();

  // this is symmetric:
  if (n1 != gamma_node()) // n2 == gamma_node()
  {
    p1 = (*p).to_point();
    p2 = (*q).to_point();
    from = (*n1).to_point();
  }
  else // n2 != gamma_node() and n1 == gamma_node()
  {
    p1 = (*q).to_point();
    p2 = (*p).to_point();
    from = (*n2).to_point();
  }

  point rot = p2.rotate90(mp);
  point through = from + (rot - mp);
  // rotate p2 counterclockwise about mp, take the vector from mp to this
  // rotated point and add this vector to origin.
  W.draw_ray(from,through,c);
}

//------------------------------------------------------------------------------
// draw_site()
// draws a site in the window W with color c
// parameters:
// p_site p     the site left of the edge looking from the source
// p_site q     the site right of the edge looking from the source
// rat_point* n1        the finite source coordinates of the source if finite
// rat_point* n2        the finite target coordinates of the target if finite
// window& W
// return value:
// none
//------------------------------------------------------------------------------

void EP_Voronoi_Diagram::
draw_site(p_site p, window& W, color c)
{
#ifdef _DEBUGAVD
   W.draw_int_node((*p).to_point(),ID_Number(*p),c);
#else
   W.draw_filled_node((*p).to_point(),c);
#endif
}


