//-----------------------------------------------------------------------------
// Freeware.  This file may be used freely to promote the ioctl90 mixer API.
//-----------------------------------------------------------------------------

// Parse.c - Parse command line parameters

#include <stdio.h>
#include <string.h>                     // memset
#include <ctype.h>                      // topupper

#include <os2.h>                        // typedefs

#include "data.h"
#include "parse.h"
#include "asciinum.h"

// Store parsed token here to make easy to see in debugger
PARSEDTOKEN Token;

// Given token which is first element of an ASCII-Z string,
// parse the token to separate the parts left and right of the colon.
// This can be useful for parsing config.sys command line options.
// Numeric parameters are converted as HEX.

void SplitToken (char *pszLine, PPARSEDTOKEN pToken)
{
   char   Ch;
   int    I;
   int    iRC;

   memset (pToken, '\0', sizeof (PARSEDTOKEN));

   if ( *pszLine == '-' || *pszLine == '/' )
   {
      pToken->fLeadingDashFound = TRUE;
      pszLine++;
   }

   // Copy all characters before the colon
   Ch = toupper (*pszLine);
   I = 0;
   while ( Ch > ' ' && Ch != ':' && I < MAXTOKENSIZE )
   {
      pToken->szKeyword[I] = Ch;          // Copy to output string
      pszLine++;
      Ch = toupper (*pszLine);
      I++;

      // For specifying options where the value is a number, the
      // separator (colon) can be omitted if the parser can determine
      // that the value is concatenated next to the keywork (keychar).
      if ( pToken->fLeadingDashFound && isdigit (Ch) && I == 1)
      {
         pToken->fSeparatorFound = TRUE; // Implied separator
         break;
      }
   }

   if (*pszLine == ':')
   {
      pToken->fSeparatorFound = TRUE;
      pszLine++;
   }

   iRC = AsciiToUlong (pszLine, &pToken->ulValue, BASE16);
   pToken->fNumericFound = (iRC == 0);

   // Copy the characters right of the colon
   Ch = toupper (*pszLine);
   I = 0;
   while (Ch > ' ' && I < MAXTOKENSIZE)
   {
      pToken->szValue[I] = Ch;        // Copy to output string
      pszLine++;
      Ch = toupper (*pszLine);
      I++;
   }

   // If token did not follow keyword:value format and if the
   // token represents an ascii number then determine, the numeric value.
   if (! pToken->fLeadingDashFound && ! pToken->fSeparatorFound)
   {
      iRC = AsciiToUlong (pToken->szKeyword, &pToken->ulValue, BASE16);
      pToken->fNumericFound = (iRC == 0);
   }
}


char *BoolTicValue (BOOL fValue)
{
   if ( fValue )
      return ("TRUE");
   else
      return ("FALSE");
}


int ParseToken (char *Parm)
{
   memset (&Token, '\0', sizeof (Token));
   SplitToken (Parm, &Token);

   if (Token.fLeadingDashFound)
   {
      if (strlen (Token.szKeyword) == 1)
      {
         switch (Token.szKeyword[0])
         {
         case 'D':
            Options.Debug = TRUE;
            break;

         case 'V':
            Options.Verbose = TRUE;
            break;

         case 'H':
         case '?':
            Options.Help = TRUE;
            break;
         }
      }
      else if (strcmp (Token.szKeyword, "HELP") == 0)
      {
         Options.Help         = TRUE;
         Options.ExtendedHelp = TRUE;
      }
   }


   if (Options.Debug)
   {
      printf ("\nToken: \"%s\"\n", Parm);
      printf ("   fLeadingDashFound %s\n", BoolTicValue (Token.fLeadingDashFound));
      printf ("   fSeparatorFound   %s\n", BoolTicValue (Token.fSeparatorFound));
      printf ("   fNumericFound     %s\n", BoolTicValue (Token.fNumericFound));
      printf ("   ulValue (hex)     %x\n", Token.ulValue);
      printf ("   szKeyword         %s\n", Token.szKeyword);
      printf ("   szValue           %s\n", Token.szValue);
   }

   return (0);
}


int ParseCommandLine (int argc, char *argv[])
{
   int iRC = 0;
   int I;

   for (I=1; I < argc; I++)
   {
      iRC = ParseToken (argv[I]);
   }

   return (iRC);
}
