/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = VTINT7F.C
 *
 * DESCRIPTIVE NAME = Virtual Touch Device Driver Int 7fh functions.
 *
 *
 * VERSION = V2.0
 *
 * DATE        09/20/91
 *
 * DESCRIPTION
 *             This module contains the VTD's Int 7fh functions.
 *             Functions called via INT 7FH router (in VDM) and VTInt7fhook().
 *
 *
 * FUNCTIONS   vtInt7fReset,              vtVMouseInstalled,
 *             vtInt7fEmulReset,          vtInt7fEmulOnOff,
 *             vtScaleThresholds,         vtInt7fSetCoOrdinates,
 *             vtInt7fSetOrigin,          vtInt7fSetThresholds,
 *             vtInt7fSetSelection,       vtInt7fSetReportRate,
 *             vtInt7fSetBlockMask,       vtInt7fGetTouchDown,
 *             vtInt7fGetLiftOff,         vtInt7fGetSelectOn,
 *             vtInt7fGetSelectOff,       vtInt7fGetCurrent,
 *             vtInt7fSetUserSub,         vtInt7fSwapUserSub,
 *             vtInt7fSetFilterType,      vtInt7fQueryStorage,
 *             vtInt7fSaveState,          vtInt7fRestoreState,
 *             vtInt7fGetEmulState,       vtInt7fGetEmulOnOff,
 *             vtInt7fGetCoOrdinates,     vtInt7fGetOrigin,
 *             vtInt7fGetThresholds,      vtInt7fGetSelection,
 *             vtInt7fGetReportRate,      vtInt7fGetBlockMask,
 *             vtInt7fGetFilterType,      vtInt7fGetDriverInfo,
 *             vtInt7fGetClickLock,       vtInt7fSetClickLock,
 *             vtInt7fDldCalibConsts.
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#include "vtdef.h"

/*
** Externals
*/

extern FPFNPDD        fpfnPTDRequestProc;
extern BOOL           fTouchEnabled;
extern TOUDEVICESTATE touDeviceState;
extern BOOL           fEventsPending;
extern BOOL           fSubActive;
extern PERVDMDATA     VDMDataDefaults;

#pragma  BEGIN_SWAP_CODE


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fReset
 *
 * DESCRIPTION   = Installed Flag and Reset (function 0)
 *   PSEUDO-CODE -
 *      Enable touch data
 *      Reset instance data to defaults
 *      Set output parms
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fReset( PINT7FPARM pin, PINT7FPARM pout )
{
  VDHCopyMem( &VDMDataDefaults, &VDMData, sizeof( PERVDMDATA ) );
  ( *fpfnPTDRequestProc )( PTDREQ_QUERYDEVICESTATE, NULL,
                           F16PFROMP( &touDeviceState ) );
  vtNotifyEmulState( CURRENT_VDM );
  fTouchEnabled = !!( touDeviceState.fTouState & TOUSTATE_CALIBRATED );
  fEventsPending = FALSE;
  fSubActive = FALSE;
  pout->p0 = ( fTouchEnabled ) ? INT7F_MINUS_ONE : 1;
  pout->p1 = ( touDeviceState.fMouState & MOUSTATE_REALMOUSE ) ?
             INT7F_MINUS_ONE : 0;
  if( pout->p0 == INT7F_MINUS_ONE )
    return( INT7FERROR_NOERROR );
  else
    return( INT7FERROR_SEEPARMS );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtVMouseInstalled
 *
 * DESCRIPTION   = Determine whether VMouse driver is installed
 *   PSEUDO-CODE -
 *      attempt to open VMouse
 *      return result
 *
 *
 * INPUT         = NONE
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = TRUE  - is a mouse driver
 * RETURN-ERROR  = FALSE - is no mouse driver
 *
 ****************************************************************************/

BOOL PRIVENTRY vtVMouseInstalled( VOID )
{
  HVDD hvdd;

  if( hvdd = VDHOpenVDD( VMD_NAME ) )
  {
    VDHCloseVDD( hvdd );
    return( TRUE );
  }
  else
    return( FALSE );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fEmulReset
 *
 * DESCRIPTION   = Mouse Emulation Reset (function 1)
 *   PSEUDO-CODE -
 *      Enable touch data
 *      Set output parms
 *      Reset instance data to defaults
 *      Save emulation parameters
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fEmulReset( PINT7FPARM pin, PINT7FPARM pout )
{
  ( *fpfnPTDRequestProc )( PTDREQ_QUERYDEVICESTATE, NULL,
                           F16PFROMP( &touDeviceState ) );
  fTouchEnabled = !!( touDeviceState.fTouState & TOUSTATE_CALIBRATED );
  pout->p0 = ( touDeviceState.fMouState & MOUSTATE_EMULALLOWED ) ?
               INT7F_MINUS_ONE : 0;
  if( pout->p0 && !vtVMouseInstalled( ) )
    pout->p0 = 1;                       /* no mouse driver installed    */
  if( ( pin->p0 <= 2 ) && ( pin->p1 <= 1 ) )
  {
    VDHCopyMem( &VDMDataDefaults, &VDMData, sizeof( PERVDMDATA ) );
    VDMData.EmulState.usEmulType = pin->p0;     /* emulation type       */
    VDMData.EmulState.fEmulBeep = pin->p1;      /* beep on/off          */
    VDMData.EmulState.usMickeys = pin->p2;      /* resolution, mickeys/inch */
    VDMData.fEmulationOn = FALSE;               /* turn emul off        */
    vtNotifyEmulState( CURRENT_VDM );
    if( pout->p0 == INT7F_MINUS_ONE )
      return( INT7FERROR_NOERROR );
    else
      return( INT7FERROR_SEEPARMS );
  }
  else
    return( INT7FERROR_INVALIDINPUT );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fEmulOnOff
 *
 * DESCRIPTION   = Mouse Emulation ON/OFF (function 2)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fEmulOnOff( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = ( touDeviceState.fMouState & MOUSTATE_EMULALLOWED ) ?
               INT7F_MINUS_ONE : 0;
  if( pin->p0 <= 1 )
  {
    VDMData.fEmulationOn = !!pin->p0;
    if( pout->p0 && !vtVMouseInstalled( ) )
      pout->p0 = 1;                     /* no mouse driver installed    */
    if( ( pout->p0 == INT7F_MINUS_ONE ) || !( pin->p0 ) )
    {                                   /* no error, OR turned emul OFF */
      vtNotifyEmulState( CURRENT_VDM );
      return( INT7FERROR_NOERROR );
    }
    else
      return( INT7FERROR_SEEPARMS );
  }
  else
    return( INT7FERROR_INVALIDINPUT );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtScaleThresholds
 *
 * DESCRIPTION   = Scale thresholds into new Z range
 *   PSEUDO-CODE -
 *                     (OldThreshold - OldZMin)
 *      NewThreshold = ------------------------ * (NewZMax - NewZMin) + NewZMin
 *                        (OldZMax - OldZMin)
 *
 *
 * INPUT         = usNewZmin - new minimum Z value
 *                 usNewZmax - new maximum Z value
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

VOID PRIVENTRY vtScaleThresholds( USHORT usNewZMin, USHORT usNewZMax )
{
  USHORT usOldZMin, usOldZMax, usTmp;

  usOldZMin = VDMData.coData.usZMin;
  usOldZMax = VDMData.coData.usZMax;
  usTmp  = VDMData.thData.usOnScreen - usOldZMin;
  usTmp *= usNewZMax - usNewZMin;
  usTmp += (usOldZMax - usOldZMin) / 2;         /* apply some rounding  */
  usTmp /= usOldZMax - usOldZMin;
  usTmp += usNewZMin;
  VDMData.thData.usOnScreen = usTmp;
  usTmp  = VDMData.thData.usPushHysteresis - usOldZMin;
  usTmp *= usNewZMax - usNewZMin;
  usTmp += (usOldZMax - usOldZMin) / 2;
  usTmp /= usOldZMax - usOldZMin;
  usTmp += usNewZMin;
  if( usTmp == VDMData.thData.usOnScreen )
    usTmp++;
  VDMData.thData.usPushHysteresis = usTmp;
  usTmp  = VDMData.thData.usPushSelection - usOldZMin;
  usTmp *= usNewZMax - usNewZMin;
  usTmp += (usOldZMax - usOldZMin) / 2;
  usTmp /= usOldZMax - usOldZMin;
  usTmp += usNewZMin;
  if( usTmp == VDMData.thData.usPushHysteresis )
    usTmp++;
  VDMData.thData.usPushSelection = usTmp;
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetCoOrdinates
 *
 * DESCRIPTION   = Set Co-Ordinate System (function 3)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetCoOrdinates( PINT7FPARM pin, PINT7FPARM pout )
{
  if( ( pin->p0 > MAX_XY )   || ( pin->p1 > MAX_XY )   ||  /* X range     */
      ( pin->p2 > MAX_XY )   || ( pin->p3 > MAX_XY )   ||  /* Y range     */
      ( pin->p4 > MAX_Z )    || ( pin->p5 > MAX_Z )    ||  /* Z range     */
      ( pin->p6 > MAX_HYST ) || ( pin->p7 > MAX_HYST ) ||  /* Hysteresis  */
      ( pin->p4 > pin->p5 )  ||                            /* Zmin > Zmax */
      ( pin->p5 - pin->p4 < 4 ) )
    return( INT7FERROR_INVALIDINPUT );
  else
  {
    vtScaleThresholds( pin->p4, pin->p5 );
    VDMData.coData.usXMin = pin->p0;
    VDMData.coData.usXMax = pin->p1;
    VDMData.coData.usYMin = pin->p2;
    VDMData.coData.usYMax = pin->p3;
    VDMData.coData.usZMin = pin->p4;
    VDMData.coData.usZMax = pin->p5;
    VDMData.coData.usXHysteresis = pin->p6;
    VDMData.coData.usYHysteresis = pin->p7;
    VDMData.fTrackVideoMode = FALSE;
    return( INT7FERROR_NOERROR );
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetOrigin
 *
 * DESCRIPTION   = Set Co-Ordinate Origin (function 4)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetOrigin( PINT7FPARM pin, PINT7FPARM pout )
{
  if( pin->p0 > ORIGIN_BOTTOMLEFT )
    return( INT7FERROR_INVALIDINPUT );
  else
  {
    VDMData.fOriginTopLeft = !pin->p0;
    return( INT7FERROR_NOERROR );
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetThresholds
 *
 * DESCRIPTION   = Set Thresholds (function 5)
 *   PSEUDO-CODE -
 *      Range check thresholds, must be between ZMin and ZMax,
 *       and; OnScreen < Push Hysteresis < Push Threshold
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetThresholds( PINT7FPARM pin, PINT7FPARM pout )
{
  if( (pin->p0 > VDMData.coData.usZMin) && (pin->p0 < VDMData.coData.usZMax) &&
      (pin->p1 > VDMData.coData.usZMin) && (pin->p1 < VDMData.coData.usZMax) &&
      (pin->p2 > VDMData.coData.usZMin) && (pin->p2 < VDMData.coData.usZMax) &&
      (pin->p0 < pin->p2) && (pin->p2 < pin->p1) )
  {
    VDMData.thData.usOnScreen = pin->p0;
    VDMData.thData.usPushSelection = pin->p1;
    VDMData.thData.usPushHysteresis = pin->p2;
    return( INT7FERROR_NOERROR );
  }
  else
    return( INT7FERROR_INVALIDINPUT );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetSelection
 *
 * DESCRIPTION   = Set Selection Mechanism (function 6)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetSelection( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;

  switch( pin->p0 )
  {                                     /* selection method             */
    case  SM_PUSH_HARDER :
    case  SM_LIFT_OFF :
      VDMData.smData.usMethod = pin->p0;
      return( INT7FERROR_NOERROR );
    case  SM_ONETOUCH :
    case  SM_ONETOUCH|SM_LIFT_OFF :
    case  SM_STABLE_POINT :
    case  SM_STABLE_POINT|SM_LIFT_OFF :
      if( !pin->p1 )
        pin->p1 = VDMData.smData.usCount_Default;
      if( !pin->p2 )
        pin->p2 = VDMData.smData.usTol_Default;
      if( !pin->p3 )
        pin->p3 = VDMData.smData.usLimit_Default;
      if( ( pin->p1 > SPBUFFERMAX ) || ( pin->p2 > VDMData.coData.usXMax ) ||
          ( pin->p3 > SPBUFFERMAX ) || ( pin->p1 > pin->p3 ) )
        return( INT7FERROR_INVALIDINPUT );
      else
      {
        flFlags = SAVEFLAGS( );
        DISABLE( );                     /* Enter critical-section       */
        VDMData.smData.usFlag = 0;      /* restart algorithm            */
        VDMData.smData.usMethod = pin->p0;
        if( pin->p1 )
          VDMData.smData.usCount = pin->p1;
        if( pin->p2 )
          VDMData.smData.usTolerance = pin->p2;
        if( pin->p3 )
          VDMData.smData.usLimit = pin->p3;
        RESTOREFLAGS( flFlags );        /* Exit critical-section        */
        return( INT7FERROR_NOERROR );
      }
      break;
    default  :
      return( INT7FERROR_INVALIDINPUT );
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetReportRate
 *
 * DESCRIPTION   = Set Data Report Rate (function 7)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetReportRate( PINT7FPARM pin, PINT7FPARM pout )
{
  if( pin->p0 > MAX_REPORT_RATE )
    return( INT7FERROR_INVALIDINPUT );
  else
  {
    VDMData.ulReportRate = (ULONG)pin->p0;
    return( INT7FERROR_NOERROR );
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetBlockMask
 *
 * DESCRIPTION   = Set Data Block Mask (function 8)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetBlockMask( PINT7FPARM pin, PINT7FPARM pout )
{
  if( pin->p0 & ~(BLOCKMASK_ALL_DATA | BLOCKMASK_PEGGED_DATA) )
    return( INT7FERROR_INVALIDINPUT );
  else
  {
    VDMData.flBlockMask = (ULONG)pin->p0;
    return( INT7FERROR_NOERROR );
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetTouchDown
 *
 * DESCRIPTION   = Get Touch-Down Info (function 9)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetTouchDown( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  pout->p0 = (USHORT)VDMData.xyzLastTouchDown.Status;
  pout->p1 = (USHORT)VDMData.xyzLastTouchDown.Count;
  pout->p2 = (USHORT)VDMData.xyzLastTouchDown.x;
  pout->p3 = (USHORT)VDMData.xyzLastTouchDown.y;
  pout->p4 = (USHORT)VDMData.xyzLastTouchDown.z;
  VDMData.xyzLastTouchDown.Count = 0;
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetLiftOff
 *
 * DESCRIPTION   = Get Lift-Off Info (function 10)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetLiftOff( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  pout->p0 = (USHORT)VDMData.xyzLastLiftOff.Status;
  pout->p1 = (USHORT)VDMData.xyzLastLiftOff.Count;
  pout->p2 = (USHORT)VDMData.xyzLastLiftOff.x;
  pout->p3 = (USHORT)VDMData.xyzLastLiftOff.y;
  pout->p4 = (USHORT)VDMData.xyzLastLiftOff.z;
  VDMData.xyzLastLiftOff.Count = 0;
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetSelectOn
 *
 * DESCRIPTION   = Selection Turned-ON Info (function 11)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> outlput parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetSelectOn( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  pout->p0 = (USHORT)VDMData.xyzLastSelectOn.Status;
  pout->p1 = (USHORT)VDMData.xyzLastSelectOn.Count;
  pout->p2 = (USHORT)VDMData.xyzLastSelectOn.x;
  pout->p3 = (USHORT)VDMData.xyzLastSelectOn.y;
  pout->p4 = (USHORT)VDMData.xyzLastSelectOn.z;
  VDMData.xyzLastSelectOn.Count = 0;
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetSelectOff
 *
 * DESCRIPTION   = Selection Turned-OFF Info (function 12)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetSelectOff( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  pout->p0 = (USHORT)VDMData.xyzLastSelectOff.Status;
  pout->p1 = (USHORT)VDMData.xyzLastSelectOff.Count;
  pout->p2 = (USHORT)VDMData.xyzLastSelectOff.x;
  pout->p3 = (USHORT)VDMData.xyzLastSelectOff.y;
  pout->p4 = (USHORT)VDMData.xyzLastSelectOff.z;
  VDMData.xyzLastSelectOff.Count = 0;
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetCurrent
 *
 * DESCRIPTION   = Get Current Position & Status (function 13)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetCurrent( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  pout->p0 = (USHORT)VDMData.xyzCurrent.Status;
  pout->p1 = (USHORT)VDMData.xyzCurrent.x;
  pout->p2 = (USHORT)VDMData.xyzCurrent.y;
  pout->p3 = (USHORT)VDMData.xyzCurrent.z;
  VDMData.xyzCurrent.Status |= XYZDATA_READ;
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetUserSub
 *
 * DESCRIPTION   = Set User-Defined Subroutine (function 14)
 *   PSEUDO-CODE -
 *      Validate input parameters
 *          ie. call mask bits are valid, user-sub address is non-zero.
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetUserSub( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags,rc;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  if( pin->p0 & ~MAX_CALLMASK )
    rc = INT7FERROR_INVALIDINPUT;
  else
  {
    VDMData.flCallMask = pin->p0;
    /* Nicky    VDMData.vpUserSub = VPFROMVADDR( pin->p2, pin->p1 );    */
    SEGMENTOF32( VDMData.fpfnUserSub ) = pin->p2;
    OFFSETOF32( VDMData.fpfnUserSub ) = pin->p1;
    rc = INT7FERROR_NOERROR;
  }
  RESTOREFLAGS( flFlags );             /* Exit critical-section         */
  return( rc );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSwapUserSub
 *
 * DESCRIPTION   = Swap User-Defined Subroutine (function 15)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSwapUserSub( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags,rc;

  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  pout->p0 = (USHORT)VDMData.flCallMask;
  pout->p1 = (SHORT)OFFSETOF32( VDMData.fpfnUserSub );
  pout->p2 = (SHORT)SEGMENTOF32( VDMData.fpfnUserSub );
  if( pin->p0 & ~MAX_CALLMASK )
    rc = INT7FERROR_INVALIDINPUT;
  else
  {
    VDMData.flCallMask = pin->p0;
    /*nicky     VDMData.vpUserSub = VPFROMVADDR( pin->p2, pin->p1 );    */
    SEGMENTOF32( VDMData.fpfnUserSub ) = pin->p2;
    OFFSETOF32( VDMData.fpfnUserSub ) = pin->p1;
    rc = INT7FERROR_NOERROR;
  }
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  return( rc );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetFilterType
 *
 * DESCRIPTION   = DownLoad Filter Constants (function 16)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetFilterType( PINT7FPARM pin, PINT7FPARM pout )
{
  if( ( pin->p0 > MAX_FILTERTYPE ) || ( pin->p1 > MAX_FILTERFREQUENCY ) )
    return( INT7FERROR_INVALIDINPUT );
  else
  {
    VDMData.ulFilterType = (ULONG)pin->p0;
    VDMData.ulFilterFreq = (ULONG)pin->p1;
    return( INT7FERROR_NOERROR );
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fQueryStorage
 *
 * DESCRIPTION   = Query Save State Storage (function 17)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fQueryStorage( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = sizeof( PERVDMDATA );
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSaveState
 *
 * DESCRIPTION   = Save Driver State (function 18)
 *   PSEUDO-CODE -
 *      Copy per-VDM data area to user's save buffer
 *
 *      VDHCopyMem(&VDMData, PFROMVADDR(pin->p2, pin->p1), sizeof(PERVDMDATA));
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSaveState( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;
  BOOL fEnabled;

  fEnabled = fTouchEnabled;
  fTouchEnabled = FALSE;
  fEventsPending = FALSE;
  while( fSubActive )
    VDHYield( !VDH_YIELD_TIME_CRITICAL );
  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  memcpy( PFROMVADDR( pin->p1, pin->p0 ), &VDMData, sizeof( PERVDMDATA ) );
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  fTouchEnabled = fEnabled;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fRestoreState
 *
 * DESCRIPTION   = Restore Driver State (function 19)
 *   PSEUDO-CODE -
 *      Copy from user's save buffer to per-VDM data area
 *      Reset emulation state if changed.       drw 17/7/91
 *
 *      VDHCopyMem(PFROMVADDR(pin->p2, pin->p1), &VDMData, sizeof(PERVDMDATA));
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fRestoreState( PINT7FPARM pin, PINT7FPARM pout )
{
  ULONG flFlags;
  BOOL fEnabled;

  fEnabled = fTouchEnabled;
  fTouchEnabled = FALSE;
  fEventsPending = FALSE;
  while( fSubActive )
    VDHYield( !VDH_YIELD_TIME_CRITICAL );
  flFlags = SAVEFLAGS( );
  DISABLE( );                           /* Enter critical-section       */
  memcpy( &VDMData, PFROMVADDR( pin->p1, pin->p0 ), sizeof( PERVDMDATA ) );
  RESTOREFLAGS( flFlags );              /* Exit critical-section        */
  vtNotifyEmulState( CURRENT_VDM );
  fTouchEnabled = fEnabled;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetEmulState
 *
 * DESCRIPTION   = Get Emul State (function 20)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetEmulState( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = VDMData.EmulState.usEmulType; /* emulation type            */
  pout->p1 = VDMData.EmulState.fEmulBeep;  /* beep on/off               */
  pout->p2 = VDMData.EmulState.usMickeys;  /* resolution, mickeys/inch  */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetEmulOnOff
 *
 * DESCRIPTION   = Get Emul On/Off (function 21)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetEmulOnOff( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = (USHORT)VDMData.fEmulationOn;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetCoOrdinates
 *
 * DESCRIPTION   = Get Co-Ordinate System (function 22)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetCoOrdinates( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = VDMData.coData.usXMin;
  pout->p1 = VDMData.coData.usXMax;
  pout->p2 = VDMData.coData.usYMin;
  pout->p3 = VDMData.coData.usYMax;
  pout->p4 = VDMData.coData.usZMin;
  pout->p5 = VDMData.coData.usZMax;
  pout->p6 = VDMData.coData.usXHysteresis;
  pout->p7 = VDMData.coData.usYHysteresis;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetOrigin
 *
 * DESCRIPTION   = Get Co-Ordinate Origin (function 23)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetOrigin( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = (USHORT)!( VDMData.fOriginTopLeft );
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetThresholds
 *
 * DESCRIPTION   = Get Thresholds (function 24)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetThresholds( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = VDMData.thData.usOnScreen;
  pout->p1 = VDMData.thData.usPushSelection;
  pout->p2 = VDMData.thData.usPushHysteresis;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetSelection
 *
 * DESCRIPTION   = Get Selection Mechanism (function 25)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetSelection( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = VDMData.smData.usMethod;
  pout->p1 = VDMData.smData.usCount;
  pout->p2 = VDMData.smData.usTolerance;
  pout->p3 = VDMData.smData.usLimit;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetReportRate
 *
 * DESCRIPTION   = Get Report Rate (function 26)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetReportRate( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = (USHORT)VDMData.ulReportRate;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetBlockMask
 *
 * DESCRIPTION   = Get Data Block Mask (function 27)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetBlockMask( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = (USHORT)VDMData.flBlockMask;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetFilterType
 *
 * DESCRIPTION   = Get Current Filter Type (function 28)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetFilterType( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = (USHORT)VDMData.ulFilterType;
  pout->p1 = (USHORT)VDMData.ulFilterFreq;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetDriverInfo
 *
 * DESCRIPTION   = Get Driver Info (function 29)
 *   PSEUDO-CODE -
 *      This should return two parameters:
 *          o device driver version
 *          o microcode level (always returns 0)
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetDriverInfo( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = ( VTD_VERSION_MAJOR << 8 ) + VTD_VERSION_MINOR;
  pout->p1 = 0;                         /* microcode level, not supported */
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fGetClickLock
 *
 * DESCRIPTION   = Get Click Locking Parameters (function 30)
 *   PSEUDO-CODE -
 *      The return parameters are those that were set (vtInt7fSetClickLock)
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fGetClickLock( PINT7FPARM pin, PINT7FPARM pout )
{
  pout->p0 = (USHORT)VDMData.clData.usClickOnOff;
  pout->p1 = (USHORT)VDMData.clData.usClickTimeout;
  pout->p2 = (USHORT)VDMData.clData.usXLockArea;
  pout->p3 = (USHORT)VDMData.clData.usYLockArea;
  return( INT7FERROR_NOERROR );
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fSetClickLock
 *
 * DESCRIPTION   = Set Click Locking Parameters (function 31)
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fSetClickLock( PINT7FPARM pin, PINT7FPARM pout )
{
  if( ( pin->p0 > 1 ) || ( pin->p2 > VDMData.coData.usXMax ) ||
      ( pin->p3 > VDMData.coData.usYMax ) )
    return( INT7FERROR_INVALIDINPUT );
  else
  {
    VDMData.clData.usClickOnOff = pin->p0;
    VDMData.clData.usClickTimeout = pin->p1;
    VDMData.clData.usXLockArea = pin->p2;
    VDMData.clData.usYLockArea = pin->p3;
    return (INT7FERROR_NOERROR);
  }
}


/****************************************************************************
 *
 * FUNCTION NAME = vtInt7fDldCalibConsts
 *
 * DESCRIPTION   = Download Calibration constants.
 *   PSEUDO-CODE - NONE
 *
 *
 * INPUT         = pin  -> input parameters
 * OUTPUT        = pout -> output parameters
 *
 * RETURN-NORMAL = INT7FERROR_NOERROR      - No Error
 * RETURN-ERROR  = INT7FERROR_INVALIDINPUT - Invalid Input
 *                 INT7FERROR_INTERFACE    - Interface Error
 *                 INT7FERROR_SEEPARMS     - Other error/status
 *
 ****************************************************************************/

RETCODE PRIVENTRY vtInt7fDldCalibConsts( PINT7FPARM pin, PINT7FPARM pout )
{
  return( INT7FERROR_NOERROR );
}

#pragma  END_SWAP_CODE
