/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/

/**************************************************************************
 *
 * SOURCE FILE NAME =  VMRATE.C
 *
 * DESCRIPTIVE NAME =  Virtual Mouse Device Driver Int 33h Rate Services
 *
 *
 * VERSION =  V2.0
 *
 * DATE       09/30/91
 *
 * DESCRIPTION This module contains the VMD's Int 33h rate services.
 *
 *
 * FUNCTIONS
 *
 *  vmInt33SetMickeyRatio()     Int 33h Set Mickey Ratio (function 15)
 *  vmSetScaleFactor()          Set scale factors
 *  vmInt33SetThreshold()       Int 33h Set Double-speed Threshold (function 19)
 *  vmSetThreshold()            Set double-speed threshold
 *  vmScaleSensitivity()        Scale Mouse Sensitivity
 *  vmInt33SetSensitivity()     Int 33h Set Sensitivity (function 26)
 *  vmInt33QuerySensitivity()   Int 33h Query Sensitivity (function 27)
 *  vmInt33SetInterruptRate()   Int 33h Set Interrupt Rate (function 28)
 *
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/
#include "vmdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

/*
**    External References
*/

extern USHORT nThreshold;
extern USHORT uxScale;
extern USHORT uyScale;
extern ULONG aulScaleTable[];
extern ULONG fsstate;

#pragma  BEGIN_SWAP_CODE

 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmInt33SetMickeyRatio()
 *
 *  DESCRIPTION    :  Int 33h Set Mickey Ratio (function 15)
 *
 *  INPUT          :
 *                    pcrf -> VDM register frame
 *  OUTPUT         :
 *      EMULATED
 *          TRUE
 *      NOT EMULATED
 *          FALSE (pass control to next VDD and/or ROM)
 *  USES           :
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT        :
 *      VDM Task-time
 *
 *  PSEUDO-CODE    :                                 REFERENCES
 *      call vmSetScaleFactor to set mickey ratio;
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33SetMickeyRatio(register PCRF pcrf)
{
  if ( ( (SHORT)CX(pcrf) > 0) && ( (SHORT)DX(pcrf) > 0) )
    vmSetScaleFactor(CX(pcrf), DX(pcrf) );
  return  TRUE;
}                                      /* vmInt33SetMickeyRatio              */

 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmSetScaleFactor()
 *
 *  DESCRIPTION    :  Set scale factors
 *
 *  INPUT     :
 *                nxMics == # mickeys per 8 pixels horizontally
 *                nyMics == # mickeys per 8 pixels vertically
 *  OUTPUT    :
 *                NONE
 *  USES      :
 *                32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT   :
 *                VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      store mickey ratio and make sure they are at least one;
 *
 ***************************************************************************/

VOID PRIVENTRY vmSetScaleFactor(ULONG nxMics,ULONG nyMics)
{
  VDMData.mstates.nxMicsPer8Pix = nxMics;
  VDMData.mstates.nyMicsPer8Pix = nyMics;
  if ( fsstate&SESSION_FGND )
  {
    uxScale = vmScaleSensitivity(VDMData.mstates.nxMicsPer8Pix,
       VDMData.mstates.scaleXMics);
    uyScale = vmScaleSensitivity(VDMData.mstates.nyMicsPer8Pix,
       VDMData.mstates.scaleYMics);
  }
}                                      /* vmSetScaleFactor                   */


 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmInt33SetThreshold()
 *
 *  DESCRIPTION    :  Int 33h Set Double-speed Threshold (function 19)
 *
 *  INPUT      :
 *               pcrf -> VDM register frame
 *  OUTPUT     :
 *               EMULATED
 *                   TRUE
 *               NOT EMULATED
 *                   FALSE (pass control to next VDD and/or ROM)
 *  USES       :
 *               32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT    :
 *                VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      call vmSetThreshold to set double speed threshold;
 *
 ***************************************************************************/

BOOL PRIVENTRY vmInt33SetThreshold(register PCRF pcrf)
{
  if ( ( SHORT )DX( pcrf ) >= 0 )
    vmSetThreshold(DX(pcrf));
  return  TRUE;
}                                      /* vmInt33SetThreshold                */


 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmSetThreshold()
 *
 *  DESCRIPTION    :  Set double-speed threshold
 *
 * INPUT       :
 *               ulThreshold == new threshold
 *  OUTPUT     :
 *               NONE
 *  USES       :
 *               32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT    :
 *               VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      store double speed threshold;
 *      call PDD to set double speed threshold;
 *
 ****************************************************************************/

VOID PRIVENTRY vmSetThreshold(ULONG ulThreshold)
{
  if ( !ulThreshold )
    ulThreshold = DEF_THRESHOLD;

  VDMData.mstates.ulThreshold = ulThreshold;

  if ( fsstate&SESSION_FGND )
    nThreshold = VDMData.mstates.ulThreshold;
}                                      /* vmSetThreshold                     */


 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmScaleSensitivity()
 *
 *  DESCRIPTION    :  Scale Mouse Sensitivity
 *
 *  INPUT      :
 *               ulValue - value to be scaled
 *               ulScale - scale factor
 *  OUTPUT     :
 *               NONE
 *  USES       :
 *               32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT    :
 *               VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      round scale factor to index number of 5;
 *      get actual scale factor from scale table using the index;
 *      return scaled value;
 *
 ***************************************************************************/

ULONG PRIVENTRY vmScaleSensitivity(ULONG ulValue,ULONG ulScale)
{
  register ULONG i;
  if (ulScale > (( 5 * MAX_SCALERANGE ) - 2 ) )   /* Defect 72280 ChangeTeam1*/
    return (1);                                   /* Defect 72280 ChangeTeam1*/
  i = (ulValue *aulScaleTable[(ulScale+2)/5])/SENS_SCALAR;
  return (i?i:1);
}                                      /* vmScaleSensitivity                 */

 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmInt33SetSensitivity()
 *
 *  DESCRIPTION    :  Int 33h Set Sensitivity (function 26)
 *
 *  INPUT      :
 *                   pcrf -> VDM register frame
 *  OUTPUT     :
 *                   EMULATED
 *                       TRUE
 *                   NOT EMULATED
 *                       FALSE (pass control to next VDD and/or ROM)
 *  USES       :
 *                   32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT    :
 *                   VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      make sure all VDM parameters are within 100;
 *      store all scale numbers;
 *      call vmSetScaleFactor to set mickey sensitivity;
 *      call vmSetThreshold to set double speed threshold;
 *
 ***************************************************************************/

BOOL PRIVENTRY vmInt33SetSensitivity(register PCRF pcrf)
{
  VDMData.mstates.scaleXMics = (ULONG)(((SHORT)BX(pcrf) > 100)?100:
                                          ((SHORT)BX (pcrf) < 0)?0:
                                                  BX(pcrf));

  VDMData.mstates.scaleYMics = (ULONG)(((SHORT)CX(pcrf) > 100)?100:
                                          ((SHORT)CX (pcrf) < 0)?0:
                                                  CX(pcrf));

  VDMData.mstates.scaleThreshold = (ULONG)((DX(pcrf) > 100)?100:
                                       ((SHORT)DX(pcrf ) < 0)?0:
                                                  DX(pcrf));

  VDMData.mstates.ulThreshold = vmScaleSensitivity(DEF_THRESHOLD,
                                (100- VDMData.mstates.scaleThreshold));

  if ( fsstate&SESSION_FGND )
  {
    uxScale = vmScaleSensitivity(VDMData.mstates.nxMicsPer8Pix,
       VDMData.mstates.scaleXMics);

    uyScale = vmScaleSensitivity(VDMData.mstates.nyMicsPer8Pix,
       VDMData.mstates.scaleYMics);

    nThreshold = VDMData.mstates.ulThreshold;
  }
  return  TRUE;
}                                      /* vmInt33SetSensitivity              */


 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmInt33QuerySensitivity()
 *
 *  DESCRIPTION    :Int 33h Query Sensitivity (function 27)
 *
 *  INPUT      :
 *               pcrf -> VDM register frame
 *  OUTPUT     :
 *               EMULATED
 *                   TRUE
 *               NOT EMULATED
 *                   FALSE (pass control to next VDD and/or ROM)
 *  USES       :
 *               32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT    :
 *               VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      set VDM BX to scaleXMics;
 *      set VDM CX to scaleYMics;
 *      set VDM DX to scaleThreshold;
 *
 ***************************************************************************/

BOOL PRIVENTRY vmInt33QuerySensitivity(register PCRF pcrf)
{
  BX(pcrf) = (WORD)VDMData.mstates.scaleXMics;
  CX(pcrf) = (WORD)VDMData.mstates.scaleYMics;
  DX(pcrf) = (WORD)VDMData.mstates.scaleThreshold;
  return  TRUE;
}                                      /* vmInt33QuerySensitivity            */

 /***************************************************************************
 *
 *  FUNCTION NAME  :  vmInt33SetInterruptRate()
 *
 *  DESCRIPTION    : Int 33h Set Interrupt Rate (function 28)
 *
 *  INPUT      :
 *                pcrf -> VDM register frame
 *  OUTPUT     :
 *                EMULATED
 *                    TRUE
 *                NOT EMULATED
 *                    FALSE (pass control to next VDD and/or ROM)
 *  USES       :
 *                32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT    :
 *                 VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      This function is not supported.
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33SetInterruptRate(register PCRF pcrf)
{
  return  TRUE;
}                                      /* vmInt33SetInterruptRate            */
#pragma  END_SWAP_CODE
