/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/

/**************************************************************************
 *
 * SOURCE FILE NAME = VMPTR.C
 *
 * DESCRIPTIVE NAME = Virtual Mouse Device Driver Int 33h Pointer Services
 *
 *
 * VERSION =   V2.0
 *
 * DATE        02/28/91
 *
 * DESCRIPTION  This module contains the VMD's Int 33h pointer services
 *
 *
 * FUNCTIONS
 *
 *    vmInt33ShowPointer()        Int 33h Show Pointer (function 1)
 *    vmShowPointer()             Show pointer
 *    vmInt33HidePointer()        Int 33h Hide Pointer (function 2)
 *    vmHidePointer()             Hide pointer
 *    vmInt33DefineGraphPointer() Int 33h Define Graphics Pointer (function 9)
 *    vmInt33DefineTextPointer()  Int 33h Define Text Pointer (function 10)
 *    vmDefineTextPointer()       Define text pointer
 *    vmInt33DefineVarPointer()   Int 33h Define variable-size Pointer (function 18)
 *    vmInt33SetCRTPage()         Int 33h Set CRT Page (function 29)
 *    vmSetCRTPage()              Set CRT Page
 *    vmInt33QueryCRTPage()       Int 33h Query CRT Page (function 30)
 *    vmResetMasks()
 *    vmUpdateGraphMasks()        Update graphics masks and notify video driver
 *
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/
#include "vmdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

/*
**    External References
*/

extern PVVINFO pvvinfoHead;
extern USHORT ausDefGraphMasks[];
extern USHORT ausDefGraphMasksJ[];                                  


#pragma  BEGIN_SWAP_CODE

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33ShowPointer()
 *
 * DESCRIPTION   :  Int 33h Show Pointer (function 1)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *              VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      reset condition-off rectangle;
 *      if mouse pointer was hidden and is about to be turned on
 *          show mouse pointer;
 *
 ****************************************************************************/

BOOL PRIVENTRY   vmInt33ShowPointer(register PCRF pcrf)
{
 if (VDMData.BIOSMode == BIOS_PS2) {    /* US-DOS */                

    /*
    ** Only do something if the pointer was previously hidden
    */

  if ( VDMData.mstates.fPtrHidden  ||  VDMData.mstates.fOffEnable )
  {
    VDMData.mstates.fOffEnable = FALSE;

    if ( VDMData.mstates.fPtrHidden )
      ++VDMData.mstates.fPtrHidden;

    if ( VDMData.mstates.fPtrHidden == PTR_VISIBLE )
      vmShowPointer(CURRENT_VDM);
  }
 } else {                               /* JAPAN-DOS */             
     if (VDMData.mstates.JDosGraphicsMode &&    /*graph mode */     
         (VDMData.mstates.fPtrHidden)) {                            
         vmShowPointer(CURRENT_VDM);                                
         ++VDMData.mstates.fPtrHidden;                              
     }                                                              
     AX(pcrf) = 0;                                                  
 }                                                                  
  return  TRUE;
}                                      /* vmInt33ShowPointer                 */


/***************************************************************************
 *
 * FUNCTION NAME :  vmShowPointer()
 *
 * DESCRIPTION   :  Show pointer
 *
 *  INPUT    :
 *             hvdm -> VDM
 *  OUTPUT   :
 *             None
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      call virtual video to turn on mouse pointer;
 *
 *************************************************************************/

VOID PRIVENTRY vmShowPointer ( HVDM hvdm )
{
  register PVVINFO pvvinfo = pvvinfoHead;
  register PVDMDATA pvd = pVDMData(hvdm);

  while ( pvvinfo )
  {
    if ( pvvinfo->vvi_vmreg.vmreg_pfnShowPtr )
      if ( (*pvvinfo->vvi_vmreg.vmreg_pfnShowPtr) (hvdm,
                                                   pvd->mstates.xCur,
                                                   pvd->mstates.yCur) )
        break;
    pvvinfo = pvvinfo->vvi_pvvinfoNext;
  }
}                                      /* vmShowPointer                      */

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33HidePointer()
 *
 * DESCRIPTION   :  Int 33h Hide Pointer (function 2)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *              32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *              VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      if mouse pointer is on
 *          hide mouse pointer
 *
 **************************************************************************/

BOOL PRIVENTRY vmInt33HidePointer( register PCRF pcrf )
{
 if (VDMData.BIOSMode == BIOS_PS2) {                                
  --VDMData.mstates.fPtrHidden;

  if ( VDMData.mstates.fPtrHidden == PTR_HIDDEN )
    vmHidePointer(CURRENT_VDM);
 } else {                                                           
    if (VDMData.mstates.fPtrHidden != PTR_HIDDEN)                   
      vmHidePointer(CURRENT_VDM);                                   
    VDMData.mstates.fPtrHidden = PTR_HIDDEN;                        
 }                                                                  
  return  TRUE;
}                                      /* vmInt33HidePointer                 */

/****************************************************************************
 *
 * FUNCTION NAME :  vmHidePointer()
 *
 * DESCRIPTION   :  Hide pointer
 *
 *  INPUT    :
 *              hvdm -> VDM
 *  OUTPUT   :
 *              None
 *  USES     :
 *              32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *              Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      call virtual video to turn off mouse pointer;
 *
 ****************************************************************************/

VOID PRIVENTRY vmHidePointer(HVDM hvdm)
{
  register PVVINFO pvvinfo = pvvinfoHead;
  while ( pvvinfo )
  {
    if (pvvinfo->vvi_vmreg.vmreg_pfnHidePtr)
      if ( ( *pvvinfo->vvi_vmreg.vmreg_pfnHidePtr )(hvdm) )
        break;
    pvvinfo = pvvinfo->vvi_pvvinfoNext;
  }
}                                      /* vmHidePointer                      */

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33DefineGraphPointer()
 *
 * DESCRIPTION   :  Int 33h Define Graphics Pointer (function 9)
 *
 *  INPUT    :
 *              pcrf -> VDM register frame
 *  OUTPUT   :
 *              EMULATED
 *                  TRUE
 *              NOT EMULATED
 *                  FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      save new hot spot coordinates;
 *      update graphics masks;
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33DefineGraphPointer(register PCRF pcrf)
{
  AssertTRUE(AX(pcrf) == INT33_DEFINEGRPOINTER);

  /*
  **  WARNING-- Redefine xHot, yHot,
  */

  VDMData.mstates.xHot = (ULONG)(LONG)(SHORT)BX(pcrf);
  VDMData.mstates.yHot = (ULONG)(LONG)(SHORT)CX(pcrf);
  vmUpdateGraphMasks(CURRENT_VDM, PFROMVADDR(ES(pcrf), DX(pcrf)));


  return ;
}                                      /* vmInt33DefineGraphPointer          */

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33DefineTextPointer()
 *
 * DESCRIPTION   :  Int 33h Define Text Pointer (function 10)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      save cursor select (hardware/software);
 *      save text and mask;
 *      save text or mask;
 *      call vmDefineTextPointer to define text cursor;
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33DefineTextPointer(register PCRF pcrf)
{
  VDMData.mstates.fHWPtr = BX(pcrf);
  VDMData.mstates.ulTextANDMask = CX(pcrf);
  VDMData.mstates.ulTextXORMask = DX(pcrf);
  vmDefineTextPointer(CURRENT_VDM);

/*
*/

  return ;
}                                      /* vmInt33DefineTextPointer           */

/****************************************************************************
 *
 * FUNCTION NAME :  vmDefineTextPointer()
 *
 * DESCRIPTION   :  Define text pointer
 *
 *  INPUT    :
 *              hvdm -> VDM
 *  OUTPUT   :
 *              None
 *  USES     :
 *              32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      call virtual video to define text cursor;
 *
 ****************************************************************************/

VOID PRIVENTRY vmDefineTextPointer(HVDM hvdm)
{
  register PVDMDATA pvd = pVDMData(hvdm);
  register PVVINFO pvvinfo = pvvinfoHead;

  while ( pvvinfo )
  {
    if ( pvvinfo->vvi_vmreg.vmreg_pfnDefTextPtr )
     ( *pvvinfo->vvi_vmreg.vmreg_pfnDefTextPtr )
        ( hvdm,
           pvd->mstates.ulTextANDMask,
           pvd->mstates.ulTextXORMask,
           pvd->mstates.fHWPtr  );

    pvvinfo = pvvinfo->vvi_pvvinfoNext;
  }
}                                      /* vmDefineTextPointer                */

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33DefineVarPointer()
 *
 * DESCRIPTION   :  Int 33h Define variable-size Pointer (function 18)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      Not supported
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33DefineVarPointer(register PCRF pcrf)
{
  /*
  ** This is an undocumented function and probably won't be supported
  */

  return  TRUE;
}                                      /* vmInt33DefineVarPointer            */

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33SetCRTPage()
 *
 * DESCRIPTION   :  Int 33h Set CRT Page (function 29)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      store video page;
 *      call vmSetCRTPage to set video page;
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33SetCRTPage(register PCRF pcrf)
{
  vmSetCRTPage(CURRENT_VDM, (ULONG)BX(pcrf));
  return  TRUE;
}                                      /* vmInt33SetCRTPage                  */

/****************************************************************************
 *
 * FUNCTION NAME :  vmSetCRTPage()
 *
 * DESCRIPTION   :  Set CRT Page
 *
 *  INPUT    :
 *             ulCRTPage = video page number
 *  OUTPUT   :
 *             None
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *            call virtual video to set video page;
 *
 ****************************************************************************/

VOID PRIVENTRY vmSetCRTPage(HVDM hvdm,ULONG ulCRTPage)
{
  register PVDMDATA pvd = pVDMData(hvdm);
  register PVVINFO pvvinfo = pvvinfoHead;

  pvd->mstates.ulVideoPage = ulCRTPage;

  while ( pvvinfo )
  {
    if ( pvvinfo->vvi_vmreg.vmreg_pfnSetVideoPage )
      ( *pvvinfo->vvi_vmreg.vmreg_pfnSetVideoPage )( hvdm, ulCRTPage );
    pvvinfo = pvvinfo->vvi_pvvinfoNext;
  }
}                                      /* vmSetCRTPage                       */

/****************************************************************************
 *
 * FUNCTION NAME :  vmInt33QueryCRTPage()
 *
 * DESCRIPTION   :  Int 33h Query CRT Page (function 30)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33QueryCRTPage(register PCRF pcrf)
{
  BX(pcrf) = (WORD)VDMData.mstates.ulVideoPage;
  return  TRUE;
}                                      /* vmInt33QueryCRTPage                */

/****************************************************************************
 *
 * FUNCTION NAME :  vmResetMasks()
 *
 * DESCRIPTION   :
 *  Update masks and Notify video driver of masksTransfer
 *   graphics masks to instance data
 *
 *  Copies screen (AND) and pointer (XOR) masks to instance data, and
 *  notifies the video driver.  Used by Int 33h Define Graphics
 *  Pointer (function 9), and on video mode resets (vmSetScreenSize) to
 *  reset the masks to defaults.
 *
 *  INPUT    :
 *            hvdm -> VDM
 *            fText == TRUE to reset text masks, FALSE to notify video
 *                        driver only
 *
 *  OUTPUT   :
 *             None
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vmResetMasks(HVDM hvdm,BOOL fText)
{
  register PVDMDATA pvd = pVDMData(hvdm);
  if (fText)
  {
    pvd->mstates.ulTextANDMask = DEF_TEXTANDMASK;
    pvd->mstates.ulTextXORMask = DEF_TEXTXORMASK;
  }

    /*
    ** Notify the virtual video driver of current TEXT masks
    */

  vmDefineTextPointer(hvdm);

    /*
    ** Notify the virtual video driver of the current GRAPHICS masks
    */

  pvd->mstates.xHot = DEF_XHOT;
  pvd->mstates.yHot = DEF_YHOT;
 if (pvd->BIOSMode == BIOS_PS2) {       /* US-DOS */                
  vmUpdateGraphMasks(hvdm, ausDefGraphMasks);
 } else {                               /* JAPAN-DOS */             
  vmUpdateGraphMasks(hvdm, ausDefGraphMasksJ);                      
 }                                                                  
}                                      /* vmResetMasks                       */

/****************************************************************************
 *
 * FUNCTION NAME :  vmUpdateGraphMasks()
 *
 * DESCRIPTION   :  Update graphics masks and notify video driver
 *
 *  Copies screen (AND) and pointer (XOR) masks to instance data, and
 *  notifies the video driver.  Used by Int 33h Define Graphics
 *  Pointer (function 9), and on video mode resets (vmSetScreenSize) to
 *  reset the masks to defaults.
 *
 *  INPUT    :
 *             hvdm -> VDM
 *             ausMasks - source masks (NULL = refresh vvideo with the
 *                                                        current mask)
 *  OUTPUT   :
 *              None
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vmUpdateGraphMasks(HVDM hvdm,PUSHORT ausMasks)
{
  register PVDMDATA pvd = pVDMData(hvdm);
  register PVVINFO pvvinfo = pvvinfoHead;

    /*
    ** Copy the masks
    */

  if ( ausMasks )
         memcpy(pvd->mstates.ausGraphMasks,
             ausMasks,
             sizeof (pvd->mstates.ausGraphMasks));

    /*
    ** Notify the virtual video driver
    */

  while ( pvvinfo )
  {
    if ( pvvinfo->vvi_vmreg.vmreg_pfnDefGraphPtr )
     ( *pvvinfo->vvi_vmreg.vmreg_pfnDefGraphPtr )
        ( hvdm,
          pvd->mstates.ausGraphMasks,
          pvd->mstates.xHot, pvd->mstates.yHot);

    pvvinfo = pvvinfo->vvi_pvvinfoNext;
  }
}                                      /* vmUpdateGraphMasks                 */

/* */                                                                 
                                                                    
BOOL PRIVENTRY vmInt33SetPtrColor(register PCRF pcrf)               
{                                                                   
    vmSetPtrColor(CURRENT_VDM, (ULONG)CX(pcrf));                    
                                                                    
    return TRUE;                                                    
}       /* vmInt33SetPtrColor */                                    
                                                                    
                                                                    
/* */                                                                 
                                                                    
VOID PRIVENTRY vmSetPtrColor(HVDM hvdm, ULONG ulPtrColor)           
{                                                                   
    register PVDMDATA pvd = pVDMData(hvdm);                         
    register PVVINFO pvvinfo = pvvinfoHead;                         
                                                                    
    pvd->mstates.ulPtrColor = ulPtrColor;                           
                                                                    
    while (pvvinfo) {                                               
        if (pvvinfo->vvi_vmreg.vmreg_pfnSetPtrColor)                
          (*pvvinfo->vvi_vmreg.vmreg_pfnSetPtrColor)                
                            (hvdm, ulPtrColor);                     
        pvvinfo = pvvinfo->vvi_pvvinfoNext;                         
    }                                                               
}       /* vmSetPtrColor */                                         

#pragma  END_SWAP_CODE

